
--    Ultrason is small Virtual Composer for MIDI instruments.
--    Copyright (C) 2015 2018 M. De Girardi

--    This program is free software; you can redistribute it and/or modify
--    it under the terms of the GNU General Public License as published by
--    the Free Software Foundation; either version 2 of the License, or
--    (at your option) any later version.

--    This program is distributed in the hope that it will be useful,
--    but WITHOUT ANY WARRANTY; without even the implied warranty of
--    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--    GNU General Public License for more details.

--    You should have received a copy of the GNU General Public License
--    along with this program; if not, write to the Free Software
--    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
-----------------------------------------------------------------
-- Authors     : E. Sens                                       --
-- Date        : Thu Feb  1 18:19:58 CET 2018                  --
-- Version     : 7.0.5                                         --
-- Description : Virtual Composer for MIDI instruments written --
--               with Ada programming language.                --
-- Ultrason version 6 : This version accept multiple files     --
-- to open the main application in autoplay and/or in loop.    --
--                                                             --
-- Dependencies : Ada, Gnat, GtkAda 2016, portmidi PragmARC.   --
-----------------------------------------------------------------
-- Accept -i -v -h -a -l switchs and then multiple filenames.  --
-- Where :
--           "-i" switch is ignore indicator for first device. --
--           "-v" switch is verbose console messages. not used.--
--           "-h" switch is help message printing en exit.     --
--           "-a" switch is auto_play indicator.               --
--           "-l" switch is in_loop indicator for play list.   --
-----------------------------------------------------------------
--                                                             --
--                                                             --
--                                       for GNU/Linux         --
--                                                             --
--                 Ultrason 7.x.x....                 and      --
--                                                             --
--                                      Microsoft Windows      --
--                                                             --
--                                                             --
-----------------------------------------------------------------

with Glib;                              use Glib;

with GNAT.Command_Line;                 use GNAT.Command_Line;

with Libsens.Exceptions;                use Libsens.Exceptions;
with Libsens.Virtual;                   use Libsens.Virtual;
with Libsens.Virtual.Work_Class;        use Libsens.Virtual.Work_Class;
with Libsens.Applications.GM_Interface; use Libsens.Applications.Gm_Interface;

with Gtk.Main;                          use Gtk.Main;

with Text_Io;                           use Text_Io;

with Libsens.Arch;                      use Libsens;

procedure Ultrason is
   
   
   
   Gm_Int    : Gm_Interface_Pointer;
   
   Error     : Gint := 0;
   
   
   Auto_Play : Boolean := False;   
   In_Loop   : Boolean := False;
   
   Verbose   : Boolean := False;
   Help      : Boolean := False;   
   Ignore    : Boolean := False;
   
   Work_Id   : Natural := 0;
   Works     : aliased Work_Array;
begin   
   
   begin
      
      loop
	 case Getopt ("i a l v h") is
	    -- Accept :=
	    -- i ignore device(s).
	    -- v verbose ;
	    -- h help ;
	    -- a auto play ;
	    -- l in loop ;
	    when ASCII.NUL => exit;
	    when 'i' =>
	       if Full_Switch = "i" then
		  Put_Line ("Got ignore");
		  ignore:= True;
	       end if;
	    when 'a' =>
	       if Full_Switch = "a" then
		  Put_Line ("Got auto play");
		  Auto_Play := True;
	       end if;
	    when 'l' =>
	       if Full_Switch = "l" then
		  In_Loop := True;
		  Put_Line ("Got in loop");
	       end if;
	    when 'v' =>		  
	       if Full_Switch = "v" then
		  Verbose := True;
		  Put_Line ("Got verbose");
	       end if;
	    when 'h' => 
	       if Full_Switch = "h" then
		  Help := True;
		  Put_Line ("Got help");
	       end if;
	       -- Put_Line ("Got b + " & Parameter);
	    when others =>
	       raise Program_Error; -- cannot occur
	 end case;
      end loop;
      
      if Help then	       
	 Put_Line("ultrason : usage : ultarson [ -a ] [ -l ] [ -i ] [ -h ] [ -v ] [Filename(s)...]");
	 Arch.Finalize;
	 return;
      end if;	
      
      declare
	 
      begin
	 
	 loop
	    declare
	       S : constant String := Get_Argument (Do_Expansion => True);
	    begin	 
	       exit when S'Length = 0;
	       Put_Line ("Adding work for " & S);
	       Works(Work_Num(Work_Id+1)) := new Work_Record;
	       Works(Work_Num(Work_Id+1)).Initialize(Work_Num(Work_Id+1), S);
	       Work_Id := Work_Id + 1;
	    end;   
	 end loop;
	 
      end;
   exception
      when Invalid_Switch    => Put_Line ("Invalid Switch " & Full_Switch);
	 raise Spec_Error;
      when Invalid_Parameter => Put_Line ("No parameter for " & Full_Switch);
	 raise Spec_Error;
   end;
   
   
   if Work_Id = 0 then
      Works(Work_Num(Work_Id+1)) := new Work_Record;
      Works(Work_Num(Work_Id+1)).Initialize(Work_Num(Work_Id+1), "");      
      Work_Id := Work_Id + 1;     
   end if;
         
   
   
   Put_Line("Program running...");
   
   if not Gtk.Main.Init_Check then
      return;
   end if;
   Put_Line("Gtk Ok !");   
   Initialize(Gm_Int, Error, Works'access, Verbose, Auto_Play, In_Loop, Help, Ignore);
   
   if Error /= 0 then
      
      Put_Line("Gm_interface error =" & Gint'Image(Error));
      
   end if;   
   
   Put_Line("Gm_interface terminated succefully.");
   
exception
   when Spec_Error =>
      Arch.Finalize;
end Ultrason;
