-- Skywalker is another attempt of A. i. written with Ada.
-- Skywalker is Copyright (C) 2024 Manuel De Girardi ; 
--
--   This program is free software; you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation; either version 2 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program; if not, write to the Free Software
--   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
--
-- Date := 2024-12-25 15:37:14 ; 
-- Version := 0.18.7r ; 
with Sky.Shared;
use Sky.Shared;
with Sky.Strings;
use Sky.Strings;
with Sky.Attributes;
use Sky.Attributes;
with Sky.Shell;
with Sky.Tools;
use Sky.Shell;
use Sky.Tools;

with Sky.Versions;
with Sky.Versions.Version_Io;
use Sky.Versions;
use Sky.Versions.Version_Io;

with Sky.Ansi.Console;
use Sky.Ansi.Console;



----------------------------------------------------------
-- Mode none

with Ada.Calendar.Formatting;
use Ada.Calendar;
with Ada.Wide_Characters.Handling;
use Ada.Wide_Characters.Handling;

with Ada.Strings.Fixed;
with Ada.Characters.Handling;
with Gnat.Directory_Operations;





with Gnat.Semaphores;
use Gnat.Semaphores;

with Text_Io;

-----------------------------------------------------------
-- Mode manager

with Ada.Containers.Vectors;
use Ada.Containers;


with Gnos;

with Gnos.Parameters;
use Gnos.Parameters;


with Gnos.Projects;
use Gnos;

with Gnos.Results;

with Gnos.Versions;
with Gnos.Versions.Editting;

use Gnos.Versions.Editting;
with Gnos.Versions.Environment;
with Gnos.Gnose;



use Ada;

with PragmARC.Ansi_Tty_Control;
use PragmARC.Ansi_Tty_Control;
use PragmARC;

with Ada.Strings.Wide_Fixed;
use Ada.Strings;
----------------------------------------------------------
-- Activity
with Sky.Accountant;
use Sky.Accountant;
with Sky.Balance;
use Sky.Balance;
with Sky.Stock;
with Sky.Stocks_Manager;
use Sky.Stock;
use Sky.Stocks_Manager;

----------------------------------------------------------
-- Organizer
with Ada.Tags;
use Ada.Tags;

package body Sky.Walk is
   
   
   Gnose_Params : Gnose_Parameters_Type;
   
   Projects_Params : Projects_Parameters_Type;
   
   package Gnos_Projects is new Gnos.Projects(Projects_Params);

   
   
   package Agnose is new Gnos.Gnose(Gnose_Params);
   use Agnose;
   use Agnose.Neural;
   use Agnose.Neural.Objects_System;
   use Agnose.Neural.Objects_System.Objects_system;
   use Agnose.Neural.Objects_System.Objects_System.Language;
   use Agnose.Neural.Objects_System.Objects_System.Language.Agnose_Projects;
   
   package Environment is new Gnos.Versions.Environment(Agnose.Neural.Objects_System.Objects_System.Language.Agnose_Projects);
   use Environment;
   
   
   
   
   use Gnat.Directory_Operations;
   use Text_Io;      
   
   use Ada.Strings.Fixed;
   use Ada.Characters;
   
   
   
   
   
   
   
   
   use Ansi_Windows;
   
   
   
   function Print_Set (Set : in Wide_String_Set;
		       Into : in Window_Record;
		       Top : in Line_Index_Range;		       
		       Total_Lines: out Natural) return Boolean is
      height : constant Positive := Into.Win.H-2;
      Width  : constant Positive := Into.Win.W-2;
      Line_Count : Natural := 0;      
   begin
      Total_Lines := 0;
      if Set.Line_Last = 0 then
	 return True;
      end if;
      
      
  Print_Loop:
      for I in Top+1 .. Positive'min(Top+Height, Set.Line_Last) loop
	 
	 declare
	    tmp : constant Wide_String := Set.Lines(I).all;
	    Line : constant Wide_String := Tmp(Tmp'First..Wide_Fixed.Index_Non_Blank((Tmp), Backward));
	    
	    Chars_Max : constant Positive := Width-1;
	    Lines_Max : constant Positive := Height;
	    
	    
	    B_Inf : Positive := 1;
	    B_Sup : Positive := Chars_Max;
	    
	    Bot   : Natural := 0;
	 begin
	    
	    if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
	       
	       
	       for Line_Window in 0 .. Integer'min(Real_Length(Line)/Chars_Max, Lines_Max) loop
		  
		  B_Inf := Line_Window*Chars_Max + 1;
		  B_Sup := Integer'min((Line_Window + 1)*Chars_Max, Real_Length(Line));
		  begin
		     
		     if Real_length(Line) <= Chars_max then
			Draw_Text(Into, Line_Count + 1, 1, White, Line);
		     else
			Draw_Text(Into, Line_Count + 1, 1, White, Line(B_Inf..B_Sup));                            
		     end if;                 
		  end;
		  
		  Line_Count := Line_Count + 1;
		  if Line_Count >= height then		     
		     exit Print_Loop;
		  end if;
		  
	       end loop;
	    else
	       
	       Line_Count := Line_Count + 1;
	    end if;                           
	 end;
	 
	 if Line_Count > height then	    
	    exit Print_Loop;
	 end if;	 
      end loop Print_Loop;
      
      Total_Lines := Line_Count;
      return True;
   exception
      when others =>
	 return False;
   end Print_Set;

   
   function Version_Update(Version : in  Sky.Versions.Version_Type) return Sky.Versions.Version_Type is
      The_Version : Sky.Versions.Version_Type := version;

   begin
      if New_Action and (not New_Description) then
	 if The_Version.Revision_Version + 1 < Revision_Version_Type'Last then
   	    The_Version.Revision_Version := The_Version.Revision_Version + 1;
   	 elsif The_Version.Major_Version + 1 < Major_Version_Type'Last then
	    The_Version.Major_Version := The_Version.Major_Version + 1;
	    The_Version.Minor_Version := 0;
	    The_Version.Revision_Version := 0;
	 else
	    raise Impl_Limit;
	 end if;
      elsif New_Description and New_Action then
	 if The_Version.Minor_Version + 1 < Minor_Version_Type'Last then
	    The_Version.Minor_Version := The_Version.Minor_Version + 1;	      
	    The_Version.Revision_Version := 0;		  	    
	 elsif The_Version.Revision_Version + 1 < Revision_Version_Type'Last then
	    The_Version.Revision_Version := The_Version.Revision_Version + 1;
	 elsif The_Version.Major_Version + 1 < Major_Version_Type'Last then
	    The_Version.Major_Version := The_Version.Major_Version + 1;
	    The_Version.Minor_Version := 0;
	    The_Version.Revision_Version := 0;
	 else
	    raise Impl_Limit;
	 end if;
      elsif New_Description then
	 The_Version.Release_Version := 'a';
	 
	 if The_Version.Minor_Version + 1 < Minor_Version_Type'Last then
	    The_Version.Minor_Version := The_Version.Minor_Version + 1;	      
	    The_Version.Revision_Version := 0;		  	    
	 elsif The_Version.Revision_Version + 1 < Revision_Version_Type'Last then
	    The_Version.Revision_Version := The_Version.Revision_Version + 1;
	 elsif The_Version.Major_Version + 1 < Major_Version_Type'Last then
	    The_Version.Major_Version := The_Version.Major_Version + 1;
	    The_Version.Minor_Version := 0;
	    The_Version.Revision_Version := 0;
	 else
	    raise Impl_Limit;
	 end if;
      elsif The_Version.Release_Version = 'a' then
	 The_Version.Release_Version := 'b';
	 if The_Version.Minor_Version + 1 < Minor_Version_Type'Last then
	    The_Version.Minor_Version := The_Version.Minor_Version + 1;	      
	    The_Version.Revision_Version := 0;		  	    
	 elsif The_Version.Revision_Version + 1 < Revision_Version_Type'Last then
	    The_Version.Revision_Version := The_Version.Revision_Version + 1;
	 elsif The_Version.Major_Version + 1 < Major_Version_Type'Last then
	    The_Version.Major_Version := The_Version.Major_Version + 1;
	    The_Version.Minor_Version := 0;
	    The_Version.Revision_Version := 0;
	 else
	    raise Impl_Limit;
	 end if;
      else
	 The_Version.Release_Version := 'r';
	 if The_Version.Revision_Version + 1 < Revision_Version_Type'Last then
	    The_Version.Revision_Version := The_Version.Revision_Version + 1;
	    
	 elsif The_Version.Minor_Version + 1 < Minor_Version_Type'Last then
	    The_Version.Minor_Version := The_Version.Minor_Version + 1;	      
	    The_Version.Revision_Version := 0;		  	    
	    
	 elsif The_Version.Major_Version + 1 < Major_Version_Type'Last then
	    The_Version.Major_Version := The_Version.Major_Version + 1;
	    The_Version.Minor_Version := 0;
	    The_Version.Revision_Version := 0;
	 else
	    raise Impl_Limit;
	 end if;
      end if;
      
      return The_Version;

   end Version_Update;

   
   
   
   
   task body CPU_Process is
      
      Cpu_File : Text_Io.File_Type;
      Cpu_Error : Integer := 0;
      Cpu : Cpu_String := (others => ' ');
      Success  : Boolean := False;
      Busy : Boolean := False;
      
   begin
      
      loop
	 select
	    when not Busy =>
	       accept Initialize do
		  Busy := True;
		  
		  
		  Errno :=  System("sh" & " -c """ & Getenv("HOME").all & "/bin/sky_cpu > " & Getenv("HOME").all & "/sky_usage.txt""" & Character ' Val (0));

		  if Is_Regular_File(Getenv("HOME").all & "/sky_usage.txt") then
		     
		     Cpu := (others => ' ');
		     
		     Text_Io.Open(Cpu_File, Text_Io.In_File, Getenv("HOME").all & "/sky_usage.txt");
		     
		     declare
			Line : constant String := Get_Line(Cpu_File);
			Pos  : Natural := Line'Length + ((Cpu_String'Length-1) - Line'Length) ;
		     begin
			
			for I in reverse Line'Range loop
			   Cpu(Pos) := Line(I);
			   Pos := Pos - 1;
			end loop;
			Cpu(Cpu'Length) := '%';
			
		     end;
		     
		     Text_Io.Close(Cpu_File);
		     
		     
		  
		  end if;
		  Busy := False;
		  
	       end Initialize;
	       
	 or
	    when not Busy =>
	       accept Cpu_Get(CPU_Usage : out CPU_String) do
		  
		 Busy := True;
		 CPU_Usage := Cpu;
	       
	       end Cpu_Get;
	       
	       Delete_File(Home_Dir & "/sky_usage.txt", Success);
	       
	       Busy := False;
	 or
	    accept Halt;
	    exit;
	 end select;
      end loop;
   end CPU_Process;
   
   
   

   
   task body None_Process is
      Lock : Binary_Semaphore(True, Default_Ceiling);
      Text : Wide_String_Access := new Wide_String ' ("");
      Old_Length    : Natural := 0; -- save text'length
      Old_Total_Length    : Natural := 0; -- save total_type'image(total_sum)'length
      Old_Pwd   : String_Access;
      
      Cpu_Lock : Binary_Semaphore(True, Default_Ceiling);
      --Application : App_Access;
      
      
      
      task Screen is
	 entry Refresh;
	 entry Halt;
      end Screen;
      
      task body Screen is
	 Prompt : String_Access;
	 
	 
	 Cpu    : Cpu_String := (others => ' ');
	 
      begin
	 
	 loop
	    
	    select
	       accept Halt;
	       exit;
	    or
	       accept Refresh do
		  Lock.Seize;

		  
		  Cpu_Lock.Seize;
		  Cpu_Process.Initialize;

		  Cpu_Process.Cpu_Get(Cpu);
		  Cpu_Lock.Release;
		  Free(Prompt);
		  Prompt := new String ' (Term_Enum'Image(Mode.App.Term) & ':' & cpu & ": " & Formatting.Image(Clock)(12..Formatting.Image(Clock)'Length) & ":" & Total_Type'Image(Total_Sum)  & "$ ");

		  if Text /= null then		     
		     
		     Put(Character'Val (13) & Prompt.all);
		     Put((Old_Length+Old_Total_Length+Cpu'length) * ' ');
		     Put(Character'Val (13) & Prompt.all);
		     W_Io.Put(Text.all);
		     
		  else
		     Put(Character'Val (13) & Prompt.all);
		     Put((Old_Length+Old_Total_Length+Cpu'length) * ' ');
		     Put(Character'Val (13) & Prompt.all);

		  end if;
		  Old_Total_Length := Total_Type'Image(Total_Sum)'Length;
		  Old_Length := 0;
		  
		  Lock.Release;

	       end Refresh;	      
	       
	    end select;
	 end loop;
      end Screen;
      
      
      task Deamon is
	 entry Halt;
	 entry Lock;
	 entry Unlock;
      end Deamon;
      task body Deamon is
	 End_Of_Task : Boolean := False;
      begin
     Main_Loop :
	 while not End_Of_Task loop
	    loop
	       select
		  accept Halt;
		  End_Of_Task := True;
		  exit Main_Loop;
	       or
		  accept Lock;
	       or
		  accept Unlock;
		  screen.Refresh;
		  exit;
	       end select;
	    end loop;
	    while not End_Of_Task loop
	       select
		  accept Halt;
		  End_Of_Task := True;	    
		  exit Main_Loop;
	       or
		  accept Lock;
		  exit;
	       or
		  accept Unlock;
	       or
		  delay 1.0;
		  screen.Refresh;
	       end select;	    
	    end loop;	    
	 end loop Main_Loop;
      end Deamon;
      
      Args_Index        : Natural := 0;
      Arguments         : Wide_String_Access;
      
      Var_Set           : Attributs(2048);
      Alias_Set         : Attributs(2048);
      Internal_Cmd : Int_Cmd_Enum := None;
      Is_Internal_Cmd : Boolean := False;
      Buffer : Wide_String_Access;
      
      Completion_Result : Wide_Result_Access;      
      History_Set   : Wide_String_Set(2048);
      History_Last  : Natural := 0;
      History_Index : Natural := 0;
      
      Errno : Integer := 0;  
   begin
      Alias_Set.List(1) := Make("ls", "ls --color");
      Alias_Set.Index := 1;
      begin
     Main_Loop:
	 loop
	    loop
	       select
		  accept Halt do

		     Deamon.Halt;

		     Screen.Halt;

		  end Halt;
		  exit Main_Loop;
	       or
		  accept Suspend do
		     Deamon.Lock;
		  end Suspend;
		  exit;
	       or
		  accept Wake_Up(Top : in Natural);
	       or
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(9) =>
			      Lock.Seize;
			      declare
				 Line : constant String_Access := 
				   new String ' (Handling.To_String(Text.all));
				 
				 Path_On_Line : Wide_String_Access;
				 buffer : String_Access := new String ' ("");
			      begin
				 if Completion_Result /= null then
				    U_Array_Free(Completion_Result.Wlines);
				 end if;
				 Completion(Line.all, Path_On_Line, Completion_Result);				    
				 if Path_On_Line /= null then
				    Buffer := new String ' (Handling.To_String(Path_On_Line.all));
				    Wide_Free(Path_On_Line);					  
				    Wide_Free(Text);
				    Text := new Wide_String ' (Handling.To_Wide_String(Buffer.all));
				    Free(Buffer);
				    U_Array_Free(Completion_Result.Wlines);
				    Wide_Result_Free(Completion_Result);
				 else
				    for I in Completion_Result.Wlines'Range loop
				       Put_Line(Delete_Local_Prefix(Handling.To_String((-Completion_Result.Wlines(I)))));
				    end loop;
				 end if;
				 
			      end;
			      Lock.Release;
			   when Wide_Character'Val(10) | Wide_Character'Val(13) =>			
			      if Text /= null and then
				Text'Length > 0 then
				 
				 Lock.Seize;
				 Text_Io.New_Line;
				 Deamon.Lock;
				 
				 if Text(1) = '-' or Text(1) = '+' then
				    if Text'Length > 1 then
				       declare
					  Val : Total_Type := 0.0;
				       begin
					  Val := Total_Type'Value(Handling.To_String(Text.all));
					  Total_Sum := Total_Sum + Val;
					  Wide_Free(Text);
					  Text := new Wide_String ' ("");
					  
				       exception
					  when Constraint_error =>
					     Put_Line("-: error");
				       end;
				    end if;
				    
				 else
				    
				    
				    
				    Internal_Cmd := Internal_Cmd_Value(Handling.To_String(Text.all));
				    Is_Internal_Cmd := True;
				    case Internal_Cmd is
				       when none =>
					  Is_Internal_Cmd := False;
				       when Alias =>
					  
					  Alias(Handling.To_String(Text.all), Alias_Set);
					  
				       when Unalias =>
					  
					  Unalias(Handling.To_String(Text.all), Alias_Set);
					  
				       when Cd =>
					  begin
					     Cpu_Lock.Seize;
					     Change_Directory(Handling.To_String(Text.all), Old_Pwd);
					     Cpu_Lock.Release;
					  exception
					     when Directory_Error =>
						Cpu_Lock.Release;
						Text_Io.Put("Directory not found");
						Text_Io.New_Line;
					  end;
				       when Set =>

					  Set(Handling.To_String(Text.all), Var_Set);
					  
				       when Unset =>

					  Unset(Handling.To_String(Text.all), Var_Set);
					  
				       when Put =>

					  Args_Index := Index(Handling.To_String(Text.all), " ")+1;
					  if Args_Index /= 0 then
					     Arguments := new Wide_String ' (Text(Args_Index..Text'Length));
					     Put(Handling.To_String(Arguments.all), Var_Set);
					  end if;
					  Text_Io.New_Line;
				       when Put_Line =>

					  Args_Index := Index(Handling.To_String(Text.all), " ")+1;
					  if Args_Index /= 0 then
					     Arguments := new Wide_String ' (Text(Args_Index..Text'length));
					     Put_Line(Handling.To_String(Arguments.all), Var_Set);
					  end if;
					  
				    end case;
				    
				    
				    if not Is_Internal_Cmd then
				       
				       --New_Line;
				       declare
					  Alias_Ptr   : String_Access;
					  Buffer      : String_Access;
					  Normal      : String_Access;				 
				       begin

					  for Iter in 1..Alias_Set.Index loop
					     declare
						Alias : constant Attribut_Record := Alias_Set.List(Iter);
						First_Space : Natural := 
						  Index(Handling.To_String(Text.all), " ");
						Non_Blank   : constant Natural :=
						  Index_Non_Blank(Handling.To_String(Text.all), Backward);
					     begin	
						if Non_Blank < First_Space then
						   First_Space := 0;
						end if;
						if First_Space /= 0 then
						   if Names(Alias) =
						     Handling.To_String(Text(Text'First..First_Space - 1)) then
						      Alias_Ptr := new String ' (Images(Alias));
						      exit;
						   end if;
						else
						   if Names(Alias) = Handling.To_String(Text(Text'First..Index_Non_Blank(Handling.To_String(Text.all), Backward))) then
						      Alias_Ptr := new String ' (Images(Alias));
						      exit;
						   end if;
						end if;
					     end;
					  end loop;

					  if Alias_Ptr /= null then
					     declare
						First_Space : constant Natural := 
						  Index(Handling.To_String(Text.all), " ");
					     begin

						if First_Space /= 0 then
						   Buffer := new String ' 
						     (Alias_Ptr.all & ' ' & Handling.To_String(Text(First_Space+1..Text ' Last)));

						else
						   Buffer := new String ' (Alias_Ptr.all);
						end if;
					     end;
					     Wide_Free(Text);

					     Normal := new String ' (Normalize_Quoted_Argument(Buffer.all));
					     Text := new Wide_String ' (Handling.To_Wide_String(Normal.all));
					     
					  end if;
				       end;
				       
				       
				       
				       if Locate_Exec_On_path(Command_Name(Handling.To_String(Text.all))) /= null then
					  
					  
					  if Command_Name(Handling.To_String(Text.all))'Length = Text'Length then
					     
					     declare
						
						Args : constant Argument_List_Access := Argument_String_To_List
						  (
						   "-c " &
						     Locate_Exec_On_path(Command_Name(Handling.To_String(Text.all))).all);
					     begin
						if Locate_Exec_On_path(Command_Name(Handling.To_String(Text.all))).all'Length /= 0 then
						   Errno := Spawn("/usr/bin/sh", Args.all);
						end if;
					     end;
					     
					  else
					     
					     declare
						
						Args : constant Argument_List_Access := new Argument_List ' (new String ' ("-c"), 
													     new String ' (Locate_Exec_On_path(Command_Name(Handling.To_String(Text.all))).all & " " & 
															     Handling.To_String(Text.all)(Command_Name(Handling.To_String(Text.all))'Length+2..Text'Last)
															  )
													    );
						
					     begin
						Normalize_Arguments(Args.all);
						Errno := Spawn("/usr/bin/sh", Args.all);
					     end;
					     
					     
					  end if;
				       else
					  New_Line;
					  Put_Line("Command not found");
				       end if;
				    end if;
				    if Errno /= 0 then	    
				       Put_Line("Error: errno = " & Integer'Image(Errno));
				    end if;
				    Old_Length := Text'Length;
				    Add_Line(History_Set, History_Last, Text.all);
				    History_Index := 0;
				    Wide_Free(Text);
				    Text := new Wide_String ' ("");			   
				 end if;
				 Deamon.Unlock;
				 Lock.Release;
			      end if;
			   when Wide_Character'Val(12) =>
			      Put(Ansi_Tty_Control.Clear_Screen);
			      --Errno := System("clear" & Character'Val(0));
			      Screen.Refresh;
			   when Wide_Character'Val(127) =>
			      Old_Length := Text'Length;
			      Buffer := new Wide_String ' (Text.all(Text'First..Text'Last-1));
			      Wide_Free(Text);
			      Text := new Wide_String ' (Buffer.all);
			      Wide_Free(Buffer);			
			      Screen.Refresh;
			   when others =>
			      null;
			end case;
		     else
			Lock.Seize;
			Buffer := new Wide_String ' (Text.all & Wchar);
			Wide_Free(Text);
			Text := new Wide_String ' (buffer.all);
			Wide_Free(Buffer);
			Old_Length := Text'Length;
			Lock.Release;
		     end if;
		     
		  end Receive;
		  Screen.Refresh;
	       or
		  accept Refresh;
		  Screen.Refresh;
		  
	       or
		  accept Page_Down;
	       or
		  accept Page_Up;
	       or
		  accept Up_Arrow do
		     
		     if (History_Last > 0) and then
		       (History_Last - History_Index) > 0 then		  
			Old_Length := Text'Length;

			Screen.Refresh;

			Wide_Free(Text);
			Text := new Wide_String ' (History_Set.Lines(History_Last - (History_Index)).all);
			History_Index := History_Index + 1;		  			

			Screen.Refresh;

		     else
			Put(Character'Val(7));
		     end if;
		  end up_arrow;
	       or
		  accept Down_Arrow do
		     
		     if (History_Index > 1) then		  
			Old_Length := Text'Length;
			Wide_Free(Text);
			
			Text := new Wide_String ' (History_Set.Lines(History_Last - (History_Index - 2)).all);
			
			History_Index := History_Index - 1;
			Screen.Refresh;
		     elsif History_Index = 1 then
			Old_Length := Text'Length;
			Wide_Free(Text);		  
			Text := new Wide_String ' ("");
			
			History_Index := History_Index - 1;
			Screen.Refresh;
		     else
			Put(Character'Val(7));
		     end if;
		  end Down_Arrow;
	       or
		  accept Left_Arrow;
	       or
		  accept Right_Arrow;
	       or
		  accept Full_Screen;
	       or
		  accept Del;
	       or
		  accept Begin_Of;
	       or
		  accept End_Of;
	       or
		  accept Overwrite;
	       or
		  accept Screen_Print;		  
	       end select;
	    end loop;
	    loop
	       select
		  
		  accept Halt do
		     Deamon.Halt;
		     Screen.Halt;
		  end Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
	       or
		  accept Wake_Up(Top : in Natural) do
		     
		     Text_Io.Put_Line("Greetings " & Getenv("USER").all & "!");
		     Deamon.Unlock;
		     delay 0.5;
		  end Wake_Up;
		  exit;
	       end select;
	    end loop;
	 end loop Main_Loop;
      end;
      --W_Io.Put_Line("None process halted");
   end None_Process;
   
   
   procedure Print_Logs (Logs : in Log_Array; Win : in Window_Record; Top : in Natural) is
      Total : Rate_Type := 0.0;
   begin
      
      Enlight_Window(Win);
      Text_Io.Put(Normal_Mode);
      for Log in Log_Enum'range loop
	 Draw_Text(Win, Log_Enum'Pos(Log) + 3, 2, Yellow, Handling.To_Wide_String(Log_Enum'Image(Log)));
	 Draw_Text(Win, Log_Enum'Pos(Log) + 3, 15, white, " : " & Handling.To_Wide_String(Rate_Type'Image(Logs(Log).Rate)));
	 Total := Total + Logs(Log).Rate;
      end loop;
      Draw_Text(Win, Log_Enum'Pos(Log_Enum'Last) + 4, 2, white, Handling.To_Wide_String((22 * '-')));
      Draw_Text(Win, Log_Enum'Pos(Log_Enum'Last) + 5, 2, white, "Total ");
      Draw_Text(Win, Log_Enum'Pos(Log_Enum'Last) + 5, 15, green, Handling.To_Wide_String(Rate_Type'Image(Total)));
   end Print_Logs;
   
   
   
   Window : Window_Type;
   
   task body Log_Process is
      
      
      Log_Win : Window_Record := (1, 1, App.Lines, App.Columns/2, Blue, Window);
      
      
      
   begin
      
  Main_Loop:
      Loop
	 loop
	    select
	       accept Receive (Wchar : in Wide_Character);
	    or
	       accept Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  Initialize(Log_Win, 1, Top+1, App.Lines, App.Columns/2, Blue, White);
		  Print_Logs(App.Logs, Log_Win, Top);
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	       or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  null;
	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  null;
	       end Down_Arrow;
	       
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt do
		  null;
	       end Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  Initialize(Log_Win, 1, Top+1, App.Lines, App.Columns/2, Blue, White);
		  Print_Logs(App.Logs, Log_Win, Top);

		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_loop;
      --W_Io.Put_Line("log process halted");
   end Log_Process;
   
   
   -------------------------------------------------------------------------------------------------------------
   --
   
   

   
   
   -------------------------------------------------------------------------------------------------------------
   --                                                                                                         --
   --                                       Money Box                                                         --
   --                                                                                                         --
   -------------------------------------------------------------------------------------------------------------
   
   type Value_Enum is (None_Value, Name_Value, Price_Ht_Value, Quantity_Value, Tva_Rate_Value,
		       Total_Ht_Value, Total_Tva_Value, Total_Ttc_Value, Delete_value);
   
   type Value_Type is
      record
	 Name : Value_Enum := None_Value;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Value_Array is array (Value_Enum) of Value_Type;
   
   type Line_Cursor_Type is
      record
	 Values  : Value_Array;
	 Val_Cur : Value_Enum := name_Value;	 
      end record;
   
   function Initialize return Line_Cursor_Type is
      Curs : Line_Cursor_Type;
   begin
      Curs.Values(Name_value) :=
	(Name_Value,
	 (others => ' '),
	 0,
	 3,
	 13
	);
      Curs.Values(Price_Ht_value) :=
	(Price_Ht_Value,
	 (others => ' '),
	 0,
	 5,
	 13
	);
      Curs.Values(Quantity_value) :=
	(Quantity_Value,
	 (others => ' '),
	 0,
	 7,
	 13
	);
      Curs.Values(TVA_Rate_value) :=
	(TVA_Rate_Value,
	 (others => ' '),
	 0,
	 9,
	 13
	);
      Curs.Values(Total_Ht_value) :=
	(Total_Ht_Value,
	 (others => ' '),
	 0,
	 11,
	 13
	);
      Curs.Values(Total_Tva_value) :=
	(Total_Tva_Value,
	 (others => ' '),
	 0,
	 13,
	 13
	);
      Curs.Values(Total_ttc_value) :=
	(Total_Ttc_Value,
	 (others => ' '),
	 0,
	 15,
	 13
	);
      Curs.Val_Cur := Name_Value;
      return Curs;
   end Initialize;
   
   
   function To_Account_Line (Curs : in Line_Cursor_Type;Account_Num : in Account_Num_Range;Line_Num : in Account_Line_Range) return Account_Line_Type is
      Line : Account_Line_Type := Null_Line;
   begin
      Line.Line_Num := Line_Num;
      Line.Description(1..Curs.Values(Name_Value).Line_Last) := 
	Curs.Values(Name_Value).Line(1..Curs.Values(Name_Value).Line_Last);

      Line.Account_num := Account_Num;
      
      Line.Unit_Ht_Price := 
	Fixed_Unit_Ht'Value(Handling.To_String(Curs.Values(Price_Ht_Value).Line(1..Curs.Values(Price_Ht_Value).Line_Last)));
      
      Line.Tva_Rate := 
	Fixed_Factor'Value(Handling.To_String(Curs.Values(Tva_Rate_Value).Line(1..Curs.Values(Tva_Rate_Value).Line_Last)));
      
      Line.Quantity := 
	Quantity_Range'Value(Handling.To_String(Curs.Values(Quantity_Value).Line(1..Curs.Values(Quantity_Value).Line_Last)));
      
      
      Line.Total_Ht := Fixed_Total(Line.Unit_Ht_Price * Fixed_Unit_Ht(Line.Quantity));
      
      Line.Total_Tva := Fixed_total(Line.Total_Ht / Fixed_Total(100.0)) * Fixed_Total(Line.Tva_Rate);
      
      Line.Total_Ttc := Line.Total_Ht + Line.Total_Tva;

      return Line;
   exception
      when others =>
	 raise Spec_Error;
   end To_Account_Line;
   
   
   function To_String (Line : in Account_Line_Type) return Wide_String is
   begin
      return 
	Handling.To_Wide_String
	(
	 Natural'Image(Line.Line_Num) & Character'Val(9) &
	   " " & Handling.To_String(Line.Description(1..16)) 
	   & "  " & Quantity_Range'Image(Line.Quantity) & Character'Val(9) &
	   Fixed_Factor'Image(Line.Tva_Rate) & Character'Val(9) &
	   Fixed_total'Image(Line.total_ht) & Character'Val(9) &
	   Fixed_total'Image(Line.total_TVA) & Character'Val(9) &
	   Fixed_total'Image(Line.total_TTC)
	);
   end To_String;
   
   
   
   
   
   
   procedure Initialize(Wins : out Ticket_Windows_Type;
			Space_Left : in Line_Range;
			Width      : in Column_Range;
			L       : in Line_Range;
			C        : in Column_Range;
			Top       : in Natural := 0) is
   begin
      
      
      Wins.Line_Num_Win := (L+2, C, Space_Left-2, 8,  
			    Cyan, Window);
      Wins.Description_Win := (L+2, C+8, Space_Left-2, 18,
      			       Cyan, Window);
      Wins.unit_Ht_Win := (L+2, C+26, Space_Left-2, 12,  
			   Cyan, Window);
      Wins.Quantity_Win := (L+2, C+38, Space_Left-2, 14,  
			    Cyan, Window);
      Wins.Total_Ht_Win := (L+2, C+52, Space_Left-2, 16,  
      			    Cyan, Window);
      Wins.TVA_Win := (L+2, C+68, Space_Left-2, 10,  
		       Cyan, Window);
      Wins.Total_TVA_Win := (L+2, C+78, Space_Left-2, 16,  
      			     Cyan, Window);
      Wins.Total_TTC_Win := (L+2, C+94, Space_Left-2, 16,  
      			     Cyan, Window);
      
      Initialize(Wins.Line_Num_Win, L+2, Top+C, Space_Left-2, 8,  Cyan, Red);
      Initialize(Wins.Description_Win, L+2, Top+C+8, Space_Left-2, 18,  Cyan, Red);
      Initialize(Wins.unit_Ht_Win, L+2, Top+C+26, Space_Left-2, 12, Cyan, Red);
      Initialize(Wins.quantity_Win, L+2, Top+C+38, Space_Left-2, 14,  Cyan, Red);
      Initialize(Wins.Total_Ht_Win, L+2, Top+C+52, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.TVA_Win, L+2, Top+C+68, Space_Left-2, 10, Cyan, Red);
      Initialize(Wins.Total_TVA_Win, L+2, Top+C+78, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.Total_TTC_Win, L+2, Top+C+94, Space_Left-2, 16, Cyan, Red);
      
      null;
   end Initialize;

   
   procedure Ticket_Wins_Draw(Wins : Ticket_Windows_Type) is
      
   begin
      Draw_Window(Wins.Line_Num_Win);
      Draw_Window(Wins.Description_Win);
      Draw_Window(Wins.Unit_Ht_Win);
      Draw_Window(Wins.Quantity_Win);
      Draw_Window(Wins.Total_Ht_Win);
      Draw_Window(Wins.TVA_Win);
      Draw_Window(Wins.Total_TVA_Win);
      Draw_Window(Wins.Total_Ttc_Win);
   end Ticket_Wins_Draw;      
   
   
   procedure Ticket_Line_Draw(Wins : Ticket_Windows_Type; Line : in Line_Range; Account_Line : in Account_Line_Type) is
      
   begin
      Draw_Text(Wins.Line_Num_Win, Line, 1, White, Handling.To_Wide_String(Voice_Num_Range'Image(Account_Line.Line_Num)));
      Draw_Text(Wins.Description_Win, Line, 1, White, Account_Line.Description(1..16));
      Draw_Text(Wins.unit_Ht_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Unit_ht'Image(Account_Line.Unit_Ht_price)));
      Draw_Text(Wins.Quantity_win, Line, 1, White, Handling.To_Wide_String(Quantity_Range'Image(Account_Line.Quantity)));
      Draw_Text(Wins.Total_Ht_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Account_Line.Total_Ht)));
      Draw_Text(Wins.TVA_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Factor'Image(Account_Line.TVA_Rate)));
      Draw_Text(Wins.Total_TVA_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Account_Line.Total_TVA)));
      Draw_Text(Wins.Total_Ttc_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Account_Line.Total_TTC)));
   end ticket_Line_Draw;      
   
   
   procedure Voice_Print (Voice : in Abstracted_Voice'Class;
			  Win : in Ticket_Windows_Type;
			  Line_Index : in Account_Line_Range;
			  Line_Max : in Positive) is
      Line_Count : natural := 0;
   begin
      if Voice.Line_Last = 0 then
	 return;
      end if;
      for Line in Line_Index+1..Natural'Min(Line_Index+Line_Max, Voice.Line_Last) loop
	 Ticket_Line_Draw(Win, Line_Count+1, Voice.Accounting(Line));
	 Line_Count := Line_Count + 1;
	 exit when Line_Count = Line_Max;
      end loop;
   end Voice_Print;


   
   task body MoneyBox_Process is
      Removed_Line_Num : Wide_String(1..4) := (others => Wide_Character ' Val (32));
      Line_Num  : Account_Line_Range;
      Line_Num_Last : Natural := 0;
      Remove_Case : Boolean := False;
      
      Account_Line_Index : Account_Line_Range := 0;
      
      Text : constant Wide_String_Access := new Wide_String ' ("");
      
      Line_Curs : Line_Cursor_Type := Initialize;
      
      Deal_Name : Deal_Enum := Deal_In;
      
      Outvoice : Outvoice_Record := Null_Outvoice;
      invoice : Invoice_Record := Null_Invoice;
      Payed   : Boolean := False;
      
      
      Validated : Boolean := False;
      
      Top : Natural := 0;
      
      Box_Dealin  : Window_Record := (1, 1, 17, App.Columns/2, Red, Window);
      Box_Dealout : Window_Record := (1, 1, 17, App.Columns/2, Green, Window);
      Voice       : Window_Record := (18, 1, App.Lines-28, App.Columns/2, Blue, Window);
      Ticket      : Window_Record := (App.Lines-10, 1, 5, App.Columns/2, white, Window);
      Balance     : Window_Record := (App.Lines-5, 1, 3, App.Columns/2, Yellow, Window);
      Cmd_Win     : Window_Record := (App.Lines-2, 1, 3, App.Columns/2, White, Window);
      
   begin
      Initialize(Box_Dealin, 1, Top+1, 17, App.Columns/2, Red, White);
      Initialize(Box_Dealout, 1, Top+1, 17, App.Columns/2, Green, White);
      Initialize(Voice, 18, Top+1, App.Lines-28, App.Columns/2, Blue, White);
      Initialize(Ticket, App.Lines-10, Top+1, 5, App.Columns/2, white, Blue);
      Initialize(Balance, App.Lines-5, Top+1, 3, App.Columns/2, Yellow, Blue);
      Initialize(Cmd_Win, App.Lines-2, Top+1, 3, App.Columns/2, White, Red);
      
      Initialize(App.Ticket_Wins, App.Lines-30, Top+App.Columns/2-2, 19, 2, 0);
      begin
     Main_Loop:
	 loop
	    loop
	       select
		  
		  accept Halt do
		     null;
		  end Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
	       or
		  accept Wake_Up (Top : in Natural) do
		     Moneybox_Process.Top := Top;
		     Initialize(Box_Dealin, 1, Top+1, 17, App.Columns/2, Red, White);
		     Initialize(Box_Dealout, 1, Top+1, 17, App.Columns/2, Green, White);
		     Initialize(Voice, 18, Top+1, App.Lines-28, App.Columns/2, Blue, White);
		     Initialize(Ticket, App.Lines-10, Top+1, 5, App.Columns/2, white, Blue);
		     Initialize(Balance, App.Lines-5, Top+1, 3, App.Columns/2, Yellow, Blue);
		     Initialize(Cmd_Win, App.Lines-2, Top+1, 3, App.Columns/2, White, Red);
      
		     Initialize(App.Ticket_Wins, App.Lines-30, Top+App.Columns/2-2, 19, 2, Top);
		     -----------------------------------------------
		     --Errno := System("clear" & Character'Val(0));
		     -----------------------------------------------
		     if Deal_Name = Deal_Out then
			Draw_Window(Box_Dealout);
			Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'image(Outvoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));
			
		     else
			Draw_Window(Box_Dealin);
			Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
			
		     end if;
		     Draw_Window(Voice);
		     Draw_Window(Balance);
		     Draw_Text(Balance, 1, 1, White, 
			       
			       " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				 ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				 ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				 ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
			      );
		     
		     Draw_Window(Cmd_Win);
		     Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");
		     Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
		     
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     

		     
		     
		     if Deal_Name = Deal_Out then
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			Draw_Text(Box_Dealout , Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     else

			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			Draw_Text(Box_Dealin, Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     end if;
		     
		  end Wake_Up;
		  
		  
		  exit;
	       or
		  
		  accept Refresh do
		     if Deal_Name = Deal_Out then
			Draw_Window(Box_Dealout);
			Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));
		     else
			Draw_Window(Box_Dealin);
			Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
		     end if;
		     Draw_Window(Voice);
		     
		     
		     
		     Draw_Window(Balance);
		     
		     
		     
		     Draw_Text(Balance, 1, 1, White, 
			       
			       " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				 ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				 ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				 ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
			      );
		     Draw_Window(Cmd_Win);
		     Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");

		     Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
		     
		     
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     
		     if Deal_Name = Deal_Out then
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealout , Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     else
			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			
			Draw_Text(Box_Dealin, Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White, Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     end if;
		     
		  end Refresh;

	       end select;
	    end loop;

	    loop
	       select
		  accept Halt;
		  exit Main_Loop;
	       or
		  accept Suspend do
		     null;
		  end Suspend;
		  exit;
	       or
		  accept Wake_Up(Top : in Natural) do
		     Moneybox_Process.Top := Top;
		     Initialize(Box_Dealin, 1, Top+1, 17, App.Columns/2, Red, White);
		     Initialize(Box_Dealout, 1, Top+1, 17, App.Columns/2, Green, White);
		     Initialize(Voice, 18, Top+1, App.Lines-28, App.Columns/2, Blue, White);
		     Initialize(Ticket, App.Lines-10, Top+1, 5, App.Columns/2, white, Blue);
		     Initialize(Balance, App.Lines-5, Top+1, 3, App.Columns/2, Yellow, Blue);
		     Initialize(Cmd_Win, App.Lines-2, Top+1, 3, App.Columns/2, White, Red);
      
		     Initialize(App.Ticket_Wins, App.Lines-30, Top+App.Columns/2-2, 19, 2, Top);
		     
		     -----------------------------------------------
		     --Errno := System("clear" & Character'Val(0));
		     -----------------------------------------------
		     if Deal_Name = Deal_Out then
			Draw_Window(Box_Dealout);
			Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));

		     else
			Draw_Window(Box_Dealin);
			Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			Draw_Window(Ticket);
			Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
		     end if;
		     Draw_Window(Voice);
		     Draw_Window(Balance);
		     Draw_Text(Balance, 1, 1, White, 
			       
			       " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				 ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				 ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				 ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
			      );
		     Draw_Window(Cmd_Win);
		     Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");

		     Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
		     
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     
		     if Deal_Name = Deal_Out then
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealout , Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     else
			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealin, Line_Curs.values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.values(Line_Curs.Val_Cur).Line(1..Line_Curs.values(Line_Curs.Val_Cur).Line_Last));
		     end if;

		  end Wake_Up;
		  
	       or when Remove_Case =>
		  accept Receive(Wchar : in Wide_Character) do
		     if Is_Control(Wchar) then
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Line_Curs := Initialize;
			      Invoice := Null_Invoice;
			      Outvoice := Null_Outvoice;
			   when Wide_Character'Val(9) =>
			      if (Text = null) or else
				Text'Length = 0 then
				 if Deal_Name = Deal_Out then
				    Deal_Name := Deal_In;
				 else
				    Deal_Name := Deal_out;
				 end if;
				 
			      end if;
			   when Wide_Character'Val(10) =>
			      if Line_Num_Last > 0 then
				 begin
				    Line_Num := 
				      Account_Line_Range'Value
				      (
				       Handling.To_String(Removed_Line_Num(1..Line_Num_Last))
				      );
				 exception
				    when Constraint_Error =>
				       null;
				 end;
			      end if;
			      if Line_Num /= 0 then
				 if Deal_Name = Deal_Out then
				    Line_Remove(Abstracted_Voice(Outvoice), Line_Num);
				 else
				    Line_Remove(Abstracted_Voice(Invoice), Line_Num);
				 end if;
				 
			      end if;				 
			      Line_Num := 0;
			      Removed_Line_Num := (others => Wide_Character'Val(32));
			      Line_Num_Last := 0;
			      Remove_Case := False;
			   when Wide_Character'Val(127) =>
			      if Line_Num_Last > 0 then
				 Removed_Line_Num(Line_Num_Last) := ' ';
				 Line_Num_Last := Line_Num_Last - 1;
			      end if;
			   when others =>
			      null;
			end case;
			if Deal_Name = Deal_Out then
			   Draw_Window(Box_Dealout);
			   Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			   Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			   Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			   Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			   Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			   Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			   Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			   Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			   Draw_Window(Ticket);
			   Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_ht)));
			   Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			   Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));
			else
			   Draw_Window(Box_Dealin);
			   Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			   Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			   Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			   Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			   Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			   Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			   Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			   Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			   Draw_Window(Ticket);
			   Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			   Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			   Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
			end if;
			Draw_Window(Voice);
			Draw_Window(Balance);
			Draw_Window(Cmd_Win);
			Draw_Text(Balance, 1, 1, White, 
				  
				  " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				    ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				    ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				    ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
				 );
			--Draw_Window(Cmd_Win);
			Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");

			Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
			Ticket_Wins_Draw(App.Ticket_Wins);
			if Deal_Name = Deal_Out then
			   for Val in Name_Value..Total_Ttc_Value loop
			      Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
					Line_Curs.values(Val).Column_Pos,
					White,
					Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			   end loop;
			   Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			   
			   Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				     Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				     White,
				     Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			else
			   
			   for Val in Name_Value..Total_Ttc_Value loop
			      Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
					Line_Curs.values(Val).Column_Pos,
					White,
					Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			   end loop;
			   Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			   Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				     Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				     White,
				     Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			end if;

			
			
			
		     elsif Is_digit(Wchar) and then
		       Line_Num_Last < Removed_Line_Num'Length then
			Removed_Line_Num(Line_Num_Last + 1) := Wchar;
			Line_Num_Last := Line_Num_Last + 1;
			Draw_Window(Cmd_Win);
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : " & Removed_Line_Num(1..Line_Num_Last));
		     else
			Put(Character'Val(7));
		     end if;
		     
		     
		     
		  end Receive;
	       or
		  when not Remove_Case and App.Initialized =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(1) =>
				 Line_Curs := Initialize;
				 Invoice := Null_Invoice;
				 Outvoice := Null_Outvoice;
			      when Wide_Character'Val(9) =>
				 if (Text = null) or else
				   Text'Length = 0 then
				    if Deal_Name = Deal_Out then
				       Deal_Name := Deal_In;
				    else
				       Deal_Name := Deal_out;
				    end if;
				    
				 end if;
			      when Wide_Character'Val(10) =>				 
				 if Line_Curs.Values(Line_Curs.Val_Cur).Line_Last /= 0 then
				    case Line_Curs.Val_Cur is
				       when None_Value =>
					  null;
				       when TVA_Rate_Value =>
					  
					  if Deal_Name = Deal_Out then
					     begin
						Line_Add(Abstracted_Voice(Outvoice), To_Account_Line(Line_Curs, 707, Outvoice.Line_count + 1));
						Outvoice.Line_Count := Outvoice.Line_Count + 1;
						if Outvoice.Line_Last >= (App.Lines-34) then
						   Account_Line_Index := Account_Line_Index + 1;
						end if;
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);
						   --Errno := System("clear" & Character'Val(0));
						   Put_Line("Outvoice error");
						   declare
						      Char : Character;
						   begin
						      Text_Io.Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;
					     end;
					  else
					     begin
						Line_Add(Abstracted_Voice(Invoice), To_Account_Line(Line_Curs, 607, Invoice.Line_Count + 1));
						Invoice.Line_Count := Invoice.Line_Count + 1;
						if Invoice.Line_Last >= (App.Lines-34) then
						   Account_Line_Index := Account_Line_Index + 1;
						end if;
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);
						   --Errno := System("clear" & Character'Val(0));
						   Put_Line("Invoice error");
						   declare
						      Char : Character;
						   begin
						      Text_Io.Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;
					     end;
					  end if;
					  Line_Curs := Initialize;				    
				       when others =>
					  Line_Curs.Val_Cur := Value_Enum'Succ(Line_Curs.Val_Cur);
				    end case;
				    Validated := True;
				 else
				    Put(Character'Val(7));
				 end if;
			      when Wide_Character ' Val (127) =>
				 if Line_Curs.Values(Line_Curs.Val_Cur).Line_Last > 0 then 
				    Line_Curs.Values(Line_Curs.Val_Cur).Line(Line_Curs.Values(Line_Curs.Val_Cur).Line_Last) := ' ';
				    Line_Curs.Values(Line_Curs.Val_Cur).Line_Last := 
				      Line_Curs.Values(Line_Curs.Val_Cur).Line_Last - 1;
				 end if;
			      when Wide_Character'Val(18) =>
				 Remove_Case := True;
				 
			      when Wide_Character'Val(20) => -- Ctrl+T : charge and billed.
				 if Deal_Name = Deal_Out then

				    Charge(Sky.Balance.Balance, Outvoice, Payed);

				    if Payed then

				       Outvoice.Date := Clock;

				       Outvoice.Account_Num := Outvoice.Accounting(1).Account_Num;

				       Outvoice.voice_Num := Sky.Balance.Outvoice_Num + 1;

				       Sky.Balance.Outvoice_Num := Sky.Balance.Outvoice_Num + 1;

				       if Save(Outvoice, "charged.txt") then
					  
					  Add(Accounts, Outvoice);

					  for Iter in 1..Outvoice.Line_Last loop
					     
					     declare
						Line : constant Account_Line_Type := Outvoice.Accounting(Iter);
					     begin
						if not Stocks_Manager.In_Stocks(Stocks_Manager.Hardware, Line.Description) then
						   
						   raise Spec_Error;
						else
						   declare
						      Stock : constant Stock_Record := (0, Line.Description, Line.Quantity, Line.Unit_Ht_Price, Line.TVA_Rate);
						   begin
						      
						      Stocks_Manager.Remove_From_stock(Stocks_Manager.Hardware, Stock.Name, Stock.Quantity);
						      
						   end;
						end if;
						
					     end;
					  end loop;
					  Outvoice := Null_Outvoice;
				       end if;
				    end if;
				 else

				    billed(Sky.Balance.Balance, invoice, Payed);

				    if Payed then

				       Invoice.Date := Clock;

				       Invoice.Account_Num := Invoice.Accounting(1).Account_Num;

				       Invoice.voice_Num := Sky.Balance.Invoice_Num + 1;

				       Sky.Balance.Invoice_Num := Sky.Balance.Invoice_Num + 1;

				       if Save(Invoice, "billed.txt") then

					  Add(Accounts, Invoice);

					  for Iter in 1..Invoice.Line_Last loop

					     declare
						Line : constant Account_Line_Type := Invoice.Accounting(Iter);
					     begin

						if not Stocks_Manager.In_Stocks(Stocks_Manager.Hardware, Line.Description) then

						   declare
						      Stock : constant Stock_Record := (Stocks_Manager.Hard_Num + 1, Line.Description, Line.Quantity, Line.Unit_Ht_Price, Line.TVA_Rate);
						   begin

						      Stocks_Manager.Hard_Num := 
							Stocks_Manager.Hard_Num + 1;
						      
						      Stocks_Manager.Stock_Add(Stocks_Manager.Hardware, Stock);

						   end;
						else

						   declare
						      Stock : constant Stock_Record := (0, Line.Description, (Line.Quantity), Line.Unit_Ht_Price, Line.TVA_Rate);
						   begin

						      Stocks_Manager.Update_Stock(Stocks_Manager.Hardware, Stock);

						   end;
						end if;
						
					     end;
					  end loop;
					  
					  Invoice := Null_Invoice;

				       end if;
				    end if;

				 end if;

			      when others =>
				 null;
			   end case;
			   if Deal_Name = Deal_Out then
			      Draw_Window(Box_Dealout);
			      Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			      Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			      Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			      Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			      Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			      Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			      Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			      Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			      Draw_Window(Ticket);
			      Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_ht)));
			      Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			      Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));
			   else
			      Draw_Window(Box_Dealin);
			      Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			      Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			      Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			      Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			      Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			      Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			      Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			      Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			      Draw_Window(Ticket);
			      Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			      Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			      Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
			   end if;
			   Draw_Window(Voice);
			   Draw_Window(Balance);
			   Draw_Text(Balance, 1, 1, White, 
				     
				     " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				       ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				       ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				       ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
				    );
			   --Draw_Window(Cmd_Win);
			   Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");
			   
			   Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
			   Ticket_Wins_Draw(App.Ticket_Wins);
			   if Deal_Name = Deal_Out then
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;
			      Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			      Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   else
			      
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;
			      Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			      
			      Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   end if;
			   if Remove_Case then
			      Draw_Window(Cmd_Win);
			      Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
			   end if;

			   
			elsif Line_Curs.Values(Line_Curs.Val_Cur).Line_Last < 32 then
			   
			   Line_Curs.Values(Line_Curs.Val_Cur).Line(Line_Curs.Values(Line_Curs.Val_Cur).Line_Last + 1) := Wchar;
			   Line_Curs.Values(Line_Curs.Val_Cur).Line_Last := 
			     Line_Curs.Values(Line_Curs.Val_Cur).Line_Last + 1;
			   if Deal_Name = Deal_Out then
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   else
			      
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   end if;
			   
			else
			   Put(Character'Val(7));
			end if;
		     end Receive;
	       or
		  when (not Remove_Case) and (not App.Initialized) =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(1) =>
				 Line_Curs := Initialize;
				 Invoice := Null_Invoice;
				 Outvoice := Null_Outvoice;
			      when Wide_Character'Val(9) =>
				 if (Text = null) or else
				   Text'Length = 0 then
				    if Deal_Name = Deal_Out then
				       Deal_Name := Deal_In;
				    else
				       Deal_Name := Deal_out;
				    end if;
				    
				 end if;
			      when Wide_Character'Val(10) =>
				 if Line_Curs.Values(Line_Curs.Val_Cur).Line_Last /= 0 then
				    case Line_Curs.Val_Cur is
				       when None_Value =>
					  null;
				       when TVA_Rate_Value =>

					  if Deal_Name = Deal_Out then
					     begin
						
						Line_Add(Abstracted_Voice(Outvoice), To_Account_Line(Line_Curs, 10, Outvoice.Line_Count + 1));
						Outvoice.Line_Count := Outvoice.Line_Count + 1;
						if Outvoice.Line_Last >= (App.Lines-34) then
						   Account_Line_Index := Account_Line_Index + 1;
						end if;
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);
						   --Errno := System("clear" & Character'Val(0));
						   Put_Line("Outvoice error");
						   declare
						      Char : Character;
						   begin
						      Text_Io.Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;
					     end;
					  else
					     begin
						Line_Add(Abstracted_Voice(Invoice), To_Account_Line(Line_Curs, 607, Invoice.Line_Count + 1));
						Invoice.Line_Count := Invoice.Line_Count + 1;
						if Invoice.Line_Last >= (App.Lines-34) then
						   Account_Line_Index := Account_Line_Index + 1;
						end if;
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);
						   --Errno := System("clear" & Character'Val(0));
						   Put_Line("Invoice error");
						   declare
						      Char : Character;
						   begin
						      Text_Io.Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;
					     end;
					  end if;
					  Line_Curs := Initialize;				    
				       when others =>
					  Line_Curs.Val_Cur := Value_Enum'Succ(Line_Curs.Val_Cur);
				    end case;
				    Validated := True;
				 else
				    Put(Character'Val(7));
				 end if;
			      when Wide_Character ' Val (127) =>
				 if Line_Curs.Values(Line_Curs.Val_Cur).Line_Last > 0 then
				    Line_Curs.Values(Line_Curs.Val_Cur).Line(Line_Curs.Values(Line_Curs.Val_Cur).Line_Last) := ' ';
				    Line_Curs.Values(Line_Curs.Val_Cur).Line_Last := 
				      Line_Curs.Values(Line_Curs.Val_Cur).Line_Last - 1;
				 end if;
			      when Wide_Character'Val(18) =>
				 Remove_Case := True;
				 
			      when Wide_Character'Val(20) => -- Ctrl+T : charge and billed.
				 if Deal_Name = Deal_Out then
				    if Outvoice.Line_Last > 0 then
				       Open(Sky.Balance.Balance, Accountant.Accounts);			    
				       Charge(Sky.Balance.Balance, Outvoice, Payed);
				       if Payed then
					  Outvoice.Date := Clock;
					  Outvoice.Account_Num := Outvoice.Accounting(1).Account_Num;
					  Outvoice.voice_Num := Sky.Balance.Outvoice_Num + 1;
					  Sky.Balance.Outvoice_Num := Sky.Balance.Outvoice_Num + 1;
					  if Save(Outvoice, "charged.txt") then
					     Add(Accounts, Outvoice);
					     
					     for Iter in 1..Outvoice.Line_Last loop
						
						declare
						   Line : constant Account_Line_Type := Outvoice.Accounting(Iter);
						begin
						   if not Stocks_Manager.In_Stocks(Stocks_Manager.Software, Line.Description) then
						      declare
							 Stock : constant Stock_Record := (Stocks_Manager.soft_Num + 1, Line.Description, Line.Quantity, Line.Unit_Ht_Price, Line.TVA_Rate);
						      begin
							 Stocks_Manager.Soft_Num := 
							   Stocks_Manager.Soft_Num + 1;
							 Stocks_Manager.Stock_Add(Stocks_Manager.Software, Stock);
						      end;
						   else
						      declare
							 Stock : constant Stock_Record := (0, Line.Description, Line.Quantity, Line.Unit_Ht_Price, Line.TVA_Rate);
						      begin
							 
							 Stocks_Manager.Add_To_stock(Stocks_Manager.Software, Stock.Name, Stock.Quantity);
						      end;
						   end if;
						   
						end;
					     end loop;
					     
					     Outvoice := Null_Outvoice;
					  end if;
					  close(Sky.Balance.Balance, Accountant.Accounts);
				       end if;
				       Open(Sky.Balance.Balance, Accountant.Accounts);
				       Outvoice := Null_Outvoice;
				       App.Initialized := True;
				    end if;
				 else
				    billed(Sky.Balance.Balance, invoice, Payed);
				    if Payed then
				       Invoice.Date := Clock;
				       Invoice.Account_Num := Invoice.Accounting(1).Account_Num;
				       Invoice.voice_Num := Sky.Balance.Invoice_Num + 1;
				       Sky.Balance.Invoice_Num := Sky.Balance.Invoice_Num + 1;
				       if Save(Invoice, "billed.txt") then
					  Add(Accounts, Invoice);
					  invoice := Null_invoice;
				       end if;

				    end if;
				    invoice := Null_invoice;
				 end if;
			      when others =>
				 null;
			   end case;
			   if Deal_Name = Deal_Out then
			      Draw_Window(Box_Dealout);
			      Draw_Text(Box_Dealout, 1, App.Columns/4-10, White, "Moneybox DealOut");
			      Draw_Text(Box_Dealout, 3, 1, White, "Name      : ");
			      Draw_Text(Box_Dealout, 5, 1, White, "Price HT  : ");
			      Draw_Text(Box_Dealout, 7, 1, White, "Quantity  : ");
			      Draw_Text(Box_Dealout, 9, 1, White, "TVA rate  : ");
			      Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			      Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			      Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			      Draw_Window(Ticket);
			      Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_ht)));
			      Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_Tva)));
			      Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Outvoice.Total_TTC)));
			   else
			      Draw_Window(Box_Dealin);
			      Draw_Text(Box_Dealin, 1, App.Columns/4-10, White, "Moneybox DealIn");
			      Draw_Text(Box_Dealin, 3, 1, White, "Name      : ");
			      Draw_Text(Box_Dealin, 5, 1, White, "Price HT  : ");
			      Draw_Text(Box_Dealin, 7, 1, White, "Quantity  : ");
			      Draw_Text(Box_Dealin, 9, 1, White, "TVA rate  : ");
			      Draw_Text(Box_Dealout, 11, 1, White, "Total HT  : ");
			      Draw_Text(Box_Dealout, 13, 1, White, "Total TVA : ");
			      Draw_Text(Box_Dealout, 15, 1, White, "Total TTC : ");
			      Draw_Window(Ticket);
			      Draw_Text(Ticket, 1, 1, White, "Total HT  : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_ht)));
			      Draw_Text(Ticket, 2, 1, White, "Total TVA : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_Tva)));
			      Draw_Text(Ticket, 3, 1, White, "Total TTC : " & Handling.To_Wide_String(Fixed_Total'Image(Invoice.Total_TTC)));
			   end if;
			   Draw_Window(Voice);
			   Draw_Window(Balance);
			   Draw_Text(Balance, 1, 1, White, 
				     
				     " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				       ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				       ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				       ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
				    );
			   --Draw_Window(Cmd_Win);
			   Draw_Text(Voice, 1, 1, White, "N     , Name            , Unit HT    , Quantity    , Total HT     , TVA %  , Total TVA       , Total Price TTC");

			   Draw_Text(Voice, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2-2) * '-')));
			   Ticket_Wins_Draw(App.Ticket_Wins);
			   if Deal_Name = Deal_Out then
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;
			      Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			      
			      Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   else
			      
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;
			      Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			      Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   end if;
			   if Remove_Case then
			      Draw_Window(Cmd_Win);
			      Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
			   end if;

			   
			elsif Line_Curs.Values(Line_Curs.Val_Cur).Line_Last < 32 then
			   
			   Line_Curs.Values(Line_Curs.Val_Cur).Line(Line_Curs.Values(Line_Curs.Val_Cur).Line_Last + 1) := Wchar;
			   Line_Curs.Values(Line_Curs.Val_Cur).Line_Last := 
			     Line_Curs.Values(Line_Curs.Val_Cur).Line_Last + 1;
			   if Deal_Name = Deal_Out then
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   else
			      
			      for Val in Name_Value..Total_Ttc_Value loop
				 Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
					   Line_Curs.values(Val).Column_Pos,
					   White,
					   Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
					Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
					White,
					Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
			   end if;
			   
			else
			   Put(Character'Val(7));
			end if;
		     end Receive;
		     
		     --if Validated then
		     --   exit;		  
		     --end if;
	       or

		  accept Refresh do
		     null;
		  end Refresh;
		  
	       or
		  accept Page_Down do
		     if Deal_Name = Deal_Out then
			if Outvoice.Line_Last >= (App.Lines-34) then
			   if Account_Line_Index + (App.Lines-34) < Outvoice.Line_Last then
			      Account_Line_Index := Account_Line_Index + (App.Lines-34);
			   else
			      Account_Line_Index := Outvoice.Line_Last - (App.Lines-34);
			   end if;
			end if;
		     else
			if Invoice.Line_Last >= (App.Lines-34) then
			   if Account_Line_Index + (App.Lines-34) < Invoice.Line_Last then
			      Account_Line_Index := Account_Line_Index + (App.Lines-34);
			   else
			      Account_Line_Index := Invoice.Line_Last - (App.Lines-34);
			   end if;
			end if;
		     end if;
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     if Deal_Name = Deal_Out then
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     else
			
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			
			Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     end if;
		     if Remove_Case then
			Draw_Window(Cmd_Win);
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
		     end if;
		  end Page_Down;
	       or
		  accept Page_Up do
		     if Deal_Name = Deal_Out then
			if Outvoice.Line_Last >= (App.Lines-34) then
			   if Account_Line_Index - (App.Lines-34) >= 0 then
			      Account_Line_Index := Account_Line_Range'Max(Account_Line_Index - (App.Lines-34), 0);
			   else
			      Account_Line_Index := 0;
			   end if;
			end if;
		     else
			if Invoice.Line_Last >= (App.Lines-34) then
			   if Account_Line_Index - (App.Lines-34) >= 0 then
			      Account_Line_Index := Account_Line_Range'Max(Account_Line_Index - (App.Lines-34), 0);
			   else
			      Account_Line_Index := 0;
			   end if;
			end if;
		     end if;
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     if Deal_Name = Deal_Out then
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     else
			
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			
			Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     end if;
		     if Remove_Case then
			Draw_Window(Cmd_Win);
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
		     end if;
		  end Page_Up;
	       or
		  accept Up_Arrow do
		     null;
		  end up_arrow;
	       or
		  accept Down_Arrow do
		     null;
		  end Down_Arrow;
	       or
		  accept Left_Arrow;
	       or
		  accept Right_Arrow;
	       or
		  accept Full_Screen;
	       or
		  accept Del;
	       or
		  accept Begin_Of do
		     Account_Line_Index := 0;
		     Ticket_Wins_Draw(App.Ticket_Wins);
		     if Deal_Name = Deal_Out then
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealout , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Outvoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);

			Draw_Text(Box_Dealout, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     else
			
			for Val in Name_Value..Total_Ttc_Value loop
			   Draw_Text(Box_Dealin , Line_Curs.values(Val).Line_Pos,
				     Line_Curs.values(Val).Column_Pos,
				     White,
				     Line_Curs.values(Val).Line(1..Line_Curs.values(Val).Line_Last));
			end loop;
			Voice_Print(Invoice, App.Ticket_Wins, Account_Line_Index, App.Lines-34);
			
			Draw_Text(Box_Dealin, Line_Curs.Values(Line_Curs.Val_Cur).Line_Pos,
				  Line_Curs.Values(Line_Curs.Val_Cur).Column_Pos,
				  White,
				  Line_Curs.Values(Line_Curs.Val_Cur).Line(1..Line_Curs.Values(Line_Curs.Val_Cur).Line_Last));
		     end if;
		     if Remove_Case then
			Draw_Window(Cmd_Win);
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
		     end if;
		  end Begin_Of;
	       or
		  accept end_Of;
	       or
		  accept Overwrite;
	       or
		  accept Screen_Print;

	       end select;
	    end loop;
	    if Validated then
	       
	       Validated := False;
	    end if;
	 end loop Main_Loop;
      end;
      
      --W_Io.Put_Line("Moneybox process halted");
   end MoneyBox_Process;
   
   
   
   
   
   
   
   
   
   
   
   -------------------------------------------------------------------------------------------------------------
   --                                                                                                         --
   --                                          Activity                                                       --
   --                                                                                                         --
   -------------------------------------------------------------------------------------------------------------
   
   
   

   
   
   
   
   procedure Initialize(Wins : out Accounting_Windows_Type;
			Space_Left : in Line_Range;
			Width      : in Column_Range;
			L          : in Line_Range;
			C          : in Column_Range;
		       Top         : in Natural := 0);
   
   
   
   
   procedure Accounting_Wins_Draw(Wins : Accounting_Windows_Type) is
      
   begin
      Draw_Window(Wins.Voice_Num_Win);
      Draw_Window(Wins.Acc_Num_Win);
      Draw_Window(Wins.Total_Ht_Win);
      Draw_Window(Wins.Total_TVA_Win);
      Draw_Window(Wins.Total_Ttc_Win);
      Draw_Window(Wins.Date_Win);
      Draw_Window(Wins.Peer_Win);
   end Accounting_Wins_Draw;      
   
   
   procedure Accounting_Line_Draw(Wins : Accounting_Windows_Type; Line : in Line_Range; Voice : in Abstracted_Voice) is
      
   begin
      Draw_Text(Wins.Voice_Num_Win, Line, 1, White, Handling.To_Wide_String(Voice_Num_Range'Image(Voice.Voice_Num)));
      Draw_Text(Wins.Acc_Num_Win, Line, 1, White, Handling.To_Wide_String(Account_Num_Range'Image(Voice.Account_Num)));
      Draw_Text(Wins.Total_Ht_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Voice.Total_Ht)));
      Draw_Text(Wins.Total_TVA_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Voice.Total_TVA)));
      Draw_Text(Wins.Total_Ttc_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Total'Image(Voice.Total_TTC)));
      Draw_Text(Wins.Date_Win, Line, 1, White, Handling.To_Wide_String(Formatting.Image(Voice.Date)));
      Draw_Text(Wins.Peer_Win, Line, 1, White, Voice.Peer_Name(1..16));
   end Accounting_Line_Draw;      
   
   
   subtype Voice_Index_Range is Positive range 1..Positive'Last;
   type Voice_Access is access all Abstracted_Voice;
   function Same(Left, Right : in Voice_Access) return Boolean is
   begin
      return Left.all = Right.all;
   end Same;
   
   package Voice_Vectors is new Ada.Containers.Vectors (Voice_Index_Range, Voice_Access, Same);
   
   function "<"(Left, Right : in Voice_Access) return Boolean is
   begin
      return Left.Date < Right.Date;
   end "<";
   
   package Sorting is new Voice_Vectors.Generic_Sorting("<");
   
   use Voice_Vectors;
   
   type State_Type is
      record
	 Date         : Time := Time_Of(1970, 1, 1, 0.0);
	 Deal_In_HT   : Fixed_Total := 0.0;
	 Deal_In_Ttc  : Fixed_Total := 0.0;
	 Tva_Out      : Fixed_Total := 0.0;
	 Deal_Out_HT  : Fixed_Total := 0.0;
	 Deal_Out_Ttc : Fixed_Total := 0.0;
	 Tva_in       : Fixed_Total := 0.0;
	 Result       : Fixed_Total := 0.0;
      end record;
   
   
   procedure Make_State (Voices : in Voice_Vectors.Vector; From : in Time; To : in Time; State : out State_Type) is
      
   begin
      
      
      
      
      
      if not Is_Empty(Voices) then
	 for Iter in 1..Length(Voices) loop
	    declare
	       Voice : constant Voice_Access := Voice_Vectors.Element(Voices, Integer(Iter));
	    begin	       
	       case Voice.Account_Num is
		  when 607 =>
		     State.Deal_In_Ht := State.Deal_In_Ht + Voice.Total_Ht;
		     State.Deal_In_TTC := State.Deal_In_TTC + Voice.Total_TTC;
		     State.Tva_Out := State.Tva_Out + Voice.Total_Tva;
		  when 707 =>
		     State.Deal_Out_Ht := State.Deal_Out_Ht + Voice.Total_Ht;
		     State.Deal_Out_TTC := State.Deal_Out_TTC + Voice.Total_TTC;
		     State.Tva_In := State.Tva_In + Voice.Total_Tva;
		  when 10 =>
		     State.Result := State.Result + Voice.Total_Ht;
		  when others =>
		     null;
	       end case;
	       exit when Voice.Date > To;
	    end;	    
	 end loop;
	 State.Result := State.Result - State.Deal_In_Ttc;
	 State.Result := State.Result + State.Deal_Out_Ttc;
	 State.Result := State.Result - State.Tva_In;
	 State.Result := State.Result + State.Tva_Out;
      end if;
   end Make_State;
   
   
   
   
   
   procedure Initialize(Wins : out Accounting_Windows_Type;
			Space_Left : in Line_Range;
			Width      : in Column_Range;
			L       : in Line_Range;
			C        : in Column_Range;
			Top       : in Natural := 0) is
   begin
      
      
      Wins.Voice_Num_Win := (L+2, C, Space_Left-2, 10,  
			     Cyan, Window);
      Wins.Acc_Num_Win := (L+2, C+10, Space_Left-2, 8,  
			   Cyan, Window);
      Wins.Total_Ht_Win := (L+2, C+18, Space_Left-2, 16,  
			    Cyan, Window);
      Wins.Total_TVA_Win := (L+2, C+34, Space_Left-2, 16,  
			     Cyan, Window);
      Wins.Total_TTC_Win := (L+2, C+50, Space_Left-2, 16,  
			     Cyan, Window);
      Wins.Date_Win := (L+2, C+66, Space_Left-2, 22,  
			Cyan, Window);
      
      Wins.Peer_Win := (L+2, C+88, Space_Left-2, Width-(C+88+2),  
			Cyan, Window);
      
      Initialize(Wins.Voice_Num_Win, L+2, Top+C, Space_Left-2, 10,  Cyan, Red);
      Initialize(Wins.Acc_Num_Win, L+2, Top+C+10, Space_Left-2, 8,  Cyan, Red);
      Initialize(Wins.Total_Ht_Win, L+2, Top+C+18, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.Total_TVA_Win, L+2, Top+C+34, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.Total_TTC_Win, L+2, Top+C+50, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.Date_Win, L+2, Top+C+66, Space_Left-2, 22, Cyan, Red);
      Initialize(Wins.Peer_Win, L+2, Top+C+88, Space_Left-2, Width-(C+88+2), Cyan, Red);
      
   end Initialize;
   

   
   
   
   task body Activity_Process is
      
      Activity_From    : Window_Record := (3, 1, 10, App.Columns/4, Blue,  Window);
      Activity_To      : Window_Record := (3, (App.Columns/4)+1, 10, App.Columns/4, Blue, Window);
      Activity_Compta  : Window_Record := (13, App.Columns/2+1, (App.Lines-20), App.Columns/2, Cyan, Window);
      Activity_Balance : Window_Record := (App.Lines-5, App.Columns/2+1, 3, App.Columns/2,  Yellow, Window);
      Activity_Cmd     : Window_Record := (App.Lines-2, 1, 3, App.Columns/2, White, Window);
      
      
      
      
      procedure State_Print(Date : in Time; Line_Max : in Line_Range) is
	 
	 
	 
	 
	 
	 
	 State : Voice_Vectors.Vector;
	 
	 Page : Voice_Vectors.Vector;
	 
	 From : State_Type;
	 To   : State_Type;
	 
	 From_Line : Abstracted_Voice(256);
	 To_Line   : Abstracted_Voice(256);
	 
	 
      begin
	 
	 for Invoice in 1..Accounts.Invoice_Last loop
	    Append(State, new Abstracted_Voice ' (abstracted_voice(Accounts.Billed(Invoice))));
	 end loop;
	 for outvoice in 1..Accounts.Outvoice_Last loop
	    Append(State, new Abstracted_Voice ' (abstracted_voice(Accounts.Charges(outvoice))));
	 end loop;
	 Sorting.Sort(State);
	 
	 if not Is_Empty(State) then
	    for Iter in 1..Length(State) loop
	       declare
		  Voice : constant Abstracted_Voice := Element(State, Voice_Index_Range(Iter)).all;
	       begin
		  if Voice.Date >= Date then
		     Append(Page, new Abstracted_Voice ' (Voice));
		  end if;
		  exit when Length(Page) = Count_Type(Line_Max);
	       end;
	    end loop;
	 end if;
	 
	 
	 
	 Make_State(State, Time_Of(1970, 1, 1, 0.0), Date, From);
	 
	 if not Is_Empty(Page) then
	    
	    From_Line := First_Element(Page).all;
	    From.Date := From_Line.date;
	 end if;
	 
	 Draw_Window(Activity_From);
	 
	 
	 
	 Draw_Text(Activity_From, 1, 1, White, "From         : " & Handling.To_Wide_String(Formatting.Image(From.Date)));
	 Draw_Text(Activity_From, 2, 1, White, "Deal  in HT  : " & Handling.To_Wide_String(Fixed_Total'Image(From.Deal_In_Ht)));
	 Draw_Text(Activity_From, 3, 1, White, "Deal  in TTC : " & Handling.To_Wide_String(Fixed_Total'Image(From.Deal_In_TTC)));
	 Draw_Text(Activity_From, 4, 1, White, "TVA  out     : " & Handling.To_Wide_String(Fixed_Total'Image(From.TVA_Out)));
	 Draw_Text(Activity_From, 5, 1, White, "Deal out HT  : " & Handling.To_Wide_String(Fixed_Total'Image(From.Deal_Out_Ht)));
	 Draw_Text(Activity_From, 6, 1, White, "Deal out TTC : " & Handling.To_Wide_String(Fixed_Total'Image(From.Deal_Out_Ht)));
	 Draw_Text(Activity_From, 7, 1, White, "TVA   in     : " & Handling.To_Wide_String(Fixed_Total'Image(From.TVA_In)));
	 Draw_Text(Activity_From, 8, 1, White, "Result       : " & Handling.To_Wide_String(Fixed_Total'Image(From.Result)));
	 
	 if not Is_Empty(Page) then
	    Make_State(State, Time_Of(1970, 1, 1, 0.0), Last_Element(Page).Date, To);
	 end if;
	 
	 if not Is_Empty(Page) then
	    To_Line := Last_Element(Page).all;
	    To.Date := To_Line.date;
	 end if;
	 
	 Draw_Window(Activity_To);
	 
	 Draw_Text(Activity_To, 1, 1, White, "To           : " & Handling.To_Wide_String(Formatting.Image(To.Date)));
	 Draw_Text(Activity_To, 2, 1, White, "Deal  in HT  : " & Handling.To_Wide_String(Fixed_Total'Image(To.Deal_In_Ht)));
	 Draw_Text(Activity_To, 3, 1, White, "Deal  in TTC : " & Handling.To_Wide_String(Fixed_Total'Image(To.Deal_In_TTC)));
	 Draw_Text(Activity_To, 4, 1, White, "TVA  out     : " & Handling.To_Wide_String(Fixed_Total'Image(To.TVA_Out)));
	 Draw_Text(Activity_To, 5, 1, White, "Deal out HT  : " & Handling.To_Wide_String(Fixed_Total'Image(To.Deal_Out_Ht)));
	 Draw_Text(Activity_To, 6, 1, White, "Deal out TTC : " & Handling.To_Wide_String(Fixed_Total'Image(To.Deal_Out_Ht)));
	 Draw_Text(Activity_To, 7, 1, White, "TVA   in     : " & Handling.To_Wide_String(Fixed_Total'Image(To.TVA_In)));
	 Draw_Text(Activity_To, 8, 1, White, "Result       : " & Handling.To_Wide_String(Fixed_Total'Image(To.Result)));
	 
	 Draw_Window(Activity_Compta);
	 
	 Draw_Text(Activity_Compta, 1, 1, White, "Voice Num, Acc N , Total HT      , Total TVA    , Total TTC     , date              , Peer Name");

	 Accounting_Wins_Draw(App.Activity_Wins);
	 
	 
	 if not Is_Empty(Page) then
	    for Iter in 1..Length(Page) loop
	       declare
		  Voice : constant Abstracted_Voice := Element(Page, Voice_Index_Range(Iter)).all;
	       begin
		  Accounting_Line_Draw(App.Activity_Wins, Line_Range(Iter), Voice);
	       end;
	    end loop;
	 end if;
	 
      end State_Print;
   begin      
      Initialize(Activity_From  , 3, 1, 10, (App.Columns)/4, Blue,  White);
      Initialize(Activity_To    , 3, App.Columns/4+1, 10, App.Columns/4, Blue, White);
      Initialize(Activity_Compta, 13, 1, (App.Lines-20), App.Columns/2, Cyan, White);
      Initialize(Activity_Balance ,App.Lines-5, 1, 3, App.Columns/2,  Yellow, White);
      Initialize(Activity_Cmd, App.Lines-2, 1, 3, App.Columns/2, White, White);
      
      
      
      Initialize(App.Activity_Wins, App.Lines-23, App.Columns/2, 13, 2, 0);
      
      
      declare
	 
	 
	 Date : Time := Clock;
	 
	 Line : Wide_String(1..App.Columns/2-2);
	 Line_Last : Natural := 0;
	 
	 
      begin
     Main_Loop:
	 loop
	    loop
	       select
		  
		  accept Halt do
		     null;
		  end Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
	       or
		  accept Wake_Up(Top : in Natural) do
		     Initialize(Activity_From  , 3, Top+1, 10, (App.Columns)/4, Blue,  White);
		     Initialize(Activity_To    , 3, Top+App.Columns/4+1, 10, App.Columns/4, Blue, White);
		     Initialize(Activity_Compta, 13, Top+1, (App.Lines-20), App.Columns/2, Cyan, White);
		     Initialize(Activity_Balance ,App.Lines-5, Top+1, 3, App.Columns/2,  Yellow, White);
		     Initialize(Activity_Cmd, App.Lines-2, Top+1, 3, App.Columns/2, White, White);
		     
		     
		     
		     Initialize(App.Activity_Wins, App.Lines-23, App.Columns/2, 13, 2, Top);
		     
		     ----------------------------------------------
		     --Errno := System("clear" & Character'Val(0));
		     ----------------------------------------------
		     State_Print(Date-86400.0, App.Lines-25);
		     
		     Draw_Window(Activity_Balance);
		     Draw_Text(Activity_Balance, 1, 1, White, 
			       
			       " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				 ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				 ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				 ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
			      );
		     Draw_Window(Activity_Cmd);
		     if Line_Last > 0 then
			Draw_text(Activity_Cmd, 1, 1, White, "Date : " & Line(Line'First..Line_Last));
		     else
			Draw_text(Activity_Cmd, 1, 1, White, "Date : ");
		     end if;
		     
		     
		  end Wake_Up;
		  exit;
	       end select;
	    end loop;

	    loop
	       
	       select
		  accept Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
		  exit;
	       or
		  accept Wake_Up(Top : in Natural);
	       or
		  
		  accept Receive(Wchar : in Wide_Character) do
		     if Is_Control(Wchar) then
			case Wchar is
			   when Wide_Character ' Val (10) =>
			      begin
				 Date := Formatting.Value(Handling.To_String(Line(1..Line_Last)));
				 Line := (others => Wide_Character'Val(0));
				 Line_Last := 0;
			      exception
				 when Constraint_Error =>
				    null;
			      end;
			   when Wide_Character'Val (20) =>
			      Accountant.Close(Sky.Balance.Balance,
					       Accountant.Accounts);
			   when Wide_Character'Val(127) =>
			      if Line_Last > 0 then
				 Line(Line_Last) := Wide_Character'Val(32);
				 Line_Last := Line_Last - 1;
			      end if;
			   when others =>
			      null;
			end case;
			----------------------------------------------
			--Errno := System("clear" & Character'Val(0));
			----------------------------------------------
			State_Print(Date-86400.0, App.Lines-25);
			
			Draw_Window(Activity_Balance);
			Draw_Text(Activity_Balance, 1, 1, White, 
				  
				  " Capital : " & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Capital)) &
				    ", Material :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.Material)) &
				    ", TVA In :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_In)) &
				    ", TVA Out :" & Handling.To_Wide_String(Fixed_Total'Image(Sky.Balance.Balance.TVA_Out)) & ""
				 );
			Draw_Window(Activity_Cmd);
			if Line_Last > 0 then
			   Draw_text(Activity_Cmd, 1, 1, White, "Date : " & Line(Line'First..Line_Last));
			else
			   Draw_text(Activity_Cmd, 1, 1, White, "Date : ");
			end if;
		     elsif Line_Last < App.Columns/2-2 then
			Line(Line_Last + 1) := Wchar;
			Line_Last := Line_Last + 1;
			Draw_Window(Activity_Cmd);
			if Line_Last > 0 then
			   Draw_text(Activity_Cmd, 1, 1, White, "Date : " & Line(Line'First..Line_Last));
			else
			   Draw_text(Activity_Cmd, 1, 1, White, "Date : ");
			end if;
		     else
			Put(Character'Val(7));
		     end if;
		  end Receive;
	       or
		  
		  accept Refresh;
		  
	       or
		  accept Page_Down;
	       or
		  accept Page_Up;
	       or
		  accept Up_Arrow do
		     null;
		  end up_arrow;
	       or
		  accept Down_Arrow do
		     null;
		  end Down_Arrow;
	       or
		  accept Left_Arrow;
	       or
		  accept Right_Arrow;
	       or
		  accept Del;
	       or
		  accept Begin_Of;
	       or
		  accept End_Of;
	       or
		  accept Overwrite;
	       or
		  accept Screen_Print;

	       end select;
	    end loop;
	    
	 end loop Main_Loop;
      end;
      --W_Io.Put_Line("Activity process halted");
   end Activity_Process;
   
   -------------------------------------------------------------------------------------------------------------
   --                                                                                                         --
   --
   -------------------------------------------------------------------------------------------------------------
   type Stock_enum is (Stock_None, Stock_Name, Stock_Quantity);
   
   type Stock_Type is
      record
	 Name : Stock_Enum := Stock_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Curs_Set is array (Stock_Enum) of Stock_Type;
   
   type Stock_Cursor_Type is
      record
	 Values  : Curs_Set;
	 Val_Cur : Stock_Enum := Stock_Name;	 
      end record;
   
   function Initialize return Stock_Cursor_Type is
      Curs : Stock_Cursor_Type;
   begin
      Curs.Values(Stock_Name) :=
	(Stock_name,
	 (others => ' '),
	 0,
	 2,
	 13
	);
      Curs.Values(Stock_Quantity) :=
	(Stock_quantity,
	 (others => ' '),
	 0,
	 4,
	 13
	);
      Curs.Val_Cur := Stock_name;
      return Curs;
   end Initialize;
   
   
   function To_Stock (Curs : in stock_Cursor_Type;Line_Num : in Stock_Num_Range;Stock_Num : in Stock_Num_Range) return Stock_Record is
      Stock : Stock_Record := Null_Stock;
   begin
      Stock.Num := Stock_Num;
      Stock.Name(1..Curs.Values(Stock_Name).Line_Last) := 
	Curs.Values(Stock_Name).Line(1..Curs.Values(Stock_Name).Line_Last);
      Stock.Quantity := 
	Quantity_Range'Value(Handling.To_String(Curs.Values(Stock_Quantity).Line(1..Curs.Values(Stock_Quantity).Line_Last)));
      
      return stock;
   exception
      when others =>
	 raise Spec_Error;
   end To_Stock;

   
   procedure Initialize(Wins : out Stock_Windows_Type;
			Space_Left : in Line_Range;
			Width      : in Column_Range;
			L       : in Line_Range;
			C        : in Column_Range;
		       Top       : in Natural := 0) is
   begin
      
      
      Wins.Stock_Num_Win := (L+2, C, Space_Left-2, 8,  
			     Cyan, Window);
      Wins.Name_Win := (L+2, C+8, Space_Left-2, 26,
			Cyan, Window);
      Wins.Quantity_Win := (L+2, C+36, Space_Left-2, 16,  
			    Cyan, Window);
      Wins.Unit_Ht_Win := (L+2, C+52, Space_Left-2, 16,  
			   Cyan, Window);
      Wins.TVA_Win := (L+2, C+68, Space_Left-2, 10,  
		       Cyan, Window);
      
      
      Initialize(Wins.Stock_Num_Win, L+2, Top+C, Space_Left-2, 8,  Cyan, Red);
      Initialize(Wins.Name_Win, L+2, Top+C+8, Space_Left-2, 26,  Cyan, Red);
      Initialize(Wins.quantity_Win, L+2, Top+C+36, Space_Left-2, 16,  Cyan, Red);
      Initialize(Wins.Unit_Ht_Win, L+2, Top+C+52, Space_Left-2, 16, Cyan, Red);
      Initialize(Wins.TVA_Win, L+2, Top+C+68, Space_Left-2, 10, Cyan, Red);

      
      null;
   end Initialize;

   
   procedure Stock_Wins_Draw(Wins : Stock_Windows_Type) is
      
   begin
      Draw_Window(Wins.Stock_Num_Win);
      Draw_Window(Wins.Name_Win);
      Draw_Window(Wins.Quantity_Win);
      Draw_Window(Wins.Unit_Ht_Win);
      Draw_Window(Wins.TVA_Win);
   end Stock_Wins_Draw;      
   
   
   procedure Stock_Line_Draw(Wins : Stock_Windows_Type; Line : in Line_Range; Stock : in Stock_Record) is
      
   begin
      Draw_Text(Wins.stock_Num_Win, Line, 1, White, Handling.To_Wide_String(stock_Num_Range'Image(Stock.Num)));
      Draw_Text(Wins.Name_Win, Line, 1, White, Stock.name(1..24));
      Draw_Text(Wins.Quantity_win, Line, 1, White, Handling.To_Wide_String(Quantity_Range'Image(Stock.Quantity)));
      Draw_Text(Wins.Unit_Ht_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Unit_Ht'Image(Stock.Unit_Ht)));
      Draw_Text(Wins.TVA_Win, Line, 1, White, Handling.To_Wide_String(Fixed_Factor'Image(Stock.TVA_Rate)));
      
   end stock_Line_Draw;      
   
   
   procedure Stocks_Print (Stocks : in Stocks_Manager.Stocks_manager;
			   Wins : in Stock_Windows_Type;
			   Line_Index : in Stock_num_Range;
			   Line_Max : in Positive) is
      Line_Count : natural := 0;
   begin
      if Stocks.Stock_Last = 0 then
	 return;
      end if;
      for Line in Line_Index+1..Natural'Min(Line_Index+Line_Max, Stocks.Stock_Last) loop
	 Stock_Line_Draw(Wins, Line_Count+1, Stocks.stocks(Line));
	 Line_Count := Line_Count + 1;
	 exit when Line_Count = Line_Max;
      end loop;
   end Stocks_Print;

   
   task body Stocks_Process is
      type Stocks_Enum is (None, Hardware, Software);
      Stocks_cur : Stocks_Enum := Software;
      
      
      
      function Format (Stock : in Stock_Record) return Wide_String is
   	 Line : Wide_String(1..App.Columns/2-2) := (others => Wide_Character'Val(32));
      begin
   	 Wide_Fixed.Move(Handling.To_Wide_String(Natural'Image(Stock.Num)), Line, Error, Left);
   	 Wide_Fixed.Replace_Slice(Line, 9, Line'Last, Stock.name);
   	 Wide_Fixed.Replace_Slice(Line, 42, Line'Last, Handling.To_Wide_String(Quantity_Range'Image(Stock.Quantity)));
   	 Wide_Fixed.Replace_Slice(Line, 55, Line'Last, Handling.To_Wide_String(Fixed_Unit_Ht'Image(Stock.Unit_Ht)));
   	 Wide_Fixed.Replace_Slice(Line, 62, Line'Last, Handling.To_Wide_String(Fixed_Factor'Image(Stock.TVA_Rate)));
   	 return Line;
      end Format;
      
      Soft_win : Window_Record := (2, 1, 10, App.Columns/2, Cyan, Window);
      Hard_win : Window_Record := (2, 1, 10, App.Columns/2, yellow, Window);
      Stocks_Win : Window_Record := (11, 1, App.Lines-13, App.Columns/2, Blue, Window);
      Cmd_Win    : Window_Record := (App.Lines-2, 1, 3, App.Columns/2, White, Window);
      
      --Stocks_index : Natural := 0;
      Stock_Line_Index : Natural := 0;
      
      Remove_Case : Boolean := False;
      Stock_Num   : Stock_Num_Range := 0;
      Line_num    : Wide_String(1..10) := (others => ' ');
      Line_Last   : Natural := 0;
      Stock_Curs : Stock_Cursor_Type := Initialize;
      
   begin
      Initialize(Stocks_Win, 11, 1, App.Lines-13, App.Columns/2, Blue, white);
      Initialize(Soft_Win, 2, 1, 10, App.Columns/2, Cyan, White);
      Initialize(Hard_Win, 2, 1, 10, App.Columns/2, yellow, White);
      Initialize(Cmd_Win, App.Lines-2, 1, 3, App.Columns/2, White, Red);
      
      Initialize(App.Stocks_Wins, App.Lines-15, App.Columns/2-4, 12, 2);
      
      
      declare
   	 Num      : Stock_Num_Range := 1;
   	 Name     : Wide_String := "Inconito.net";
   	 Line     : Wide_String(1..32) := (others => Wide_Character'Val(32));
   	 Qt       : Quantity_Range := 0.0;
   	 Ht_Price : Fixed_Unit_Ht := 0.0;
   	 TVA      : Fixed_Factor := 0.0;
   	 Stock    : Stock_Record;
      begin
   	 
     Main_Loop:
   	 loop
   	    loop
   	       select
   		  accept Halt;
   		  exit Main_Loop;
   	       or
   		  accept Suspend;
   		  exit;
   	       or
   		  accept Wake_Up(Top : in Natural);
   	       or
		  when not Remove_Case =>  
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character ' Val (9) =>
				 case Stocks_Cur is
				    when None =>
				       Stocks_Cur := Software;
				    when Hardware =>
				       Stocks_Cur := Software;
				    when Software =>
				       Stocks_Cur := Hardware;
				 end case;
			      when Wide_Character'Val(5) =>
				 Remove_Case := True;
			      when Wide_Character'Val(10) =>				 
				 if Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last /= 0 then
				    case Stock_Curs.Val_Cur is
				       when Stock_None =>
					  null;
					  
				       when Stock_Quantity =>
					  
					  if Stocks_Cur = Software then
					     declare
						Stock : Stock_Record;
					     begin

						
						Stock := To_Stock(Stock_Curs, Stocks_Manager.Software.Stock_Last + 1, Stocks_Manager.Soft_Num);

						Stocks_Manager.Add_To_Stock(Stocks_Manager.Software, Stock.Name, Stock.Quantity);
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);

						   Put_Line("Stocks Error");
						   declare
						      Char : Character;
						   begin
						      Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;						
					     end;
					     Stock_Curs := Initialize;				    
					  elsif Stocks_Cur = Hardware then
					     declare
						Stock : Stock_Record;
					     begin
						--if Stock.Quantity > 0.0 then
						Stock := To_Stock(Stock_Curs, Stocks_Manager.Hardware.Stock_Last + 1, Stocks_Manager.Hard_Num);
						Stocks_Manager.Add_To_Stock(Stocks_Manager.Hardware, Stock.Name, Stock.Quantity);
					     exception
						when Spec_Error =>
						   Put(Ansi_Tty_Control.Clear_Screen);

						   Put_Line("Stocks Error");
						   declare
						      Char : Character;
						   begin
						      Put_Line("Press any key to continue");
						      Get_Immediate(Char);
						   end;						
					     end;
					     Stock_Curs := Initialize;				    

					  end if;
					  
					  
				       when others =>
					  Stock_Curs.Val_Cur := Stock_Enum'Succ(Stock_Curs.Val_Cur);
				    end case;
				    
				 else
				    Put(Character'Val(7));
				 end if;
			      when Wide_Character ' Val (127) =>
				 if Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last > 0 then
				    Stock_Curs.Values(Stock_Curs.Val_Cur).Line(Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last) := ' ';
				    Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last := 
				      Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last - 1;
				 end if;
			      when Wide_Character'Val(18) =>
				 Remove_Case := False;			      
				 
			      when others =>
				 null;
			   end case;
			elsif Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last < 32 then
			   
			   Stock_Curs.Values(Stock_Curs.Val_Cur).Line(Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last + 1) := Wchar;
			   Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last := 
			     Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last + 1;
			   if Stocks_cur = software then
			      for Val in Stock_Name..Stock_Quantity loop
				 Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					   Stock_Curs.values(Val).Column_Pos,
					   White,
					   Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					White,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));
			   elsif Stocks_Cur = Hardware then
			      
			      for Val in Stock_Name..Stock_Quantity loop
				 Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					   Stock_Curs.values(Val).Column_Pos,
					   White,
					   Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			      end loop;

			      Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					White,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));
			   end if;
			   
			else
			   Put(Character'Val(7));
			end if;
			Draw_Window(Stocks_Win);
			Draw_Text(Stocks_Win, 1, 1, White, "N    , Name                        , " & 
				    "Quantity      , Unit HT     , TVA rate");
			Draw_Text(Stocks_Win, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2)-2) * '-'));
			Draw_Window(Cmd_Win);
			Stock_Wins_Draw(App.Stocks_Wins);
			case Stocks_Cur is			
			   when None =>
			      null;
			   when Software =>
			      Draw_Window(Soft_Win);
			      Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
			      Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
			      Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");
			      
			      for Val in Stock_Name..Stock_Quantity Loop
				 Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					   Stock_Curs.values(Val).Column_Pos,
					   White,
					   Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			      end loop;

			      Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			      Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					White,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

			   when Hardware =>
			      Draw_Window(Hard_Win);
			      Draw_Text(Hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks : Update stock");
			      Draw_Text(Hard_Win, 2, 1, White, "Name      : ");
			      Draw_Text(Hard_Win, 4, 1, White, "Quantity  : ");
			      
			      for Val in Stock_Name..Stock_quantity loop
				 Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					   Stock_Curs.values(Val).Column_Pos,
					   White,
					   Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			      end loop;
			      Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);
			      
			      Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					White,
					Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));
			      
			      
			end case;
			if Remove_Case then
			   
			   Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
			end if;

		     end Receive;
		     
	       or
		  when Remove_Case =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character ' Val (9) =>
				 case Stocks_Cur is
				    when None =>
				       Stocks_Cur := Software;
				    when Hardware =>
				       Stocks_Cur := Software;
				    when Software =>
				       Stocks_Cur := Hardware;
				 end case;
			      when Wide_Character'Val(10) =>				 
				 if Line_Last > 0 then
				    begin
				       Stock_Num := Stock_Num_Range'Value(Handling.To_String(Line_num(Line'First..Line_Last)));
				       if Stock_Num /= 0 then
					  begin
					     case Stocks_Cur is
						when None =>
						   null;
						when Software =>
						   Stocks_Manager.Stock_Remove(Stocks_Manager.Software, Stock_Num);
						when Hardware =>
						   Stocks_Manager.Stock_Remove(Stocks_Manager.Hardware, Stock_Num);
					     end case;
					  exception
					     when others =>
						null;
					  end;

				       end if;
				    exception
				       when others =>
					  null;
				    end;				 
				 end if;
				 Line_Num := (others => ' ');
				 Line_Last := 0;
				 Remove_Case := False;
				 
			      when Wide_Character'Val(127) =>
				 if Line_Last > 0 then
				    Line(Line_Last) := Wide_Character'Val(32);
				    Line_Last := Line_Last - 1;
				 else
				    Put(Character'Val(7));
				 end if;
			      when others =>
				 null;
			   end case;
			   Draw_Window(Stocks_Win);
			   Draw_Text(Stocks_Win, 1, 1, White, "N    , Name                        , " & 
				       "Quantity      , Unit HT     , TVA rate");
			   Draw_Text(Stocks_Win, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2)-2) * '-'));
			   
			   Stock_Wins_Draw(App.Stocks_Wins);
			   Draw_Window(Cmd_Win);
			   case Stocks_Cur is			
			      when None =>
				 null;
			      when Software =>
				 Draw_Window(Soft_Win);
				 Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
				 Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
				 Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");
				 
				 for Val in Stock_Name..Stock_quantity loop
				    Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					      Stock_Curs.values(Val).Column_Pos,
					      White,
					      Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
				 end loop;
				 Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);
				 
				 Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					   Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					   White,
					   Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

			      when Hardware =>
				 Draw_Window(Hard_Win);
				 Draw_Text(hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks  : update stock");
				 Draw_Text(hard_Win, 2, 1, White, "Name      : ");
				 Draw_Text(hard_Win, 4, 1, White, "Quantity  : ");
				 
				 for Val in Stock_Name..Stock_quantity loop
				    Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					      Stock_Curs.values(Val).Column_Pos,
					      White,
					      Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
				 end loop;
				 Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

				 Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
					   Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
					   White,
					   Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

				 
			   end case;
			   if Remove_Case then
			      if Line_Last > 0 then
				 
				 Draw_Text(Cmd_Win, 1, 1, White, "Line num : " & Line_num(Line'First..Line_last));
			      else
				 Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
			      end if;   		     
			   end if;   		     
			elsif Line_Last < Line_num'Length then
			   Line_Num(Line_Last + 1) := Wchar;
			   Line_Last := Line_Last + 1;
			   Draw_Window(Cmd_Win);
			   if Line_Last > 0 then
			      
			      Draw_Text(Cmd_Win, 1, 1, White, "Line num : " & Line_num(Line'First..Line_last));
			   else
			      Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
			   end if;   		     
			else
			   Put(Character'Val(7));
			end if;
		     end Receive;
   	       or

   		  accept Refresh do
   		     Draw_Window(Stocks_Win);
   		     Draw_Text(Stocks_Win, 1, 1, White, "N    , Name                        , " & 
   				 "Quantity      , Unit HT     , TVA rate");
   		     Draw_Text(Stocks_Win, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2)-2) * '-'));
		     
   		     Stock_Wins_Draw(App.Stocks_Wins);
		     Draw_Window(Cmd_Win);
		     case Stocks_Cur is			
   			when None =>
   			   null;
   			when Software =>
			   Draw_Window(Soft_Win);
			   Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
			   Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");
			   
			   for Val in Stock_Name..Stock_quantity loop
			      Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;
			   Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);
			   
			   Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

   			when Hardware =>
			   Draw_Window(Hard_Win);
			   Draw_Text(hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks  : update stock");
			   Draw_Text(hard_Win, 2, 1, White, "Name      : ");
			   Draw_Text(hard_Win, 4, 1, White, "Quantity  : ");
			   
			   for Val in Stock_Name..Stock_quantity loop
			      Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;
			   Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));
			   
   			   
   		     end case;
		     if Remove_Case then
			
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
		     end if;   		     
   		  end Refresh;
		  
   	       or
   		  accept Page_Down do
		     Stock_Wins_Draw(App.Stocks_Wins);
		     case Stocks_Cur is
			when None =>
			   null;
			when Software =>
			   if Stock_Line_Index < Stocks_Manager.Software.Stock_Last then
			      Stock_Line_Index := Stock_Line_Index + Stock_Num_Range'Min (Stock_Line_Index+(App.Lines-19), Stocks_Manager.Software.Stock_Last);
			   else
			      Put(Character'Val(7));
			   end if;
			   Draw_Window(Soft_Win);
			   Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
			   Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");
			   
			   for Val in Stock_Name..Stock_Quantity Loop
			      Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;

			   Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

			when Hardware =>
			   if Stock_Line_Index < Stocks_Manager.hardware.Stock_Last then
			      Stock_Line_Index := Stock_Line_Index + Stock_Num_Range'Min (Stock_Line_Index+(App.Lines-19), Stocks_Manager.hardware.Stock_Last);
			   else
			      Put(Character'Val(7));
			   end if;
			   Draw_Window(Hard_Win);
			   Draw_Text(Hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks : Update stock");
			   Draw_Text(Hard_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Hard_Win, 4, 1, White, "Quantity  : ");
			   
			   for Val in Stock_Name..Stock_quantity loop
			      Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;
			   Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);
			   
			   Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));
			   
		     end case;
		  end Page_Down;
   	       or
   		  accept Page_Up do
		     Stock_Wins_Draw(App.Stocks_Wins);
		     case Stocks_Cur is
			when None =>
			   null;
			when Software =>
			   if Stock_Line_Index - (App.Lines-19) > 0 then
			      Stock_Line_Index := Stock_Line_Index + Stock_Line_Index - (App.Lines-19);
			   else
			      Stock_Line_Index := 0;
			   end if;

			   Draw_Window(Soft_Win);
			   Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
			   Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");

			   for Val in Stock_Name..Stock_Quantity Loop
			      Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;

			   Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

			when Hardware =>
			   if Stock_Line_Index - (App.Lines-19) > 0 then
			      Stock_Line_Index := Stock_Line_Index + Stock_Line_Index - (App.Lines-19);
			   else
			      Stock_Line_Index := 0;
			   end if;

			   Draw_Window(Hard_Win);
			   Draw_Text(Hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks : Update stock");
			   Draw_Text(Hard_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Hard_Win, 4, 1, White, "Quantity  : ");

			   for Val in Stock_Name..Stock_quantity loop
			      Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;

			   Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

		     end case;
		  end Page_Up;
   	       or
   		  accept Up_Arrow do
   		     null;
   		  end up_arrow;
   	       or
   		  accept Down_Arrow do
   		     null;
   		  end Down_Arrow;
   	       or
   		  accept Left_Arrow;
   	       or
   		  accept Right_Arrow;
   	       or
   		  accept Full_Screen;
   	       or
   		  accept Del;
   	       or
   		  accept Begin_Of;
   	       or
   		  accept End_Of;
   	       or
   		  accept Overwrite;
   	       or
   		  accept Screen_Print;

   	       end select;
   	    end loop;
	    
   	    loop
   	       select
		  
   		  accept Halt do
   		     null;
   		  end Halt;
   		  exit Main_Loop;
   	       or
   		  accept Suspend;
   	       or
   		  accept Wake_Up (Top : in Natural) do
		     Initialize(Stocks_Win, 11, Top+1, App.Lines-13, App.Columns/2, Blue, white);
		     Initialize(Soft_Win, 2, Top+1, 10, App.Columns/2, Cyan, White);
		     Initialize(Hard_Win, 2, Top+1, 10, App.Columns/2, yellow, White);
		     Initialize(Cmd_Win, App.Lines-2, Top+1, 3, App.Columns/2, White, Red);
		     
		     Initialize(App.Stocks_Wins, App.Lines-15, App.Columns/2-4, 12, 2, top);
		     -----------------------------------------------
   		     --Errno := System("clear" & Character'Val(0));
		     -----------------------------------------------
   		     Draw_Window(Stocks_Win);
   		     Draw_Text(Stocks_Win, 1, 1, White, "N    , Name                        , " & 
   				 "Quantity      , Unit HT     , TVA rate");
   		     Draw_Text(Stocks_Win, 2, 1, Blue, Handling.To_Wide_String(((App.Columns/2)-2) * '-'));
		     Draw_Window(Cmd_Win);
   		     Stock_Wins_Draw(App.Stocks_Wins);
		     
   		     case Stocks_Cur is			
   			when None =>
   			   null;
   			when Software =>
			   Draw_Window(Soft_Win);
			   Draw_Text(Soft_Win, 1, App.Columns/4-10, White, "Software Stocks : Update stock");
			   Draw_Text(Soft_Win, 2, 1, White, "Name      : ");
			   Draw_Text(Soft_Win, 4, 1, White, "Quantity  : ");
			   
			   
			   for Val in Stock_Name..Stock_Quantity loop
			      Draw_Text(Soft_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;
			   Stocks_Print(Stocks_Manager.Software, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Soft_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

   			when Hardware =>
			   Draw_Window(Hard_Win);
			   Draw_Text(hard_Win, 1, App.Columns/4-10, White, "Hardware Stocks  : update stock");
			   Draw_Text(hard_Win, 2, 1, White, "Name      : ");
			   Draw_Text(hard_Win, 4, 1, White, "Quantity  : ");
			   

			   for Val in Stock_Name..Stock_Quantity loop
			      Draw_Text(Hard_Win , Stock_Curs.values(Val).Line_Pos,
					Stock_Curs.values(Val).Column_Pos,
					White,
					Stock_Curs.values(Val).Line(1..Stock_Curs.values(Val).Line_Last));
			   end loop;
			   Stocks_Print(Stocks_Manager.Hardware, App.Stocks_Wins, stock_Line_Index, App.Lines-19);

			   Draw_Text(Hard_Win, Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Pos,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Column_Pos,
				     White,
				     Stock_Curs.Values(Stock_Curs.Val_Cur).Line(1..Stock_Curs.Values(Stock_Curs.Val_Cur).Line_Last));

   			   
   		     end case;
		     
		     if Remove_Case then
			
			Draw_Text(Cmd_Win, 1, 1, White, "Line num : ");
		     end if;
   		  end Wake_Up;
   		  exit;
   	       end select;
   	    end loop;
   	 end loop Main_Loop;
      end;
      
      --W_Io.Put_Line("Stocks process halted");
   end Stocks_Process;
   
   
   task body Bank_Process is      
      
   begin
      
  Main_Loop:
      loop
	 loop
	    select
	       accept Receive (Wchar : in Wide_Character);
	    or
	       accept Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  W_Io.Put_Line("Bank");
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	       or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  null;
	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  null;
	       end Down_Arrow;
	       
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  W_Io.Put_Line("Bank");
		  Text_Io.Put_Line("Greetings " & Getenv("USER").all & "!");
		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Bank process halted");
   end Bank_Process;
   
   use Abstract_Vectors;
   
   
   procedure Highlight_Plan_Draw (Obj : Abstract_Access; Line : in Positive;Width : in Positive; Top : in Natural) is
      Win : Window_Record := (Line, Top+2, 6, Width, Blue, Window);      
   begin
      
      Initialize(Win, Line, Top+2, 6, Width, Blue, White);
      Enlight_Window(Win);
      Text_Io.Put(Normal_Mode);
      
      Draw_Text(Win, 1, 1, White, "Name : " & Obj.Info.Name.all);
      
      Draw_Text(Win, 2, 1, White, "Date : " & Handling.To_Wide_String(Formatting.Image(Obj.Date)));
      
      Draw_Text(Win, 3, 1, White, "Locality : " & Obj.Locality.all);
      Draw_Text(Win, 4, 1, White, "define : " & Obj.Define.all);
      
   end Highlight_Plan_Draw;
   
   procedure Plan_Draw (Obj : Abstract_Access; Line : in Positive;Width : in Positive; Top : in Natural) is
      Win : Window_Record := (Line, Top+2, 6, Width, Blue, Window);      
   begin
      
      Initialize(Win, Line, Top+2, 6, Width, Blue, White);
      Draw_Window(Win);
      Draw_Text(Win, 1, 1, White, "Name : " & Obj.Info.Name.all);
      
      Draw_Text(Win, 2, 1, White, "Date : " & Handling.To_Wide_String(Formatting.Image(Obj.Date)));
      
      Draw_Text(Win, 3, 1, White, "Locality : " & Obj.Locality.all);
      
      Draw_Text(Win, 4, 1, White, "define : " & Obj.Define.all);
      
   end Plan_Draw;
   



   procedure Planning_Draw (Planning : in Abstract_Vectors.Vector;Index : in Natural; Width : in Positive; Top : in Natural) is
      
      Count : Natural := 1;
   begin
      
      if Abstract_Vectors.Is_empty(Planning) then
	 return;
      end if;
      
      for Plan in Index+1..natural'Min(Index + (App.Lines-5) / 6, Integer(Abstract_Vectors.Length(Planning))) loop
	 
	 declare
	    E : Abstract_Access := Abstract_Vectors.Element(Planning, Abstract_Index(Plan));
	    
	 begin
	    
	    if Clock >= E.Date and then
	      E.Date + E.Period > Clock then
	       
	       Highlight_Plan_Draw(E, Count+1, width, Top);
	       
	    else
	       
	       Plan_Draw(E, Count+1, width, Top);
	       
	    end if;
	    Count := Count + 6;
	 end;
      end loop;

   end Planning_Draw;
   
   
   
   
   
   
   procedure To_Planning (Obj : in Abstract_Access;Vector : in out Abstract_Vectors.Vector) is
   begin
      if Obj = null then
	 return;
      end if;
      if Obj.all in Event_Class | Rendezvous_Class | Meeting_Class | Job_Class then
	 Vector := Vector & Obj;
      end if;
      
      if Obj.Info.Childs_Num /= 0 then
	 for Child in 1..Obj.Info.Childs_Num loop
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(Obj.Childs, Abstract_Index(Child));
	    begin
	       To_Planning(E, Vector);
	    end;
	 end loop;
      end if;
      
   end To_Planning;
   
   function "<" (Left, Right : in Abstract_Access) return Boolean is
   begin
      if Left = null or Right = null then
	 raise Program_Error;
      end if;
      return Left.Date < Right.Date;
   end "<";
   
   package Date_Sorting is new Abstract_Vectors.Generic_Sorting("<");
   
   
   task body Planning_Process is
      
      
      Main_Win : Window_Record := (1, 1, App.Lines-3, App.Columns/2, White, Window);
      
      Planning : Abstract_Vectors.Vector;
      Plan_Index : Natural := 0;
      
      
   begin
      
  Main_Loop:
      loop
	 loop
	    
	    select
	       accept Receive (Wchar : in Wide_Character);
	    or
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  
		  null;
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  If Plan_Index > 0 then
		     Plan_Index := Plan_Index - 1;
		     Enlight_Window(Main_Win);
		  
		     Text_Io.Put(Normal_Mode);
		  
		     Planning_Draw(Planning, Plan_Index, App.Columns/2-2, 0);
		  end if;
		     

	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  if Abstract_Vectors.Length(Planning) /= 0 then
		     if Plan_Index + 1 < Integer(Abstract_Vectors.Length(Planning)) then
			Plan_Index := Plan_Index + 1;
			Enlight_Window(Main_Win);
		  
			Text_Io.Put(Normal_Mode);
			
			Planning_Draw(Planning, Plan_Index, App.Columns/2-2, 0);
		     end if;
		     
		  end if;
	       end Down_Arrow;
	       
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  
		  Initialize(Main_Win, 1, TOP+1, App.Lines-3, App.Columns/2, White, Cyan);
		  
		  Enlight_Window(Main_Win);
		  
		  Text_Io.Put(Normal_Mode);
		  
		  Planning := Abstract_Vectors.Empty_Vector;
		  
		  To_Planning(App.Terminal.Root, Planning);
		  
		  Plan_Index := 0;
		  if Abstract_Vectors.Length(Planning) /= 0 then
		     
		     Date_Sorting.Sort(Planning);
		     
		     for I in 1..Abstract_Vectors.Length(Planning) loop
		  	
		  	declare
		  	   E : constant Abstract_Access := Element(Planning, Abstract_Index(I));
		  	begin
		  	   
		  	   if E.date+E.Period < Clock then
		  	      
		  	      Plan_Index := Plan_Index + 1;
		  	   end if;
		  	end;
		     end loop;
		     
		     Planning_Draw(Planning, Plan_Index, App.Columns/2-2, Top);
		  
		  end if;
		  
	       end Wake_Up;
	       
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Planning process halted");
   end Planning_Process;
   
   
   task body Library_Process is

   begin
      
  Main_Loop:
      loop
	 loop
	    select
	       accept Receive (Wchar : in Wide_Character);
	    or
	       accept Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  W_Io.Put_Line("Library");
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	    or
	       accept Refresh;
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  null;
	       end up_arrow;
	    or
	       accept Down_Arrow do
		  null;
	       end Down_Arrow;
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Del;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Overwrite;
	    or
	       accept Screen_Print;

	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  W_Io.Put_Line("Library");
		  Text_Io.Put_Line("Greetings " & Getenv("USER").all & "!");
		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Library process halted");
   end Library_Process;
   
   ---------------------------------------------------------------------------------------------------------------
   --                                                                                                           --
   --                                        Ada projects manager                                               --
   --                                                                                                           --
   ---------------------------------------------------------------------------------------------------------------
   
   
   
   procedure Load_File (Set : in out Wide_String_Set; Name : in String; Last : out Line_Index_Range) is
      
      File : W_Io.File_Type;
      
   begin
      W_Io.Open(File, W_Io.In_File, Name, Form => "WCEM=8");
      Last := 0;
      
      while not W_Io.End_Of_File(File) loop

   	 declare
   	    Line : constant Wide_String :=W_Io.Get_Line(File);
   	 begin

   	    Set.Lines(Last+1) := new Wide_String ' (Line);
   	    Last := Last + 1;

   	 end;
      end loop;
      Set.Line_Last := Last;
      W_Io.Close(File);
   end Load_File;

   
   use Projects_Dll;
   
   function Get_Info(Projects : in Projects_Record; Project_Num : in Project_Num_Range) return Project_Info_Record is
      Curs : Projects_Dll.Cursor;
      First : Projects_Dll.Cursor;
   begin
      if Projects_Dll.Length(Projects.List) /= 0 then
	 Curs := Projects_Dll.First(Projects.List);
	 First := Curs;
	 loop
	    declare
	       Current : constant Project.Project_Access := Projects_Dll.Element(Curs);	       
	    begin
	       if Current.Project_Num = Project_Num then
		  return Project_Info_Record(Current.all);
	       end if;
	    end;
	    Curs := Projects_Dll.Next(Curs);
	    exit when Curs = First;
	 end loop;
	 
      end if;
      return Project_Info_Record(Null_Project);
   exception
      when others =>
	 return Project_Info_Record(Null_Project);
   end Get_Info;
   
   function Get_Project(Projects : in Projects_Record; Project_Num : in Project_Num_Range) return Project.Project_Access is
      Curs : Projects_Dll.Cursor;
      
   begin

      if Projects_Dll.Length(Projects.List) /= 0 then

	 Curs := Projects_Dll.First(Projects.List);

	 for Ier in 1..Projects_Dll.Length(Projects.List) loop
	    declare
	       Current : constant Project.Project_Access := Projects_Dll.Element(Curs);	       
	    begin

	       if Current.Project_Num = Project_Num then

		  return Current;
	       end if;
	    end;

	    Curs := Projects_Dll.Next(Curs);

	 end loop;
	 
      end if;

      return null;
   exception
      when others =>
	 return null;
   end Get_Project;
   
   
   procedure Component_Info_Draw (Name   : in String;
				  Line_Pos : in Line_Range;
				  Width    : in Column_Range;
				  Top      : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top+2, 3, Width, Cyan, Window);
   begin

      Initialize(Win, Line_Pos, Top+2, 3, Width, Cyan, Yellow);

      Draw_Window(Win);

      Draw_Text(Win, 1, 1, White, "Library name : " & Handling.To_Wide_String(Name));

   end Component_Info_Draw;
   procedure Enlight_Component_Info_Draw (Name : in String;
					  Line_Pos : in Line_Range;
					  Width    : in Column_Range;
					  Top      : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top+2, 3, Width, Cyan, Window);
   begin

      Initialize(Win, Line_Pos, Top+2, 3, Width, Cyan, Yellow);

      enlight_Window(Win);

      Put(Normal_Mode);

      Draw_Text(Win, 1, 1, White, "Library name : " & Handling.To_Wide_String(Name));

   end Enlight_Component_Info_Draw;
   
   procedure Components_Print(Project : in Project_Record;
			      Component_Index : in Natural; 			       
			      Highlighted : in Natural;
			      lines       : in Line_Range;
			      columns       : in Column_Range;
			      Top           : in Natural := 0) is
      Line_Pos : Line_Range := 7;
   begin

      if Project.Lib_Last > 0 then

	 for Iter in Component_Index+1 .. Project.Lib_Last loop

	    declare
	       Libname : constant String_Access := Project.Libs(Iter).Unit_Name;
	    begin

	       if Libname /= null then

		  if Highlighted /= 0 and Iter = Highlighted then		     

		     Enlight_Component_Info_Draw(Libname.all, Line_Pos, Columns/2-2, top);

		  else

		     Component_Info_Draw(Libname.all, Line_Pos, Columns/2-2, top);

		  end if;
		  Line_Pos := Line_Pos + 3;
		  exit when Line_Pos >= Lines + 7;
	       end if;

	    end;
	    
	 end loop;

      end if;

   end Components_Print;

   
   procedure Archive_Info_Draw (Info : in Project_Info_Record;
				Line_Pos : in Line_Range;
				Width    : in Column_Range;
				Top      : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top + 2, 5, Width, Blue, Window);
      
      
   begin
      Initialize(Win, Line_Pos, Top+2, 5, Width, Blue, Red);

      Draw_Window(Win);

      Draw_Text(Win, 1, 1, White, "Project Name : " & Handling.To_Wide_String(Info.Project_Name.all));
      Draw_Text(Win, 2, 1, White, "Version      : " & Handling.To_Wide_String(Version_io.To_String(Info.Version)));
      Draw_Text(Win, 3, 1, White, "Elapsed      : " & Handling.To_Wide_String(Long_Integer'Image(Long_Integer(Long_Integer(Info.Days) * 86400 + Long_Integer(Info.Elapsed/3600.0))/3600)) & " hours");

   end Archive_Info_Draw;
   procedure Enlight_Archive_Info_Draw (Info : in Project_Info_Record;
					Line_Pos : in Line_Range;
					Width    : in Column_Range;
					Top      : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top + 2, 5, Width, Blue, Window);
      
      
   begin
      
      Initialize(Win, Line_Pos, Top + 2, 5, Width, Blue, Red);
      
      enlight_Window(Win);
      Put(Normal_Mode);
      Draw_Text(Win, 1, 1, White, "Project Name : " & Handling.To_Wide_String(Info.Project_Name.all));
      Draw_Text(Win, 2, 1, White, "Version      : " & Handling.To_Wide_String(Version_Io.To_String(Info.Version)));
      Draw_Text(Win, 3, 1, White, "Elapsed      : " & Handling.To_Wide_String(Long_Integer'Image(Long_Integer(Long_Integer(Info.Days) * 86400 + Long_Integer(Info.Elapsed/3600.0))/3600)) & " hours");
      
   end Enlight_Archive_Info_Draw;
   
   
   procedure Archives_Print(Archives : in Projects_Record;
			    Project_Index : in Natural; 			       
			    Highlighted : in Project_Num_Range;
			    lines       : in Line_Range;
			    columns       : in Column_Range;
			    Top           : in Natural := 0) is
      Curs : Projects_Dll.Cursor;
      First : Projects_Dll.Cursor;
      Line_Pos : Line_Range := 7;
   begin

      if Projects_Dll.Length(Archives.List) /= 0 then

	 Curs := Projects_Dll.First(Archives.List);
	 First := Curs;

	 for I in 1..Project_Index loop
	    Curs := Projects_Dll.Next(Curs);
	 end loop;

	 loop
	    declare
	       The_Project : constant Project.Project_Access := Projects_Dll.Element(Curs);	       
	    begin

	       if The_Project /= null then

		  if The_Project.Project_Num = Highlighted then		     
		     
		     Enlight_Archive_Info_Draw(Project_Info_Record(The_Project.all), Line_Pos, Columns/2-2, top);
		     
		     
		     
		  else
		     
		     Archive_Info_Draw(Project_Info_Record(The_Project.all), Line_Pos, Columns/2-2, top);
		     

		  end if;
		  Line_Pos := Line_Pos + 5;
		  exit when Line_Pos >= Lines;
	       end if;

	    end;
	    if Curs /= Last(Archives.List) then

	       Curs := Projects_Dll.Next(Curs);

	    else
	       exit;
	    end if;

	 end loop;
      end if;
   end Archives_Print;



   
   procedure Project_Info_Draw (Info : in Project_Info_Record;
				Line_Pos : in Line_Range;
				Width    : in Column_Range;
			       Top        : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top + 2, 5, Width, Cyan, Window);
   begin
      Initialize(Win, Line_Pos, Top + 2, 5, Width, Cyan, White);
      Draw_Window(Win);
      Draw_Text(Win, 1, 1, White, "Project Name : " & Handling.To_Wide_String(Info.Project_Name.all));
      Draw_Text(Win, 2, 1, White, "Version      : " & Handling.To_Wide_String(Version_io.To_String(Info.Version)));
      Draw_Text(Win, 3, 1, White, "Project path : " & Handling.To_Wide_String(Info.Source_Path.all));
   end Project_Info_Draw;
   procedure Enlight_Project_Info_Draw (Info : in Project_Info_Record;
					Line_Pos : in Line_Range;
					Width    : in Column_Range;
				       Top        : in Natural := 0) is
      Win : Window_Record := (Line_Pos, Top + 2, 5, Width, Cyan, Window);
   begin
      Initialize(Win, Line_Pos, Top + 2, 5, Width, Cyan, White);
      enlight_Window(Win);
      Put(Normal_Mode);
      Draw_Text(Win, 1, 1, White, "Project Name : " & Handling.To_Wide_String(Info.Project_Name.all));
      Draw_Text(Win, 2, 1, White, "Version      : " & Handling.To_Wide_String(Version_Io.To_String(Info.Version)));
      Draw_Text(Win, 3, 1, White, "Project path : " & Handling.To_Wide_String(Info.Source_Path.all));
   end Enlight_Project_Info_Draw;
   
   
   
   
   procedure Projects_Print(Projects : in Projects_Record;
			    Project_Index : in Natural; 			       
			    Highlighted : in Project_Num_Range;
			    lines       : in Line_Range;
			    columns       : in Column_Range;
			   Top            : in Natural := 0) is
      Curs : Projects_Dll.Cursor;
      First : Projects_Dll.Cursor;
      Line_Pos : Line_Range := 7;
   begin
      if Projects_Dll.Length(Projects.List) /= 0 then
	 Curs := Projects_Dll.First(Projects.List);
	 First := Curs;
	 for I in 1..Project_Index loop
	    Curs := Projects_Dll.Next(Curs);
	 end loop;
	 loop
	    declare
	       The_Project : constant Project.Project_Access := Projects_Dll.Element(Curs);	       
	    begin
	       if The_Project /= null then

		  if The_Project.Project_Num = Highlighted then		     
		     
		     Enlight_Project_Info_Draw(Project_Info_Record(The_Project.all), Line_Pos, Columns/2-2, top);
		     Put(Normal_Mode);
		  else

		     Project_Info_Draw(Project_Info_Record(The_Project.all), Line_Pos, Columns/2-2, top);
		     
		  end if;
		  Line_Pos := Line_Pos + 5;
		  exit when Line_Pos >= Lines;
	       end if;
	       
	    end;
	    if Curs /= Last(Projects.List) then
	       Curs := Projects_Dll.Next(Curs);
	    else
	       exit;
	    end if;
	    
	 end loop;
      end if;
   end Projects_Print;

   
   type Part_Enum is (Creator_None, Creator_Abstract, Creator_Name, Creator_Version, Creator_Author, Creator_Library);
   
   type Part_Record is
      record	 
	 Tag  : String(1..12) := (others => Character'Val(32));
	 Part : Part_Enum := Creator_None;
	 Line : Wide_String(1..78) := (others => Wide_Character ' Val (32)) ; 
	 Last : Natural := 0;
	 Win : Window_Record;
	 Line_Pos : Line_Range := 1;
	 Column_Pos : Column_Range := 13;
      end record;
   
   type Part_Array is array (Part_Enum) of Part_Record;
   
   type Page_Curs_Type is
      record
	 Parts : Part_Array;
	 Part_Cur : Part_Enum := Creator_None;
      end record;
   
   type Win_Set is array (Part_Enum range Creator_Abstract .. Creator_Library) of Window_Record;
   
   
   procedure Cursor_Set(Curs : in out Page_Curs_Type; From : in Project_Record) is
   begin
      
      Wide_Fixed.Move(Handling.To_Wide_String(From.Project_Name.all), Curs.Parts(Creator_Name).Line, Error, Left);
      Curs.Parts(Creator_Name).Last := From.Project_Name'Length;
      
      Wide_Fixed.Move(Handling.To_Wide_String(Version_Io.To_String(From.Version)), Curs.Parts(Creator_Version).Line, Error, Left);
      Curs.Parts(Creator_Version).Last := Version_Io.To_String(From.Version)'Length;
      
      Wide_Fixed.Move(Handling.To_Wide_String(From.Copyright.all), Curs.Parts(Creator_Author).Line, Error, Left);
      Curs.Parts(Creator_Author).Last := From.Copyright'Length;
      
      Curs.Part_Cur := Creator_Library;
      
      
   end Cursor_Set;
   
   
   
   function Initialize (Wins : Win_Set) return Page_Curs_Type is
      Page_Curs : Page_Curs_Type;
   begin
      Page_Curs.Part_Cur := Creator_Abstract;
      Page_Curs.Parts(Creator_Abstract) :=
	(
	 "Abstract  : ",
	 Creator_Abstract,
	 (others => Wide_Character ' Val (32)),
	 0,
	 Wins(Creator_Abstract),
	 1,
	 13
	);
      Page_Curs.Parts(Creator_Name) :=
	(
	 "Name      : ",
	 Creator_Name,	 
	 (others => Wide_Character ' Val (32)),
	 0,
	 Wins(Creator_Name),
	 1,
	 13
	);
      Page_Curs.Parts(Creator_Version) :=
	(
	 "Version   : ",
	 Creator_Version,
	 (others => Wide_Character ' Val (32)),
	 0,
	 Wins(Creator_Version),
	 1,
	 13
	);
      Page_Curs.Parts(Creator_Author) :=
	(
	 "Author    : ",
	 Creator_Author,
	 (others => Wide_Character ' Val (32)),
	 0,
	 Wins(Creator_Author),
	 1,
	 13
	);
      
      Page_Curs.Parts(Creator_Library) :=
	(
	 "Library   : ",
	 Creator_Library,
	 (others => Wide_Character ' Val (32)),
	 0,
	 Wins(Creator_Library),
	 1,
	 13
	);
      
      return Page_Curs;
   end Initialize;
   procedure Page_Print(Page : in Page_Curs_Type) is
   begin
      for Part in Creator_Abstract .. Page.Part_Cur loop
	 Draw_Window(Page.Parts(Part).Win);
	 Draw_Text(Page.Parts(Part).Win,
		   Page.Parts(Part).Line_Pos,
		   1,
		   White,
		   Handling.To_Wide_String(Page.Parts(Part).Tag));
	 Draw_Text(Page.Parts(Part).Win,
		   Page.Parts(Part).Line_Pos,
		   Page.Parts(Part).Column_Pos,
		   White,
		   Page.Parts(Part).Line(1..Page.Parts(Part).Last));
      end loop;
   end Page_Print;
   
   
   procedure List_Print (Project : in Project_Record; Win : in Window_Record ; Lib_Index : in Natural) is
      Line_Pos : Natural := 1;
   begin
      Draw_Window(Win);

      for Lib in Lib_Index + 1 .. Project.Lib_Last loop
	 
	 Draw_Text(Win, Line_Pos, 1, White, 
		   Handling.To_Wide_String(Natural'Image(Lib) & " - " &
					     Project.Libs(Lib).Unit_Name.all));
	 Line_Pos := Line_Pos + 1;
	 exit when Line_Pos > Win.win.H-2;
      end loop;

   end List_Print;
   
   
   type Creator_Record is
      record
	 Curs    : Page_Curs_Type;
	 New_Project : Project.Project_Access;
      end record;
   
   
   
   function To_Project (Creator : in Creator_Record; Num : in Project_Num_Range) return Project.Project_Access is
      
      
   begin
      Creator.New_Project.Project_Num := Num;
      Creator.New_Project.Project_Name :=
	new String '
	(
	 Handling.To_String(Creator.Curs.Parts(Creator_Name).Line(1..Creator.Curs.Parts(Creator_Name).Last))
	);
      Creator.New_Project.Copyright :=
	new String '
	(
	 Handling.To_String(Creator.Curs.Parts(Creator_Author).Line(1..Creator.Curs.Parts(Creator_Author).Last))
	);
      Creator.New_Project.Version := Version_Io.To_Version(From => Handling.To_String(Creator.Curs.Parts(Creator_Version).Line(1..Creator.Curs.Parts(Creator_Version).Last)));
      
      return Creator.New_Project;
   end To_Project;
   
   procedure Write_To_Append (Filename : in String; Line : in String) is
      File : File_Type;
   begin
      Open(File, Append_File, Filename);      
      Put_Line(File, Line);
      Close(File);
   end Write_To_Append;
   
   procedure List_Save (Project : in Project_Record) is
      Filename : constant String := Project.Project_Name.all & "-" & Version_Io.To_String(Project.Version) & '/' & "Project.lst";
      File : File_Type;
   begin
      if Project.Lib_Last /= 0 then
	 Create(File, Out_File, filename);
	 Put_Line(File, Project.Project_Name.all);
	 Put_Line(File, Project.Project_Name.all);
	 for lib in 1..Project.Lib_Last loop
	    Put_Line(File, Project.Libs(Lib).Unit_Name.all);
	 end loop;
	 Close(File);
      end if;
   end List_Save;
   
   
   procedure Make_Project (Creator : in Creator_Record) is
      
      Info : constant Project_Info_Record := Project_Info_record(Creator.New_Project.all);
      
   begin
      --  mkdir $project-$2
      Make_Dir(Creator.New_Project.Project_Name.all & "-" & Version_Io.To_String(Creator.New_Project.Version));
      
      --  cd $project-$2
      Change_Dir(Creator.New_Project.Project_Name.all & "-" & Version_Io.To_String(Creator.New_Project.Version));            
      
      --  gunzip ./.Ada/Ada_Empty-6.4.0.tar.gz
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("../.Ada/Ada_Empty-6.4.0.tar.gz");	 
      Begin
	 Spawn(Locate_Exec_On_Path("gunzip").all, Args.all, Success);
      End;
      
      
      
      --  tar -x -v -f ./.Ada/Ada_Empty-6.4.0.tar
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("-x -v -f ../.Ada/Ada_Empty-6.4.0.tar");	 
      Begin
	 Spawn(Locate_Exec_On_Path("tar").all, Args.all, Success);
      End;
      
      
      
      --  gzip ./.Ada/Ada_Empty-6.4.0.tar
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("../.Ada/Ada_Empty-6.4.0.tar");	 
      Begin
	 Spawn(Locate_Exec_On_Path("gzip").all, Args.all, Success);
      End;
      

      
      --  make pre
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("pre tail head");	 
      Begin
	 Spawn(Locate_Exec_On_Path("make").all, Args.all, Success);
      End;
      
      

      Write_To_Append("Abstract.txt", Handling.To_String(Creator.Curs.Parts(Creator_Abstract).Line(1..Creator.Curs.Parts(Creator_Abstract).Last)));
      Write_To_Append("Project.lst", Handling.To_String(Creator.Curs.Parts(Creator_Name).Line(1..Creator.Curs.Parts(Creator_Name).Last)));
      Write_To_Append("Project.lst", Handling.To_String(Creator.Curs.Parts(Creator_Name).Line(1..Creator.Curs.Parts(Creator_Name).Last)));

      for Lib in 1..Creator.New_Project.Lib_Last loop
	 Write_To_Append("Project.lst", Creator.New_Project.Libs(Lib).Unit_Name.all);
      end loop;
      Write_To_Append("Versions.lst", Version_Io.To_String(Creator.New_Project.Version));

      Write_To_Append("Copyright", Handling.To_String(Creator.Curs.Parts(Creator_Author).Line(1..Creator.Curs.Parts(Creator_Author).Last)));
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("prepare project all");	 
      Begin
	 Spawn(Locate_Exec_On_Path("make").all, Args.all, Success);
      End;
      


      Change_Dir("..");
      

   end Make_Project;
   
   -- upgrade of 1 the Revision of Version
   function Update (Info : Project_Info_Record; Version : in Sky.Versions.Version_Type; Sessions : in out Session_Array; Logs : in out Log_Array) return Integer is
      Errno : Integer := 0;
   begin
      
      
      
      Put_Line("version " & Info.Project_Name.all & ' ' &
		 Integer'Image(Info.Version.Major_Version)(2..Integer'Image(Info.Version.Major_Version)'Last) & ' ' &
		 Integer'Image(Info.Version.Minor_Version)(2..Integer'Image(Info.Version.Minor_Version)'Last) & ' ' &
		 Integer'Image(Info.Version.Revision_Version)(2..Integer'Image(Info.Version.Revision_Version)'Last) & ' ' &
		 Info.Version.Release_Version);
      
      Put_Line("Upgrade " & Integer'Image(Version.Major_Version)(2..Integer'Image(Version.Major_Version)'Last) & ' ' &
		 Integer'Image(Version.Minor_Version)(2..Integer'Image(Version.Minor_Version)'Last) & ' ' &
		 Integer'Image(Version.Revision_Version)(2..Integer'Image(Version.Revision_Version)'Last) & ' ' &
		 Version.Release_Version);

      Text_Io.Put_Line("cp -r " & Info.Project_Name.all & '-' & To_String(Info.Version) & ' ' & Info.Project_Name.all & '-' & To_String(Version) & Character'Val(0));      
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("-r " & Info.Project_Name.all & '-' & To_String(Info.Version) & ' ' & Info.Project_Name.all & '-' & To_String(Version));
      Begin
	 Spawn(Locate_Exec_On_Path("cp").all, Args.all, Success);
      End;

      
      Text_Io.Put_Line("cp " & Info.Project_Name.all & '-' & To_String(Info.Version) & "/log/Todo.lst" & '-' & To_String(Info.Version) & ' ' &  Info.Project_Name.all & '-' & To_String(Version) & "/log/Todo.lst" & '-' & To_String(Version) & Character'Val(0));

      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Info.Version) & "/log/Todo.lst" & '-' & To_String(Info.Version) & ' ' &  Info.Project_Name.all & '-' & To_String(Version) & "/log/Todo.lst" & '-' & To_String(Version));	 
      Begin
	 Spawn(Locate_Exec_On_Path("cp").all, Args.all, Success);
      End;
      
      Text_Io.Put_Line("emacs " & Info.Project_Name.all & '-' & To_String(Version) & "/log/Todo.lst" & '-' & To_String(Version) & Character'Val(0));
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & "/log/Todo.lst" & '-' & To_String(Version));	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      
      declare
	 File : Text_Io.File_Type;
      begin
	 Create(File, Out_File, Info.Project_Name.all & '-' & To_String(Version) & "/Versions.lst");
	 Put_Line(File, To_String(Version));
	 Close(File);
      end;
      
      Open(App.Sessions(Sky.Logs.Project));
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & '/' & Info.Project_Name.all & '/' & Info.Project_Name.all & "/src/" & Info.Project_Name.all & ".adb");	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      Close(Sessions(Sky.Logs.Project));
      
      

      Statistic(Sessions, Logs);

      

      

      Open(App.Sessions(Howto));

      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & '/' & "HOWTO");	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      

      Close(Sessions(Howto));
      
      

      Statistic(Sessions, Logs);

      
      Open(App.Sessions(HISTORY));
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & '/' & "HISTORY");	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      
      Close(Sessions(HISTORY));
      Statistic(Sessions, Logs);
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & "/doc/man/man1/" & Info.Project_Name.all & ".1.gz");	 
      Begin
	 Spawn(Locate_Exec_On_Path("gunzip").all, Args.all, Success);
      End;
      
      
      

      
      Open(App.Sessions(Manpage));
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & "/doc/man/man1/" & Info.Project_Name.all & ".1");	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & "/doc/man/man1/" & Info.Project_Name.all & ".1");	 
      Begin
	 Spawn(Locate_Exec_On_Path("gzip").all, Args.all, Success);
      End;
      
      
      
      Close(Sessions(Manpage));
      

      
      Statistic(Sessions, Logs);
	    
      
      
      Open(App.Sessions(Makefile));
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List(Info.Project_Name.all & '-' & To_String(Version) & "/Makefile");	 
      Begin
	 Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Success);
      End;
      
      
	   
      
      Close(Sessions(Makefile));
      
      
      Statistic(Sessions, Logs);
      
      
      Declare
	 Success : Boolean;
	 Args : Argument_List_Access := Argument_String_To_List("-C " & Info.Project_Name.all & '-' & To_String(Version) & " head tail msg all enlight-sources clean");	 
      Begin
	 Spawn(Locate_Exec_On_Path("make").all, Args.all, Success);
      End;
      
	   
      
      
      
      
      
      
      
      
       return errno;
								      
								      
								      
      
   end Update;
   
   
   procedure Remove_Name(Project : in out Project_Record; Lib_Num : in Natural) is
   begin
      if Lib_Num = 0 then
	 return;
      end if;
      if Lib_Num < Project.Lib_Last then
	 for Iter in Lib_Num..Project.Lib_Last - 1 loop
	    Project.Libs(Iter) := Project.Libs(Iter + 1);
	    Project.Libs(Project.Lib_last) := Null_Component;
	    Project.Lib_Last := Project.Lib_Last - 1;
	 end loop;
      else
	 Project.Libs(Project.Lib_last) := Null_Component;
	 Project.Lib_Last := Project.Lib_Last - 1;
      end if;
   end Remove_Name;
   
   type Project_Manager_Record is new Creator_Record;

   
   
   task body Manager_Process is
      
      Creator_Abs : Window_Record := (1, 1, 3, App.Columns/2, Blue, Window);
      Creator_N     : Window_Record := (4, 1, 3, App.Columns/2, Blue, Window);
      Creator_Ver   : Window_Record := (App.Lines-8, 1, 3, App.Columns/2, Blue, Window);
      Creator_Auth   : Window_Record := (App.Lines-5, 1, 3, App.Columns/2, Blue, Window);
      Creator_List     : Window_Record := (7, 1, App.Lines-15, App.Columns/2, Blue, Window);
      Creator_Lib  : Window_Record := (App.Lines-2, 1, 3, App.Columns/2, Blue, Window);
      
      Manager_Abs : Window_Record := (1, 1, 3, App.Columns/2, Blue, Window);
      Manager_N     : Window_Record := (4, 1, 3, App.Columns/2, Blue, Window);
      Manager_Ver   : Window_Record := (App.Lines-8, 1, 3, App.Columns/2, Blue, Window);
      Manager_Auth   : Window_Record := (App.Lines-5, 1, 3, App.Columns/2, Blue, Window);
      Manager_List     : Window_Record := (7, 1, App.Lines-15, App.Columns/2, Blue, Window);
      Manager_Cmd  : Window_Record := (App.Lines-2, 1, 3, App.Columns/2, Blue, Window);
      
      
      Project_Info : Project_Info_Record;
      
      
      
      Project_Header   : Window_Record := (1, 1, 5, App.Columns/2, Blue, Window);
      
      Project_Selector : Window_Record := (6, 1, App.Lines - 5, App.Columns/2, Blue, Window);
      
      
      Components_Win   : Window_Record := (6,  1, (App.Lines - 8)/2, App.Columns/2, Blue, Window);
      
      Make_Win         : Window_Record := (6 + (App.Lines - 8)/2+1, 1, (App.Lines - 8)/2, App.Columns/2, White, Window);
      
      
      
      
      
      type Step_Enum is (Prj_None, Prj_Selector, Prj_Archives, Prj_Components, Prj_Creator, Prj_Manager, Prj_Editor, Prj_Lib_Remove, Prj_Upgrade, Prj_help);
      
      type Upgrade_Cmd_Enum is (None, Remove, Add, Make, Upgrade, quit, help);
      
      
      function Value (Line : in String) return Upgrade_Cmd_Enum is
	 
      begin
	 if Index(Line, " ") /= 0 then
	    return Upgrade_Cmd_Enum'Value(Line(Line'First..Index(Line, " ")-1));
	 else
	    return Upgrade_Cmd_Enum'Value(Line);
	 end if;
	 
      exception
	 when Constraint_Error =>
	    return None;
      end Value;
      
      
      type Lang_Enum is (En, Fr, C);
      Lang      : Lang_Enum := En;
      
      Color     : constant String := Character'Val(27) & "[01;36m";
      Normal    : constant String := Character'Val(27) & "[00m";
      
      
      En_Msg : constant String := "Hello";
      Fr_Msg : constant String := "Salut";
      C_Msg  : constant String := "Error";
      
      User_Val  : constant String := "USER";
      Lang_Val  : constant String := "LANG";
      
      The_World : constant String := "world";
      The_Msg   : constant String := "Hello";   
      
      
      The_User  : String_Access;
      The_Lang  : String_Access;
      
      
      procedure Draw_Header(Project : in Project_Record; Project_Header : in Window_Record) is
      begin
	 if The_User /= null then            
	    case Lang is
	       when Fr =>
		  Draw_Text(Project_Header, 1, 1, White, Wide_Fixed."*"((The_User'Length+Fr_Msg'Length+10), '-'));
		  Draw_Text(Project_Header, 2, 1, White, Handling.To_Wide_String("-- " & Fr_Msg & ' ' & Color & The_User.all & Normal & " ! " & " --"));
		  Draw_Text(Project_Header, 3, 1, White, Wide_Fixed."*"((The_User'Length+fr_Msg'Length+10), '-'));
		  
		  Draw_Text(Project_Header, 1, (The_User'Length+fr_Msg'Length+10)+2, White, "Project name  : " & Handling.To_Wide_String(Project.Project_Name.all));
		  Draw_Text(Project_Header, 2, (The_User'Length+fr_Msg'Length+10)+2, White, "Elapsed       : " & Handling.To_Wide_String(Long_Integer'Image(Long_Integer(Project.Days) * 86400 + Long_Integer(Project.Elapsed) / 3600)) & " hours");
		  
	       when En =>
		  Draw_Text(Project_Header, 1, 1, White, Wide_Fixed."*"((The_User'Length+En_Msg'Length+10), '-'));
		  Draw_Text(Project_Header, 2, 1, White, Handling.To_Wide_String("-- " & En_Msg & ' ' & Color & The_User.all & Normal & " ! " & " --"));
		  Draw_Text(Project_Header, 3, 1, White, Wide_Fixed."*"((The_User'Length+En_Msg'Length+10), '-'));
		  Draw_Text(Project_Header, 1, (The_User'Length+En_Msg'Length+10)+2, White, "Project name  : " & Handling.To_Wide_String(Project.Project_Name.all));
		  Draw_Text(Project_Header, 2, (The_User'Length+En_Msg'Length+10)+2, White, "Elapsed       : " & Handling.To_Wide_String(Long_Integer'Image((Long_Integer(Project.Days) * 86400 + Long_Integer(Project.Elapsed)) / 3600)) & " hours");
		  
	       when C =>
		  Draw_Text(Project_Header, 1, 1, White, Wide_Fixed."*"((The_User'Length+C_Msg'Length+10), '-'));
		  Draw_Text(Project_Header, 2, 1, White, Handling.To_Wide_String("-- " & C_Msg & ' ' & Color & The_User.all & Normal & " ! " & " --"));
		  Draw_Text(Project_Header, 3, 1, White, Wide_Fixed."*"((The_User'Length+C_Msg'Length+10), '-'));
		  Draw_Text(Project_Header, 1, (The_User'Length+C_Msg'Length+10)+2, White, "Project name  : " & Handling.To_Wide_String(Project.Project_Name.all));
		  Draw_Text(Project_Header, 2, (The_User'Length+C_Msg'Length+10)+2, White, "Elapsed       : " & Handling.To_Wide_String(Long_Integer'Image((Long_Integer(Project.Days) * 86400 + Long_Integer(Project.Elapsed)) / 3600)) & " hours");
		  
	    end case;
	 else
	    Draw_Text(Project_Header, 1, 1, White, Wide_Fixed."*"((The_World'Length+The_Msg'Length+10), '-'));
	    Draw_Text(Project_Header, 2, 1, White, Handling.To_Wide_String("-- " & The_Msg & ' ' & Color & The_world & Normal & " ! " & " --"));
	    Draw_Text(Project_Header, 3, 1, White, Wide_Fixed."*"((The_World'Length+The_Msg'Length+10), '-'));
	    Draw_Text(Project_Header, 1, (The_World'Length+The_Msg'Length+10)+2, White, "Project name  : " & Handling.To_Wide_String(Project.Project_Name.all));
	    Draw_Text(Project_Header, 2, (The_World'Length+The_Msg'Length+10)+2, White, "Elapsed       : " & Handling.To_Wide_String(Long_Integer'Image((Long_Integer(Project.Days) * 86400 + Long_Integer(Project.Elapsed)) / 3600)) & " hours");

	 end if;
      end Draw_Header;
      
      
      
      
      Arch_Num : Project_Num_Range := 1;
      Arch_Index : Project_Num_Range := 0;
      Arch_Count : Natural := 0;
      
      Arch_Curs : Projects_Dll.Cursor;
      
      Component_Cur : Natural := 0;
      Component_Index : Natural := 0;

      
      
      Step_Cur : Step_Enum := Prj_Selector;
      
      Project_Index : Project_Num_Range := 0;
      
      Project_Count : Natural := 0;    
      
      
      Curs : Projects_Dll.Cursor;
      
      Creator : Creator_Record;
      Project_Manager : Project_Manager_Record;
      
      Upgraded : Project.Project_Access;
      
      
      Lib_Index : Natural := 0;
      
      Lib_Name  : Wide_String(1..3) := (others => Wide_Character'Val(32));
      Lib_Name_Last : Natural := 0;
      
      Line_Remove : Boolean := False;
      Line_Num : Natural := 0;
      
      
      
      Cmd_line : Wide_String(1..256) := (others => Wide_Character'Val(0));
      Cmd_Last : Natural := 0;
      Cmd : Upgrade_Cmd_Enum := None;
      
      
      Make_On_Bin      : constant String_Access := Locate_Exec_On_Path("make");
      Args             : Argument_List_Access := Argument_String_To_List("all");
      Outfilename      : constant String := "sky-spawn.stdout";
      File_Set         : Wide_String_Set(65536);
      File_Index       : Natural := 0;
      Line_Last        : Natural := 0;
      Success : Boolean := False;
      Total_Lines      : Natural := 0;
      
      
      Gnose_Context : constant access Gnos_Projects.Projects_Context_Record := 
	new gnos_Projects.Projects_Context_Record;
      
      Project_Context : constant access Agnose_Projects.Projects_Context_Record := 
	new Agnose_Projects.Projects_Context_Record;

      Top : Natural := 0;
      
      Pid : Process_Id;
      
      
   begin
      Initialize(Creator_Abs, 1, 1, 3, App.Columns/2, Blue, White);
      Initialize(Creator_N, 4, 1, 3, App.Columns/2, Blue, White);
      Initialize(Creator_Ver, App.Lines-8, 1, 3, App.Columns/2, Blue, White);
      Initialize(Creator_Auth, App.Lines-5, 1, 3, App.Columns/2, Blue, White);
      Initialize(Creator_List, 7, 1, App.Lines-15, App.Columns/2, Blue, White);
      Initialize(Creator_Lib, App.Lines-2, 1, 3, App.Columns/2, Blue, white);
      
      Initialize(Manager_Abs, 1, 1, 3, App.Columns/2, Blue, White);
      Initialize(Manager_N, 4, 1, 3, App.Columns/2, Blue, White);
      Initialize(Manager_Ver, App.Lines-8, 1, 3, App.Columns/2, Blue, White);
      Initialize(Manager_Auth, App.Lines-5, 1, 3, App.Columns/2, Blue, White);
      Initialize(Manager_List, 7, 1, App.Lines-15, App.Columns/2, Blue, White);
      Initialize(Manager_cmd, App.Lines-2, 1, 3, App.Columns/2, Blue, white);
      
      
      Initialize(Project_Header, 1, 1, 5, App.Columns/2, Blue, White);
      Initialize(Project_Selector, 6, 1, App.Lines - 5, App.Columns/2, Blue, White);
      
      Initialize(Components_Win, 6,  1, (App.Lines - 8)/2+1, App.Columns/2, Blue, Red);
      
      Initialize(Make_Win, 6 + (App.Lines - 8)/2+1, 1, (App.Lines - 8)/2, App.Columns/2, White, Magenta);
      
      
      declare
	 
	 Project_Num : Project_Num_Range := 0;
      begin
	 
	 The_User := Getenv(User_Val);
	 The_Lang := Getenv(Lang_Val);
	 if The_Lang /= null then
	    begin
	       Lang := Lang_Enum'Value(The_Lang.all(1..2));
	    exception
	       when Constraint_Error =>
		  W_Io.Put_Line("Uncknow language ; ");
	    end;
	 end if;	       
	 
	 
	 
     Main_Loop:
	 loop
	    loop
	       select
		  
		  accept Halt do
		     null;
		  end Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
	       or
		  accept Wake_Up(Top : in Natural) do
		     Manager_Process.Top := Top;
		     Initialize(Creator_Abs, 1, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_N, 4, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_Ver, App.Lines-8, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_Auth, App.Lines-5, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_List, 7, Top+1, App.Lines-15, App.Columns/2, Blue, White);
		     Initialize(Creator_Lib, App.Lines-2, Top+1, 3, App.Columns/2, Blue, white);
		     
		     Initialize(Manager_Abs, 1, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_N, 4, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_Ver, App.Lines-8, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_Auth, App.Lines-5, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_List, 7, 1, App.Lines-15, App.Columns/2, Blue, White);
		     Initialize(Manager_cmd, App.Lines-2, Top+1, 3, App.Columns/2, Blue, white);
		     
		     
		     Initialize(Project_Header, 1, Top+1, 5, App.Columns/2, Blue, White);
		     Initialize(Project_Selector, 6, Top+1, App.Lines - 5, App.Columns/2, Blue, White);
		     
		     Initialize(Components_Win, 6,  Top+1, (App.Lines - 8)/2+1, App.Columns/2, Blue, Red);
		     
		     Initialize(Make_Win, 6 + (App.Lines - 8)/2+1, Top+1, (App.Lines - 8)/2, App.Columns/2, White, Magenta);
		     
		     -----------------------------------------------
		     --Errno := System("clear" & Character'Val(0));
		     -----------------------------------------------
		     if Curs = Projects_Dll.No_Element then
			Project_Num := 0;
			if Projects_Dll.Length(App.Projects.List) = 1 then
			   Curs := first(App.Projects.List);			   
			   Project_Num := First_Element(App.Projects.List).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			elsif Projects_Dll.Length(App.Projects.List) > 1 then
			   Curs := First(App.Projects.List);			   
			   Project_Num := First_Element(App.Projects.List).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			end if;
			
		     end if;
		     if Arch_Curs = Projects_Dll.No_Element then
			if Length(App.Archives.List) > 0 then
			   Arch_Curs := First(App.Archives.List);
			end if;
		     end if;
		     
		  end Wake_Up;
		  case Step_Cur is
		     when Prj_Archives =>
			Draw_Window(Project_Header);
			Draw_Window(Project_Selector);
			if Length(App.Archives.List) /= 0 then
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			end if;
		     when Prj_Components =>
			null;
		     when Prj_Selector =>
			Draw_Window(Project_Header);
			if Project_Manager.New_Project /= null then
			   Draw_Header(Project_Manager.New_Project.all, Project_Header);
			else
			   Draw_Header(Null_Project, Project_Header);
			end if;
			
			Draw_Window(Project_Selector);
			Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
		     when Prj_Creator =>
			Draw_Window(Creator_Abs);
			Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
			Draw_Window(Creator_N);
			Draw_Text(Creator_N, 1, 1, White, "Name      : ");
			Draw_Window(Creator_Ver);
			Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
			Draw_Window(Creator_Auth);
			Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
			Draw_Window(Creator_List);
			if Creator.New_Project /= null then
			   List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			end if;
			Draw_Window(Creator_Lib);
			Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
			Page_Print(Creator.Curs);
			Draw_Text(
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				  White,
				  Creator.Curs.Parts
				    (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				 );
		     when Prj_Lib_Remove =>
			Draw_Window(Creator_Abs);
			Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
			Draw_Window(Creator_N);
			Draw_Text(Creator_N, 1, 1, White, "Name      : ");
			Draw_Window(Creator_Ver);
			Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
			Draw_Window(Creator_Auth);
			Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
			Draw_Window(Creator_List);
			
			List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			
			Draw_Window(Creator_Lib);
			Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
			Page_Print(Creator.Curs);
			Draw_Text(
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				  Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				  White,
				  Creator.Curs.Parts
				    (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				 );
			if Lib_Name_Last > 0 then
			   Draw_Text(Creator_Lib, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			else
			   Draw_Text(Creator_Lib, 1, 1, White, "Line num : ");
			end if;
		     when Prj_Manager =>
			Draw_Window(Manager_Abs);
			Draw_Text(Manager_Abs, 1, 1, White, "Abstract  : ");
			Draw_Window(Manager_N);
			Draw_Text(Manager_N, 1, 1, White, "Name      : " & Handling.To_Wide_String(Project_Manager.New_Project.Project_Name.all));
			Draw_Window(Manager_Ver);
			Draw_Text(Manager_Ver, 1, 1, White, "Version   : " & Handling.To_Wide_String(Version_Io.To_String(Project_Manager.New_Project.Version)));
			Draw_Window(Manager_Auth);
			Draw_Text(Manager_Auth, 1, 1, White, "Author    : " & Handling.To_Wide_String(Project_Manager.New_Project.Copyright.all));
			Draw_Window(Manager_List);
			
			List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);
			
			Draw_Window(Manager_cmd);
			Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			
			Page_Print(Project_Manager.Curs);
			Draw_Text(
				  Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Win,
				  Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Line_Pos,
				  Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Column_Pos,
				  White,
				  Project_Manager.Curs.Parts
				    (Project_Manager.Curs.Part_Cur).Line(1..Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Last)
				 );
			if Line_Remove then
			   if Lib_Name_Last > 0 then
			      Draw_Text(Manager_cmd, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			   else
			      Draw_Text(Manager_cmd, 1, 1, White, "Line num : ");
			   end if;
			elsif Cmd_Last > 0 then
			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			else
			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			end if;
		     when Prj_Upgrade =>
			Draw_Window(Components_Win);
			Component_Cur := 0;
			Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, Top);
			
			Draw_Window(Make_Win);
			if not Print_Set(File_Set, Make_Win, File_Index, Total_Lines) then
			   raise Program_Error;
			end if;
			if Cmd_Last > 0 then

			   Draw_Window(Manager_cmd);
			   
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));

			else

			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");

			end if;
		     when others =>
			null;
		  end case;
		  exit;
	       or
		  accept Refresh do
		     null;
		  end Refresh;

	       end select;
	    end loop;

	    loop
	       select
		  accept Halt;
		  exit Main_Loop;
	       or
		  accept Suspend;
		  exit;
	       or
		  accept Wake_Up(Top : in Natural) do
		     Manager_Process.Top := Top;
		     Initialize(Creator_Abs, 1, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_N, 4, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_Ver, App.Lines-8, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_Auth, App.Lines-5, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Creator_List, 7, Top+1, App.Lines-15, App.Columns/2, Blue, White);
		     Initialize(Creator_Lib, App.Lines-2, Top+1, 3, App.Columns/2, Blue, white);
		     
		     Initialize(Manager_Abs, 1, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_N, 4, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_Ver, App.Lines-8, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_Auth, App.Lines-5, Top+1, 3, App.Columns/2, Blue, White);
		     Initialize(Manager_List, 7, 1, App.Lines-15, App.Columns/2, Blue, White);
		     Initialize(Manager_cmd, App.Lines-2, Top+1, 3, App.Columns/2, Blue, white);
		     
		     
		     Initialize(Project_Header, 1, Top+1, 5, App.Columns/2, Blue, White);
		     Initialize(Project_Selector, 6, Top+1, App.Lines - 5, App.Columns/2, Blue, White);
		     
		     Initialize(Components_Win, 6,  Top+1, (App.Lines - 8)/2+1, App.Columns/2, Blue, Red);
		     
		     Initialize(Make_Win, 6 + (App.Lines - 8)/2+1, Top+1, (App.Lines - 8)/2, App.Columns/2, White, Magenta);
		     		     -----------------------------------------------
		     --Errno := System("clear" & Character'Val(0));
		     -----------------------------------------------
		     if Curs = Projects_Dll.No_Element then
			Project_Num := 0;
			if Projects_Dll.Length(App.Projects.List) = 1 then
			   Curs := first(App.Projects.List);			   
			   Project_Num := First_Element(App.Projects.List).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			elsif Projects_Dll.Length(App.Projects.List) > 1 then
			   Curs := First(App.Projects.List);			   
			   Project_Num := First_Element(App.Projects.List).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			end if;
			
		     end if;
		     if Arch_Curs = Projects_Dll.No_Element then
			if Length(App.Archives.List) > 0 then
			   Arch_Curs := First(App.Archives.List);
			end if;
		     end if;

		  end Wake_Up;
		  
	       or
		  when Step_Cur = Prj_Selector =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(9) =>

				 Project_Num := Projects_Dll.Element(Curs).Project_Num;
				 
				 if Project_Num = 0 then
				    null; -- open archives;
				    Arch_Count := 0;
				    Arch_Index := 0;
				    if Length(App.Archives.List) > 0 then
				       Arch_Curs := First(App.Archives.List);
				    end if;
				    Step_Cur := Prj_Archives;				    
				 else		
				    
				    Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				    
				    
				    
				    
				    if Project_Manager.New_Project /= null then
				       
				       
				       Gnos_Projects.Initialize(Gnose_Context.all, Projects_Params);   
				       
				       Project_Context.Project := new Agnose_Projects.Project_Record_Type(With_Ada);
				       
				       
				       
				       
				       Agnose_Projects.Load_project(Project_Context.Project.all, Project_Manager.New_Project.Project_Name.all & 
								      '-' & Version_Io.To_String(Project_Manager.New_Project.Version));   
				       
				       
				       
				       
				       
				       Set_Start_Time;
				       Calendar.Split(Clock, Gnos.Versions.Start_Date.Year, Gnos.Versions.Start_Date.Month, Gnos.Versions.Start_Date.Day, Gnos.Versions.Start_Date.Hours);
				       
				       Open(App.Sessions(Sky.Logs.Project));
				       
				       Project_Manager.Curs := Initialize((Manager_Abs,
									   Manager_N,
									   Manager_Ver,
									   Manager_Auth,
									   Manager_Cmd));
				       
				       Step_Cur := Prj_Upgrade;
				       Component_Cur := 0;
				       Project.Open(Project_Manager.New_Project.all, clock);
				       -----------------------------------------------
				       --Errno := System("clear" & Character'Val(0));
				       -----------------------------------------------
				       
				    end if;
				 end if;
			      when Wide_Character'Val(10) => 
				 Project_Num := Element(Curs).Project_Num;
				 if Project_Num = 0 then
				    ----------------------------------------------
				    --Errno := System("clear" & Character'Val(0));
				    ----------------------------------------------
				    Creator.Curs := Initialize((Creator_Abs,
								Creator_N,
								Creator_Ver,
								Creator_Auth,
								Creator_Lib));
				    Project_Free(Creator.New_Project);
				    Creator.New_Project := new Project_Record(256);
				    
				    Step_Cur := Prj_Creator;
				    
				 else
				    Project_Manager.Curs := Initialize((Manager_Abs,
									Manager_N,
									Manager_Ver,
									Manager_Auth,
									Manager_Cmd));
				    
				    
				    Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				    
				    
				    Gnos_Projects.Initialize(Gnose_Context.all, Projects_Params);   
				    
				    Project_Context.Project := new Agnose_Projects.Project_Record_Type(With_Ada);
				    
				    Agnose_Projects.Load_project(Project_Context.Project.all, Project_Manager.New_Project.Project_Name.all & 
								   '-' & Version_Io.To_String(Project_Manager.New_Project.Version));
				    
				    
				    
				    Set_Start_Time;
				    Calendar.Split(Clock, Gnos.Versions.Start_Date.Year, Gnos.Versions.Start_Date.Month, Gnos.Versions.Start_Date.Day, Gnos.Versions.Start_Date.Hours);
				    
				    Open(App.Sessions(Sky.Logs.Project));
				    
				    Project.Open(Project_Manager.New_Project.all, Clock);
				    Cursor_Set(Project_Manager.Curs, From => Project_Manager.New_Project.all);
				    if Project_Manager.New_Project /= null then
				       Step_Cur := Prj_Manager;
				       
				    end if;
				 end if;
			      when Wide_Character'Val(127) =>
				 
				 if App.Archives.Projects_Num + 1 <= Project_Num_Range'Last then
				    if Project_Num /= 0 then
				       null;-- move archive to projects.List;
				       Arch_Curs := Projects_Dll.last(App.Archives.List);
				       
				       declare
					  El : constant Project.Project_Access := Get_Project(App.Projects, Project_Num);
				       begin
					  if El.Project_Name /= null and then
					    El.Project_Name'Length /= 0 then
					     
					     Errno := System("mv " & El.Project_Name.all & 
							       '-' & Versions.Version_Io.To_String(El.Version) &
							       " Archives/" & Character'Val(0));
					     if Errno = 0 then
						App.Archives.Projects_Num := App.Archives.Projects_Num + 1;
						El.Project_Num := App.Archives.Projects_Num;
						Projects_Dll.Append(App.Archives.List, El);
						Delete(App.Projects.List, Curs);
						if Length(App.Projects.List) > 0 then
						   Curs := First(App.Projects.List);
						   Project_Num := First_Element(App.Projects.List).Project_Num;
						   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
						   Project_Count := 0;
						   Project_Index := 0;
						end if;
						Projects_Manager.Archives_Save(App.Archives, "Archives/Projects.lst");
						Projects_Manager.Manager_Save(App.Archives, "Projects.lst");
					     end if;
					  end if;
				       end;
				    end if;				    
				 end if;
			      when others =>
				 null;
			   end case;
			else
			   null;
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_None =>
			   null;
			when Prj_Archives =>
			   Draw_Window(Project_Header);
			   Draw_Window(Project_Selector);
			   if Length(App.Archives.List) /= 0 then
			      Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			   end if;
			when prj_Components =>
			   
			   null;
			   
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when Prj_Creator =>
			   Draw_Window(Creator_Abs);
			   Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Creator_N);
			   Draw_Text(Creator_N, 1, 1, White, "Name      : ");
			   Draw_Window(Creator_Ver);
			   Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
			   Draw_Window(Creator_Auth);
			   Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
			   Draw_Window(Creator_List);
			   Draw_Window(Creator_Lib);
			   Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
			   Page_Print(Creator.Curs);
			   Draw_Text(
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				     White,
				     Creator.Curs.Parts
				       (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				    );
			   
			when Prj_Manager =>
			   Draw_Window(Manager_Abs);
			   Draw_Text(Manager_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Manager_N);
			   Draw_Text(Manager_N, 1, 1, White, "Name      : " & Handling.To_Wide_String(Project_Manager.New_Project.Project_Name.all));
			   Draw_Window(Manager_Ver);
			   Draw_Text(Manager_Ver, 1, 1, White, "Version   : " & Handling.To_Wide_String(Version_Io.To_String(Project_Manager.New_Project.Version)));
			   Draw_Window(Manager_Auth);
			   Draw_Text(Manager_Auth, 1, 1, White, "Author    : " & Handling.To_Wide_String(Project_Manager.New_Project.Copyright.all));
			   Draw_Window(Manager_List);
			   
			   List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);
			   
			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   
			   Page_Print(Project_Manager.Curs);
			   Draw_Text(
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Win,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Line_Pos,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Column_Pos,
				     White,
				     Project_Manager.Curs.Parts
				       (Project_Manager.Curs.Part_Cur).Line(1..Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Last)
				    );
			   if Line_Remove then
			      if Lib_Name_Last > 0 then
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			      else
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : ");
			      end if;
			   elsif Cmd_Last > 0 then
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			   else
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   end if;
			when Prj_Upgrade =>
			   Draw_Window(Components_Win);

			   Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, top);

			   Draw_Window(Make_Win);
			   if Cmd_Last > 0 then

			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));

			   else

			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");

			   end if;
			when others =>
			   null;
		     end case;
	       or
		  when Step_Cur = Prj_creator =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(1) =>
				 Creator.Curs := Initialize((Creator_Abs,
							     Creator_N,
							     Creator_Ver,
							     Creator_Auth,
							     Creator_Lib));
				 Project_Free(Creator.New_Project);
				 Creator.New_Project := new Project_Record(256);
				 
			      when Wide_Character'Val(5) => 
				 Step_Cur := Prj_Lib_Remove;
			      when Wide_Character'Val(10) => 
				 if Creator.Curs.Part_Cur = Creator_Library then
				    
				    if Creator.Curs.Parts(Creator.Curs.Part_Cur).Last /= 0 then

				       Lib_Add
					 (Creator.New_Project.all, 
					  Handling.To_String
					    (Creator.Curs.Parts(Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)));

				       Creator.Curs.Parts(Creator.Curs.Part_Cur).Line := (others => ' ');
				       Creator.Curs.Parts(Creator.Curs.Part_Cur).Last := 0;
				    end if;
				 else
				    Creator.Curs.Part_Cur := Part_Enum'Succ(Creator.Curs.Part_Cur);
				 end if;
			      when Wide_Character'Val(12) =>
				 ----------------------------------------------
				 --Errno := System("clear" & Character'Val(0));
				 null;
				 ----------------------------------------------
			      when Wide_Character'Val(18) => 
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;
				 

				 
				 
				 ----------------------------------------------
				 --Errno := System("clear" & Character'Val(0));
				 ----------------------------------------------
				 if Curs = Projects_Dll.No_Element then
				    Project_Num := 0;
				    if Projects_Dll.Length(App.Projects.List) = 1 then
				       Curs := first(App.Projects.List);			   
				       Project_Num := First_Element(App.Projects.List).Project_Num;
				       Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				    elsif Projects_Dll.Length(App.Projects.List) > 1 then
				       Curs := First(App.Projects.List);			   
				       Project_Num := First_Element(App.Projects.List).Project_Num;
				       Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				    end if;
				    
				 end if;
				 
				 
				 
				 
				 if Arch_Curs = Projects_Dll.No_Element then
				    if Length(App.Archives.List) > 0 then
				       Arch_Curs := First(App.Archives.List);
				    end if;
				 end if;
				 
			      when Wide_Character'Val(20) =>
				 
				 
				 Creator.New_Project := To_Project(Creator, App.Projects.Projects_Num + 1);
				 
				 
				 Make_Project(Creator);
				 
				 Curs := Last(App.Projects.List);
				 
				 Insert(App.Projects.List, Curs, Creator.New_Project);

				 App.Projects.Projects_Num := App.Projects.Projects_Num + 1;

				 Project_Num := App.Projects.Projects_Num;

				 

				 
				 Creator.Curs := Initialize((Creator_Abs,
							     Creator_N,
							     Creator_Ver,
							     Creator_Auth,
							     Creator_Lib));
				 
				 Project_Num := First_Element(App.Projects.List).Project_Num;
				 Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				 Curs := Previous(Curs);
				 
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

			      when Wide_Character'Val(127) =>
				 
				 if Creator.Curs.Parts(Creator.Curs.Part_Cur).Last > 0 then
				    Creator.Curs.Parts(Creator.Curs.Part_Cur).Line(Creator.Curs.Parts(Creator.Curs.Part_Cur).Last) := Wide_Character'Val(32);
				    Creator.Curs.Parts(Creator.Curs.Part_Cur).Last := 
				      Creator.Curs.Parts(Creator.Curs.Part_Cur).Last - 1;
				 else
				    Text_Io.Put(Character'Val(7));
				 end if;
			      when others =>
				 null;
			   end case;
			   case Step_Cur is
			      when Prj_None =>
				 null;
			      when Prj_Selector =>
				 Draw_Window(Project_Header);
				 
				 if Project_Manager.New_Project /= null then
				    Draw_Header(Project_Manager.New_Project.all, Project_Header);
				 end if;
				 
				 Draw_Window(Project_Selector);
				 Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			      when Prj_Creator =>
				 Draw_Window(Creator_Abs);
				 Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
				 Draw_Window(Creator_N);
				 Draw_Text(Creator_N, 1, 1, White, "Name      : ");
				 Draw_Window(Creator_Ver);
				 Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
				 Draw_Window(Creator_Auth);
				 Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
				 Draw_Window(Creator_List);
				 
				 List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
				 
				 Draw_Window(Creator_Lib);
				 Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
				 Page_Print(Creator.Curs);
				 Draw_Text(
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
					   White,
					   Creator.Curs.Parts
					     (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
					  );				 
			      when Prj_Lib_Remove =>
				 if Lib_Name_Last > 0 then
				    Draw_Text(Creator_Lib, 1, 1, White, "Line num  : " & Lib_Name(1..Lib_Name_last));
				 else
				    Draw_Text(Creator_Lib, 1, 1, White, "Line num  : ");
				 end if;
			      when others =>
				 null;
			   end case;
			elsif Creator.Curs.Parts(Creator.Curs.Part_Cur).Last + 1 <= 
			  Creator.Curs.Parts(Creator.Curs.Part_Cur).Line'Length then
			   
			   Creator.Curs.Parts(Creator.Curs.Part_Cur).Line(Creator.Curs.Parts(Creator.Curs.Part_Cur).Last + 1) := Wchar;
			   Creator.Curs.Parts(Creator.Curs.Part_Cur).Last :=
			     Creator.Curs.Parts(Creator.Curs.Part_Cur).Last + 1;
			   Draw_Window(Creator.Curs.Parts(Creator.Curs.Part_Cur).Win);
			   Page_Print(Creator.Curs);
			   Draw_Text(
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				     White,
				     Creator.Curs.Parts
				       (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				    );
			   
			else
			   Text_Io.Put(Character ' Val (7));
			end if;
		     end Receive;
		     
	       or
		  when Step_Cur = Prj_Manager =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then			   
			   case Wchar is
			      when Wide_Character'Val(10) => 
				 begin
				    Cmd := Value(Handling.To_String(Cmd_Line(Cmd_Line'First..Cmd_Last)));
				    case Cmd is
				       when None =>
					  null;
				       when Help =>
					  null;
				       when Make =>
					  if Cmd_Last = 4 then
					     Errno := System("make" & Character'Val(0));
					  else
					     Change_Dir(Project_Manager.New_Project.Project_Name.all & "-" & Version_Io.To_String(Project_Manager.New_Project.Version));            
					     
					     Free(Args);
					     Args := Argument_String_To_List(Handling.To_String(Cmd_Line(6..Cmd_Last)) & Character'Val(0));
					     
					     -------------------------------------------------------------------
					     
					     -- Spawn(Make_On_Bin.all, Args.all, Outfilename, Success, errno);
					     
					     
					     -- replaced by
					     
					     Pid := Non_Blocking_Spawn(Make_On_Bin.all, Args.all, Outfilename);
					     
					     Wait_Process(Pid, Success);
					     
					     -------------------------------------------------------------------
					     
					     
					     
					     --Errno := System("make " &  Handling.To_String(Cmd_Line(6..Cmd_Last)) & Character'Val(0));
					     Load_File(File_Set, Outfilename, Line_Last);
					     Change_Dir("..");
					     
					     
					  end if;
				       when Add =>
					  if Cmd_Last > 4 then
					     Lib_Add(Project_Manager.New_Project.all, Handling.To_String(Cmd_Line(5..Cmd_Last)));
					     List_Save(Project_Manager.New_Project.all);
					  else
					     Put(Character'Val(7));
					  end if;
				       when Remove =>
					  begin
					     Line_Num := Natural'Value(Handling.To_String(Cmd_Line(Cmd_Line'First+7..Cmd_Last)));
					     
					     Remove_Name(Project_Manager.New_Project.all, Line_Num);
					     List_Save(Project_Manager.New_Project.all);
					  exception
					     when others =>
						null;
					  end;
				       when Upgrade =>
					  
					  
					  
					  ----------------------------------------------------------------------------------------------------------
					  --
					  --
					  --
					  --
					  --
					  --
					  
					  Upgraded := new Project_Record ' (Project_Manager.New_project.all);
					  
					  

					  
					  
					  Text_Io.Put_Line("Update version...");
					  
					  
					  
					  Text_Io.Put_Line("-- check for New Specifications...");
					  
					  Versions.New_Description := Environment.Check_Modification_To_New_Description(Project_Context.all);
					  

					  Put_Line("New Description : " & Boolean'Image(Versions.New_Description));					  
					  
					  
					  Text_Io.Put_Line("-- check for New Implementation...");
					  
					  Versions.New_Action := Environment.Check_Modification_To_New_Action(Project_Context.all);

					  Put_Line("New Implementation : " & Boolean'Image(Versions.New_Action));					  
					  
					  Text_Io.Put_Line("-- Version Update...");
					  -- update Version !
					  if Project_Context.Project.Project_Prefix /= null then
					     
					     Text_Io.Put_Line("Project prefix : " & Project_Context.Project.Project_Prefix.all);
					     
					     Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
					     declare
						
						Node_Name : access String := new String ' (Project_Context.Project.Project_Name(Project_Context.Project.Project_Prefix'Length+1..Project_Context.Project.Project_Name'Last));

					     begin
						
						
						if Versions.New_Description then
						   Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
						   
						   Text_Io.Put_Line("-- Editting minor version...");
						   
						   Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
						   Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						   
						   for File_Id in Specifications_Filenames_Search.File_Content'Range loop
						      Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
						      Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						   end loop;
						   
						   
						end if;
						
						if Versions.New_Action then
						   Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
						   
						   Text_Io.Put_Line("-- Editting revision version...");
						   
						   Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
						   Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						   for File_Id in Implementations_Filenames_Search.File_Content'Range loop
						      Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
						      Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						   end loop;
						   
						   
						end if;
						
						if Versions.New_Action or Versions.New_Description then
						   Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
						   
						   Text_Io.Put_Line("-- Editting major version...");
						   To_Major(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						   To_Minor(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						   To_Revision(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						   
						else
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Text_Io.Put_Line("-- Editting release version...");
						   To_Release(Project_Context.Project.Project_Name.all,
							      Project_Context.Project.Project_Prefix.all);
						end if;
						Versions.New_Action := False;	       
						Versions.New_Description := False;
						
					     end;
					  else

					     Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
					     declare
						
						Node_Name : access String := new String ' ("");

					     begin
						
						
						if Versions.New_Description then
						   Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
						   
						   
						   Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
						   Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						   
						   for File_Id in Specifications_Filenames_Search.File_Content'Range loop
						      Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
						      Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						   end loop;
						   
						   
						end if;
						
						if Versions.New_Action then
						   Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
						   Upgraded.Version := 
						     Version_Update(Upgraded.Version);
						   Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
						   
						   Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
						   Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						   for File_Id in Implementations_Filenames_Search.File_Content'Range loop
						      Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
						      Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						   end loop;
						   
						   
						end if;
						
						if Versions.New_Action or Versions.New_Description then
						   Text_Io.Put_Line("-- Editting revision version...");
						   To_Revision(Project_Context.Project.Project_Name.all, "");
						   Text_Io.Put_Line("-- Editting minor version...");
						   To_Minor(Project_Context.Project.Project_Name.all, "");
						   Text_Io.Put_Line("-- Editting major version...");
						   To_Major(Project_Context.Project.Project_Name.all, "");
						else
						   if Environment.Version_Update /= 0 then
						      raise Program_Error;
						   end if;		  
						   Text_Io.Put_Line("-- Editting release version...");
						   To_Release(Project_Context.Project.Project_Name.all,
							      "");
						end if;
						Versions.New_Action := False;	       
						Versions.New_Description := False;
						
					     end;
					     
					  end if;
					  
					  --
					  --
					  --
					  --
					  --
					  --
					  --
					  -----------------------------------------------------------------------------------------------------------
					  

					  --  --
					  --  --
					  --  --
					  --  --
					  --  --
					  --  --
					  --  --
					  --  -----------------------------------------------------------------------------------------------------------
					  
					  
					  
					  
					  
					  
					  if Upgraded /= null then
					     
					     App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
					     Upgraded.Project_Num := App.Projects.Projects_Num;
					     Curs := Last(App.Projects.List);
					     Projects_DLL.Delete(App.Projects.List, Curs);
					     if Upgraded.Version.Revision_Version + 1 < Sky.Versions.Revision_Version_Type'Last then
						Upgraded.Version.Revision_Version := 
						  Upgraded.Version.Revision_Version + 1;
					     end if;
					     
					     Append(App.Projects.List, Upgraded);
					     Projects_Manager.Manager_Save(App.Projects, "./Projects.lst");
					     if Update(Get_Info(App.Projects, Project_Manager.New_Project.Project_Num), Upgraded.Version, App.Sessions, App.Logs) = 0 then
						
						Clear(App.Projects.List);
						Projects_Manager.Manager_Load(App.Projects, "./Projects.lst");
					     end if;
					     
					     App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
					     Append(App.Projects.List, 
						    new Project_Record ' (Null_Project));
					     
					  end if;
					  Project.Close(Project_Manager.New_Project.all, Clock);
					  Project_Count := 0;
					  Project_Index := 0;
					  Curs := First(App.Projects.List);			   
					  Project_Num := First_Element(App.Projects.List).Project_Num;
					  Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
					  
					  
					  
					  
					  Close(App.Sessions(Sky.Logs.Project));
				       
					  Statistic(App.Sessions, App.Logs);
					  Step_Cur := Prj_Selector;

				       when Quit =>
					  null;
				    end case;
				    
				    Cmd_Last := 0;
				    Cmd_Line := (others => Wide_Character'Val(32));
				 exception
				    when others =>
				       null;
				 end;
			      when Wide_Character'Val(18) => 
				 Project.Close(Project_Manager.New_Project.all, Clock);
				 
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

				 Draw_Window(Project_Header);
				 
				 
				 Draw_Window(Project_Selector);
				 Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);

			      when Wide_Character'Val(21) =>
				 
				 ----------------------------------------------------------------------------------------------------------
				 --
				 --
				 --
				 --
				 --
				 --
				 
				 
				 

				 
				 Upgraded := new Project_Record ' (Project_Manager.New_project.all);

				 Text_Io.Put_Line("Update version...");
				 
				 
				 
				 Text_Io.Put_Line("-- check for New Specifications...");
				 
				 Versions.New_Description := Environment.Check_Modification_To_New_Description(Project_Context.all);
				 


				 
				 
				 Text_Io.Put_Line("-- check for New Implementation...");
				 
				 Versions.New_Action := Environment.Check_Modification_To_New_Action(Project_Context.all);

				 Text_Io.Put_Line("-- Version Update...");
				 -- update Version !
				 if Project_Context.Project.Project_Prefix /= null then

				    Text_Io.Put_Line("Project prefix : " & Project_Context.Project.Project_Prefix.all);
				    Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
				    declare

				       Node_Name : access String := new String ' (Project_Context.Project.Project_Name(Project_Context.Project.Project_Prefix'Length+1..Project_Context.Project.Project_Name'Last));

				    begin
				       
				       
				       if Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  Text_Io.Put_Line("-- Editting minor version...");
					  
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version), Node_Name.all);
					  
					  for File_Id in Specifications_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  Text_Io.Put_Line("-- Editting revision version...");
					  
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version), Node_Name.all);
					  for File_Id in Implementations_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action or Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  Text_Io.Put_Line("-- Editting major version...");
					  To_Major(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  To_Minor(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  To_Revision(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  
				       else
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  Text_Io.Put_Line("-- Editting release version...");
					  To_Release(Project_Context.Project.Project_Name.all,
						     Project_Context.Project.Project_Prefix.all);
				       end if;
				       Versions.New_Action := False;	       
				       Versions.New_Description := False;
				       
				    end;
				 else

				    Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 

				    declare
				       
				       Node_Name : access String := new String ' ("");

				    begin
				       
				       
				       if Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  --  if Environment.Version_Update /= 0 then
					  --     raise Program_Error;
					  --  end if;		  
					  
					  
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version), Node_Name.all);
					  
					  for File_Id in Specifications_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  --  if Environment.Version_Update /= 0 then
					  --     raise Program_Error;
					  --  end if;		  		  
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version), Node_Name.all);
					  for File_Id in Implementations_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Project_Manager.New_Project.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action or Versions.New_Description then
					  Text_Io.Put_Line("-- Editting revision version...");
					  To_Revision(Project_Context.Project.Project_Name.all, "");
					  Text_Io.Put_Line("-- Editting minor version...");
					  To_Minor(Project_Context.Project.Project_Name.all, "");
					  Text_Io.Put_Line("-- Editting major version...");
					  To_Major(Project_Context.Project.Project_Name.all, "");
				       else
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  --  if Environment.Version_Update /= 0 then
					  --     raise Program_Error;
					  --  end if;		  
					  Text_Io.Put_Line("-- Editting release version...");
					  To_Release(Project_Context.Project.Project_Name.all,
						     "");
				       end if;
				       Versions.New_Action := False;	       
				       Versions.New_Description := False;
				       
				    end;
				    
				 end if;

				 
				 
				 

				 
				 
				 
				 --
				 --
				 --
				 --
				 --
				 --
				 --
				 -----------------------------------------------------------------------------------------------------------
				 
				 
				 if Upgraded /= null then
				    
				    App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
				    Upgraded.Project_Num := App.Projects.Projects_Num;
				    Curs := Last(App.Projects.List);
				    Projects_DLL.Delete(App.Projects.List, Curs);
				    if Upgraded.Version.Revision_Version + 1 < Sky.Versions.Revision_Version_Type'Last then
				       Upgraded.Version.Revision_Version := 
					 Upgraded.Version.Revision_Version + 1;
				    end if;
				    
				    Append(App.Projects.List, Upgraded);
				    Projects_Manager.Manager_Save(App.Projects, "./Projects.lst");
				    if Update(Get_Info(App.Projects, Project_Manager.New_Project.Project_Num), Upgraded.Version, App.Sessions, App.Logs) = 0 then
				       
				       Clear(App.Projects.List);
				       Projects_Manager.Manager_Load(App.Projects, "./Projects.lst");
				    end if;
				    
				    App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
				    Append(App.Projects.List, 
					   new Project_Record ' (Null_Project));
				    
				 end if;
				 Project.Close(Project_Manager.New_Project.all, Clock);
				 Project_Count := 0;
				 Project_Index := 0;
				 Curs := First(App.Projects.List);			   
				 Project_Num := First_Element(App.Projects.List).Project_Num;
				 Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				 

				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

				 
			      when Wide_Character'Val(127) =>
				 if Cmd_Last > 0 then
				    Cmd_Line(Cmd_Last) :=  Wide_Character'Val(32);
				    Cmd_Last := Cmd_Last - 1;
				 else
				    Put(Character'Val(7));
				 end if;
			      when others =>
				 null;
			   end case;
			elsif Cmd_Last < Cmd_Line'Length then
			   Cmd_Line(Cmd_Last + 1) := Wchar;
			   Cmd_Last := Cmd_Last + 1;
			else
			   Put(Character'Val(7));
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when Prj_Creator =>
			   Draw_Window(Creator_Abs);
			   Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Creator_N);
			   Draw_Text(Creator_N, 1, 1, White, "Name      : ");
			   Draw_Window(Creator_Ver);
			   Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
			   Draw_Window(Creator_Auth);
			   Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
			   Draw_Window(Creator_List);
			   Draw_Window(Creator_Lib);
			   Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
			   Page_Print(Creator.Curs);
			   Draw_Text(
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				     White,
				     Creator.Curs.Parts
				       (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				    );
			when Prj_Lib_Remove =>
			   Draw_Window(Creator_Abs);
			   Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Creator_N);
			   Draw_Text(Creator_N, 1, 1, White, "Name      : ");
			   Draw_Window(Creator_Ver);
			   Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
			   Draw_Window(Creator_Auth);
			   Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
			   Draw_Window(Creator_List);
			   
			   List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			   
			   Draw_Window(Creator_Lib);
			   Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
			   Page_Print(Creator.Curs);
			   Draw_Text(
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
				     Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
				     White,
				     Creator.Curs.Parts
				       (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
				    );
			   if Lib_Name_Last > 0 then
			      Draw_Text(Creator_Lib, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			   else
			      Draw_Text(Creator_Lib, 1, 1, White, "Line num : ");
			   end if;
			when Prj_Manager =>
			   Draw_Window(Manager_Abs);
			   Draw_Text(Manager_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Manager_N);
			   Draw_Text(Manager_N, 1, 1, White, "Name      : " & Handling.To_Wide_String(Project_Manager.New_Project.Project_Name.all));
			   Draw_Window(Manager_Ver);
			   Draw_Text(Manager_Ver, 1, 1, White, "Version   : " & Handling.To_Wide_String(Version_Io.To_String(Project_Manager.New_Project.Version)));
			   Draw_Window(Manager_Auth);
			   Draw_Text(Manager_Auth, 1, 1, White, "Author    : " & Handling.To_Wide_String(Project_Manager.New_Project.Copyright.all));
			   
			   Draw_Window(Manager_List);
			   
			   List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);
			   
			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   Page_Print(Project_Manager.Curs);
			   Draw_Text(
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Win,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Line_Pos,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Column_Pos,
				     White,
				     Project_Manager.Curs.Parts
				       (Project_Manager.Curs.Part_Cur).Line(1..Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Last)
				    );
			   if Line_Remove then
			      if Lib_Name_Last > 0 then
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			      else
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : ");
			      end if;
			   elsif Cmd_Last > 0 then
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			   else
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   end if;
			when Prj_Upgrade =>
			   Draw_Window(Components_Win);
			   Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, top);
			   
			   Draw_Window(Make_Win);
			   if Cmd_Last > 0 then
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			   else
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   end if;
			when others =>
			   null;
		     end case;
		     
	       or
		  when Step_Cur = Prj_archives =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(9) =>			
				 if Length(App.Archives.List) > 0 then
				    Arch_Num := Element(Arch_Curs).Project_Num;
				    Put_Line("Arch_Num : " & Integer'Image(Integer(Arch_Num)));
				    
				    if App.Projects.Projects_Num + 1 <= Project_Num_Range'Last then
				       
				       if Arch_Num /= 0 then
					  null;-- move archive to projects.List;
					  Curs := Projects_Dll.First(App.Projects.List);
					  Project_Count := 0;
					  Project_Index := 0;
					  declare
					     El : constant Project.Project_Access := Get_Project(App.Archives, Arch_Num);
					  begin
					     if El.Project_Name /= null and then
					       El.Project_Name'Length /= 0 then
						
						Errno := System("mv Archives/" & El.Project_Name.all & '-' & 
								  Versions.Version_Io.To_String(El.Version) & 
								  " ." & Character'Val(0));
						
						
						
						
						
						
						if Errno = 0 then
						   El.Project_Num := App.Projects.Projects_Num + 1;
						   App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
						   
						   Insert(App.Projects.List, Curs, El, Curs);
						   Delete(App.Archives.List, Arch_Curs);
						   Projects_Manager.Archives_Save(App.Archives, "Archives/Projects.lst");
						   Projects_Manager.Manager_Save(App.Archives, "Projects.lst");
						end if;
					     end if;
					  end;
				       end if;				    
				    end if;
				 end if;
				 if Length(App.Archives.List) > 0 then
				    Arch_Curs := First(App.Archives.List);
				 end if;
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

			      when Wide_Character'Val(10) => 
				 null; -- nothing to be done.
			      when Wide_Character'Val(18) => 
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;
			      when Wide_Character'Val(127) =>
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;
			      when others =>
				 null;
			   end case;
			else
			   null;
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_Archives =>
			   Draw_Window(Project_Selector);
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			when Prj_None =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when others =>
			   null;
		     end case;
		     
	       or
		  when Step_Cur = Prj_Components =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(10) => 
				 null;
			      when Wide_Character'Val(18) => 
				 Step_Cur := Prj_Manager;
			      when others =>
				 null;
			   end case;
			else
			   null;
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_None =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when others =>
			   null;
		     end case;
		     
		     
	       or
		  when Step_Cur = Prj_Editor =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(10) => 
				 null;
			      when Wide_Character'Val(18) => 
				 Step_Cur := Prj_Manager;
			      when others =>
				 null;
			   end case;
			else
			   null;
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_None =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when others =>
			   null;
		     end case;
	       or
		  when Step_Cur = Prj_Lib_Remove =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(10) => 
				 declare
				    Line_Num : Natural := 0;
				 begin
				    Line_Num := Natural'Value(Handling.To_String(Lib_Name(1..Lib_Name_Last)));
				    Remove_Name(Creator.New_Project.all, Line_Num);
				    
				 exception
				    when others =>
				       null;
				 end;
				 Step_Cur := Prj_Creator;
				 Lib_Name := (others => Wide_Character'Val(32));
				 Lib_Name_Last := 0;
			      when Wide_Character'Val(12) =>
				 -----------------------------------------------
				 --Errno := System("clear" & Character'Val(0));
				 -----------------------------------------------
				 null;
			      when Wide_Character'Val(18) => 
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

			      when Wide_Character'Val(127) => 
				 if Lib_Name_Last > 0 then
				    Lib_Name(Lib_Name_Last) := Wide_Character'Val(32);
				    Lib_Name_Last := Lib_Name_Last - 1;
				 else
				    Put(Character'Val(7));
				 end if;
			      when others =>
				 null;
			   end case;
			   case Step_Cur is
			      when Prj_None =>
				 null;
			      when Prj_Selector =>
				 Draw_Window(Project_Header);
				 if Project_Manager.New_Project /= null then
				    Draw_Header(Project_Manager.New_Project.all, Project_Header);
				 end if;
				 Draw_Window(Project_Selector);
				 Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			      when Prj_Creator =>
				 Draw_Window(Creator_Abs);
				 Draw_Text(Creator_Abs, 1, 1, White, "Abstract  : ");
				 Draw_Window(Creator_N);
				 Draw_Text(Creator_N, 1, 1, White, "Name      : ");
				 Draw_Window(Creator_Ver);
				 Draw_Text(Creator_Ver, 1, 1, White, "Version   : ");
				 Draw_Window(Creator_Auth);
				 Draw_Text(Creator_Auth, 1, 1, White, "Author    : ");
				 Draw_Window(Creator_List);
				 
				 List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
				 
				 Draw_Window(Creator_Lib);
				 Draw_Text(Creator_Lib, 1, 1, White, "Library   : ");
				 Page_Print(Creator.Curs);
				 Draw_Text(
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Win,
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Line_Pos,
					   Creator.Curs.Parts(Creator.Curs.Part_Cur).Column_Pos,
					   White,
					   Creator.Curs.Parts
					     (Creator.Curs.Part_Cur).Line(1..Creator.Curs.Parts(Creator.Curs.Part_Cur).Last)
					  );				 
			      when Prj_Lib_Remove =>
				 Draw_Window(Creator_Lib);
				 if Lib_Name_Last > 0 then
				    Draw_Text(Creator_Lib, 1, 1, White, "Line num  : " & Lib_Name(1..Lib_Name_last));
				 else
				    Draw_Text(Creator_Lib, 1, 1, White, "Line num  : ");
				 end if;

			      when others =>
				 null;
			   end case;
			elsif Lib_Name_Last < Lib_Name'Length then
			   Draw_Window(Creator_Lib);
			   if Lib_Name_Last > 0 then
			      Draw_Text(Creator_Lib, 1, 1, White, "Line num  : " & Lib_Name(1..Lib_Name_last));
			   else
			      Draw_Text(Creator_Lib, 1, 1, White, "Line num  : ");
			   end if;
			   
			   Lib_Name(Lib_Name_Last + 1) := Wchar;
			   Lib_Name_Last := Lib_Name_Last + 1;
			else
			   Put(Character'Val(7));
			   
			end if;
		     end Receive;
	       or
		  when Step_Cur = Prj_help =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is
			      when Wide_Character'Val(10) => 
				 null;
			      when Wide_Character'Val(18) => 
				 Step_Cur := Prj_Manager;
			      when others =>
				 null;
			   end case;
			else
			   null;
			end if;
		     end Receive;
		     case Step_Cur is
			when Prj_None =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when others =>
			   null;
		     end case;
	       or
		  when Step_Cur = Prj_Upgrade =>
		     accept Receive(Wchar : in Wide_Character) do
			if Is_Control(Wchar) then
			   case Wchar is			      
			      when Wide_Character'Val(10) => 
				 if Cmd_Last = 0 then
				    if Component_Cur > 0 then
				       Args := Argument_String_To_List(Project_Manager.New_Project.Project_Name.all & '-' & Versions.Version_Io.To_String(Project_Manager.New_Project.Version)
									 & Directory_Separator & 
									 Project_Manager.New_Project.Project_Name.all & Directory_Separator & 
									 Project_Manager.New_Project.Libs(Component_Cur).Unit_Name.all & Directory_Separator);
				       
				       
				       
				       
				       Spawn(Locate_Exec_On_Path("emacs").all, Args.all, Outfilename, Success, errno);
				       
				       
				       Put(Ansi_Tty_Control.Clear_Screen);
				       --Errno := System("clear" & Character'Val(0));
				       if App.Desktop(2) /= null then
					  case App.Desktop(2).Name is
					     when None =>
						null;
					     when Log =>
						Log_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when MoneyBox =>
						Moneybox_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Activity =>
						Activity_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Stocks =>
						Stocks_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Bank =>
						bank_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Planning =>
						planning_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Library =>
						Library_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Manager =>
						null;
					     when Contacts =>
						Contacts_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Menu =>
						null;
					     when Organizer =>
						organizer_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when Editor=>
						Editor_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
					     when others =>
						null;
					  end case;
					  
				       end if;
				       Load_File(File_Set, Outfilename, Line_Last);
				    end if;
				 else
				    
				    begin
				       Cmd := Value(Handling.To_String(Cmd_Line(Cmd_Line'First..Cmd_Last)));
				       case Cmd is
					  when None =>
					     null;
					  when Help =>
					     null;
					  when Make =>
					     if Cmd_Last = 4 then
						Free(Args);
						Args := new Argument_List(1..0);
						
						-------------------------------------------------------------------
						
						-- Spawn(Make_On_Bin.all, Args.all, Outfilename, Success, errno);
						
						-- replaced by
						
						Pid := Non_Blocking_Spawn(Make_On_Bin.all, Args.all, Outfilename);
						
						Wait_Process(Pid, Success);
						
						--------------------------------------------------------------------
						Load_File(File_Set, Outfilename, Line_Last);
						
						
					     elsif Cmd_Last > 5 then
						
						
						begin
						   Change_Dir(Project_Manager.New_Project.Project_Name.all & "-" & Version_Io.To_String(Project_manager.New_Project.Version));            
						   
						   Free(Args);
						   Args := Argument_String_To_List(Handling.To_String(Cmd_Line(6..Cmd_Last)) & Character'Val(0));
						   
						   -----------------------------------------------------------------
						   
						   -- Spawn(Make_On_Bin.all, Args.all, Outfilename, Success, errno);						   						   
						   -- replaced by
						
						   Pid := Non_Blocking_Spawn(Make_On_Bin.all, Args.all, Outfilename);
						   
						   Wait_Process(Pid, Success);
						   
						   --------------------------------------------------------------------
						   Load_File(File_Set, Outfilename, Line_Last);
						   Change_Dir("..");
						   
						   
						   
						exception
						   when Name_Error =>
						      Put_Line("name error");
						   when Constraint_Error =>
						      Put_Line("constraint error");
						   when others =>
						      Put_Line("others error");
						end ;
						
					     end if;
					  when Add =>
					     if Cmd_Last > 4 then
						Lib_Add(Project_Manager.New_Project.all, Handling.To_String(Cmd_Line(5..Cmd_Last)));
						List_Save(Project_Manager.New_Project.all);
					     else
						Put(Character'Val(7));
					     end if;
					  when Remove =>
					     begin
						Line_Num := Natural'Value(Handling.To_String(Cmd_Line(Cmd_Line'First+7..Cmd_Last)));
						
						Remove_Name(Project_Manager.New_Project.all, Line_Num);
						List_Save(Project_Manager.New_Project.all);
					     exception
						when others =>
						   null;
					     end;
					  when Upgrade =>
					     

					     
					     
					     ----------------------------------------------------------------------------------------------------------
					     --
					     --
					     --
					     --
					     --
					     --
					     
					     Upgraded := new Project_Record ' (Project_Manager.New_project.all);
					     
					     
					     
					     
					     
					     Text_Io.Put_Line("Update version...");
					     
					     
					     
					     Text_Io.Put_Line("-- check for New Specifications...");
					     
					     Versions.New_Description := Environment.Check_Modification_To_New_Description(Project_Context.all);
					     

					     
					     
					     Text_Io.Put_Line("-- check for New Implementation...");
					     
					     Versions.New_Action := Environment.Check_Modification_To_New_Action(Project_Context.all);
					     
					     Text_Io.Put_Line("-- Version Update...");
					     -- update Version !
					     if Project_Context.Project.Project_Prefix /= null then
						Text_Io.Put_Line("Project prefix : " & Project_Context.Project.Project_Prefix.all);
						Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
						declare

						   Node_Name : access String := new String ' (Project_Context.Project.Project_Name(Project_Context.Project.Project_Prefix'Length+1..Project_Context.Project.Project_Name'Last));

						begin
						   
						   
						   if Versions.New_Description then
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Text_Io.Put_Line("-- Editting minor version...");
						      
						      Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
						      Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						      
						      for File_Id in Specifications_Filenames_Search.File_Content'Range loop
							 Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
							 Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						      end loop;
						      
						      
						   end if;
						   
						   if Versions.New_Action then
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Text_Io.Put_Line("-- Editting revision version...");
						      
						      Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
						      Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						      for File_Id in Implementations_Filenames_Search.File_Content'Range loop
							 Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
							 Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						      end loop;
						      
						      
						   end if;
						   
						   if Versions.New_Action or Versions.New_Description then
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Text_Io.Put_Line("-- Editting major version...");
						      To_Major(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						      To_Minor(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						      To_Revision(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
						      
						   else
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Text_Io.Put_Line("-- Editting release version...");
						      To_Release(Project_Context.Project.Project_Name.all,
								 Project_Context.Project.Project_Prefix.all);
						   end if;
						   Versions.New_Action := False;	       
						   Versions.New_Description := False;
						   
						end;
					     else
						Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
						declare
						   
						   Node_Name : access String := new String ' ("");

						begin
						   
						   
						   if Versions.New_Description then
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      
						      
						      Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
						      Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						      
						      for File_Id in Specifications_Filenames_Search.File_Content'Range loop
							 Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
							 Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						      end loop;
						      
						      
						   end if;
						   
						   if Versions.New_Action then
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
						      Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
						      for File_Id in Implementations_Filenames_Search.File_Content'Range loop
							 Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
							 Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
						      end loop;
						      
						      
						   end if;
						   
						   if Versions.New_Action or Versions.New_Description then
						      Text_Io.Put_Line("-- Editting revision version...");
						      To_Revision(Project_Context.Project.Project_Name.all, "");
						      Text_Io.Put_Line("-- Editting minor version...");
						      To_Minor(Project_Context.Project.Project_Name.all, "");
						      Text_Io.Put_Line("-- Editting major version...");
						      To_Major(Project_Context.Project.Project_Name.all, "");
						   else
						      Upgraded.Version := 
							Version_Update(Upgraded.Version);
						      Text_Io.Put_Line("-- Editting release version...");
						      To_Release(Project_Context.Project.Project_Name.all,
								 "");
						   end if;
						   Versions.New_Action := False;	       
						   Versions.New_Description := False;
						   
						end;
						
					     end if;

					     
					     
					     

					     
					     
					     
					     --
					     --
					     --
					     --
					     --
					     --
					     --
					     -----------------------------------------------------------------------------------------------------------
					     
					     
					     
					     if Upgraded /= null then
						
						App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
						Upgraded.Project_Num := App.Projects.Projects_Num;
						Curs := Last(App.Projects.List);
						Projects_DLL.Delete(App.Projects.List, Curs);
						
						Append(App.Projects.List, Upgraded);
						Projects_Manager.Manager_Save(App.Projects, "./Projects.lst");
						if Update(Get_Info(App.Projects, Project_Manager.New_Project.Project_Num), Upgraded.Version, App.Sessions, App.Logs) = 0 then
						   
						   Clear(App.Projects.List);
						   Projects_Manager.Manager_Load(App.Projects, "./Projects.lst");
						end if;
						
						App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
						Append(App.Projects.List, 
						       new Project_Record ' (Null_Project));
						
					     end if;
					     Project.Close(Project_Manager.New_Project.all, Clock);
					     Project_Count := 0;
					     Project_Index := 0;
					     Curs := First(App.Projects.List);			   
					     Project_Num := First_Element(App.Projects.List).Project_Num;
					     Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
					     
					     Step_Cur := Prj_Selector;
					     Close(App.Sessions(Sky.Logs.Project));
				       
					     Statistic(App.Sessions, App.Logs);
					  when Quit =>
					     null;
				       end case;
				       
				       
				       
				       
				       Cmd_Last := 0;
				       Cmd_Line := (others => Wide_Character'Val(32));
				    exception
				       when others =>
					  null;
				    end;
				 end if;
				 if Cmd_Last = 0 then
				    
				    null;
				    
				 else
				    
				    null;
				    
				 end if;
			      when Wide_Character'Val(12) => 
				 ----------------------------------------------
				 --Errno := System("clear" & Character'Val(0));
				 ---------------------------------------------
				 null;
			      when Wide_Character'Val(18) => 
				 Step_Cur := Prj_manager;
				 Cursor_Set(Project_Manager.Curs, From => Project_Manager.New_Project.all);
				 Project.Close(Project_Manager.New_Project.all, Clock);
				 
			      when Wide_Character'Val(21) =>
				 
				 
				 
				 ----------------------------------------------------------------------------------------------------------
				 --
				 --
				 --
				 --
				 --
				 --
				 Upgraded := new Project_Record ' (Project_Manager.New_project.all);
				 
				 Text_Io.Put_Line("Update version...");
				 
				 
				 
				 Text_Io.Put_Line("-- check for New Specifications...");
				 
				 Versions.New_Description := Environment.Check_Modification_To_New_Description(Project_Context.all);
				 
				 
				 
				 Text_Io.Put_Line("-- check for New Implementation...");
				 
				 Versions.New_Action := Environment.Check_Modification_To_New_Action(Project_Context.all);
				 
				 Text_Io.Put_Line("-- Version Update...");
				 -- update Version !
				 if Project_Context.Project.Project_Prefix /= null then
				    
				    Text_Io.Put_Line("Project prefix : " & Project_Context.Project.Project_Prefix.all);
				    
				    Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
				    declare
				       
				       Node_Name : access String := new String ' (Project_Context.Project.Project_Name(Project_Context.Project.Project_Prefix'Length+1..Project_Context.Project.Project_Name'Last));

				    begin
				       
				       
				       if Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  Text_Io.Put_Line("-- Editting minor version...");
					  
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
					  
					  for File_Id in Specifications_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  Text_Io.Put_Line("-- Editting revision version...");
					  
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all, "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
					  for File_Id in Implementations_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action or Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  Text_Io.Put_Line("-- Editting major version...");
					  To_Major(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  To_Minor(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  To_Revision(Project_Context.Project.Project_Name.all, Project_Context.Project.Project_Prefix.all);
					  
				       else
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Text_Io.Put_Line("-- Editting release version...");
					  To_Release(Project_Context.Project.Project_Name.all,
						     Project_Context.Project.Project_Prefix.all);
				       end if;
				       Versions.New_Action := False;	       
				       Versions.New_Description := False;
				       
				    end;
				 else

				    Text_Io.Put_Line("Project name : " & Project_Context.Project.Project_Name.all); 
				    declare
				       
				       Node_Name : access String := new String ' ("");

				    begin
				       
				       
				       if Versions.New_Description then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  
					  
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
					  
					  for File_Id in Specifications_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Specifications_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action then
					  Put_Line("Current version is : " & Versions.Version_Io.To_String(Upgraded.Version));
					  Upgraded.Version := 
					    Version_Update(Upgraded.Version);
					  Put_Line("version is updated :" & Versions.Version_Io.To_String(Upgraded.Version));
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Date", Formatting.Image(Clock), Node_Name.all);
					  Main_Description(Project_Context.Project.Project_Name.all, "", "Version", Versions.Version_Io.To_String(Upgraded.Version), Node_Name.all);
					  for File_Id in Implementations_Filenames_Search.File_Content'Range loop
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Date", Formatting.Image(Clock));
					     Update(Handling.To_String(Results."-"(Implementations_Filenames_Search.File_Content(File_Id))), "Version", Versions.Version_Io.To_String(Upgraded.Version));
					  end loop;
					  
					  
				       end if;
				       
				       if Versions.New_Action or Versions.New_Description then
					  Text_Io.Put_Line("-- Editting revision version...");
					  To_Revision(Project_Context.Project.Project_Name.all, "");
					  Text_Io.Put_Line("-- Editting minor version...");
					  To_Minor(Project_Context.Project.Project_Name.all, "");
					  Text_Io.Put_Line("-- Editting major version...");
					  To_Major(Project_Context.Project.Project_Name.all, "");
				       else
					  if Environment.Version_Update /= 0 then
					     raise Program_Error;
					  end if;		  
					  Text_Io.Put_Line("-- Editting release version...");
					  To_Release(Project_Context.Project.Project_Name.all,
						     "");
				       end if;
				       Versions.New_Action := False;	       
				       Versions.New_Description := False;
				       
				    end;
				    
				 end if;
				 
				 --
				 --
				 --
				 --
				 --
				 --
				 --
				 -----------------------------------------------------------------------------------------------------------
				 
				 
				 
				 
				 if Upgraded /= null then
				    
				    App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
				    Upgraded.Project_Num := App.Projects.Projects_Num;
				    Curs := Last(App.Projects.List);
				    Projects_DLL.Delete(App.Projects.List, Curs);
				    
				    Append(App.Projects.List, Upgraded);
				    Projects_Manager.Manager_Save(App.Projects, "./Projects.lst");
				    
				    if Update(Get_Info(App.Projects, Project_Manager.New_Project.Project_Num), Upgraded.Version, App.Sessions, App.Logs) = 0 then
				       
				       Clear(App.Projects.List);
				       Projects_Manager.Manager_Load(App.Projects, "./Projects.lst");
				       
				    end if;
				    
				    App.Projects.Projects_Num := App.Projects.Projects_Num + 1;
				    Append(App.Projects.List, 
					   new Project_Record ' (Null_Project));
				    
				    Projects_Manager.Manager_Save(App.Projects, "./Projects.lst");
				 end if;
				 Project.Close(Project_Manager.New_Project.all, Clock);
				 Project_Count := 0;
				 Project_Index := 0;
				 Curs := First(App.Projects.List);			   
				 Project_Num := First_Element(App.Projects.List).Project_Num;
				 Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
				 Close(App.Sessions(Sky.Logs.Project));
				       
				 Statistic(App.Sessions, App.Logs);
				 Step_Cur := Prj_Selector;

			      when Wide_Character'Val(127) =>
				 if Cmd_Last > 0 then
				    Cmd_Line(Cmd_Last) :=  Wide_Character'Val(32);
				    Cmd_Last := Cmd_Last - 1;
				 else
				    Close(App.Sessions(Sky.Logs.Project));
				       
				    Statistic(App.Sessions, App.Logs);
				    Step_Cur := Prj_Selector;

				 end if;
			      when others =>
				 null;
			   end case;
			elsif Cmd_Last < Cmd_Line'Length then
			   Cmd_Line(Cmd_Last + 1) := Wchar;
			   Cmd_Last := Cmd_Last + 1;
			else
			   Put(Character'Val(7));
			end if;
			
		     end Receive;
		     case Step_Cur is
			when Prj_None =>
			   null;
			when Prj_Selector =>
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   end if;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			   
			when Prj_Manager =>

			   Draw_Window(Manager_Abs);
			   Draw_Text(Manager_Abs, 1, 1, White, "Abstract  : ");
			   Draw_Window(Manager_N);
			   Draw_Text(Manager_N, 1, 1, White, "Name      : " & Handling.To_Wide_String(Project_Manager.New_Project.Project_Name.all));
			   Draw_Window(Manager_Ver);
			   Draw_Text(Manager_Ver, 1, 1, White, "Version   : " & Handling.To_Wide_String(Version_Io.To_String(Project_Manager.New_Project.Version)));
			   Draw_Window(Manager_Auth);
			   Draw_Text(Manager_Auth, 1, 1, White, "Author    : " & Handling.To_Wide_String(Project_Manager.New_Project.Copyright.all));

			   Draw_Window(Manager_List);

			   List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);

			   Draw_Window(Manager_cmd);
			   Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");

			   
			   
			   Draw_Text(
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Win,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Line_Pos,
				     Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Column_Pos,
				     White,
				     Project_Manager.Curs.Parts
				       (Project_Manager.Curs.Part_Cur).Line(1..Project_Manager.Curs.Parts(Project_Manager.Curs.Part_Cur).Last)
				    );

			   if Line_Remove then
			      if Lib_Name_Last > 0 then
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : " & Lib_Name(1..Lib_Name_last));
			      else
				 Draw_Text(Manager_cmd, 1, 1, White, "Line num : ");
			      end if;
			   elsif Cmd_Last > 0 then
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			   else
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   end if;

			when Prj_Upgrade =>
			   Draw_Window(Components_Win);
			   Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, top);

			   Draw_Window(Make_Win);
			   if not Print_Set(File_Set, Make_Win, File_Index, Total_lines) then
			      raise Program_Error;
			   end if;
			   if Cmd_Last > 0 then
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			   else
			      Draw_Window(Manager_cmd);
			      Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			   end if;
			when others =>
			   null;
		     end case;
	       or

		  accept Refresh do
		     null;
		  end Refresh;
		  
	       or
		  accept Page_Down do
		     case Step_Cur is
			when Prj_Selector =>
			   null;
			when Prj_Creator =>
			   if Lib_Index + (Creator_List.Win.H-2) < Creator.New_Project.Lib_Last then
			      Lib_Index := Lib_Index + (Creator_List.Win.H-2);
			      Draw_Window(Creator_List);
			      
			      List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			   else
			      Put(character'Val(0));
			   end if;
			when Prj_Manager =>
			   if Lib_Index + (Manager_List.Win.H-2) < Project_Manager.New_Project.Lib_Last then
			      Lib_Index := Lib_Index + (Manager_List.Win.H-2);
			      Draw_Window(Manager_List);
			      
			      List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);
			   else
			      Put(character'Val(0));
			   end if;
			when Prj_Upgrade =>
			   if File_Index < Integer'Min(File_Index + (App.Lines-8)/2-3, Line_Last) then
			      File_Index := Integer'Min(File_Index + (App.Lines-8)/2-3, Line_Last);
			   end if;
			   Draw_Window(Make_Win);
			   if not Print_Set(File_Set, Make_Win, File_Index, Total_lines) then
			      raise Program_Error;
			   end if;
			when others =>
			   null;
		     end case;
		  end Page_Down;
	       or
		  accept Page_Up do
		     case Step_Cur is
			when Prj_Selector =>
			   null;
			when Prj_Creator =>
			   if Lib_Index - (Creator_List.Win.H-2) >= 0 then
			      
			      Lib_Index := Lib_Index - (Creator_List.Win.H-2);
			      Draw_Window(Creator_List);
			      
			      List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			   elsif Lib_Index /= 0 then
			      Lib_Index := 0;
			      Draw_Window(Creator_List);
			      
			      List_Print(Creator.New_Project.all, Creator_List, Lib_Index);
			   else
			      Put(character'Val(0));
			   end if;
			when Prj_Manager =>
			   if Lib_Index - (Creator_List.Win.H-2) >= 0 then
			      
			      Lib_Index := Lib_Index - (Manager_List.Win.H-2);
			      Draw_Window(Creator_List);
			      
			      List_Print(Project_Manager.New_Project.all, Manager_List, Lib_Index);
			   elsif Lib_Index /= 0 then
			      Lib_Index := 0;
			      Draw_Window(Manager_List);
			      
			      List_Print(Project_manager.New_Project.all, Manager_List, Lib_Index);
			   else
			      Put(character'Val(0));
			   end if;
			when Prj_Upgrade =>
			   if File_Index - (App.Lines-8)/2-2 >= 0 then 
			      File_Index := File_Index - (App.Lines-8)/2-2;
			   else
			      File_Index := 0;
			   end if;
			   Draw_Window(Make_Win);
			   if not Print_Set(File_Set, Make_Win, File_Index, Total_lines) then
			      raise Program_Error;
			   end if;
			when others =>
			   null;
		     end case;
		  end Page_Up;
	       or
		  accept Up_Arrow do
		     case Step_Cur is
			when Prj_Archives =>
			   if Arch_Curs /= First(App.Archives.List) then
			      Previous(Arch_Curs);
			      if Arch_Count > 0 then
				 Arch_Count := Arch_Count - 1;
			      elsif Arch_Index > 0 then
				 Arch_Index := Arch_Index - 1;
			      end if;
			      
			   end if;
			   
			   Arch_Num := Projects_dll.Element(Arch_Curs).Project_Num;
			   
			   Draw_Window(Project_Selector);
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			   
			when Prj_Selector =>
			   if Curs /= First(App.Projects.List) then
			      
			      Previous(Curs);
			      if Project_Count > 0 then
				 Project_Count := Project_Count - 1;
			      elsif Project_Index > 0 then
				 Project_Index := Project_Index - 1;
			      end if;
			      
			      declare
				 Current : constant Project.Project_Access := Element(Curs);
			      begin	
				 if Current /= null then
				    
				    Project_Num := Current.Project_Num;
				    
				 end if;
			      end;
			      
			      
			      
			   end if;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   else
			      Draw_Header(Null_Project, Project_Header);
			   end if;
			   
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when Prj_Creator =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Upgrade =>
			   if Project_Manager.New_Project /= null and then
			     Component_Cur - 1 >= 0 then
			      Component_Cur := Component_Cur - 1;
			      
			      Component_Index := Integer'Max(0, Component_Cur - ((App.Lines-8)/2-2)/3);
			      
			      Draw_Window(Components_Win);
			      Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, top);
			      
			      
			      if Cmd_Last > 0 then
				 Draw_Window(Manager_cmd);
				 Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			      else
				 Draw_Window(Manager_cmd);
				 Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			      end if;
			   else
			      Put(Character'Val(7));
			   end if;
			when others =>
			   null;
		     end case;
		     
		  end up_arrow;
	       or
		  accept Down_Arrow do
		     
		     case Step_Cur is
			when Prj_Archives =>
			   if Arch_Curs /= Last(App.Archives.List) then
			      
			      Arch_Curs := Next(Arch_Curs);
			      
			      if Arch_Count < (App.Lines-7)/5-1 then
				 
				 Arch_Count := Arch_Count + 1;
			      elsif Arch_Index < Natural(Projects_Dll.Length(App.Archives.List)) then
				 
				 Arch_Index := Arch_Index + 1;
			      end if;

			      Arch_Num := Projects_dll.Element(Arch_Curs).Project_Num;

			      
			   end if;
			   
			   
			   Draw_Window(Project_Selector);
			   
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			when Prj_Selector =>			   
			   
			   if Curs /= Last(App.Projects.List) then
			      
			      Curs := Next(Curs);
			      
			      if Project_Count < (App.Lines-7)/5-1 then
				 Project_Count := Project_Count + 1;
			      elsif Project_Index < Natural(Projects_Dll.Length(App.Projects.List)) then
				 Project_Index := Project_Index + 1;
			      end if;
			      
			      
			      
			      declare
				 Current : constant Project.Project_Access := Element(Curs);
			      begin	
				 
				 Project_Num := Current.Project_Num;
				 
			      end;
			      
			      
			   end if;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   else
			      Draw_Header(Null_Project, Project_Header);
			   end if;
			   
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			   
			when Prj_Creator =>
			   null;
			when Prj_Manager =>
			   null;
			when Prj_Upgrade =>
			   if Project_Manager.New_Project /= null and then
			     Component_Cur + 1 <= Project_Manager.New_Project.Lib_Last then
			      Component_Cur := Component_Cur + 1;
			      
			      Component_Index := Integer'Max(0, Component_Cur - ((App.Lines-8)/2-2)/3);
			      
			      Draw_Window(Components_Win);
			      Components_Print(Project_Manager.New_Project.all, Component_Index , Component_cur, (App.Lines-8)/2-2, App.Columns, top);
			      
			      
			      if Cmd_Last > 0 then
				 Draw_Window(Manager_cmd);
				 Draw_Text(Manager_cmd, 1, 1, White, "Command   : " & Cmd_Line(Cmd_Line'First..Cmd_Last));
			      else
				 Draw_Window(Manager_cmd);
				 Draw_Text(Manager_cmd, 1, 1, White, "Command   : ");
			      end if;
			   end if;
			when others =>
			   null;
		     end case;
		  end Down_Arrow;
	       or
		  accept Left_Arrow;
	       or
		  accept Right_Arrow;
	       or
		  accept Full_Screen;
	       or
		  accept Del;
	       or
		  accept Begin_Of do
		     case Step_Cur is
			when Prj_Selector =>
			   Curs := First(App.Projects.List);
			   Project_Num := Element(Curs).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   else
			      Draw_Header(Null_Project, Project_Header);
			   end if;
			   Project_Count := 0;
			   Project_Index := 0;
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when Prj_Archives =>
			   Arch_Curs := First(App.Archives.List);
			   Arch_Num := Element(Arch_Curs).Project_Num;
			   Draw_Window(Project_Selector);
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			   Arch_Count := 0;
			   Arch_Index := 0;
			when Prj_Components =>
			   null;
			when others =>
			   null;
		     end case;
		  end Begin_Of;
	       or
		  accept End_Of do
		     case Step_Cur is
			when Prj_Selector =>
			   Curs := Last(App.Projects.List);
			   Project_Num := Element(Curs).Project_Num;
			   Project_Manager.New_Project := Get_Project(App.Projects, Project_Num);
			   Draw_Window(Project_Header);
			   if Project_Manager.New_Project /= null then
			      Draw_Header(Project_Manager.New_Project.all, Project_Header);
			   else
			      Draw_Header(Null_Project, Project_Header);
			   end if;
			   Project_Count := Integer(Length(App.Projects.List) - 1);
			   Project_Index := Integer'max(0, Integer(Length(App.Projects.List)) - ((App.Lines-7)/5)-1);
			   Draw_Window(Project_Selector);
			   Projects_Print(App.Projects, Project_Index, Project_Num, App.Lines, App.Columns, top);
			when Prj_Archives =>
			   Arch_Curs := Last(App.Archives.List);
			   Arch_Num := Element(Arch_Curs).Project_Num;
			   Draw_Window(Project_Selector);
			   Archives_Print(App.Archives, Arch_Index, Arch_Num, App.Lines, App.Columns, top);
			   Arch_Count := Integer(Length(App.Archives.List) - 1);
			   Arch_Index := Integer'max(0, Integer(Length(App.Archives.List)) - ((App.Lines-7)/5)-1);
			when Prj_Components =>
			   null;
			when others =>
			   null;
		     end case;
		  end End_Of;
	       or
		  accept Overwrite;
	       or
		  accept Screen_Print;

	       end select;
	    end loop;
	    
	 end loop Main_Loop;
      end;
      --W_Io.Put_Line("Manager process halted");
   end Manager_Process;
   
   
      procedure Highlight_Contact_Draw (Obj : in Profile_Class; Line : in Positive;Width : in Positive; Top : in Natural) is
      Win : Window_Record := (Line, Top+2, 7, Width, Blue, Window);      
   begin
      
      Initialize(Win, Line, Top+2, 7, Width, Blue, White);
      Enlight_Window(Win);
      Text_Io.Put(Normal_Mode);
      
      Draw_Text(Win, 1, 1, White, "Name : " & Obj.Info.Name.all);
      
      Draw_Text(Win, 2, 1, White, "Born : " & Handling.To_Wide_String(Formatting.Image(Obj.Born)));
      
      Draw_Text(Win, 3, 1, White, "Addr : " & Obj.Addr.all);
      
      Draw_Text(Win, 4, 1, White, "Phone : " & Obj.Phone.all);
      Draw_Text(Win, 5, 1, White, "Email : " & Obj.Email.all);
      
   end Highlight_Contact_Draw;
   
   procedure Contact_Draw (Obj : in Profile_Class; Line : in Positive;Width : in Positive; Top : in Natural) is
      Win : Window_Record := (Line, Top+2, 7, Width, Blue, Window);      
   begin
      
      Initialize(Win, Line, Top+2, 7, Width, Blue, White);
      Draw_Window(Win);
      Draw_Text(Win, 1, 1, White, "Name : " & Obj.Info.Name.all);
      
      Draw_Text(Win, 2, 1, White, "Born : " & Handling.To_Wide_String(Formatting.Image(Obj.Born)));
      
      Draw_Text(Win, 3, 1, White, "Addr : " & Obj.Addr.all);
      
      Draw_Text(Win, 4, 1, White, "Phone : " & Obj.Phone.all);
      Draw_Text(Win, 5, 1, White, "Email : " & Obj.Email.all);
      
   end Contact_Draw;
   



   procedure Contacts_Draw (Contacts : in Abstract_Vectors.Vector;Index : in Natural; Width : in Positive; Top : in Natural) is
      
      Count : Natural := 1;
   begin
      
      if Abstract_Vectors.Is_empty(Contacts) then
	 return;
      end if;
      
      for Contact in Index+1..natural'Min(Index + (App.Lines-5) / 7, Integer(Abstract_Vectors.Length(Contacts))) loop
	 
	 declare
	    E : Abstract_Access := Abstract_Vectors.Element(Contacts, Abstract_Index(Contact));
	    
	 begin
	    
	    Contact_Draw(Profile_Class(E.all), Count+1, width, Top);
	       
	    
	    Count := Count + 7;
	 end;
      end loop;

   end Contacts_Draw;
   
   
   
   
   
   
   procedure To_contacts (Obj : in Abstract_Access;Vector : in out Abstract_Vectors.Vector) is
   begin
      if Obj = null then
	 return;
      end if;
      if Obj.all in Profile_Class then
	 Vector := Vector & Obj;
      end if;
      
      if Obj.Info.Childs_Num /= 0 then
	 for Child in 1..Obj.Info.Childs_Num loop
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(Obj.Childs, Abstract_Index(Child));
	    begin
	       To_Contacts(E, Vector);
	    end;
	 end loop;
      end if;
      
   end To_Contacts;
   
   function Name_Inf (Left, Right : in Abstract_Access) return Boolean is
   begin
      
      if Left = null or Right = null then
	 raise Program_Error;
      end if;
      if Left.Info.Name = null or Right.Info.Name = null then
	 raise Program_Error;
      end if;
      return Left.Info.Name.all < Right.Info.Name.all;
   end Name_inf;
   
   package Name_Sorting is new Abstract_Vectors.Generic_Sorting(Name_inf);
      

   task body Contacts_Process is
      
      Main_Win : Window_Record := (1, 1, App.Lines-3, App.Columns/2, White, Window);
      
      Contacts : Abstract_Vectors.Vector;
      Contact_Index : Natural := 0;

      
      
      
   begin
  Main_Loop:
      loop
	 loop
	    select
	       accept Receive (Wchar : in Wide_Character);
	    or
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  null;
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	       or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  If Contact_Index > 0 then
		     Contact_Index := Contact_Index - 1;
		     Enlight_Window(Main_Win);
		  
		     Text_Io.Put(Normal_Mode);
		  
		     Contacts_Draw(Contacts, Contact_Index, App.Columns/2-2, 0);
		  end if;
		     

	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  if Abstract_Vectors.Length(Contacts) /= 0 then
		     if Contact_Index + 1 < Integer(Abstract_Vectors.Length(Contacts)) then
			Contact_Index := Contact_Index + 1;
			Enlight_Window(Main_Win);
		  
			Text_Io.Put(Normal_Mode);
			
			Contacts_Draw(Contacts, Contact_Index, App.Columns/2-2, 0);
		     end if;
		     
		  end if;
	       end Down_Arrow;
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  Initialize(Main_Win, 1, TOP+1, App.Lines-3, App.Columns/2, White, Cyan);
		  
		  Enlight_Window(Main_Win);
		  
		  Text_Io.Put(Normal_Mode);
		  
		  Contacts := Abstract_Vectors.Empty_Vector;
		  
		  To_Contacts(App.Terminal.Root, Contacts);
		  
		  Contact_Index := 0;
		  if Abstract_Vectors.Length(Contacts) /= 0 then
		     
		     Name_Sorting.Sort(Contacts);
		     
		     
		     Contacts_Draw(Contacts, Contact_Index, App.Columns/2-2, Top);
		  
		  end if;
		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Contacts process halted");
   end Contacts_Process;
   
   
   subtype Menu_Range is Positive range 1..13;
   
   type Menu_List is array (Menu_Range) of String_Access;
   
   Main_Menu : constant Menu_List := 
     
     (new String ' ("Projects menu"),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(1))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(2))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(3))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(4))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(5))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(6))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(7))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(8))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(9))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(10))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(11))),
      new String ' (Mode_Enum'Image(Mode_Enum'Val(12))));
   
   
   
   
   
   
   task body Menu_Process is
      Menu_Index : Menu_Range := 1;
      
      subtype Line_Type is String(1..48);
      
      Title : Line_Type := (others => Character'Val(32));
      Item  : Line_Type := (others => Character'Val(32));
      
      Menu_Win : Window_Record := (1, 1, App.Lines, App.Columns/2, Blue, Window);
      
      
      type Project_List is array (Positive range 1..App.Lines-8) of String_Access;
      
      
      Project_Last : natural := 0;
      
      
      Projects_Menu : Project_List := (1 => new String ' ("Main menu"), others => null);
				      
      
      procedure Menu_Print(Win : in Window_Record; Selected : in Menu_Range) is
	 
      begin
	 
      	 Draw_Window(Win);
      	 Fixed.Move("Main menu", Title, Error, Left);
      	 Draw_Text(Win, 3, App.Columns/4-8, White, Handling.To_Wide_String(Title));
	 
	 for Iter in Main_Menu'Range loop
	    Item := (others => Character'Val(32));
	    Fixed.Move(Main_Menu(Iter).all, Item, Error, Left);
	    if Iter /= Selected then
	       Draw_Text(Win, Iter*2+5, App.Columns/4-8, White, Handling.To_Wide_String(Item));
	    else
	       Enlight_Draw_Text(Win, Iter*2+5, App.Columns/4-8, White, Handling.To_Wide_String(Item));
	       Text_Io.Put(Normal_Mode);
	    end if;
	 end loop;
	    
      end Menu_Print;
      
      
      procedure Projects_Print(Win : in Window_Record; Selected : in Positive) is
	 
      begin
	 
      	 Draw_Window(Win);
      	 Fixed.Move("Projects menu", Title, Error, Left);
      	 Draw_Text(Win, 3, App.Columns/4-8, White, Handling.To_Wide_String(Title));
	 
	 for Iter in 1..Project_Last loop
	    
	    Item := (others => Character'Val(32));
	    
	    Fixed.Move(Projects_Menu(Iter).all, Item, Error, Left);
	    
	    if Iter /= Selected then
	       
	       Draw_Text(Win, Iter*2+5, App.Columns/4-8, White, Handling.To_Wide_String(Item));
	       
	    else
	       
	       Enlight_Draw_Text(Win, Iter*2+5, App.Columns/4-8, White, Handling.To_Wide_String(Item));
	       
	       Text_Io.Put(Normal_Mode);
	    end if;
	 end loop;
	    
      end Projects_Print;
      
      
      
      
      
      Is_Projects : Boolean := False;
      
      
   begin
      Initialize(Menu_Win, 1, 1, App.Lines, App.Columns/2, Blue, White);
      
  Main_Loop:
      loop
	 loop
	    select
	       when not Is_Projects =>
	       accept Receive (Wchar : in Wide_Character) do
		  if Is_Control(Wchar) then
		     case Wchar is
			when Wide_Character'Val(10) =>
			   if Menu_Index = 1 then
			      Projects_Print(Menu_Win, Menu_Index);
			      Is_Projects := True;
			   elsif Menu_Index /= 11 then
			      App.Desktop(2) := App.Desktop(1);
			      App.Desktop(1) := Machines(Menu_Index-1);
			      App.Mode := Mode_Enum'Val(Menu_Index-1);
			   end if;
			when others =>
			   null;
		     end case;		  
		  end if;
	       end Receive;
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Wake_Up(0);
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Wake_Up(0);
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when others =>
			null;
		  end case;
	       end if;
	    or
	       when Is_Projects =>
		  accept Receive (Wchar : in Wide_Character) do
		     if Is_Control(Wchar) then
			case Wchar is
			   when Wide_Character'Val(10) =>
			      if Menu_Index = 1 then				 
				 Menu_Print(Menu_Win, Menu_Index);
				 Is_Projects := False;				 
			      else
				 
				 declare
				    
				    Project_Name : String := Projects_Menu(Menu_Index).all(1..Index(Projects_Menu(Menu_Index).all, "-", Backward)-1);
				    Success : Boolean := False;
				    Args : Argument_List_Access;
				    Args_String : String_Access;
				 begin
				    Text_Io.New_Line;
				    Text_Io.Put_Line("Program name : " & Projects_Menu(Menu_Index).all & "/bin/" & Project_Name);
				    Text_Io.Put("Arguments :");
				    begin
				       Args_String := new String ' (Text_Io.Get_Line);
				       begin
					  Args := Argument_String_To_List(Args_String.all);
					  Normalize_Arguments(Args.all);
					  Change_Dir(Projects_Menu(Menu_Index).all);
					  Spawn("bin/" & Project_Name, Args.all, Success);
					  Change_Dir("..");
				       end;
				    exception
				       when others =>
					  null;
				    end;
				    Projects_Print(Menu_Win, Menu_Index);
				 end;
			      end if;
			   when others =>
			      null;
			end case;		  
		     end if;
		  end Receive;


	    or
	       accept Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do		  
		  Menu_Print(Menu_Win, Menu_index);
		  Project_Last := 1;
		  Menu_Index := 1;
		  Is_Projects := False;
		  declare
		     Curs : Projects_Dll.Cursor := First(App.Projects.List);
		  begin
		     if not Projects_Dll.Is_Empty(App.Projects.List) then
			Project_Last := Natural(Projects_Dll.Length(App.Projects.List));
			for Project in 1..Projects_Dll.Length(App.Projects.List)-1 loop
			   declare
			      E : Sky.Project.Project_Access := Projects_Dll.Element(curs);
			   begin
			      
			      if E /= null then
				 Projects_Menu(Integer(Project)+1) := 
				   new String ' (E.Project_Name.all &
						   '-' & To_String(E.Version));
			      end if;
			   end;
			   Projects_Dll.Next(Curs);
			end loop;
		     end if;
		  end;
	       
	       end Wake_Up;
	    or
	       
	       accept Suspend;
	       exit;
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       when not Is_Projects =>
		  accept Up_Arrow do
		     if Menu_Index > 1 then
			Menu_Index := Menu_Index - 1;
		     end if;
		  end Up_Arrow;
		  Menu_Print(Menu_Win, Menu_Index);
	    or
	       when not Is_Projects =>
		  accept Down_Arrow do
		     if Menu_Index < Menu_Range'Last then
			Menu_Index := Menu_Index + 1;
		     end if;
		  end Down_Arrow;
		  Menu_Print(Menu_Win, Menu_Index);
		  
	    or
	       when Is_Projects =>
		  accept Up_Arrow do
		     
		     if Menu_Index > 1 then
			Menu_Index := Menu_Index - 1;
		     end if;
		  end Up_Arrow;
		  Projects_Print(Menu_Win, Menu_Index);
	    or
	       when Is_Projects =>
		  accept Down_Arrow do
		     if Menu_Index < Project_Last then
			Menu_Index := Menu_Index + 1;
		     end if;
		  end Down_Arrow;
		  
		  Projects_Print(Menu_Win, Menu_Index);
		  
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  
		  Menu_Print(Menu_Win, Menu_index);
		  Project_Last := 1;
		  Menu_Index := 1;
		  Is_Projects := False;
		  declare
		     Curs : Projects_Dll.Cursor := First(App.Projects.List);
		  begin
		     if not Projects_Dll.Is_Empty(App.Projects.List) then
			Project_Last := Natural(Projects_Dll.Length(App.Projects.List));
			for Project in 1..Projects_Dll.Length(App.Projects.List)-1 loop
			   declare
			      E : Sky.Project.Project_Access := Projects_Dll.Element(Curs);
			   begin
			      
			      if E /= null then
				 Projects_Menu(Integer(Project)+1) := 
				   new String ' (E.Project_Name.all &
						   '-' & To_String(E.Version));
			      end if;
			   end;
			   Projects_Dll.Next(Curs);
			end loop;
		     end if;
		  end;
	       end Wake_Up;	    	       
	       exit;	       
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Menu process halted");
   end Menu_Process;
   
   
   
   ---------------------------------------------------------------------------
   --                                                                       --
   --                             Organizer                                 --
   --                                                                       --
   ---------------------------------------------------------------------------
   
   use Abstract_Vectors;
   
   function Id (Name : in Class_Enum) return Abstract_Access is
   begin
      
      if Name = Planning then
   	 return new Planning_Class;
      elsif Name = Content then
   	 return new Content_Class;
      elsif Name = Universe then
   	 return new Universe_Class;
      elsif Name = Event then
   	 return new Event_Class;
      elsif Name = Meeting then
   	 return new Meeting_Class;
      elsif Name = Rendezvous then
   	 return new Rendezvous_Class;
      elsif Name = Job then
   	 return new Job_Class;
      elsif Name = Profile then
	 return new Profile_Class;
      elsif Name = Book then
      	 return new Book_Class;
      elsif Name = CDROM then
      	 return new CDROM_Class;
      elsif Name = DVD then
      	 return new DVD_Class;
      elsif Name = Car then
      	 return new Car_Class;
      elsif Name = House then
      	 return new House_Class;
      end if;

      return null;
   end Id;            
   
   type Object_Counter is array (Class_Enum'Range) of Natural;
   
   procedure Global_Content_Print (Object : Abstract_Access;
				   Win       : in Window_Type) is

      
      Objects : Abstract_Vectors.Vector;
      
      Counter : Object_Counter := (others => 0);
      Win_Index : Natural := 0;
   begin

      if Object /= null then
	 
	 Objects := Object.Childs;
	 
	 for I in 1..Last_Index(Objects) loop
	    
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(Objects, I);
	    begin
	       
	       for J in Class_Enum'Succ(None)..Class_Enum'Last loop
		  
		  if E.Tag = J then
		     
		     Counter(J) := Counter(J) + 1;
		     
		  end if;
	       end loop;
	    end;
	 end loop;
	 
	 for I in Class_Enum'Succ(None)..Class_Enum'Last loop
	    
	    Draw(Win, Win_Index+1, 3, White, Handling.To_Wide_String(Class_Enum'Image(I)) &
		   Wide_Fixed."*"((16 - Class_Enum'Image(I)'Length),' ') &
		" : Total  = " & Handling.To_Wide_String(Natural'Image(Counter(I))) &
		   " " & Handling.To_Wide_String(Class_Enum'Image((i))));
	    
	    Win_Index := Win_Index + 1;
	 end loop;
      else
	 
	 Draw(Win, Win_Index+1, 3, White, "Obj = null");
	 
      end if;

      
      
   end Global_Content_Print;

   procedure Global_Read(File : W_Io.File_Type;
   			 End_Of_File : out Boolean;
   			 Object : out Abstract_Access) is
      Tag : Class_Enum := None;
      Name : Wide_String(1..8) := (others => Wide_Character'Val(32));
      Name_Last : Natural := 0;
      Index : Natural := 0;
   begin

      End_Of_File := False;
      while not W_Io.End_Of_File(File) loop

	 declare
	    Name : Wide_String := W_Io.Get_Line(File);
   	 begin
   	    

   	    Index := Wide_Fixed.Index_Non_Blank(Name, backward);
   	    if index = 0 then
   	       raise Constraint_Error;
   	    else
   	       if Wide_Characters.Handling.Is_Control(Name(Index)) then
   		  raise Constraint_Error;
   	       end if;
   	    end if;

   	    Tag := Class_Enum'Value(Handling.To_String(Name));

   	    exit;
   	 exception
   	    when Constraint_Error =>

   	       W_Io.Get_Line(File, Name, Name_Last);

   	       begin

   		  Tag := Class_Enum'Value(Handling.To_String(Name(Name'First..Name_Last)));

   		  exit;
		  
   	       end;
   	 end;
      end loop;
      


      Object := Id(Tag);

      End_Of_File := W_Io.End_Of_File(File);
      if not End_Of_File then
	 
	 Object.Read(File);
	 
      end if;

      
   exception
      when Constraint_Error =>
   	 raise;
      when others =>
   	 null;
   end Global_Read;
   
   
   
   procedure Global_Print(Object : Abstract_Access;
   			  Header       : in Window_Type;
			  Main : in Window_Type) is
   begin
      if Object = null then
	 return;
      end if;
      if Object.all'Tag = Universe_Class'Tag then
   	 Universe_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Planning_Class'Tag then
   	 Planning_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Content_Class'Tag then
   	 Content_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Event_Class'Tag then
   	 Event_Class(Object.all).Print(Header, Main);      	 
      elsif Object.all'Tag = Job_Class'Tag then
   	 Job_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Meeting_Class'Tag then
   	 Meeting_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Rendezvous_Class'Tag then
   	 Rendezvous_Class(Object.all).Print(Header, Main);
      elsif Object.all'Tag = Profile_Class'Tag then
   	 Profile_Class(Object.all).Print(Header, Main);      	 
	 
      end if;      
   end Global_Print;
   
   procedure Parent(T : in out Terminal_Type) is
      
      V : Abstract_Vectors.Vector := T.Path;
   begin
      if not Is_Empty(V) then
   	 if Last_Index(V) >= 1 then
	    T.Cur := Last_Element(V);
	    if Last_Index(V) > 1 then
	       Delete(V, Last_Index(V));
	    end if;
   	    
   	 end if;
      end if;
      T.Path := V;
   end Parent;
   
   procedure Switch(T : in out Terminal_Type;
   		    Num : in Abstract_Index;
   		    Success : out boolean) is
      Vector : Abstract_Vectors.Vector;	 
   begin
      Success := False;
      
      if T.Cur /= null then
   	 Vector := T.Cur.Childs;
   	 if not Is_Empty(Vector) then
   	    if Num <= Last_Index(Vector) then
   	       T.Path := T.Path & T.Cur;
   	       T.Cur := Abstract_Vectors.Element(Vector, Num);
	       
   	       if T.Cur /= null then
   		  
   		  Success := True;
   	       else
   		  raise Program_Error;		     
   	       end if;
   	    end if;
   	 end if;
      end if;	 
   end Switch;

   

procedure Restore_Vector(T : in out Terminal_Type;
   			    File : W_Io.File_Type) is
      
      Success : Boolean := False;
      N, O : Abstract_Access;	 
      Child_Childs, Child_Numb : Natural := 0;
      End_Of_File : Boolean := False;
   begin
      
      if T.Root = null then
	 Global_Read(File, End_Of_File, T.Root);
	 T.Cur := T.Root;
	 if T.Cur.Info.Childs_Num /= 0 then
	    for Child in 1..T.Cur.Info.Childs_Num loop
	       if not W_Io.End_Of_File(File) then
		  Restore_Vector(T, File);
	       end if;
	       Parent(T);
	    end loop;
	 end if;
      else
	 for Child in 1..T.Cur.Info.Childs_Num loop
	    Global_Read(File, End_Of_File, N);	    
	    T.Cur.Childs := T.Cur.Childs & N;
	    if not W_Io.End_Of_File(File) then
	       if N.Info.Childs_num /= 0 then
		  Switch(T, Last_Index(T.Cur.Childs), Success);
		  Restore_Vector(T, File);
		  Parent(T);
	       end if;
	    end if;
	 end loop;
      end if;

   end Restore_Vector;

   
   
   
   procedure Restore(T : in out Terminal_Type;
   		     Filename : in String) is
      
      End_Of_File : Boolean := False;
      File : W_Io.File_Type;
   begin
      
      W_Io.Open(File, W_Io.in_File, Filename);
      if not W_Io.End_Of_File(File) then
   	 Restore_Vector(T, File);
      end if;
      
      W_Io.Close(File);
   end Restore;
   
   
   procedure Search(T : in out Terminal_Type;
   		    name : in Wide_String;
   		    Path_Index : in out Natural;
   		    Success : out boolean) is
      
      
      
      Vector : Abstract_Vectors.Vector := T.Cur.Childs;
      
   begin
      Success := False;
      
      if not Is_Empty(Vector) then
	 for I in 1..Last_Index(Vector) loop
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(Vector, I);
	       Term : Terminal_Type := T;
	    begin
	       if E /= null then						
		  if E.Info.Name.all = Name then
		     Switch(Term, I,  success);
		     if Success then
			T := Term;
			exit;
		     end if;
			
		  else
		     Switch(Term, I,  success);
		     Path_Index := Path_Index + 1;
		     if Success then
			Search(Term, Name, Path_Index, Success);
			if Success then
			T := Term;
			exit;
		     end if;
		     end if;	
		  end if;
	       else
		  raise Program_Error;
	       end if;
	    end;
	 end loop;
	 
      end if;
      
   end Search;
   
   
   
   
   
   procedure Save_Object (Object : Abstract_Access;			  
   			  File : W_Io.File_Type;
   			  Index  : Natural := 0) is
      O : constant Abstract_Access := Object;
      New_Index : Natural := Index;
   begin      

      O.all.Write(File);

      
      
      if not Is_Empty(O.Childs) then

	 
   	 for I in 1..Last_Index(O.Childs) loop

   	    declare
   	       E : constant Abstract_Access :=
   		 Abstract_Vectors.Element(O.Childs, I);
   	    begin

   	       Save_Object(E, File, New_Index+1);	       
	       
   	    end;
   	 end loop;	 	 

      end if;
      
   end Save_Object;
   
   
   
   
   
   procedure Save (Object : in Abstract_Access;
   		   Filename : in String) is
      File : W_Io.File_Type;
      
   begin

      W_Io.Create(File, W_Io.Out_File, Filename);

      Save_Object(Object, File);

      W_Io.Close(File);

   end Save;


   
   type Organizer_Type;
   
   task type Date_Process (Organizer : access Organizer_Type) is
      entry Start;
      entry Stop;
      entry Halt;
   end Date_Process;
   
   
   type Organizer_Type is limited
      record
	 Main_Win    : Window_Record := (1, 1, App.lines, App.Columns/2, Blue, Window);
	 Header_Win  : Window_Record := (2, 2, 3, App.Columns/2-2, white, Window);
	 content_Win : Window_Record := (5, 2, 22, (App.Columns/2-2)/2, white, Window);
	 Cmd_Win     : Window_Record := (27, 2, 3, (App.Columns/2-2)/2, Blue, Window);
	 Proc        : Date_Process(Organizer_Type'Access);
	 Obj_Header  : Window_Record := (5, (App.Columns/2-2)/2+2, 6, (App.Columns/2-2)/2, Blue, Window);
	 Obj_Main    : Window_Record := (11, (App.Columns/2-2)/2+2, 7, (App.Columns/2-2)/2, white, Window);
	 Obj_Attr    : Window_Record := (18, (App.Columns/2-2)/2+2, 12, (App.Columns/2-2)/2, Blue, Window);
	 List_Win    : Window_Record := (30, 2, App.Lines-29, (App.Columns/2-2), white, Window);
      end record;
   
   task body Date_Process is
   begin
  Main_Loop:
      loop
	 loop
	    select
	       accept Start;
	       exit;
	    or
	       accept Stop;
	    or
	       accept Halt;
	       exit Main_Loop;		      
	    end select;
	 end loop;
	 loop
	    select
	       accept Start;
	    or
	       accept Stop;
	       exit;
	    or
	       accept Halt;
	       exit Main_Loop;		      
	    or
	       delay 1.0;
	       Text_Io.Put(Save_Cursor);
	       Draw_Text(Organizer.Header_Win, 1, App.Columns/4-10, White, Handling.To_Wide_String(Formatting.Image(Clock)));
	       Text_Io.Put(Restore_Cursor);
	    end select;
	 end loop;
      end loop Main_Loop;
   end Date_Process;
   
   procedure Initialize (Organizer : out Organizer_Type; Top : Natural := 0) is
   begin
      Initialize(Organizer.Main_Win, 1, Top+1, App.lines, App.Columns/2, Blue, white);
      Initialize(Organizer.Header_Win, 2, Top+2, 3, App.Columns/2-2, white, Cyan);
      
      Initialize(Organizer.Content_Win, 5, Top+2, 22, (App.Columns/2-2)/2, white, Cyan);
      Initialize(Organizer.Cmd_Win, 27, Top+2, 3, (App.Columns/2-2)/2, Blue, White);
      Initialize(Organizer.Obj_Header, 5, Top+(App.Columns/2-2)/2+2, 6, (App.Columns/2-2)/2, Blue, White);
      Initialize(Organizer.Obj_Main, 11, Top+(App.Columns/2-2)/2+2, 7, (App.Columns/2-2)/2, white, Cyan);
      Initialize(Organizer.Obj_Attr, 18, Top+(App.Columns/2-2)/2+2, 12, (App.Columns/2-2)/2, Blue, White);
      Initialize(Organizer.List_Win, 30, Top+2, App.Lines-29, (App.Columns/2-2), white, Red);
   end Initialize;
   
   procedure Draw (Organizer : in Organizer_Type; Terminal : in Terminal_Type) is
   begin
      Enlight_Window(Organizer.Main_Win);
      Enlight_Window(Organizer.Header_win);
      Enlight_Window(Organizer.Content_Win);
      Enlight_Window(Organizer.cmd_Win);
      Enlight_Window(Organizer.Obj_Header);
      Enlight_Window(Organizer.Obj_Main);
      Enlight_Window(Organizer.Obj_Attr);
      Enlight_Window(Organizer.List_Win);
      Text_Io.Put(Normal_Mode);
   end Draw;
   
   type Cmd_Enum is (Null_Cmd, Make, List, Switch, Root, Parent, Print, Search, Tree, Attribute, remove);
   
   function Cmd_Value (Line : in Wide_String) return Cmd_Enum is
      Cmd : Cmd_Enum  := Null_Cmd;
      First_Space : constant Natural := Wide_Fixed.Index(Line, " ");
   begin
      
      if First_Space /= 0 then
	 begin
	    
	    Cmd := Cmd_Enum'Value(Handling.To_String(Line(Line'First..First_Space-1)));
	    
	 exception
	    when others =>
	       Cmd := Null_Cmd;
	 end;
	 
      elsif Wide_Fixed.Index_Non_Blank(Line) /= 0 then
	 begin
	    Cmd := Cmd_Enum'Value(Handling.To_String(Line(Wide_Fixed.Index_Non_Blank(Line)..Wide_Fixed.Index_Non_Blank(Line, Backward))));
	 exception
	    when others =>
	       Cmd := Null_Cmd;
	 end;
	 
      end if;
      return Cmd;
   end Cmd_Value;
   
   
   
   ---------------------------------------------------------------------
   --                        Planning curs
   
   type Planning_enum is (Planning_None, Planning_Name, Planning_From, Planning_To);
   
   type Planning_Type is
      record
   	 Name : Planning_Enum := Planning_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Planning_Curs_Set is array (Planning_Enum) of Planning_Type;
   
   type Planning_Cursor_Type is
      record
   	 Values  : Planning_Curs_Set;
   	 Val_Cur : Planning_Enum := Planning_Name;	 
      end record;
   
   function Initialize return Planning_Cursor_Type is
      Curs : Planning_Cursor_Type;
   begin
      Curs.Values(Planning_Name) :=
   	(Planning_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 8
   	);
      Curs.Values(Planning_From) :=
   	(Planning_From,
   	 (others => ' '),
   	 0,
   	 2,
   	 8
   	);
      Curs.Values(Planning_To) :=
   	(Planning_To,
   	 (others => ' '),
   	 0,
   	 3,
   	 8
   	);
      Curs.Val_Cur := Planning_name;
      return Curs;
   end Initialize;
   
   
   function To_Planning (Curs : in Planning_Cursor_Type; Index : in Abstract_Index) return Planning_Class is
      Planning : Planning_Class := Create;
   begin
      Planning.info.Index := Index;
      Planning.info.Name := new Wide_String '
   	(Curs.Values(Planning_Name).Line(1..Curs.Values(Planning_Name).Line_Last));
      Planning.From := 
   	Formatting.Value(Handling.To_String(Curs.Values(Planning_From).Line(1..Curs.Values(Planning_From).Line_Last)));
      Planning.To := 
   	Formatting.Value(Handling.To_String(Curs.Values(Planning_To).Line(1..Curs.Values(Planning_To).Line_Last)));
      
      return Planning;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Planning;
   
   
   ---------------------------------------------------------------------------------
   --                            Attribute curs
   
   type Attribute_Enum is (Attribute_None, Attribute_Name, Attribute_Image);
   
   type Attribute_Type is
      record
   	 Name : Attribute_Enum := Attribute_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Attribute_Curs_Set is array (Attribute_Enum) of Attribute_Type;
   
   type Attribute_Cursor_Type is
      record
   	 Values  : Attribute_Curs_Set;
   	 Val_Cur : Attribute_Enum := Attribute_Name;	 
      end record;
   
   function Initialize return Attribute_Cursor_Type is
      Curs : Attribute_Cursor_Type;
   begin
      Curs.Values(Attribute_Name) :=
   	(Attribute_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 8
   	);
      Curs.Values(Attribute_Image) :=
   	(Attribute_Image,
   	 (others => ' '),
   	 0,
   	 2,
   	 9
   	);
      Curs.Val_Cur := Attribute_name;
      return Curs;
   end Initialize;
   
   
   
   ---------------------------------------------------------------------------------
   --                            Event curs
   
   type Event_Enum is (Event_None, Event_Name, Event_Date, Event_Period, Event_Locality, Event_Define);
   
   type Event_Type is
      record
   	 Name : Event_Enum := Event_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Event_Curs_Set is array (Event_Enum) of Event_Type;
   
   type Event_Cursor_Type is
      record
   	 Values  : Event_Curs_Set;
   	 Val_Cur : Event_Enum := Event_Name;	 
      end record;
   
   function Initialize return Event_Cursor_Type is
      Curs : Event_Cursor_Type;
   begin
      Curs.Values(Event_Name) :=
   	(Event_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 8
   	);
      Curs.Values(Event_Date) :=
   	(Event_Date,
   	 (others => ' '),
   	 0,
   	 2,
   	 8
   	);
      Curs.Values(Event_Period) :=
   	(Event_Period,
   	 (others => ' '),
   	 0,
   	 3,
	 9
   	);
      Curs.Values(Event_Locality) :=
   	(Event_Locality,
   	 (others => ' '),
   	 0,
   	 4,
   	 12
   	);
      Curs.Values(Event_Define) :=
   	(Event_Define,
   	 (others => ' '),
   	 0,
   	 5,
   	 10
   	);
      Curs.Val_Cur := Event_name;
      return Curs;
   end Initialize;
   
   
   function To_Event (Curs : in Event_Cursor_Type;Index : in Abstract_Index) return Event_Class is
      Event : Event_Class := Create;
   begin
      Event.info.Index := Index;
      Event.info.Name := new Wide_String ' (Curs.Values(Event_Name).Line(1..Curs.Values(Event_Name).Line_Last));
      Event.Date := Formatting.Value(Handling.To_String(Curs.Values(Event_Date).Line(1..Curs.Values(Event_Date).Line_Last)));
      Event.Period := Duration(Float'Value(Handling.To_String(Curs.Values(Event_Period).Line(1..Curs.Values(Event_Period).Line_Last))) * 60.0);
      Event.Locality := new Wide_String ' (Curs.Values(Event_Locality).Line(1..Curs.Values(Event_Locality).Line_Last));
      Event.Define := new Wide_String ' (Curs.Values(Event_Define).Line(1..Curs.Values(Event_Define).Line_Last));
      
      return Event;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Event;
   
   ---------------------------------------------------------------------------------
   --                            Job curs
   
   type Job_Enum is (Job_None, Job_Name, Job_Date, Job_Period, Job_Locality, Job_Define);
   
   type Job_Type is
      record
   	 Name : Job_Enum := Job_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Job_Curs_Set is array (Job_Enum) of Job_Type;
   
   type Job_Cursor_Type is
      record
   	 Values  : Job_Curs_Set;
   	 Val_Cur : Job_Enum := Job_Name;	 
      end record;
   
   function Initialize return Job_Cursor_Type is
      Curs : Job_Cursor_Type;
   begin
      Curs.Values(Job_Name) :=
   	(Job_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 9
   	);
      Curs.Values(Job_Date) :=
   	(Job_Date,
   	 (others => ' '),
   	 0,
   	 2,
   	 9
   	);
      Curs.Values(Job_Period) :=
   	(Job_Period,
   	 (others => ' '),
   	 0,
   	 3,
   	 9
   	);
      Curs.Values(Job_Locality) :=
   	(Job_Locality,
   	 (others => ' '),
   	 0,
   	 4,
   	 12
   	);
      Curs.Values(Job_Define) :=
   	(Job_Define,
   	 (others => ' '),
   	 0,
   	 5,
   	 10
   	);
      Curs.Val_Cur := Job_name;
      return Curs;
   end Initialize;
   
   
   function To_Job (Curs : in Job_Cursor_Type;Index : in Abstract_Index) return Job_Class is
      Job : Job_Class := Create;
   begin
      Job.info.Index := Index;
      Job.info.Name := new Wide_String ' (Curs.Values(Job_Name).Line(1..Curs.Values(Job_Name).Line_Last));
      Job.Date := Formatting.Value(Handling.To_String(Curs.Values(Job_Date).Line(1..Curs.Values(Job_Date).Line_Last)));
      Job.Period := Duration(Float'Value(Handling.To_String(Curs.Values(Job_Period).Line(1..Curs.Values(Job_Period).Line_Last))) * 60.0);
      Job.Locality := new Wide_String ' (Curs.Values(Job_Locality).Line(1..Curs.Values(Job_Locality).Line_Last));
      Job.Define := new Wide_String ' (Curs.Values(Job_Define).Line(1..Curs.Values(Job_Define).Line_Last));
      
      return Job;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Job;

   ---------------------------------------------------------------------------------
   --                            Meeting curs
   
   type Meeting_Enum is (Meeting_None, Meeting_Name, Meeting_Date, Meeting_Period, Meeting_Locality, Meeting_Define);
   
   type Meeting_Type is
      record
   	 Name : Meeting_Enum := Meeting_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Meeting_Curs_Set is array (Meeting_Enum) of Meeting_Type;
   
   type Meeting_Cursor_Type is
      record
   	 Values  : Meeting_Curs_Set;
   	 Val_Cur : Meeting_Enum := Meeting_Name;	 
      end record;
   
   function Initialize return Meeting_Cursor_Type is
      Curs : Meeting_Cursor_Type;
   begin
      Curs.Values(Meeting_Name) :=
   	(Meeting_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 9
   	);
      Curs.Values(Meeting_Date) :=
   	(Meeting_Date,
   	 (others => ' '),
   	 0,
   	 2,
   	 9
   	);
      Curs.Values(Meeting_Period) :=
   	(Meeting_Period,
   	 (others => ' '),
   	 0,
   	 3,
   	 9
   	);
      Curs.Values(Meeting_Locality) :=
   	(Meeting_Locality,
   	 (others => ' '),
   	 0,
   	 4,
   	 12
   	);
      Curs.Values(Meeting_Define) :=
   	(Meeting_Define,
   	 (others => ' '),
   	 0,
   	 5,
   	 10
   	);
      Curs.Val_Cur := Meeting_name;
      return Curs;
   end Initialize;
   
   
   function To_Meeting (Curs : in Meeting_Cursor_Type;Index : in Abstract_Index) return Meeting_Class is
      Meeting : Meeting_Class := Create;
   begin
      Meeting.info.Index := Index;
      Meeting.info.Name := new Wide_String ' (Curs.Values(Meeting_Name).Line(1..Curs.Values(Meeting_Name).Line_Last));
      Meeting.Date := Formatting.Value(Handling.To_String(Curs.Values(Meeting_Date).Line(1..Curs.Values(Meeting_Date).Line_Last)));
      Meeting.Period := Duration(Float'Value(Handling.To_String(Curs.Values(Meeting_Period).Line(1..Curs.Values(Meeting_Period).Line_Last))) * 60.0);
      Meeting.Locality := new Wide_String ' (Curs.Values(Meeting_Locality).Line(1..Curs.Values(Meeting_Locality).Line_Last));
      Meeting.Define := new Wide_String ' (Curs.Values(Meeting_Define).Line(1..Curs.Values(Meeting_Define).Line_Last));
      
      return Meeting;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Meeting;
   
   ---------------------------------------------------------------------------------
   --                            Rendezvous curs
   
   type Rendezvous_Enum is (Rendezvous_None, Rendezvous_Name, Rendezvous_Date, Rendezvous_Period, Rendezvous_Locality, Rendezvous_Define);
   
   type Rendezvous_Type is
      record
   	 Name : Rendezvous_Enum := Rendezvous_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Rendezvous_Curs_Set is array (Rendezvous_Enum) of Rendezvous_Type;
   
   type Rendezvous_Cursor_Type is
      record
   	 Values  : Rendezvous_Curs_Set;
   	 Val_Cur : Rendezvous_Enum := Rendezvous_Name;	 
      end record;
   
   function Initialize return Rendezvous_Cursor_Type is
      Curs : Rendezvous_Cursor_Type;
   begin
      Curs.Values(Rendezvous_Name) :=
   	(Rendezvous_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 9
   	);
      Curs.Values(Rendezvous_Date) :=
   	(Rendezvous_Date,
   	 (others => ' '),
   	 0,
   	 2,
   	 9
   	);
      Curs.Values(Rendezvous_Period) :=
   	(Rendezvous_Period,
   	 (others => ' '),
   	 0,
   	 3,
   	 9
   	);
      Curs.Values(Rendezvous_Locality) :=
   	(Rendezvous_Locality,
   	 (others => ' '),
   	 0,
   	 4,
   	 12
   	);
      Curs.Values(Rendezvous_Define) :=
   	(Rendezvous_Define,
   	 (others => ' '),
   	 0,
   	 5,
   	 10
   	);
      Curs.Val_Cur := Rendezvous_name;
      return Curs;
   end Initialize;
   
   
   function To_Rendezvous (Curs : in Rendezvous_Cursor_Type;Index : in Abstract_Index) return Rendezvous_Class is
      Rendezvous : Rendezvous_Class := Create;
   begin
      Rendezvous.info.Index := Index;
      Rendezvous.info.Name := new Wide_String ' (Curs.Values(Rendezvous_Name).Line(1..Curs.Values(Rendezvous_Name).Line_Last));
      Rendezvous.Date := Formatting.Value(Handling.To_String(Curs.Values(Rendezvous_Date).Line(1..Curs.Values(Rendezvous_Date).Line_Last)));
      Rendezvous.Period := Duration(Float'Value(Handling.To_String(Curs.Values(Rendezvous_Period).Line(1..Curs.Values(Rendezvous_Period).Line_Last))) * 60.0);
      Rendezvous.Locality := new Wide_String ' (Curs.Values(Rendezvous_Locality).Line(1..Curs.Values(Rendezvous_Locality).Line_Last));
      Rendezvous.Define := new Wide_String ' (Curs.Values(Rendezvous_Define).Line(1..Curs.Values(Rendezvous_Define).Line_Last));
      
      return Rendezvous;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Rendezvous;

   ---------------------------------------------------------------------------
   --                         Content curs
   type Content_enum is (Content_None, Content_Name, Content_Description, Content_Price);
   
   type Content_Type is
      record
   	 Name : Content_Enum := Content_None;
   	 Line : Wstring_45 := (others => ' ');
   	 Line_Last : Natural := 0;
   	 Line_Pos      : Line_Range := 1;
   	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Content_Curs_Set is array (Content_Enum) of Content_Type;
   
   type Content_Cursor_Type is
      record
   	 Values  : Content_Curs_Set;
   	 Val_Cur : Content_Enum := Content_Name;	 
      end record;
   
   function Initialize return Content_Cursor_Type is
      Curs : Content_Cursor_Type;
   begin
      Curs.Values(Content_Name) :=
   	(Content_name,
   	 (others => ' '),
   	 0,
   	 1,
   	 8
   	);
      Curs.Values(Content_Description) :=
   	(Content_Description,
   	 (others => ' '),
   	 0,
   	 2,
   	 9
   	);
      Curs.Values(Content_Price) :=
   	(Content_Price,
   	 (others => ' '),
   	 0,
   	 3,
   	 9
   	);
      
      Curs.Val_Cur := Content_name;
      return Curs;
   end Initialize;
   
   
   function To_Content (Curs : in Content_Cursor_Type;Index : in Abstract_Index) return Content_Class is
      Content : Content_Class := Create;
   begin
      Content.info.Index := Index;
      Content.info.Name := new Wide_String '
   	(Curs.Values(Content_Name).Line(1..Curs.Values(Content_Name).Line_Last));
      Content.Description := new Wide_String '
   	(Curs.Values(Content_Description).Line(1..Curs.Values(Content_Description).Line_Last));
      Content.Price := 
   	Fixed_Unit_Ttc'Value(Handling.To_String(Curs.Values(Content_Price).Line(1..Curs.Values(Content_Price).Line_Last)));
      
      return Content;
   exception
      when others =>
   	 raise Spec_Error;
   end To_Content;
   
   
   
   procedure Add_To (Terminal : in out Terminal_Type; Obj : Abstract_Access) is
   begin
      if Terminal.Root = null then
	 Obj.Info.Index := 1;	 
	 Terminal.Root := Obj;
	 Terminal.Cur := Obj;
	 Terminal.Path := Terminal.Path & Obj;
      else
	 Obj.Info.Index := Abstract_Index(Abstract_Vectors.Length(Terminal.Cur.Childs) + 1);
	 Terminal.Cur.Childs := Terminal.Cur.Childs & Obj;
	 Terminal.Cur.Info.Childs_Num := Terminal.Cur.Info.Childs_Num + 1;
      end if;
   end Add_To;
   
   
   
   
   
   ---------------------------------------------------------------------------
   --                           Profile curs
   
   
   type Profile_enum is (Profile_None, Profile_Full_Name, Profile_Born, Profile_Addr, Profile_Phone, Profile_Email);
   
   type Profile_Type is
      record
	 Name : Profile_Enum := Profile_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Profile_Curs_Set is array (Profile_Enum) of Profile_Type;
   
   type Profile_Cursor_Type is
      record
	 Values  : Profile_Curs_Set;
	 Val_Cur : Profile_Enum := Profile_Full_Name;	 
      end record;
   
   function Initialize return Profile_Cursor_Type is
      Curs : Profile_Cursor_Type;
   begin
      Curs.Values(Profile_Full_Name) :=
	(Profile_Full_Name,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Values(Profile_Born) :=
	(Profile_Born,
	 (others => ' '),
	 0,
	 2,
	 8
	);
      Curs.Values(Profile_Addr) :=
	(Profile_Addr,
	 (others => ' '),
	 0,
	 3,
	 8
	);
      Curs.Values(Profile_Phone) :=
	(Profile_Phone,
	 (others => ' '),
	 0,
	 4,
	 9
	);
      Curs.Values(Profile_Email) :=
	(Profile_Email,
	 (others => ' '),
	 0,
	 5,
	 9
	);
      Curs.Val_Cur := Profile_Full_Name;
      return Curs;
   end Initialize;
   
   
   function To_Profile (Curs : in Profile_Cursor_Type;Index : in Abstract_Index) return Profile_Class is
      Profile : Profile_Class := Create;
   begin
      Profile.Info.Index := Index;
      Profile.info.Name := new Wide_String '
	(Curs.Values(Profile_Full_Name).Line(1..Curs.Values(Profile_Full_Name).Line_Last));
      
      Profile.Full_Name := new Wide_String '
	(Curs.Values(Profile_Full_Name).Line(1..Curs.Values(Profile_Full_Name).Line_Last));
      
      Profile.Born := Formatting.Value(Handling.To_String(Curs.Values(Profile_Born).Line(1..Curs.Values(Profile_Born).Line_Last)));
      
      Profile.Addr := new Wide_String '
	(Curs.Values(Profile_Addr).Line(1..Curs.Values(Profile_Addr).Line_Last));
      
      Profile.Phone := new Wide_String '
	(Curs.Values(Profile_Phone).Line(1..Curs.Values(Profile_Phone).Line_Last));
      
      Profile.Email := new Wide_String '
	(Curs.Values(Profile_Email).Line(1..Curs.Values(Profile_Email).Line_Last));
      
      
      return Profile;
   exception
      when others =>
	 raise Spec_Error;
   end To_Profile;
   
   
   

   ---------------------------------------------------------------------------
   --                           Universe curs
   
   
   type Universe_enum is (Universe_None, Universe_Name);
   
   type Universe_Type is
      record
	 Name : Universe_Enum := Universe_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Universe_Curs_Set is array (Universe_Enum) of Universe_Type;
   
   type Universe_Cursor_Type is
      record
	 Values  : Universe_Curs_Set;
	 Val_Cur : Universe_Enum := Universe_Name;	 
      end record;
   
   function Initialize return Universe_Cursor_Type is
      Curs : Universe_Cursor_Type;
   begin
      Curs.Values(Universe_Name) :=
	(Universe_Name,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Val_Cur := Universe_Name;
      return Curs;
   end Initialize;
   
   
   
   function To_Universe (Curs : in Universe_Cursor_Type;Index : in Abstract_Index) return Universe_Class is
      Universe : Universe_Class := Create;
   begin
      Universe.Info.Index := Index;
      Universe.info.Name := new Wide_String '
	(Curs.Values(Universe_Name).Line(1..Curs.Values(Universe_Name).Line_Last));
      return Universe;
   exception
      when others =>
	 raise Spec_Error;
   end To_Universe;

   

   
   
   ---------------------------------------------------------------------------
   --                           Car curs
   
   
   type Car_enum is (Car_None, Car_Name, Car_Mark, Car_Model, Car_mat);
   
   type Car_Type is
      record
	 Name : Car_Enum := Car_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Car_Curs_Set is array (Car_Enum) of Car_Type;
   
   type Car_Cursor_Type is
      record
	 Values  : Car_Curs_Set;
	 Val_Cur : Car_Enum := Car_Name;	 
      end record;
   
   function Initialize return Car_Cursor_Type is
      Curs : Car_Cursor_Type;
   begin
      Curs.Values(Car_Name) :=
	(Car_Name,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Values(Car_Mark) :=
	(Car_Mark,
	 (others => ' '),
	 0,
	 2,
	 8
	);
      Curs.Values(Car_Model) :=
	(Car_Model,
	 (others => ' '),
	 0,
	 3,
	 9
	);
      Curs.Values(Car_Mat) :=
	(Car_Mat,
	 (others => ' '),
	 0,
	 4,
	 7
	);
      Curs.Val_Cur := Car_Name;
      return Curs;
   end Initialize;
   
   function To_Car (Curs : in Car_Cursor_Type;Index : in Abstract_Index) return Car_Class is
      Car : Car_Class := Create;
   begin
      Car.Info.Index := Index;
      Car.info.Name := new Wide_String '
	(Curs.Values(Car_Name).Line(1..Curs.Values(Car_Name).Line_Last));
      Car.Mark := new Wide_String '
	(Curs.Values(Car_Mark).Line(1..Curs.Values(Car_Mark).Line_Last));
      Car.Model := new Wide_String '
	(Curs.Values(Car_Model).Line(1..Curs.Values(Car_Model).Line_Last));
      Car.Mat := new Wide_String '
	(Curs.Values(Car_Mat).Line(1..Curs.Values(Car_Mat).Line_Last));
      return Car;
   exception
      when others =>
	 raise Spec_Error;
   end To_Car;

   
   ---------------------------------------------------------------------------
   --                           House curs
   
   
   type House_enum is (House_None, House_Name, House_Addr, House_Surface);
   
   type House_Type is
      record
	 Name : House_Enum := House_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type House_Curs_Set is array (House_Enum) of House_Type;
   
   type House_Cursor_Type is
      record
	 Values  : House_Curs_Set;
	 Val_Cur : House_Enum := House_Name;	 
      end record;
   
   function Initialize return House_Cursor_Type is
      Curs : House_Cursor_Type;
   begin
      Curs.Values(House_Name) :=
	(House_Name,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Values(House_addr) :=
	(House_addr,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Values(House_Surface) :=
	(House_Surface,
	 (others => ' '),
	 0,
	 1,
	 8
	);
      Curs.Val_Cur := House_Name;
      return Curs;
   end Initialize;
   
      function To_House (Curs : in House_Cursor_Type;Index : in Abstract_Index) return House_Class is
      House : House_Class := Create;
   begin
      House.Info.Index := Index;
      House.info.Name := new Wide_String '
	(Curs.Values(House_Name).Line(1..Curs.Values(House_Name).Line_Last));
      House.Addr := new Wide_String '
	(Curs.Values(House_Addr).Line(1..Curs.Values(House_Addr).Line_Last));
      House.Surf := new Wide_String '
	(Curs.Values(House_Surface).Line(1..Curs.Values(House_surface).Line_Last));
      return House;
   exception
      when others =>
	 raise Spec_Error;
   end To_house;

   ---------------------------------------------------------------------------
   --                           Media curs
   
   
   type Media_enum is (Media_None, Media_title, Media_Author, Media_Editor);
   
   type Media_Type is
      record
	 Name : Media_Enum := Media_None;
	 Line : Wstring_45 := (others => ' ');
	 Line_Last : Natural := 0;
	 Line_Pos      : Line_Range := 1;
	 Column_Pos    : Column_Range := 1;
      end record;
   
   type Media_Curs_Set is array (Media_Enum) of Media_Type;
   
   type Media_Cursor_Type is
      record
	 Values  : Media_Curs_Set;
	 Val_Cur : Media_Enum := Media_Title;
      end record;
   
   function Initialize return Media_Cursor_Type is
      Curs : Media_Cursor_Type;
   begin
      Curs.Values(Media_title) :=
	(Media_Title,
	 (others => ' '),
	 0,
	 1,
	 9
	);
      Curs.Values(Media_Author) :=
	(Media_Author,
	 (others => ' '),
	 0,
	 1,
	 10
	);Curs.Values(Media_Editor) :=
	(Media_Editor,
	 (others => ' '),
	 0,
	 1,
	 10
	);
      Curs.Val_Cur := Media_Title;
      return Curs;
   end Initialize;
   
   
   function To_CDROM (Curs : in Media_Cursor_Type;Index : in Abstract_Index) return CDROM_Class is
      CDROM : CDROM_Class := Create;
   begin
      CDROM.Info.Index := Index;
      CDROM.info.Name := new Wide_String '
	(Curs.Values(Media_title).Line(1..Curs.Values(Media_title).Line_Last));
      CDROM.Author := new Wide_String '
	(Curs.Values(Media_Author).Line(1..Curs.Values(Media_Author).Line_Last));
      CDROM.Editor := new Wide_String '
	(Curs.Values(Media_Editor).Line(1..Curs.Values(Media_Editor).Line_Last));
      return CDROM;
   exception
      when others =>
	 raise Spec_Error;
   end To_CDROM;
   
   
   function To_DVD (Curs : in Media_Cursor_Type;Index : in Abstract_Index) return DVD_Class is
      DVD : DVD_Class := Create;
   begin
      DVD.Info.Index := Index;
      DVD.info.Name := new Wide_String '
	(Curs.Values(Media_title).Line(1..Curs.Values(Media_title).Line_Last));
      DVD.Author := new Wide_String '
	(Curs.Values(Media_Author).Line(1..Curs.Values(Media_Author).Line_Last));
      DVD.Editor := new Wide_String '
	(Curs.Values(Media_Editor).Line(1..Curs.Values(Media_Editor).Line_Last));
      return DVD;
   exception
      when others =>
	 raise Spec_Error;
   end To_DVD;
   
   function To_Book (Curs : in Media_Cursor_Type;Index : in Abstract_Index) return Book_Class is
      Book : Book_Class := Create;
   begin
      Book.Info.Index := Index;
      Book.info.Name := new Wide_String '
	(Curs.Values(Media_title).Line(1..Curs.Values(Media_title).Line_Last));
      Book.Author := new Wide_String '
	(Curs.Values(Media_Author).Line(1..Curs.Values(Media_Author).Line_Last));
      Book.Editor := new Wide_String '
	(Curs.Values(Media_Editor).Line(1..Curs.Values(Media_Editor).Line_Last));
      return Book;
   exception
      when others =>
	 raise Spec_Error;
   end To_Book;
   
   
   function Child (Cur : in Abstract_Access;Index : in Natural) return Abstract_Access is
   begin
      if Index = 0 then
	 return Cur;
      end if;
      if Cur = null then
	 return null;
      end if;
      return Abstract_Vectors.Element(Cur.Childs, Abstract_Index(Index));
   exception
      when others =>
	 return null;
   end Child;
   
   function Child (Cur : in Abstract_Access;Name : in Wide_String) return Abstract_Access is
      Obj : Abstract_Access;
   begin
      if Cur = null then
	 return null;
      end if;
      if Cur.Info.Name.all = Name then
	 return Cur;
      end if;
      if not Abstract_Vectors.Is_Empty(Cur.Childs) then
	 for Child in 1..Abstract_Vectors.Length(Cur.Childs) loop
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(Cur.Childs, Abstract_Index(Child));
	    begin
	       if E.Info.Name /= null and then
		 E.Info.Name.all = Name then
		  return E;
	       end if;
	    end;
	 end loop;
      end if;
      return null;
      
   end Child;
   
   
   
   function To_String(Info : in Abstract_Info) return Wide_String is
   begin
      if Info.Name = null then
	 raise Program_Error;
      end if;
      return Handling.To_Wide_String(Abstract_Index'Image(Info.Index)) & " " 
	&	Handling.To_Wide_String(Class_Enum'Image(Info.Tag)) & 
	" " & Info.Name.all & " " 
	& Handling.To_Wide_String(Formatting.Image(Info.Created));
   end To_String;
	   
   function To_Tree(Info : in Abstract_Info; Level : in Natural) return Wide_String is
   begin
      if Info.Name = null then
	 raise Program_Error;
      end if;

      return Wide_Fixed."*"(Level, ' ') & '+' & ' ' & 
	Handling.To_Wide_String(Abstract_Index'Image(Info.Index)) & " " 
	& Handling.To_Wide_String(Class_Enum'Image(Info.Tag)) & " " 
	& Info.Name.all & " " 
	& Handling.To_Wide_String(Formatting.Image(Info.Created));
   end To_Tree;
   
   
   
   
   procedure Tree (Obj : in Abstract_Access; Level : in Natural; Set : in out Wide_String_Set; Last : in out Natural) is
   begin

      if Obj /= null then
	 if Level = 0 then

	    Add_Line(Set, Last, To_tree(Obj.Info, Level+1));
	 end if;

	 if not Abstract_Vectors.Is_Empty(Obj.Childs) then

	    Add_Line(Set, Last, Wide_Fixed."*"(Level+1, ' ') & "| ");

	    for Child in 1..Abstract_Vectors.Length(Obj.Childs) loop

	       declare
		  E : Abstract_Access := Abstract_Vectors.Element(Obj.Childs, Abstract_Index(Child));
	       begin
		  Add_Line(Set, Last, To_tree(E.Info, Level+2));
		  Tree(E, Level+3, Set, Last);

	       end;
	    end loop;
	 end if;
      end if;

   end Tree;
   
   
   procedure Tree_Free (O : in out Abstract_Access) is
   begin
      if O = null then
	 return;
      end if;
      
      if Abstract_Vectors.Length(O.Childs) /= 0 then
	 for Child in 1 .. Abstract_Index(Abstract_Vectors.Length(O.Childs)) loop
	    
	    declare
	       E : Abstract_Access := Abstract_Vectors.Element(O.Childs, Child);
	    begin
	       
	       Tree_Free(E);
	    end;
	 end loop;
      end if;
      Obj_free(O);
   end Tree_Free;
   
   procedure Remove (T : in out Terminal_Type; Index : in Abstract_Index) is
   begin

      if Index > Abstract_Index(Abstract_Vectors.Length(T.Cur.Childs)) then
	 return;
      end if;
      
      
      declare
	 E : Abstract_Access := Abstract_Vectors.Element(T.Cur.Childs, Index);
      begin
	 Tree_Free(E);
	 
      end;
      
      
      Abstract_Vectors.Delete(T.Cur.Childs, Index);
      
      T.Cur.Info.Childs_Num := T.Cur.Info.Childs_Num - 1;
      
      
      if not Abstract_Vectors.Is_empty(T.Cur.Childs)and then
	Index <= Abstract_Index(Abstract_Vectors.Length(T.Cur.Childs)) then
	 
	 
	 

	 for Child in Index .. Abstract_Index(Abstract_Vectors.Length(T.Cur.Childs)) loop
	    
	    declare
	       E : constant Abstract_Access := Abstract_Vectors.Element(T.Cur.Childs, Index);
	    begin
	       E.Info.Index := E.Info.Index - 1;
	       
	    end;
	 end loop;
      
	 
      end if;
      
   end Remove;
	 
   
   
   
   
   -----------------------------------
   -- Organizer process             --
   -----------------------------------
   
   task body Organizer_Process is
      
      
      
      
      Attribute_Curs : Attribute_Cursor_Type;
      
      Universe_Curs : Universe_Cursor_Type;
      Profile_Curs : Profile_Cursor_Type;
      Event_Curs : Event_Cursor_Type;
      Job_Curs : Job_Cursor_Type;
      Meeting_Curs : Meeting_Cursor_Type;
      Rendezvous_Curs : Rendezvous_Cursor_Type;
      Content_Curs : Content_Cursor_Type;
      Planning_Curs : Planning_Cursor_Type;
      Media_Curs : Media_Cursor_Type;
      Car_Curs : Car_Cursor_Type;
      House_Curs : House_Cursor_Type;
      
      Obj_Tag : Class_Enum := None;
      
      Cmd_Line : Wide_String (1..App.Columns/2-6);
      Line_Last : Natural := 0;
      
      Cmd : Cmd_Enum := Null_cmd;
      
      Organizer : Organizer_Type;
      
      Verax : Boolean := False;
            
      Top : Natural := 0;
      
      In_Print : Boolean := False;
      Print_Index : Natural := 0;
      
      Printed : Abstract_Access;
      
      Path_Index : Natural := 0;
      Success : Boolean := False;
      
      Childs_Set : Wide_String_Set(4096);
      Line_Index  : Natural := 0;
      Last : Natural := 0;
      Total_Line : Natural := 0;
      
      Level : Natural := 0;
      
      Is_Initialized : Boolean := False;
      
      Child_Index : Abstract_Index := 1;
      
      On_List : Boolean := False;
      On_Tree : Boolean := False;
      
      Name : String_Access;
      
   begin
      Initialize(Organizer, 0);

	 
      
  Main_Loop:
      loop
	 loop
	    select
	       when Obj_Tag = None and Cmd /= Attribute =>
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      null;
			   when Wide_Character'Val(10) =>
			      if Line_Last /= 0 then
				 In_Print := False;
				 Cmd := Cmd_Value(Cmd_Line(1..Line_Last));
				 case Cmd is
				    when Null_Cmd =>
				       null;
				    when Make =>
				       declare
					  Tag_Name : String_Access;
				       begin
					  if Wide_Fixed.Index(Cmd_Line(Cmd_Line'First..Line_Last), " ") /= 0 then
					     Tag_Name := new String ' (Handling.To_String(Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last)));
					     Obj_Tag := Class_Enum'Value(Tag_Name.all);
					     case Obj_Tag is
						when None =>
						   null;
						when universe =>
						   Universe_Curs := Initialize;
						when Profile =>
						   Profile_Curs := Initialize;
						when Event =>
						   Event_Curs := Initialize;
						when Meeting =>
						   Meeting_Curs := Initialize;
						when Job =>
						   Job_Curs := Initialize;
						when Rendezvous =>
						   Rendezvous_Curs := Initialize;
						when Content =>
						   Content_Curs := Initialize;
						when Planning =>
						   Planning_Curs := Initialize;
						when Book..DVD =>
						   Media_Curs := Initialize;
						when Car =>
						   Car_Curs := Initialize;
						when House =>
						   House_Curs := Initialize;
						when others =>
						   null;
					     end case;
					  end if;
				       exception
					  when others =>
					     null;
				       end;
				    when Attribute =>
				       Attribute_Curs := Initialize;
				    when Parent =>
				       Parent(App.Terminal);
				    when Switch =>
				       if App.Terminal.Cur /= null then
					  if Wide_Fixed.Index(Cmd_Line(Cmd_Line'First..Line_Last), " ") /= 0 then
					     declare
						Index_name : Wide_String_Access;
						Index : Abstract_Index := 1;
					     begin
						Index_name := new Wide_String ' (Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last));
						Index := Abstract_Index'Value(Handling.To_String(Index_Name.all));
						Switch(App.Terminal, Index, Verax);
					     exception
						when others =>
						   null;
					     end;
					  end if;
				       end if;
				    when Root =>
				       App.Terminal.Cur := App.Terminal.Root;
				       App.Terminal.Path := Abstract_Vectors.Empty_Vector;
				       App.Terminal.Path := App.Terminal.Path & App.Terminal.Root;
				    when Print =>				       
				       if App.Terminal.Cur /= null then
					  if Wide_Fixed.Index(Cmd_Line(Cmd_Line'First..Line_Last), " ") /= 0 then
					     declare
						Index_name : Wide_String_Access;
						Index : Natural := 0;
					     begin
						
						Index_name := new Wide_String ' (Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last));
						Print_Index := Natural'Value(Handling.To_String(Index_Name.all));
						Printed := Child(App.Terminal.Cur, Print_Index);
						In_Print  := True;
						Enlight_Window(Organizer.Obj_Header);
						Text_Io.Put(Normal_Mode);
						Enlight_Window(Organizer.Obj_Main);
						Text_Io.Put(Normal_Mode);
						Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
						
						Enlight_Window(Organizer.Obj_Attr);
						Text_Io.Put(Normal_Mode);
						Print(Printed.definition, Organizer.Obj_Attr.Win);
					     exception
						when others =>
						   declare
						      Name : Wide_String_Access;
						      Index : Abstract_Index := 1;
						   begin
						      Index_name := new Wide_String ' (Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last));
						      Printed := Child(App.Terminal.Cur, Index_Name.all);
						      In_Print  := True;
						      Enlight_Window(Organizer.Obj_Header);
						      Text_Io.Put(Normal_Mode);
						      Enlight_Window(Organizer.Obj_Main);
						      Text_Io.Put(Normal_Mode);
						      Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
						      Enlight_Window(Organizer.Obj_Attr);
						      Text_Io.Put(Normal_Mode);
						      Print(Printed.definition, Organizer.Obj_Attr.Win);
						   exception
						      when others =>
							 null;
						   end;
					     end;
					  else
					     In_Print := True;
					     Enlight_Window(Organizer.Obj_Header);
					     Text_Io.Put(Normal_Mode);
					     Enlight_Window(Organizer.Obj_Main);
					     Text_Io.Put(Normal_Mode);
					     App.Terminal.Cur.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
					     Enlight_Window(Organizer.Obj_Attr);
					     Text_Io.Put(Normal_Mode);

					     Print(App.Terminal.Cur.Definition, Organizer.Obj_Attr.Win);

					     Print_Index := 0;
					     Printed := null;
					  end if;
				       
				       end if;
				    when Search =>
				       if App.Terminal.Cur /= null then
					  if Wide_Fixed.Index(Cmd_Line(Cmd_Line'First..Line_Last), " ") /= 0 then
					     declare
						Index_name : Wide_String_Access;
						Index : Abstract_Index := 1;
					     begin
						Index_name := new Wide_String ' (Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last));
						Search(App.Terminal, Index_Name.all, Path_Index, Success);
						if Success then
						   Enlight_Window(Organizer.Obj_Header);
						   Text_Io.Put(Normal_Mode);
						   Enlight_Window(Organizer.Obj_Main);
						   Text_Io.Put(Normal_Mode);
						   App.Terminal.Cur.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
						   Enlight_Window(Organizer.Obj_Attr);
						Text_Io.Put(Normal_Mode);
						Print(Printed.definition, Organizer.Obj_Attr.Win);
						   In_Print := True;
						end if;
						Success := False;
						Path_Index := 0;
					     exception
						when Constraint_Error =>
						   null;
					     end;
					  end if;
				       end if;
				    when List =>
				       Set_Free(Childs_Set);
				       Enlight_Window(Organizer.List_Win);
				       Text_Io.Put(Normal_Mode);
				       Last := 0;
				       Total_Line := 0;
				       Line_Index := 0;
				       On_List := True;
				       On_Tree := False;
				       if App.Terminal.Cur /= null then
					  if not Abstract_Vectors.Is_Empty(App.Terminal.Cur.Childs) then
					     for Child in 1..Abstract_Vectors.Length(App.Terminal.Cur.Childs) loop
						declare
						   E : constant Abstract_Access := Abstract_Vectors.Element(App.Terminal.Cur.Childs, Abstract_Index(Child));
						begin
						   Add_Line(Childs_Set, Last, To_String(E.Info));
						end;
					     end loop;
					     
					     
					  end if;
					  
				       end if;
				       Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
				    when Tree =>
				       On_Tree := True;
				       On_List := False;
				       Enlight_Window(Organizer.List_Win);
				       Text_Io.Put(Normal_Mode);
				       Total_Line := 0;
				       Line_Index := 0;
				       Set_Free(Childs_Set);
				       if App.Terminal.Cur /= null then
					  Tree(App.Terminal.Cur, Level, Childs_Set, Last);
					  Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
				       end if;				       
				    when Remove =>
				       declare
					  Name : Wide_String_Access;
					  Index : Abstract_Index := 1;
				       begin

					  if Wide_Fixed.Index(Cmd_Line, " ") /= 0 then

					     name := new Wide_String ' (Cmd_Line(Wide_Fixed.Index(Cmd_Line, " ")+1..Line_Last));

					     if Wide_Fixed.Index_Non_Blank(Name.all) /= 0 then

						Index := Abstract_Index'Value(Handling.To_String(Name.all));
						
						Remove(App.terminal, Index);
						Save(App.Terminal.root, Organizer_Filename);
					     end if;
					  end if;
					  if On_List then
					     Set_Free(Childs_Set);
					     Enlight_Window(Organizer.List_Win);
					     Text_Io.Put(Normal_Mode);
					     Last := 0;
					     Total_Line := 0;
					     Line_Index := 0;
					     if App.Terminal.Cur /= null then
						if not Abstract_Vectors.Is_Empty(App.Terminal.Cur.Childs) then
						   for Child in 1..Abstract_Vectors.Length(App.Terminal.Cur.Childs) loop
						      declare
							 E : constant Abstract_Access := Abstract_Vectors.Element(App.Terminal.Cur.Childs, Abstract_Index(Child));
						      begin
							 Add_Line(Childs_Set, Last, To_String(E.Info));
						      end;
						   end loop;
						   
						   
						end if;
						
					     end if;
					     Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
					  elsif On_Tree then
					     Total_Line := 0;
					     Line_Index := 0;
					     Enlight_Window(Organizer.List_Win);
					     Text_Io.Put(Normal_Mode);
					     Set_Free(Childs_Set);
					     if App.Terminal.Cur /= null then
						Tree(App.Terminal.Cur, Level, Childs_Set, Last);
						Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
					     end if;
					  end if;
					  
					  
					  
					  Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
				       exception
					  when others =>
					     null;
				       end;
				 end case;
			      end if;
			      Line_Last := 0;
			   when Wide_Character'Val(12) =>
			      Last := 0;
			      Printed := null;
			      Print_Index := 0;
			      Initialize(Organizer, Top);
			      Draw(Organizer, App.Terminal);
			      Enlight_Window(Organizer.Content_Win);
			      Text_Io.Put(Normal_Mode);
			      Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			      Enlight_Window(Organizer.cmd_Win);
			      Text_Io.Put(Normal_Mode);
			      if App.Terminal.Cur = null then
				 Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			      elsif App.Terminal.Cur.Info.Name /= null then
				 Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			      else
				 Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			      end if;
			   when Wide_Character'Val(127) =>
			      if Line_Last > 0 then
				 Cmd_Line(Line_Last) := Wide_Character'Val(32);
				 Line_Last := Line_Last - 1;
			      else
				 Text_Io.Put(Character'Val(7));
			      end if;
			   when others =>
			      null;
			end case;
		     else
			if App.Terminal.Root /= null and then
			  App.Terminal.cur.Info.Name /= null then
			   if Line_Last + 1 <= (Cmd_Line'Length-App.Terminal.cur.Info.Name'Length) then
			      Cmd_Line(Line_Last + 1) := Wchar;
			      Line_Last := Line_Last + 1;
			   else
			      Text_Io.Put(Character'Val(7));
			      
			   end if;
			elsif Line_Last + 1 <= Cmd_Line'Length then
			   Cmd_Line(Line_Last + 1) := Wchar;
			   Line_Last := Line_Last + 1;
			else
			   Text_Io.Put(Character'Val(7));
			   
			end if;
			
		     end if;
		     
		  end Receive;
		  Enlight_Window(Organizer.Content_Win);
		  Text_Io.Put(Normal_Mode);
		  Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
		  Enlight_Window(Organizer.cmd_Win);
		  Text_Io.Put(Normal_Mode);
		  if App.Terminal.Cur = null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
		  elsif App.Terminal.Cur.Info.Name /= null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
		  else
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
		  end if;
		  case Obj_Tag is
		     when None =>
			case Cmd is
			   when Attribute =>
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Image : ");
			      
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Attribute_Name..Attribute_Image loop
				 Draw_Text(Organizer.Obj_Main, Attribute_Curs.values(Val).Line_Pos,
					   Attribute_Curs.values(Val).Column_Pos,
					   White,
					   Attribute_Curs.values(Val).Line(1..Attribute_Curs.values(Val).Line_Last));
			      end loop;
				       
			      Draw_Text(Organizer.Obj_Main, Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Pos,
					Attribute_Curs.Values(Attribute_Curs.Val_Cur).Column_Pos,
					White,
					Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(1..Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     when Profile =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Born : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Addr : ");
			Draw_Text(Organizer.Obj_Main, 4, 1, White, "Phone : ");
			Draw_Text(Organizer.Obj_Main, 5, 1, White, "Email : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when Content =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Define : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Price : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when Planning =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "From : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "To   : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when Universe =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when Event .. Rendezvous =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Date : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Period : ");
			Draw_Text(Organizer.Obj_Main, 4, 1, White, "Locality : ");
			Draw_Text(Organizer.Obj_Main, 5, 1, White, "Define : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when Book..CDROM =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Author : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Editor : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Title : ");
		     when Car =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Mark : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Model : ");
			Draw_Text(Organizer.Obj_Main, 4, 1, White, "Mat : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when House =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Addr : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Surf : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		     when others =>
			Enlight_Window(Organizer.Obj_Main);
			Text_Io.Put(Normal_Mode);
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		  end case;
		  if App.Terminal.Root /= null then
		     Child_Index := Abstract_Index(Abstract_Vectors.Length(App.Terminal.Cur.childs)+1);
		  else
		     Child_Index := 1;
		  end if;
		     
		     
		     
		  or
	       when cmd = attribute =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Attribute_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Enlight_Window(Organizer.Obj_Attr);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last /= 0 then
				 case Attribute_Curs.Val_Cur is
				    when Attribute_None =>
				       null;
				       
				    when Attribute_Name =>
				       Name := new String ' (Handling.To_String(Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(1..Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last)));
				       Attribute_Curs.Val_Cur := Attribute_Enum'Succ(Attribute_Curs.Val_Cur);
				    when Attribute_Image =>
				       if App.Terminal.Cur /= null then
					  if App.Terminal.Cur.Definition.Index < App.Terminal.Cur.Definition.List'Last then
					     App.Terminal.Cur.Definition.List(App.Terminal.Cur.Definition.Index + 1) :=
					       Make(Name.all, Handling.To_String(Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(1..Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last)));
					     
					     App.Terminal.Cur.Definition.Index := App.Terminal.Cur.Definition.Index + 1;
					  end if;
				       end if;
				       
				       attribute_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Free(Name);
				       Attribute_Curs.Val_Cur := Attribute_None;
				       Cmd := Null_cmd;
				 end case;
				 
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last > 0 then
				 Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last) := ' ';
				 Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last := 
				   Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      
			   when others =>
			      null;
			end case;
		     elsif Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last < 45 then
			
			Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last := 
			  Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Attribute_Name..Attribute_image loop
			   Draw_Text(Organizer.Obj_Main, Attribute_Curs.values(Val).Line_Pos,
				     Attribute_Curs.values(Val).Column_Pos,
				     White,
				     Attribute_Curs.values(Val).Line(1..Attribute_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Pos,
				  Attribute_Curs.Values(Attribute_Curs.Val_Cur).Column_Pos,
				  White,
				  Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(1..Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Attribute_Curs.Val_Cur is
		     when Attribute_none =>
			
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.Obj_Main);
			Enlight_Window(Organizer.Obj_Attr);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur /= null then
			   for I in 1..App.Terminal.Cur.Definition.Index loop
			      Draw_Text(Organizer.Obj_Attr, I, 1, white, Handling.To_Wide_String(App.Terminal.Cur.Definition.List(I).Name.all));
			      Draw_Text(Organizer.Obj_Attr, I, App.Terminal.Cur.Definition.List(I).Name'Length+1, Red, "=");
			      Draw_Text(Organizer.Obj_Attr, I, App.Terminal.Cur.Definition.List(I).Name'Length+2, white, Handling.To_Wide_String(App.Terminal.Cur.Definition.List(I).Value.all));
			   end loop;
			end if;
			
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			
			Enlight_Window(Organizer.Obj_Main);
			Enlight_Window(Organizer.cmd_Win);
			Enlight_Window(Organizer.Obj_Attr);
			
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur /= null then
			   for I in 1..App.Terminal.Cur.Definition.Index loop
			      Draw_Text(Organizer.Obj_Attr, I, 1, white, Handling.To_Wide_String(App.Terminal.Cur.Definition.List(I).Name.all));
			      Draw_Text(Organizer.Obj_Attr, I, App.Terminal.Cur.Definition.List(I).Name'Length+1, Red, "=");
			      Draw_Text(Organizer.Obj_Attr, I, App.Terminal.Cur.Definition.List(I).Name'Length+2, white, Handling.To_Wide_String(App.Terminal.Cur.Definition.List(I).Value.all));
			   end loop;
			end if;
			
			
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
			
			
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Image : ");
			
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			for Val in Attribute_Name..Attribute_Image loop
			   Draw_Text(Organizer.Obj_Main, Attribute_Curs.values(Val).Line_Pos,
				     Attribute_Curs.values(Val).Column_Pos,
				     White,
				     Attribute_Curs.values(Val).Line(1..Attribute_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Pos,
				  Attribute_Curs.Values(Attribute_Curs.Val_Cur).Column_Pos,
				  White,
				  Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line(1..Attribute_Curs.Values(Attribute_Curs.Val_Cur).Line_Last));
			
			null;
		  end case;
		  
	    or
	       
	       when Obj_Tag = Profile =>
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Profile_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last /= 0 then
				 case Profile_Curs.Val_Cur is
				    when Profile_None =>
				       null;
					  
				    when Profile_Email =>
				       
				       declare
					  Profile : constant Profile_Class := To_Profile(Profile_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Profile_Class '(Profile));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				 
				       Profile_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Profile_Curs.Val_Cur := Profile_Enum'Succ(Profile_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Pos,
					   Profile_Curs.Values(Profile_Curs.Val_Cur).Column_Pos,
					   White,
					   Profile_Curs.Values(Profile_Curs.Val_Cur).Line(1..Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last > 0 then
				 Profile_Curs.Values(Profile_Curs.Val_Cur).Line(Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last) := ' ';
				 Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last := 
				   Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Born : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Addr : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Phone : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Email : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Profile_Full_Name..Profile_Email loop
				 Draw_Text(Organizer.Obj_Main, Profile_Curs.values(Val).Line_Pos,
					   Profile_Curs.values(Val).Column_Pos,
					   White,
					   Profile_Curs.values(Val).Line(1..Profile_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Pos,
					Profile_Curs.Values(Profile_Curs.Val_Cur).Column_Pos,
					White,
					Profile_Curs.Values(Profile_Curs.Val_Cur).Line(1..Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last < 45 then
			
			Profile_Curs.Values(Profile_Curs.Val_Cur).Line(Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last := 
			  Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last + 1;
			
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Born : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Addr : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Phone : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Email : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Profile_Full_Name..Profile_Email loop
				 Draw_Text(Organizer.Obj_Main, Profile_Curs.values(Val).Line_Pos,
					   Profile_Curs.values(Val).Column_Pos,
					   White,
					   Profile_Curs.values(Val).Line(1..Profile_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Pos,
					Profile_Curs.Values(Profile_Curs.Val_Cur).Column_Pos,
					White,
					Profile_Curs.Values(Profile_Curs.Val_Cur).Line(1..Profile_Curs.Values(Profile_Curs.Val_Cur).Line_Last));
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Root = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
			
			
	    or
	       when Obj_Tag = Content =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Content_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Content_Curs.Values(Content_Curs.Val_Cur).Line_Last /= 0 then
				 case Content_Curs.Val_Cur is
				    when Content_None =>
				       null;
					  
				    when Content_Price =>
				       
				       
				       declare
					  Content : constant Content_Class := To_Content(Content_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Content_Class '(Content));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				
					  Content_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Content_Curs.Val_Cur := Content_Enum'Succ(Content_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Content_Curs.Values(Content_Curs.Val_Cur).Line_Pos,
					   Content_Curs.Values(Content_Curs.Val_Cur).Column_Pos,
					   White,
					   Content_Curs.Values(Content_Curs.Val_Cur).Line(1..Content_Curs.Values(Content_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Content_Curs.Values(Content_Curs.Val_Cur).Line_Last > 0 then
				 Content_Curs.Values(Content_Curs.Val_Cur).Line(Content_Curs.Values(Content_Curs.Val_Cur).Line_Last) := ' ';
				 Content_Curs.Values(Content_Curs.Val_Cur).Line_Last := 
				   Content_Curs.Values(Content_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Define : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Price : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Content_Name..Content_Price loop
				 Draw_Text(Organizer.Obj_Main, Content_Curs.values(Val).Line_Pos,
					   Content_Curs.values(Val).Column_Pos,
					   White,
					   Content_Curs.values(Val).Line(1..Content_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Content_Curs.Values(Content_Curs.Val_Cur).Line_Pos,
					Content_Curs.Values(Content_Curs.Val_Cur).Column_Pos,
					White,
					Content_Curs.Values(Content_Curs.Val_Cur).Line(1..Content_Curs.Values(Content_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Content_Curs.Values(Content_Curs.Val_Cur).Line_Last < 45 then
			
			Content_Curs.Values(Content_Curs.Val_Cur).Line(Content_Curs.Values(Content_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Content_Curs.Values(Content_Curs.Val_Cur).Line_Last := 
			  Content_Curs.Values(Content_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Content_Name..Content_Price loop
			   Draw_Text(Organizer.Obj_Main, Content_Curs.values(Val).Line_Pos,
				     Content_Curs.values(Val).Column_Pos,
				     White,
				     Content_Curs.values(Val).Line(1..Content_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Content_Curs.Values(Content_Curs.Val_Cur).Line_Pos,
				  Content_Curs.Values(Content_Curs.Val_Cur).Column_Pos,
				  White,
				  Content_Curs.Values(Content_Curs.Val_Cur).Line(1..Content_Curs.Values(Content_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
	    or
	       when Obj_Tag = Planning =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Planning_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last /= 0 then
				 case Planning_Curs.Val_Cur is
				    when Planning_None =>
				       null;
					  
				    when Planning_To =>
				       
				       declare
					  Planning : constant Planning_Class := To_Planning(Planning_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Planning_Class '(Planning));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				 
				       Planning_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Planning_Curs.Val_Cur := Planning_Enum'Succ(Planning_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Pos,
					   Planning_Curs.Values(Planning_Curs.Val_Cur).Column_Pos,
					   White,
					   Planning_Curs.Values(Planning_Curs.Val_Cur).Line(1..Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last > 0 then
				 Planning_Curs.Values(Planning_Curs.Val_Cur).Line(Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last) := ' ';
				 Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last := 
				   Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "From : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "To   : ");
			      
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Planning_Name..Planning_To loop
				 Draw_Text(Organizer.Obj_Main, Planning_Curs.values(Val).Line_Pos,
					   Planning_Curs.values(Val).Column_Pos,
					   White,
					   Planning_Curs.values(Val).Line(1..Planning_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Pos,
					Planning_Curs.Values(Planning_Curs.Val_Cur).Column_Pos,
					White,
					Planning_Curs.Values(Planning_Curs.Val_Cur).Line(1..Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last < 45 then
			
			Planning_Curs.Values(Planning_Curs.Val_Cur).Line(Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last := 
			  Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Planning_Name..Planning_To loop
			   Draw_Text(Organizer.Obj_Main, Planning_Curs.values(Val).Line_Pos,
				     Planning_Curs.values(Val).Column_Pos,
				     White,
				     Planning_Curs.values(Val).Line(1..Planning_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Pos,
				  Planning_Curs.Values(Planning_Curs.Val_Cur).Column_Pos,
				  White,
				  Planning_Curs.Values(Planning_Curs.Val_Cur).Line(1..Planning_Curs.Values(Planning_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
			   
			   
	    or
	       when Obj_Tag = Event =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Event_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Event_Curs.Values(Event_Curs.Val_Cur).Line_Last /= 0 then
				 case Event_Curs.Val_Cur is
				    when Event_None =>
				       null;
					  
				    when Event_Define =>
				       
				       
				       declare
					  Event : constant Event_Class := To_Event(Event_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Event_Class '(Event));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;

				       Event_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Event_Curs.Val_Cur := Event_Enum'Succ(Event_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Event_Curs.Values(Event_Curs.Val_Cur).Line_Pos,
					   Event_Curs.Values(Event_Curs.Val_Cur).Column_Pos,
					   White,
					   Event_Curs.Values(Event_Curs.Val_Cur).Line(1..Event_Curs.Values(Event_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Event_Curs.Values(Event_Curs.Val_Cur).Line_Last > 0 then
				 Event_Curs.Values(Event_Curs.Val_Cur).Line(Event_Curs.Values(Event_Curs.Val_Cur).Line_Last) := ' ';
				 Event_Curs.Values(Event_Curs.Val_Cur).Line_Last := 
				   Event_Curs.Values(Event_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Date : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Period : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Locality : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Define : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Event_Name..Event_Define loop
				 Draw_Text(Organizer.Obj_Main, Event_Curs.values(Val).Line_Pos,
					   Event_Curs.values(Val).Column_Pos,
					   White,
					   Event_Curs.values(Val).Line(1..Event_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Event_Curs.Values(Event_Curs.Val_Cur).Line_Pos,
					Event_Curs.Values(Event_Curs.Val_Cur).Column_Pos,
					White,
					Event_Curs.Values(Event_Curs.Val_Cur).Line(1..Event_Curs.Values(Event_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Event_Curs.Values(Event_Curs.Val_Cur).Line_Last < 45 then
			
			Event_Curs.Values(Event_Curs.Val_Cur).Line(Event_Curs.Values(Event_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Event_Curs.Values(Event_Curs.Val_Cur).Line_Last := 
			  Event_Curs.Values(Event_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Event_Name..Event_Define loop
			   Draw_Text(Organizer.Obj_Main, Event_Curs.values(Val).Line_Pos,
				     Event_Curs.values(Val).Column_Pos,
				     White,
				     Event_Curs.values(Val).Line(1..Event_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Event_Curs.Values(Event_Curs.Val_Cur).Line_Pos,
				  Event_Curs.Values(Event_Curs.Val_Cur).Column_Pos,
				  White,
				  Event_Curs.Values(Event_Curs.Val_Cur).Line(1..Event_Curs.Values(Event_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
	    or
	       when Obj_Tag = Job =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Job_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Job_Curs.Values(Job_Curs.Val_Cur).Line_Last /= 0 then
				 case Job_Curs.Val_Cur is
				    when Job_None =>
				       null;
					  
				    when Job_Define =>
				       
				       declare
					  Job : constant Job_Class := To_Job(Job_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Job_Class '(Job));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				       
				       Job_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Job_Curs.Val_Cur := Job_Enum'Succ(Job_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Job_Curs.Values(Job_Curs.Val_Cur).Line_Pos,
					   Job_Curs.Values(Job_Curs.Val_Cur).Column_Pos,
					   White,
					   Job_Curs.Values(Job_Curs.Val_Cur).Line(1..Job_Curs.Values(Job_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Job_Curs.Values(Job_Curs.Val_Cur).Line_Last > 0 then
				 Job_Curs.Values(Job_Curs.Val_Cur).Line(Job_Curs.Values(Job_Curs.Val_Cur).Line_Last) := ' ';
				 Job_Curs.Values(Job_Curs.Val_Cur).Line_Last := 
				   Job_Curs.Values(Job_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Date : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Period : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Locality : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Define : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Job_Name..Job_Define loop
				 Draw_Text(Organizer.Obj_Main, Job_Curs.values(Val).Line_Pos,
					   Job_Curs.values(Val).Column_Pos,
					   White,
					   Job_Curs.values(Val).Line(1..Job_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Job_Curs.Values(Job_Curs.Val_Cur).Line_Pos,
					Job_Curs.Values(Job_Curs.Val_Cur).Column_Pos,
					White,
					Job_Curs.Values(Job_Curs.Val_Cur).Line(1..Job_Curs.Values(Job_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Job_Curs.Values(Job_Curs.Val_Cur).Line_Last < 45 then
			
			Job_Curs.Values(Job_Curs.Val_Cur).Line(Job_Curs.Values(Job_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Job_Curs.Values(Job_Curs.Val_Cur).Line_Last := 
			  Job_Curs.Values(Job_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Job_Name..Job_Define loop
			   Draw_Text(Organizer.Obj_Main, Job_Curs.values(Val).Line_Pos,
				     Job_Curs.values(Val).Column_Pos,
				     White,
				     Job_Curs.values(Val).Line(1..Job_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Job_Curs.Values(Job_Curs.Val_Cur).Line_Pos,
				  Job_Curs.Values(Job_Curs.Val_Cur).Column_Pos,
				  White,
				  Job_Curs.Values(Job_Curs.Val_Cur).Line(1..Job_Curs.Values(Job_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
	    or
	       when Obj_Tag = Meeting =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Meeting_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last /= 0 then
				 case Meeting_Curs.Val_Cur is
				    when Meeting_None =>
				       null;
					  
				    when Meeting_Define =>
				       
				 
				       declare
					  Meeting : constant Meeting_Class := To_Meeting(Meeting_Curs, Child_index);
				       begin
					  Add_To(App.Terminal, new Meeting_Class '(Meeting));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				       
				       Meeting_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Meeting_Curs.Val_Cur := Meeting_Enum'Succ(Meeting_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Pos,
					   Meeting_Curs.Values(Meeting_Curs.Val_Cur).Column_Pos,
					   White,
					   Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line(1..Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last > 0 then
				 Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line(Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last) := ' ';
				 Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last := 
				   Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Date : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Period : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Locality : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Define : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Meeting_Name..Meeting_Define loop
				 Draw_Text(Organizer.Obj_Main, Meeting_Curs.values(Val).Line_Pos,
					   Meeting_Curs.values(Val).Column_Pos,
					   White,
					   Meeting_Curs.values(Val).Line(1..Meeting_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Pos,
					Meeting_Curs.Values(Meeting_Curs.Val_Cur).Column_Pos,
					White,
					Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line(1..Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last < 45 then
			
			Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line(Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last := 
			  Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Meeting_Name..Meeting_Define loop
			   Draw_Text(Organizer.Obj_Main, Meeting_Curs.values(Val).Line_Pos,
				     Meeting_Curs.values(Val).Column_Pos,
				     White,
				     Meeting_Curs.values(Val).Line(1..Meeting_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Pos,
				  Meeting_Curs.Values(Meeting_Curs.Val_Cur).Column_Pos,
				  White,
				  Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line(1..Meeting_Curs.Values(Meeting_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
	    or
	       when Obj_Tag = Rendezvous =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Rendezvous_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last /= 0 then
				 case Rendezvous_Curs.Val_Cur is
				    when Rendezvous_None =>
				       null;
					  
				    when Rendezvous_Define =>
				       
				       declare
					  Rendezvous : constant Rendezvous_Class := To_Rendezvous(Rendezvous_Curs, Child_Index);
				       begin
					  Add_To(App.Terminal, new Rendezvous_Class '(Rendezvous));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				 
				       Rendezvous_Curs := Initialize;
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       Rendezvous_Curs.Val_Cur := Rendezvous_Enum'Succ(Rendezvous_Curs.Val_Cur);
				 end case;
				 Draw_Text(Organizer.Obj_Main, Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Pos,
					   Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Column_Pos,
					   White,
					   Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line(1..Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last));
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last > 0 then
				 Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line(Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last) := ' ';
				 Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last := 
				   Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Date : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Period : ");
			      Draw_Text(Organizer.Obj_Main, 4, 1, White, "Locality : ");
			      Draw_Text(Organizer.Obj_Main, 5, 1, White, "Define : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Rendezvous_Name..Rendezvous_Define loop
				 Draw_Text(Organizer.Obj_Main, Rendezvous_Curs.values(Val).Line_Pos,
					   Rendezvous_Curs.values(Val).Column_Pos,
					   White,
					   Rendezvous_Curs.values(Val).Line(1..Rendezvous_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Pos,
					Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Column_Pos,
					White,
					Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line(1..Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last < 45 then
			
			Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line(Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last := 
			  Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Rendezvous_Name..Rendezvous_Define loop
			   Draw_Text(Organizer.Obj_Main, Rendezvous_Curs.values(Val).Line_Pos,
				     Rendezvous_Curs.values(Val).Column_Pos,
				     White,
				     Rendezvous_Curs.values(Val).Line(1..Rendezvous_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Pos,
				  Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Column_Pos,
				  White,
				  Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line(1..Rendezvous_Curs.Values(Rendezvous_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
	    or
	       when Obj_Tag = Universe =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Content_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Enlight_Window(Organizer.Obj_Attr);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last /= 0 then
				 case Universe_Curs.Val_Cur is
				    when Universe_None =>
				       null;
				       
				    when Universe_Name =>
				       
				       declare
					  Universe : constant Universe_Class := To_Universe(Universe_Curs, Child_index);
				       begin
					  Add_To(App.Terminal, new Universe_Class '(Universe));
					  Save(App.Terminal.root, Organizer_Filename);
				       end;
				 
				       
				       --------------------------------
				       Universe_Curs := Initialize;
				       --------------------------------
				       
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       
				       Universe_Curs.Val_Cur := Universe_Enum'Succ(Universe_Curs.Val_Cur);
				 end case;
				 
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last > 0 then
				 Universe_Curs.Values(Universe_Curs.Val_Cur).Line(Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last) := ' ';
				 Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last := 
				   Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
			      for Val in Universe_Name..Universe_Name loop
				 Draw_Text(Organizer.Obj_Main, Universe_Curs.values(Val).Line_Pos,
					   Universe_Curs.values(Val).Column_Pos,
					   White,
					   Universe_Curs.values(Val).Line(1..Universe_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Pos,
					Universe_Curs.Values(Universe_Curs.Val_Cur).Column_Pos,
					White,
					Universe_Curs.Values(Universe_Curs.Val_Cur).Line(1..Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last < 45 then
			
			Universe_Curs.Values(Universe_Curs.Val_Cur).Line(Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last := 
			  Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Universe_Name..Universe_name loop
			   Draw_Text(Organizer.Obj_Main, Universe_Curs.values(Val).Line_Pos,
				     Universe_Curs.values(Val).Column_Pos,
				     White,
				     Universe_Curs.values(Val).Line(1..Universe_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Pos,
				  Universe_Curs.Values(Universe_Curs.Val_Cur).Column_Pos,
				  White,
				  Universe_Curs.Values(Universe_Curs.Val_Cur).Line(1..Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  Enlight_Window(Organizer.Obj_Main);
		  Text_Io.Put(Normal_Mode);
		  Draw_Text(Organizer.Obj_Main, 1, 1, White, "Name : ");
		  for Val in Universe_Name..Universe_Name loop
		     Draw_Text(Organizer.Obj_Main, Universe_Curs.values(Val).Line_Pos,
			       Universe_Curs.values(Val).Column_Pos,
			       White,
			       Universe_Curs.values(Val).Line(1..Universe_Curs.values(Val).Line_Last));
		  end loop;
		  
		  Draw_Text(Organizer.Obj_Main, Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Pos,
			    Universe_Curs.Values(Universe_Curs.Val_Cur).Column_Pos,
			    White,
			    Universe_Curs.Values(Universe_Curs.Val_Cur).Line(1..Universe_Curs.Values(Universe_Curs.Val_Cur).Line_Last));
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Text_Io.Put(Normal_Mode);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when others =>
			null;
		  end case;
		  
		  	    or
	       when Obj_Tag = Book or Obj_Tag = CDROM or Obj_Tag = DVD =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_Control(Wchar) then
			
			case Wchar is
			   when Wide_Character'Val(1) =>
			      Content_Curs := Initialize;
			      Enlight_Window(Organizer.Obj_Header);
			      Enlight_Window(Organizer.Obj_Main);
			      Enlight_Window(Organizer.Obj_Attr);
			      Text_Io.Put(Normal_Mode);
			      Obj_Tag := None;
			   when Wide_Character'Val(10) =>				 
			      if Media_Curs.Values(Media_Curs.Val_Cur).Line_Last /= 0 then
				 case Media_Curs.Val_Cur is
				    when Media_None =>
				       null;
				       
				    when Media_Editor =>
				       case Obj_Tag is
					  when Book =>
					     
					     declare
						Book : constant Book_Class := To_Book(Media_Curs, Child_index);
					     begin
						Add_To(App.Terminal, new Book_Class '(Book));
						Save(App.Terminal.root, Organizer_Filename);
					     end;
					  when CDROM =>
					     declare
						CDROM : constant CDROM_Class := To_CDROM(Media_Curs, Child_index);
					     begin
						Add_To(App.Terminal, new CDROM_Class '(CDROM));
						Save(App.Terminal.root, Organizer_Filename);
					     end;
					     
					  when DVD =>
					     declare
						DVD : constant DVD_Class := To_DVD(Media_Curs, Child_index);
					     begin
						Add_To(App.Terminal, new DVD_Class '(DVD));
						Save(App.Terminal.root, Organizer_Filename);
					     end;
					  when others =>
					     null;
				       end case;
					     
				       --------------------------------
				       Media_Curs := Initialize;
				       --------------------------------
				       
				       Enlight_Window(Organizer.Obj_Main);
				       Text_Io.Put(Normal_Mode);
				       Obj_Tag := None;
				    when others =>
				       
				       Media_Curs.Val_Cur := Media_Enum'Succ(Media_Curs.Val_Cur);
				 end case;
				 
			      else
				 Put(Character'Val(7));
			      end if;
			   when Wide_Character ' Val (127) =>
			      if Media_Curs.Values(Media_Curs.Val_Cur).Line_Last > 0 then
				 Media_Curs.Values(Media_Curs.Val_Cur).Line(Media_Curs.Values(Media_Curs.Val_Cur).Line_Last) := ' ';
				 Media_Curs.Values(Media_Curs.Val_Cur).Line_Last := 
				   Media_Curs.Values(Media_Curs.Val_Cur).Line_Last - 1;
			      end if;
			      Enlight_Window(Organizer.Obj_Main);
			      Text_Io.Put(Normal_Mode);
			      Draw_Text(Organizer.Obj_Main, 2, 1, White, "Author : ");
			      Draw_Text(Organizer.Obj_Main, 3, 1, White, "Editor : ");
			      Draw_Text(Organizer.Obj_Main, 1, 1, White, "Title : ");
			      for Val in Media_title..Media_Editor loop
				 Draw_Text(Organizer.Obj_Main, Media_Curs.values(Val).Line_Pos,
					   Media_Curs.values(Val).Column_Pos,
					   White,
					   Media_Curs.values(Val).Line(1..Media_Curs.values(Val).Line_Last));
			      end loop;
			      
			      Draw_Text(Organizer.Obj_Main, Media_Curs.Values(Media_Curs.Val_Cur).Line_Pos,
					Media_Curs.Values(Media_Curs.Val_Cur).Column_Pos,
					White,
					Media_Curs.Values(Media_Curs.Val_Cur).Line(1..Media_Curs.Values(Media_Curs.Val_Cur).Line_Last));
			   when others =>
			      null;
			end case;
		     elsif Media_Curs.Values(Media_Curs.Val_Cur).Line_Last < 45 then
			
			Media_Curs.Values(Media_Curs.Val_Cur).Line(Media_Curs.Values(Media_Curs.Val_Cur).Line_Last + 1) := Wchar;
			Media_Curs.Values(Media_Curs.Val_Cur).Line_Last := 
			  Media_Curs.Values(Media_Curs.Val_Cur).Line_Last + 1;
			
			for Val in Media_Title..Media_Editor loop
			   Draw_Text(Organizer.Obj_Main, Media_Curs.values(Val).Line_Pos,
				     Media_Curs.values(Val).Column_Pos,
				     White,
				     Media_Curs.values(Val).Line(1..Media_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Media_Curs.Values(Media_Curs.Val_Cur).Line_Pos,
				  Media_Curs.Values(Media_Curs.Val_Cur).Column_Pos,
				  White,
				  Media_Curs.Values(Media_Curs.Val_Cur).Line(1..Media_Curs.Values(Media_Curs.Val_Cur).Line_Last));
			
		     else
			Put(Character'Val(7));
		     end if;
		     
		  end Receive;
		  Enlight_Window(Organizer.Obj_Main);
		  Text_Io.Put(Normal_Mode);
		  
		  case Obj_Tag is		     
		     when None =>
			Draw(Organizer, App.Terminal);
			Text_Io.Put(Normal_Mode);
			Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
			Enlight_Window(Organizer.cmd_Win);
			Text_Io.Put(Normal_Mode);
			if App.Terminal.Cur = null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			elsif App.Terminal.Cur.Info.Name /= null then
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# " & Cmd_Line(Cmd_Line'First..Line_Last));
			else
			   Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# " & Cmd_Line(Cmd_Line'First..Line_Last));
			end if;
		     when Book .. DVD =>
			Draw_Text(Organizer.Obj_Main, 2, 1, White, "Author : ");
			Draw_Text(Organizer.Obj_Main, 3, 1, White, "Editor : ");
			Draw_Text(Organizer.Obj_Main, 1, 1, White, "Title : ");
			for Val in Media_Title..Media_Editor loop
			   Draw_Text(Organizer.Obj_Main, Media_Curs.values(Val).Line_Pos,
				     Media_Curs.values(Val).Column_Pos,
				     White,
				     Media_Curs.values(Val).Line(1..Media_Curs.values(Val).Line_Last));
			end loop;
			
			Draw_Text(Organizer.Obj_Main, Media_Curs.Values(Media_Curs.Val_Cur).Line_Pos,
				  Media_Curs.Values(Media_Curs.Val_Cur).Column_Pos,
				  White,
				  Media_Curs.Values(Media_Curs.Val_Cur).Line(1..Media_Curs.Values(Media_Curs.Val_Cur).Line_Last));
		     when others =>
			null;
		  end case;
		  
		  
		  
	    or
	       accept Halt do
		  Organizer.Proc.Halt;
	       end Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  Organizer_Process.Top := Top;
		  Initialize(Organizer, Top);
		  Draw(Organizer, App.Terminal);
		  Obj_Tag := None;
		  if Top = 0 then
		     Organizer.Proc.Start;
		  end if;
		  Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
		  if In_Print then
		     if Print_Index /= 0 then
			if Printed /= null then
			   Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			end if;
		     else
			if Printed /= null then
			   Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			else
			   App.Terminal.Cur.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			end if;
		     end if;
		  end if;
		  if Last /= 0 then
		     Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
		  end if;
		  if App.Terminal.Root = null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# ");
		  elsif App.Terminal.Cur.Info.Name /= null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# ");
		  else
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# ");
		  end if;
	       end Wake_Up;
	    or
	       
	       accept Suspend do
		  Organizer.Proc.Stop;
	       end Suspend;
	       exit;
	    or
	       accept Page_Down do
		  if Line_Index + (Organizer.List_Win.H-2) <= Last then
		     Line_Index := Line_Index + (Organizer.List_Win.H-2);
		     Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
		  else
		     Text_Io.Put(Character'Val(7));		     
		  end if;
	       end Page_Down;
	    or
	       accept Page_Up do
		  if Line_Index > 0 then
		     Line_Index := Line_Index - (Organizer.List_Win.H-2);
		     Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
		  else
		     Text_Io.Put(Character'Val(7));		     
		  end if;
	       end Page_Up;
	    or
	       accept Up_Arrow do
		  null;
	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  null;
	       end Down_Arrow;
	       
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt do
		  Organizer.Proc.Halt;
	       end Halt;
	       exit Main_Loop;
	    or
	       accept Suspend do
		  Organizer.Proc.Stop;
	       end Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  Initialize(Organizer, Top);
		  Organizer_Process.Top := Top;
		  Draw(Organizer, App.Terminal);
		  if Top = 0 then
		     Organizer.Proc.Start;
		  end if;
		  Obj_Tag := None;
		  Global_Content_Print(App.Terminal.Cur, Organizer.Content_Win.win);
		  if In_Print then
		     if Print_Index /= 0 then
			if Printed /= null then
			   Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			end if;
		     else
			if Printed /= null then
			   Printed.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			else
			   App.Terminal.Cur.Print(Organizer.Obj_Header.Win, Organizer.Obj_Main.Win);
			end if;
		     end if;
		  end if;
		  if Last /= 0 then
		     Success := Print_Set(Childs_Set, Organizer.List_Win, Line_Index, Total_Line);
		  end if;
		  if App.Terminal.Root = null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "# ");
		  elsif App.Terminal.Cur.Info.Name /= null then
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, App.Terminal.Cur.Info.Name.all & "# ");
		  else
		     Draw_Text(Organizer.Cmd_Win, 1, 1, White, "null# ");
		  end if;
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Organizer process halted");
   end Organizer_Process;
   
   
   task body Editor_Process is
      
      End_Of_Program : Boolean := False;
   begin

  Main_Loop:
      loop
	 loop
	    select
	       accept Receive (Wchar : in Wide_Character) do
		  if Is_Control(Wchar) then
		     case Wchar is
			when Wide_Character'Val(4) =>
			   End_Of_Program := True;
			when others =>
			   null;
		     end case;
		  else
		     case Wchar is
			when Wide_Character ' Val(32) =>
			   null;
			when others =>
			   null;
		     end case;
		  end if;
	       end Receive;
	       if End_Of_Program then
		  App.Desktop(2) := App.Desktop(1);
		  App.Desktop(1) := Machines(3);
		  App.Mode := Planning;
		  End_Of_Program := False;
		  exit;
	       end if;
	    or
	       accept Halt;
	       exit Main_loop;
	    or
	       accept Wake_Up(Top : in Natural) do
		  null;
	       end Wake_Up;
	    or
	       accept Suspend;
	       exit;
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow do
		  null;
	       end Up_Arrow;
	       
	    or
	       accept Down_Arrow do
		  null;
	       end Down_Arrow;
	       
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Del;
	    or
	       accept Overwrite;      
	    end select;
	 end loop;
	 loop
	    select
	       
	       accept Halt;
	       exit Main_Loop;
	    or
	       accept Suspend;
	    or
	       accept Wake_Up(Top : in Natural) do
		  null;		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
      end loop Main_Loop;
      --W_Io.Put_Line("Editor process halted");
   end Editor_Process;
   
   
   -------------------------------------------------------------------------------------------------------------------------------
   --                                                    Application                                                            --
   -------------------------------------------------------------------------------------------------------------------------------
   --                                                                                                                           --
   --                                                                                                                           --
   --                                                                                                                           --
   
   
   
   
   task body Console_Process is
      On_Esc : Boolean := False;
      End_Of_This_Task : Boolean := False;
   begin
      loop
	 select
	    accept Initialize do
	       App.Desktop := (1 => Machines(0), 2 => Machines(12));
	       for Mode in Machines'Range loop
		  if Machines(Mode) /= null then
		     
		     case Machines(Mode).Name is
			when None =>
			   None_Record(Machines(Mode).all).Proc.Suspend;
			when Log =>
			   Log_Record(Machines(Mode).all).Proc.Suspend;
			when MoneyBox =>
			   Moneybox_Record(Machines(Mode).all).Proc.Suspend;
			when Activity =>
			   Activity_Record(Machines(Mode).all).Proc.Suspend;
			when Stocks =>
			   Stocks_Record(Machines(Mode).all).Proc.Suspend;
			when Bank =>
			   bank_Record(Machines(Mode).all).Proc.Suspend;
			when Planning =>
			   planning_Record(Machines(Mode).all).Proc.Suspend;
			when Library =>
			   Library_Record(Machines(Mode).all).Proc.Suspend;
			when Manager =>
			   Manager_Record(Machines(Mode).all).Proc.Suspend;
			when Contacts =>
			   Contacts_Record(Machines(Mode).all).Proc.Suspend;
			when Menu =>
			   Menu_Record(Machines(Mode).all).Proc.Suspend;
			when Organizer =>
			   Organizer_Record(Machines(Mode).all).Proc.Suspend;
			when Editor=>
			   Editor_Record(Machines(Mode).all).Proc.Suspend;
			   
			when others =>
			   null;
		     end case;
		  end if;
	       end loop;

	    end Initialize;
	 or
	    accept Receive(Wchar : in Wide_Character) do
	       case App.Desktop(1).Name is
		  when None =>
		     None_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Organizer =>
		     Organizer_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Editor =>
		     Editor_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Planning =>
		     Planning_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Manager =>
		     Manager_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Moneybox =>
		     Moneybox_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Activity =>
		     Activity_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Stocks =>
		     Stocks_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Bank =>
		     Bank_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Log =>
		     Log_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Menu =>
		     Menu_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Contacts =>
		     Contacts_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when Library =>
		     Library_Record(App.Desktop(1).all).Proc.Receive(Wchar);
		  when others =>
		     null;
	       end case;
	    end Receive;
	 or
	    
	    accept Halt do
	       for Log in Log_Enum'Range loop
		  Close(App.Sessions(Log));
	       end loop;
	       

	       Statistic(App.Sessions, App.Logs);

	    
	       for Mode in Machines'Range loop
		  if Machines(Mode) /= null then
		     case Machines(Mode).Name is
			when None =>
			   None_Record(Machines(Mode).all).Proc.Halt;
			when Log =>
			   Log_Record(Machines(Mode).all).Proc.Halt;
			when MoneyBox =>
			   Moneybox_Record(Machines(Mode).all).Proc.Halt;
			when Activity =>
			   Activity_Record(Machines(Mode).all).Proc.Halt;
			when Stocks =>
			   Stocks_Record(Machines(Mode).all).Proc.Halt;
			when Bank =>
			   bank_Record(Machines(Mode).all).Proc.Halt;
			when Planning =>
			   planning_Record(Machines(Mode).all).Proc.Halt;
			when Library =>
			   Library_Record(Machines(Mode).all).Proc.Halt;
			when Manager =>
			   Manager_Record(Machines(Mode).all).Proc.Halt;
			when Contacts =>
			   Contacts_Record(Machines(Mode).all).Proc.Halt;
			when Menu =>
			   Menu_Record(Machines(Mode).all).Proc.Halt;
			when Organizer =>
			   organizer_Record(Machines(Mode).all).Proc.Halt;
			when Editor=>
			   Editor_Record(Machines(Mode).all).Proc.Halt;
			   
			when others =>
			   null;
		     end case;
		  end if;
	       end loop;
	    end Halt;
	       
	    --W_Io.Put_Line("Halting system.");
	    exit;
	 or
	    accept Mode (Mode : out Mode_Enum) do
	       if App.Mode /= Quit then
		  Mode := App.Desktop(1).Name;
	       else
		  Mode := Quit;
		  
	       end if;
	    end Mode;
	    
	 or
	    accept Reset do
	       Put(Ansi_Tty_Control.Clear_Screen);
	       --Errno := System("clear" & Character'Val(0));
	       for Log in Log_Enum'Range loop
		  Close(App.Sessions(Log));
	       end loop;
	       

	       Statistic(App.Sessions, App.Logs);

	       
	       if App.Desktop(1) /= null then
		  
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Suspend;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Suspend;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Suspend;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Suspend;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Suspend;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Suspend;
		     when Planning =>
			planning_Record(App.Desktop(1).all).Proc.Suspend;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Suspend;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Suspend;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Suspend;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Suspend;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Suspend;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Suspend;
		     when others =>
			null;
		  end case;
	       end if;
	       if App.Desktop(2) /= null then
		  
		  case App.Desktop(2).Name is
		     when None =>
			None_Record(App.Desktop(2).all).Proc.Suspend;
		     when Log =>
			Log_Record(App.Desktop(2).all).Proc.Suspend;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(2).all).Proc.Suspend;
		     when Activity =>
			Activity_Record(App.Desktop(2).all).Proc.Suspend;
		     when Stocks =>
			Stocks_Record(App.Desktop(2).all).Proc.Suspend;
		     when Bank =>
			bank_Record(App.Desktop(2).all).Proc.Suspend;
		     when Planning =>
			planning_Record(App.Desktop(2).all).Proc.Suspend;
		     when Library =>
			Library_Record(App.Desktop(2).all).Proc.Suspend;
		     when Manager =>
			Manager_Record(App.Desktop(2).all).Proc.Suspend;
		     when Contacts =>
			Contacts_Record(App.Desktop(2).all).Proc.Suspend;
		     when Menu =>
			Menu_Record(App.Desktop(2).all).Proc.Suspend;
		     when Organizer =>
			Organizer_Record(App.Desktop(2).all).Proc.Suspend;
		     when Editor=>
			Editor_Record(App.Desktop(2).all).Proc.Suspend;
		     when others =>
			null;
		  end case;
	       end if;
	       if App.Desktop(1).Name = None then
		  null;
	       elsif App.Desktop(1).Name = Log then
		  App.Desktop(2) := Machines(12);
		  App.Desktop(1) := Machines(10);
	       elsif App.Desktop(1).Name = Menu then
		  App.Desktop(1) := Machines(0);
		  App.Desktop(2) := Machines(10);
	       elsif App.Desktop(1).Name = Moneybox and
		 App.Desktop(2).Name = Activity then
		  App.Desktop(1) := Machines(4);
		  App.Desktop(2) := Machines(1);
	       elsif App.Desktop(1).Name =  Manager and
		 App.Desktop(2).Name = organizer then
		  App.Desktop(1) := Machines(1);
		  App.Desktop(2) := Machines(4);
	       elsif App.Desktop(1).name in Organizer .. moneybox then
		  App.Desktop(2) := App.desktop(1);
		  App.Desktop(1) := Machines(Mode_Enum'Pos(Mode_Enum'Succ(App.Desktop(1).Name)));
	       elsif App.Desktop(1).Name = Activity then
		  App.Desktop(1) := Machines(5);
		  App.Desktop(2) := Machines(6);

	       elsif App.Desktop(1).Name = Stocks then
		  App.Desktop(1) := Machines(6);
		  App.Desktop(2) := Machines(7);
	       elsif App.Desktop(1).Name = Log then
	       
		  App.Desktop(1) := App.Desktop(2);
		  App.Desktop(2) := Machines(9);
	       
	       end if;
	       if App.Desktop(2) /= null then
		  case App.Desktop(2).Name is
		     when None =>
			None_Record(App.Desktop(2).all).Proc.Suspend;			
		     when Log =>
			Log_Record(App.Desktop(2).all).Proc.Suspend;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(2).all).Proc.Suspend;
		     when Activity =>
			Activity_Record(App.Desktop(2).all).Proc.Suspend;
		     when Stocks =>
			Stocks_Record(App.Desktop(2).all).Proc.Suspend;
		     when Bank =>
			bank_Record(App.Desktop(2).all).Proc.Suspend;
		     when Planning =>
			planning_Record(App.Desktop(2).all).Proc.Suspend;
		     when Library =>
			Library_Record(App.Desktop(2).all).Proc.Suspend;
		     when Manager =>
			Manager_Record(App.Desktop(2).all).Proc.Suspend;
		     when Contacts =>
			Contacts_Record(App.Desktop(2).all).Proc.Suspend;
		     when Menu =>
			Menu_Record(App.Desktop(2).all).Proc.Suspend;
		     when Organizer =>
			organizer_Record(App.Desktop(2).all).Proc.Suspend;
		     when Editor=>
			Editor_Record(App.Desktop(2).all).Proc.Suspend;
		     when others =>
			null;
		  end case;
		  case App.Desktop(2).Name is
		     when None =>
			null;
		     when Log =>
			Log_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Activity =>
			Activity_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Stocks =>
			Stocks_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Bank =>
			bank_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Planning =>
			planning_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Library =>
			Library_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Manager =>
			Manager_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Contacts =>
			Contacts_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Menu =>
			null;
		     when Organizer =>
			organizer_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when Editor=>
			Editor_Record(App.Desktop(2).all).Proc.Wake_Up(App.Columns/2);
		     when others =>
			null;
		  end case;
	       end if;
	       delay 0.25;
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			Open(App.Sessions(None));
			None_Record(App.Desktop(1).all).Proc.Wake_Up(0);
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when MoneyBox =>
			Open(App.Sessions(Moneybox));
			Moneybox_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Activity =>
			Open(App.Sessions(Activity));
			Activity_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Stocks =>
			Open(App.Sessions(Stocks));
			Stocks_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Bank =>
			Open(App.Sessions(Bank));
			bank_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Planning =>
			Open(App.Sessions(Planning));
			Planning_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Menu =>
			--Errno := System("clear" & Character'Val(0));
			Menu_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Organizer =>
			Open(App.Sessions(Organizer));
			Organizer_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when Editor=>
			Open(App.Sessions(Computer));
			Editor_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     when others =>
			null;
		  end case;
	       end if;
	    end Reset;

	    
	    
	    
	 or
	    accept Switch (Mode : in Mode_Enum) do
	       for Log in Log_Enum'Range loop
		     Close(App.Sessions(Log));
		  end loop;
		  

		  Statistic(App.Sessions, App.Logs);

      
	       if Mode /= App.Mode then
		  Errno := System("clear" & Character'Val(0));
		  if App.Desktop(1) /= null then
		     case App.Desktop(1).Name is
			when None =>
			   None_Record(App.Desktop(1).all).Proc.Suspend;
			when Log =>
			   Log_Record(App.Desktop(1).all).Proc.Suspend;
			when MoneyBox =>
			   Moneybox_Record(App.Desktop(1).all).Proc.Suspend;
			when Activity =>
			   Activity_Record(App.Desktop(1).all).Proc.Suspend;
			when Stocks =>
			   Stocks_Record(App.Desktop(1).all).Proc.Suspend;
			when Bank =>
			   bank_Record(App.Desktop(1).all).Proc.Suspend;
			when Planning =>
			   planning_Record(App.Desktop(1).all).Proc.Suspend;
			when Library =>
			   Library_Record(App.Desktop(1).all).Proc.Suspend;
			when Manager =>
			   Manager_Record(App.Desktop(1).all).Proc.Suspend;
			when Contacts =>
			   Contacts_Record(App.Desktop(1).all).Proc.Suspend;
			when Menu =>
			   Menu_Record(App.Desktop(1).all).Proc.Suspend;
			when Organizer =>
			   Organizer_Record(App.Desktop(1).all).Proc.Suspend;
			when Editor=>
			   Editor_Record(App.Desktop(1).all).Proc.Suspend;
			when others =>
			   null;
		     end case;
		  end if;
		  if App.Desktop(2) /= null then
		     case App.Desktop(2).Name is
			when None =>
			   None_Record(App.Desktop(2).all).Proc.Suspend;
			when Log =>
			   Log_Record(App.Desktop(2).all).Proc.Suspend;
			when MoneyBox =>
			   Moneybox_Record(App.Desktop(2).all).Proc.Suspend;
			when Activity =>
			   Activity_Record(App.Desktop(2).all).Proc.Suspend;
			when Stocks =>
			   Stocks_Record(App.Desktop(2).all).Proc.Suspend;
			when Bank =>
			   bank_Record(App.Desktop(2).all).Proc.Suspend;
			when Planning =>
			   planning_Record(App.Desktop(2).all).Proc.Suspend;
			when Library =>
			   Library_Record(App.Desktop(2).all).Proc.Suspend;
			when Manager =>
			   Manager_Record(App.Desktop(2).all).Proc.Suspend;
			when Contacts =>
			   Contacts_Record(App.Desktop(2).all).Proc.Suspend;
			when Menu =>
			   Menu_Record(App.Desktop(2).all).Proc.Suspend;
			when Organizer =>
			   Organizer_Record(App.Desktop(2).all).Proc.Suspend;
			when Editor=>
			   Editor_Record(App.Desktop(2).all).Proc.Suspend;
			when others =>
			   null;
		     end case;
		  end if;
		  if Mode /= Quit then
		     App.Desktop(2) := App.Desktop(1);
		     App.Desktop(1) := Machines(Mode_Enum'Pos(Mode));
		     case App.desktop(2).Name is
			when None =>
			   null;
			when Log =>
			   Log_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when MoneyBox =>
			   Moneybox_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Activity =>
			   Activity_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Stocks =>
			   Stocks_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Bank =>
			   bank_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Planning =>
			   Planning_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Library =>
			   Library_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Manager =>
			   Manager_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Contacts =>
			   Contacts_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Menu =>
			   null;
			when Organizer =>
			   Organizer_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when Editor=>
			   Editor_Record(App.Desktop(2).all).Proc.Wake_up(App.Columns/2);
			when others =>
			   null;
		     end case;
		     delay 0.25;
		     case App.desktop(1).Name is
		     	when None =>
		     	   None_Record(App.Desktop(1).all).Proc.Wake_Up(0);
		     	when Log =>
		     	   Log_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when MoneyBox =>
		     	   Moneybox_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Activity =>
		     	   Activity_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Stocks =>
		     	   Stocks_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Bank =>
		     	   bank_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Planning =>
		     	   Planning_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Library =>
		     	   Library_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Manager =>
		     	   Manager_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Contacts =>
		     	   Contacts_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Menu =>
		     	   --Errno := System("clear" & Character'Val(0));
		     	   Menu_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Organizer =>
		     	   Organizer_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when Editor=>
		     	   Editor_Record(App.Desktop(1).all).Proc.Wake_up(0);
		     	when others =>
		     	   null;
		     end case;
		     
		  end if;
		  
		  App.Mode := Mode;
		  
		  
	       end if;
	    end Switch; 	
	    case App.Mode is
	       when Quit =>
		  
		  if End_Of_This_Task then
		     accept Halt;
		     
		     for Mode in Machines'Range loop
			
			if Machines(Mode) /= null then

			   case Machines(Mode).Name is
			      when None =>
				 None_Record(Machines(Mode).all).Proc.Halt;
			      when Log =>
				 Log_Record(Machines(Mode).all).Proc.Halt;
			      when MoneyBox =>
				 Moneybox_Record(Machines(Mode).all).Proc.Halt;
			      when Activity =>
				 Activity_Record(Machines(Mode).all).Proc.Halt;
			      when Stocks =>
				 Stocks_Record(Machines(Mode).all).Proc.Halt;
			      when Bank =>
				 bank_Record(Machines(Mode).all).Proc.Halt;
			      when Planning =>
				 planning_Record(Machines(Mode).all).Proc.Halt;
			      when Library =>
				 Library_Record(Machines(Mode).all).Proc.Halt;
			      when Manager =>
				 Manager_Record(Machines(Mode).all).Proc.Halt;
			      when Contacts =>
				 Contacts_Record(Machines(Mode).all).Proc.Halt;
			      when Menu =>
				 Menu_Record(Machines(Mode).all).Proc.Halt;
			      when Organizer =>
				 organizer_Record(Machines(Mode).all).Proc.Halt;
			      when Editor=>
				 Editor_Record(Machines(Mode).all).Proc.Halt;
				 
			      when others =>
				 null;
			   end case;
			end if;
			
		     end loop;		     
		     
		     
		     exit;		   		     
	       
		  else
		     
		     
		     W_Io.Put("Realy quit ?");
		     
		  end if;		  
		  End_Of_This_Task := True;
	       when None =>
		  Open(App.Sessions(None));
		  End_Of_This_Task := False;
	       when Organizer..Library =>
		  if App.Desktop(1) /= null then
		     case App.Desktop(1).Name is
			when None =>
			   
			   None_Record(App.Desktop(1).all).Proc.Wake_Up(0);
			   
			when Log =>
			   Log_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when MoneyBox =>
			   Open(App.Sessions(Moneybox));
			   Moneybox_Record(App.Desktop(1).all).Proc.Wake_up(0);
			   
			when Activity =>
			   Open(App.Sessions(Activity));
			   Activity_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Stocks =>
			   Open(App.Sessions(stocks));
			   Stocks_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Bank =>
			   Open(App.Sessions(bank));
			   bank_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Planning =>
			   Open(App.Sessions(planning));
			   Planning_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Library =>
			   Library_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Manager =>
			   Manager_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Contacts =>
			   Contacts_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Menu =>
			   --Errno := System("clear" & Character'Val(0));
			   Menu_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Organizer =>
			   Open(App.Sessions(Organizer));
			   Organizer_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when Editor=>
			   Open(App.Sessions(computer));
			   Editor_Record(App.Desktop(1).all).Proc.Wake_up(0);
			when others =>
			   null;
		     end case;
		  end if;
	       
	    end case;
	    
	    

	    
	 or
	    accept Set_On_Esc(Is_Escape : in Boolean) do            
	       On_Esc := Is_Escape;
	    end Set_On_Esc;
	 or
	    accept Page_Down do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Page_Down;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Page_Down;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Page_Down;
		     when others =>
			null;
		  end case;
	       end if;
	    end Page_Down;
	 or
	    accept Page_Up do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Page_Up;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Page_Up;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Page_Up;
		     when others =>
			null;
		  end case;
	       end if;
	    end Page_Up;
	 or
	    accept Up_Arrow do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Up_Arrow;
		     when others =>
			null;
		  end case;
	       end if;
	    end up_arrow;
	 or
	    accept Down_Arrow do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Down_Arrow;
		     when others =>
			null;
		  end case;
	       end if;
	    end Down_Arrow;
	 or
	    accept Left_Arrow;
	 or
	    accept Right_Arrow;   	 
	 or
	    accept Del;
	 or
	    accept Begin_Of do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.Begin_Of;
		     when others =>
			null;
		  end case;
	       end if;
	    end Begin_Of;
	 or
	    accept End_Of do
	       if App.Desktop(1) /= null then
		  case App.Desktop(1).Name is
		     when None =>
			None_Record(App.Desktop(1).all).Proc.End_Of;
		     when Log =>
			Log_Record(App.Desktop(1).all).Proc.End_Of;
		     when MoneyBox =>
			Moneybox_Record(App.Desktop(1).all).Proc.End_Of;
		     when Activity =>
			Activity_Record(App.Desktop(1).all).Proc.End_Of;
		     when Stocks =>
			Stocks_Record(App.Desktop(1).all).Proc.End_Of;
		     when Bank =>
			bank_Record(App.Desktop(1).all).Proc.End_Of;
		     when Planning =>
			Planning_Record(App.Desktop(1).all).Proc.End_Of;
		     when Library =>
			Library_Record(App.Desktop(1).all).Proc.End_Of;
		     when Manager =>
			Manager_Record(App.Desktop(1).all).Proc.End_Of;
		     when Contacts =>
			Contacts_Record(App.Desktop(1).all).Proc.End_Of;
		     when Menu =>
			Menu_Record(App.Desktop(1).all).Proc.End_Of;
		     when Organizer =>
			Organizer_Record(App.Desktop(1).all).Proc.End_Of;
		     when Editor=>
			Editor_Record(App.Desktop(1).all).Proc.End_Of;
		     when others =>
			null;
		  end case;
	       end if;
	    end End_Of;
	 or
	    accept Overwrite;
	    
	 end select;	 
	 end loop;
	 --W_Io.Put_Line("System halted.");
      end Console_Process;

end Sky.Walk ; 
