-- elhoim is full object organizer with editor and command interpreter.
-- Elhoim is Copyright (C) 2023 Manuel De Girardi ; 
--
--   This program is free software; you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation; either version 2 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program; if not, write to the Free Software
--   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
--
-- Date := "2023-05-26 17:40:38"
-- Version := "0.6.6b"
with Ansi_Console;
with Windows_Console;

with PragmARC.Menu_Handler;

with PragmARC.Rem_Nn_Wrapper;
use PragmARC.Rem_Nn_Wrapper;


with PragmARC.Ansi_Tty_Control;
use PragmARC.Ansi_Tty_Control;
with PragmARC.Wrapping;
use PragmARC;

with Ada.Wide_Text_Io;
with Ada.Text_Io;
with Ada.Characters.Handling;
use Ada.Characters;
with Ada.Strings.Wide_Fixed;
use Ada.Strings.Wide_Fixed;
with Ada.Strings.Fixed;
use Ada.Strings;
with Ada.Wide_Characters.Handling;
use Ada.Wide_Characters.Handling;
with Ada.Characters.Conversions;
use Ada.Strings;
with Ada.Calendar.Formatting;
use Ada.Calendar;

with Ada.Command_Line;
use Ada.Command_Line;

with Ada.Strings.UTF_Encoding.Strings;
with Ada.Strings.UTF_Encoding.Wide_Strings;

with Ada.Containers;
use Ada.Containers;

with Gnat.Directory_Operations;
use Gnat.Directory_Operations;
with Gnat.Command_Line;
with Gnat.Regexp;

use Gnat;

with System.Os_Constants;

with El.Interf;
with El.Result ;
with El.Attrib ;
with El.Params ;
with El.Strings;
with El.Window ;
with El.Shared ;
with El.Operat ;
with El.Outils ;
with El.Errors ;
with El.Langua ;

package body El.Center is
   
   use Interf;
   use Result ;
   use Attrib ;
   use Params ;
   use Strings;
   use Window ;
   use Shared ;
   use Operat ;
   use Outils ;
   use Errors ;
   
   use Ansi_Console;
   use Classes.Windows;
   
   
   package W_Io renames Ada.Wide_Text_Io;
   package T_Io renames Ada.Text_Io;
   use Gnat;
   use Os_Lib;
   
   Os_Name : constant System.Os_Constants.Os_Type := System.Os_Constants.Target_Os;
   
   ------------------------------------------------------------------------------------------
   --                                     Dialog mode                                      --
   ------------------------------------------------------------------------------------------
   
   function Count (Set : in Wide_String_Set;
		   Doc_Index : in Natural;			  
		   Line_index : in Natural; 
		   Wchar_Index : in Natural;
		   Height : in Line_Range;
		   Width : in Column_Range) return Natural is

      Total_Lines : Natural := 0;      
   begin
      
  Line_Count :
      for I in Doc_Index+1..Line_Index loop
	 
	 declare
	    tmp : constant Wide_String := 
	      (Set.lines(I).all);
	    Line : constant Wide_String := Tmp(Tmp'First..Wide_Fixed.Index_Non_Blank((Tmp), Backward));
	    
	    Chars_Max : constant Positive := Width;
	    Lines_Max : constant Positive := Height;
	 begin
	    exit when (Total_Lines >= Height);
	    if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
	       
	       for Line_Window in 0..Integer'min(Real_Length(Line)/Chars_Max, Lines_Max) loop		  
		  
		  Total_lines := Total_Lines + 1;
		  exit when (Total_Lines >= Height);
	       end loop;
	    else
	       Total_lines := Total_Lines + 1;
	       
	    end if;                           
	    
	 end;
	 exit when (Total_Lines >= Height);
      end loop Line_Count;
      return Total_Lines;
   end Count;

   
   
   function Print_Set (Set : in Wide_String_Set;
		       Into : in Window_Record;
		       Top : in Line_Index_Range;		       
		       Total_Lines: out Natural) return Boolean is
      height : constant Positive := Into.Win.H-2;
      Width  : constant Positive := Into.Win.W-2;
      Line_Count : Natural := 0;      
   begin
      Total_Lines := 0;
      if Set.Line_Last = 0 then
	 return True;
      end if;
      
      
  Print_Loop:
      for I in Top+1 .. Positive'min(Top+Height, Set.Line_Last) loop
	 
	 declare
	    tmp : constant Wide_String := Set.Lines(I).all;
	    Line : constant Wide_String := Tmp(Tmp'First..Wide_Fixed.Index_Non_Blank((Tmp), Backward));
	    
	    Chars_Max : constant Positive := Width-1;
	    Lines_Max : constant Positive := Height;
	    
	    
	    B_Inf : Positive := 1;
	    B_Sup : Positive := Chars_Max;

	    Bot   : Natural := 0;
	 begin
	    
	    
	    if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
	       
	       
	       
	       for Line_Window in 0 .. Integer'min(Real_Length(Line)/Chars_Max, Lines_Max) loop

		     B_Inf := Line_Window*Chars_Max + 1;
		     B_Sup := Integer'min((Line_Window + 1)*Chars_Max, Real_Length(Line));
		     begin
			
			if Real_length(Line) <= Chars_max then
			   Draw_Text(Into, Line_Count + 1, 1, White, Line);
			else
			   Draw_Text(Into, Line_Count + 1, 1, White, Line(B_Inf..B_Sup));                            
			end if;                 
		     end;
		     
		     Line_Count := Line_Count + 1;
		     if Line_Count >= height then		     
			exit Print_Loop;
		     end if;

		  end loop;
	    else

	       Line_Count := Line_Count + 1;
	    end if;                           
	 end;

	 if Line_Count > height then	    
	    exit Print_Loop;
	 end if;	 
      end loop Print_Loop;

      Total_Lines := Line_Count;
      return True;
   exception
      when others =>
	 return False;
   end Print_Set;
   
   
   procedure Load_File (Set : in out Wide_String_Set; Name : in String; Last : out Line_Index_Range) is
      
      File : W_Io.File_Type;
      
   begin
      W_Io.Open(File, W_Io.In_File, Name, Form => "WCEM=8");
      Last := 0;
      
      while not W_Io.End_Of_File(File) loop

   	 declare
   	    Line : constant Wide_String :=W_Io.Get_Line(File);
   	 begin

   	    Set.Lines(Last+1) := new Wide_String ' (Line);
   	    Last := Last + 1;

   	 end;
      end loop;
      Set.Line_Last := Last;
      W_Io.Close(File);
   end Load_File;
   
   
   
   function Word_Count (Line : in Wide_String; Max_Word : in Line_Index_Range) return Word_range is
      Number_Of_Word : Word_range:= 0;
      
      Set : Wide_String_Set(Max_Word);
      Set_Index : Natural := 0;
      Top : Natural := 0;
      Bot : Natural := 0;
   begin
      if Line'Length = 0 then
	 return 0;
      end if;
      if Index_Non_blank(Line) = 0 then
	 return 0;
      end if;
      Top := 1;
      for Wchar in Line'Range loop
	 if Is_Letter (Line(Wchar)) then
	    Bot := Bot + 1;
	 else
	    if Bot > Top then
	       Number_Of_Word := Number_Of_Word + 1;
	    end if;
	    Number_Of_Word := Number_Of_Word + 1;
	    Top := Wchar+1;
	    Bot := Positive'Min(Top, Line'Last);
	 end if;
	 
      end loop;
      return Number_Of_Word;
   end Word_Count;


   
   procedure Add_Line (Set : in out Wide_String_Set;
		       Last : in out Line_Index_Range;
		       Line : in Wide_String) is           
   begin
      Last := Set.Line_Last;
      if Last = Set.Lines'Last then
	 Wide_Free (Set.Lines ( 1 ) );
	 for Line in 1..Last - 1 loop
	    Set.Lines(Line) := Set.Lines (Line + 1);
	 end loop;
	 Set.lines(Last) := new Wide_String ' (Line);
      else
	 Set.Lines(Last+1) := new Wide_String ' (Line);
	 Last := Last + 1;
      end if;
      Set.Line_Last := Last;
   end Add_Line;
   
   
   procedure Devel_Ai_Draw_Proposition(Center : access Center_Record);
   procedure Devel_Ai_Draw_Neural_States(Center : access Center_Record);
   
   procedure Devel_Ai_Draw_Class_ranking(Center : access Center_Record);
   
   procedure Devel_Ai_Draw_Top_class(Center : access Center_Record);
   procedure Devel_Ai_Draw_Goal_Path(Center : access Center_Record);
   
   
   task body Dialog_Process is
      
      Suspended : Boolean := False;
      
      task Clock_Tasking is
	 entry Halt;
	 entry Suspend;
	 entry Start;
      end Clock_Tasking;
      
      Dialog_Lock : Binary_Semaphore(True, Default_Ceiling);
      
      
     
      task body Clock_Tasking is
	 Lock : Boolean := True;
      begin
	 Clock_Loop :
	     loop
		while Lock loop
		   select
		      accept Suspend;
		   or
		      accept Start do
			 Lock := False;
		      end Start;
		      exit;
		   or
		      accept Halt;
		      exit Clock_Loop;
		   end select;
		end loop;
		loop
		   select
		      accept Start;
		   or
		      accept Suspend do
			 Lock := True;
		      end Suspend;
		      exit;
		   or
		      accept Halt;
		      exit Clock_Loop;
		   or
		      delay 1.0;
		      case Center.Current_Mode is
			 when Dialog_Mode =>
			    Dialog_Lock.Seize;
			    T_Io.Put(Save_Cursor);
			    Draw_Text(Center.Frames(Header).Win, 1, Center.Frames(Header).Win.Win.W/2-10, Yellow, Handling.To_Wide_String(Formatting.Image(Clock)));
			    T_Io.Put(Restore_Cursor);
			    Dialog_Lock.Release;
			 when others =>
			    null;
		      end case;
		   end select;
		end loop;
	     end loop Clock_Loop;
	     Center.Global_Lock.Seize;
	     W_Io.Put_Line("Clocking halted");
	     Center.Global_Lock.Release;
      end Clock_Tasking;
      
      
      
      Height : Positive := Center.Frames(Dialog_View).Win.H-2;
      Width  : Positive := Center.Frames(Dialog_View).Win.W-2;
      
      
      
      
      
      
      subtype Agent_Enum is El.Tag_Name range Planning .. House;
      
      type Abstract_Manager(Agent_Max : Positive);
      
      type Manager_Access is access all Abstract_Manager'Class;
      
      type Abstract_Ai_Agent;
      task type Partial_Agent(Agent : access Abstract_Ai_Agent ' Class) is
	 entry Switch_Manager(To : in Manager_Access);
	 entry Receive (Obj : in Config.Server.Engine.Intelligency.Classes.Abstracted_Access);
	 entry Receive (Line : in Wide_String);
	 entry Halt;
      end Partial_Agent;
			      
			      
      type Ai_Agent_Access is  access all Abstract_Ai_Agent'Class;
      
      type Agent_Table is array (Positive range <>) of Ai_Agent_Access;
      
      type Abstract_Manager(Agent_Max : Positive) is tagged limited
	 record	   
	    Agents : Agent_Table(1..Agent_Max);
	    Agent_Last : Natural := 0;
	 end record;
      
      type Abstract_Ai_Agent is abstract tagged limited
	 record
	    Agent : Partial_Agent (Abstract_Ai_Agent ' access) ;
	    Manager : Manager_Access;
	 end record;
      
      
      task Deamon_Network is
	 entry Receive (Feedback : in Input_type;
			Input    : in Input_Type;
			Output   : in Output_Type);
	 
	 entry Halt;
      end Deamon_Network;
      
      
      
      
      
      task body Partial_Agent is
      begin
	 loop
	    select
	       accept Switch_Manager(To : in Manager_Access);
	    or
	       accept Receive (Obj : in Config.Server.Engine.Intelligency.Classes.Abstracted_Access);
	    or
	       accept Receive (Line : in Wide_String);
	    or
	       accept Halt;
	       exit;
	    end select;
	 end loop;
      end Partial_Agent;
      
      
      
      
      
      
      
      
      
      
      Dialog_Initialized  : Boolean := False;
      
      
      
      task body Deamon_Network is
	 
      begin
	 loop
	    select
	       
	       accept Receive (Feedback : in Input_type;
			Input    : in Input_Type;
			Output   : in Output_Type) do
		  Center.Samples(1).Input := Feedback;
		     
		  Center.Samples(1).output := Neural_Retrain.Output_Type(Input);
		     
		  Center.Samples(2).Input := Input;
		     
		  Center.Samples(2).output := Output;
	       end Receive;
		  
	    or
	       accept Halt;
	       exit;
	    or
	       delay 90.0;
	       if not Suspended and Dialog_initialized then
		  Dialog_Process.Dialog;
	       end if;	       
	    end select;
	 end loop;
      end Deamon_Network;
      
      
      
      
      Conv       : Real := 0.006;
      Epoch_Max : Natural := 100;
      Reuse      : constant Boolean := True;
      Rms_Error  : Real := 0.0;
      
      End_Of_This_Task      : Boolean := False;
      
      Is_Overwrite        : Boolean := False;
      
      Wchar_Max        : Positive := Cols-4;      
      Wline            : Wide_String_Access := new Wide_String ' ("");
      Wbuffer          : Wide_String_Access;
      Wline_Length     : Natural := 0;
      Current_Pos      : Natural := 1;
      Left_Buffer      : Wide_String_Access;
      Word_Number      : Word_Range := 0;
      
      Response_Buffer  : Wide_String_Set(1); 
      internal_Buffer  : Wide_String_Set(1); 
      
	  
      Dialog_Buffer_Last  : Line_Index_Range := 0;
      Dialog_Buffer_Index : Line_Index_Range := 0;
      Dialog_Buffer       : Wide_String_Set(2880); -- 24 h self : 1440 * 2.
      Samples_Last        : Line_Index_Range := 0;
      Samples_Index       : Line_Index_Range := 0;
      Samples_Set         : Wide_String_Set(1024); 
      Samples_Loaded      : Boolean := False;
      Glossary_Last       : Line_Index_Range := 0;
      Glossary_Index      : Line_Index_Range := 0;
      Glossary_Set        : Wide_String_Set(4096); 
      Glossary_Loaded     : Boolean := False;
      Total_Lines         : Line_Index_Range := 0;
      Total               : Line_Index_Range := 0;
      View                : Boolean := False; -- Samples if false, Glossary else. Tab to chage.
      
      
   begin
      while not End_Of_This_Task loop
	 loop
	    select
	       accept Initialize(User_Text : in Wide_String) do
		  
		  declare
		     
		  begin
		     Initialize(Center.Frames(Header).Win, 1, 1, 3, Cols,  White, Red);
		     Initialize(Center.Frames(Global_View).Win, 4, 1, Lines-3, Cols,  green, Yellow);
		     Initialize(Center.Frames(Input).Win, 6, 2, 3, Cols-2,  Cyan, White);
		     Initialize(Center.Frames(Resp).Win, 10, 2, 5, Cols/2-2,  Cyan, White);
		     Initialize(Center.Frames(Output).Win, 16, 2, 5, Cols/2-2,  Cyan, White);
		     Initialize(Center.Frames(Dialog_View).Win, 22, 2, Lines/8*5, Cols/2-2,  Cyan, White);
		     Initialize(Center.Frames(Buffer).Win, Lines-5, 2, 5, Cols/2-2,  Cyan, White);
		     Initialize(Center.Frames(Right_View).Win, 10, Cols/2, Lines-10, Cols/2,  yellow, green);
		  end;
		  
		  Dialog_Lock.Seize;
		  
		  Draw_Window(Center.Frames(Header).Win);
		  Draw_Window(Center.Frames(Global_View).Win);
		  Draw_Window(Center.Frames(Input).Win);
		  Draw_Window(Center.Frames(Resp).Win);
		  Draw_Window(Center.Frames(Output).Win);
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  Draw_Window(Center.Frames(Buffer).Win);
		  Draw_Window(Center.Frames(Right_View).Win);
		  -------------------------------------------------------------------
		  --                  Drawing the windows titles                   --
		  -------------------------------------------------------------------	    
		  Draw_Text(Center.Frames(Global_View).Win, 1, 1, Green, "Input");
		  Draw_Text(Center.Frames(Global_View).Win, 5, 1, Magenta, "Response");
		  Draw_Text(Center.Frames(Global_View).Win, 11, 1, Yellow, "Internal");	    
		  Draw_Text(Center.Frames(Global_View).Win, 17, 1, Cyan, "Dialog");
		  Draw_Text(Center.Frames(Global_View).Win, 5, Columns/4*3-8, Blue, "Samples/Glossary");
		  -------------------------------------------------------------------
		  --                  Drawing the buffer states                    --
		  -------------------------------------------------------------------
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  case View is
		     when False =>		  
			if Os_Lib.Is_Regular_File(Center.Set_Filename.all) then
			   Load_File(Samples_Set, Center.Set_Filename.all, Samples_Last);
			elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & Center.Set_Filename.all) then	       
			   Load_File(Samples_Set, "data" & Dir_Separator & Center.Set_Filename.all, Samples_Last);
			end if;      	    
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
			Samples_Loaded := True;
		     when True =>
			if Os_Lib.Is_Regular_File("data" & Dir_Separator & "glossary.txt") then
			   Load_File(Glossary_Set, "data" & Dir_Separator & "glossary.txt", Glossary_Last);
			   if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			   end if;
			end if;
			Glossary_Loaded := True;
		  end case;	 
		  
		  if User_Text'Length /= 0 then
		     
		     Center.Input := String2input (User_Text, Center.Words);
		     Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Wline.all);
		     if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			Dialog_Buffer_Index := Dialog_Buffer_Index + Wline'Length/(Center.Frames(Dialog_View).Win.Win.W-2);
		     end if;
		     
		     declare
			Request : Request_NN;
		     begin
			Request.Respond (Center.Input, Center.Output);
		     exception
			when others =>
			   null;
		     end;
		     
		     Center.Dialog_Response := new Wide_String ' (Output2string (Center.Output, Center.Words));
		     declare
			Request : Request_NN;
		     begin
			Request.Respond (Neural_Retrain.Input_Type(Center.Output), Center.Feedback);
		     exception
			when others =>
			   null;
		     end;
		     
		     Center.Internal := new Wide_String ' (Output2string(Center.Feedback, Center.Words));
		     Center.Input := Neural_Retrain.Input_Type(Center.Feedback);
		     if Center.Dialog_Response /= null and then Center.Dialog_Response'Length /= 0 then
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Dialog_Response.all);			   			   
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + Center.Dialog_Response'Length/(Center.Frames(Dialog_View).Win.Win.W-2);
			end if;
			Total := 0;
			Add_Line(Center.Response_Buffer, Center.Response_Buffer.Line_Last, Center.Dialog_Response.all);
			if Print_Set(Center.Response_Buffer, Center.Frames(Resp).Win, 0, Total) then
			   null;
			end if;
		     else
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, "");	      			      
			
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			end if;
			Draw_Text(Center.Frames(Resp).Win, 1, 1, White, "");
		     end if;
		     if Center.Internal /= null and then Center.Internal'Length /= 0 then
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Internal.all);
			
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + Center.internal'Length/(Center.Frames(Dialog_View).Win.Win.W-2);
			end if;
			Total := 0;
			Add_Line(Center.Internal_Buffer, Total, Center.Internal.all);
			if Print_Set(Center.Internal_Buffer, Center.Frames(Output).Win, 0, Total) then
			   null;
			end if;
			
		     else
			Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "");			      
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			end if;
			Draw_Text(Center.Frames(Output).Win, 1, 1, White, "");
		     end if;			   
		  end if;

		  Total_Lines := 0;
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index, Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;

		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Dialog_Lock.Release;
		  Clock_Tasking.Start;
	       end Initialize;
	       exit;
	    or
	       accept Halt do
		  End_Of_This_Task := True;
		  Clock_Tasking.Halt;	       
		  Deamon_Network.Halt;
	       end Halt;
	       exit;
	    or
	       accept Lock;
	    or
	       accept Wake_Up do
		  Suspended := False;	       
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Header).Win);
		  Draw_Window(Center.Frames(Global_View).Win);
		  Draw_Window(Center.Frames(Input).Win);
		  Draw_Window(Center.Frames(Resp).Win);
		  Draw_Window(Center.Frames(Output).Win);
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  Draw_Window(Center.Frames(Buffer).Win);
		  Draw_Window(Center.Frames(Right_View).Win);
		  -------------------------------------------------------------------
		  --                  Drawing the windows titles                   --
		  -------------------------------------------------------------------	    
		  Draw_Text(Center.Frames(Global_View).Win, 1, 1, Green, "Input");
		  Draw_Text(Center.Frames(Global_View).Win, 5, 1, Magenta, "Response");
		  Draw_Text(Center.Frames(Global_View).Win, 11, 1, Yellow, "Internal");	    
		  Draw_Text(Center.Frames(Global_View).Win, 17, 1, Cyan, "Dialog");
		  Draw_Text(Center.Frames(Global_View).Win, 5, Columns/4*3-8, Blue, "Samples/Glossary");
		  -------------------------------------------------------------------
		  --                  Drawing the buffer states                    --
		  -------------------------------------------------------------------
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  case View is
		     when False =>		  
			if Os_Lib.Is_Regular_File(Center.Set_Filename.all) then
			   Samples_Set.Line_Last := 0;
			   Load_File(Samples_Set, Center.Set_Filename.all, Samples_Last);
			elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & Center.Set_Filename.all) then	       
			   Load_File(Samples_Set, "data" & Dir_Separator & Center.Set_Filename.all, Samples_Last);
			end if;      	    
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
			Samples_Loaded := True;
		     when True =>
			if Os_Lib.Is_Regular_File("data" & Dir_Separator & "glossary.txt") then
			   Glossary_Set.Line_Last := 0;
			   Load_File(Glossary_Set, "data" & Dir_Separator & "glossary.txt", Glossary_Last);
			   if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			   end if;
			end if;
			Glossary_Loaded := True;
		  end case;
		  
		  if Print_Set(Center.Response_Buffer, Center.Frames(Resp).Win, 0, Total) then
		     null;
		  end if;
		  Total := 0;
		  if Print_Set(Center.Internal_Buffer, Center.Frames(Output).Win, 0, Total) then
		     null;
		  end if;
		  
		  Total_Lines := 0;
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index, Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;

		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  Dialog_Lock.Release;
		  Clock_Tasking.Start;
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
	 while not End_Of_This_Task Loop
	    select
	       accept Wake_Up do
		  Suspended := False;	       
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Header).Win);
		  Draw_Window(Center.Frames(Global_View).Win);
		  Draw_Window(Center.Frames(Input).Win);
		  Draw_Window(Center.Frames(Resp).Win);
		  Draw_Window(Center.Frames(Output).Win);
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  Draw_Window(Center.Frames(Buffer).Win);
		  Draw_Window(Center.Frames(Right_View).Win);
		  -------------------------------------------------------------------
		  --                  Drawing the windows titles                   --
		  -------------------------------------------------------------------	    
		  Draw_Text(Center.Frames(Global_View).Win, 1, 1, Green, "Input");
		  Draw_Text(Center.Frames(Global_View).Win, 5, 1, Magenta, "Response");
		  Draw_Text(Center.Frames(Global_View).Win, 11, 1, Yellow, "Internal");	    
		  Draw_Text(Center.Frames(Global_View).Win, 17, 1, Cyan, "Dialog");
		  Draw_Text(Center.Frames(Global_View).Win, 5, Columns/4*3-8, Blue, "Samples/Glossary");
		  -------------------------------------------------------------------
		  --                  Drawing the buffer states                    --
		  -------------------------------------------------------------------
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  case View is
		     when False =>		  
			if Os_Lib.Is_Regular_File(Center.Set_Filename.all) then
			   Samples_Set.Line_Last := 0;
			   Load_File(Samples_Set, Center.Set_Filename.all, Samples_Last);
			elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & Center.Set_Filename.all) then	       
			   Load_File(Samples_Set, "data" & Dir_Separator & Center.Set_Filename.all, Samples_Last);
			end if;      	    
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
			Samples_Loaded := True;
		     when True =>
			if Os_Lib.Is_Regular_File("data" & Dir_Separator & "glossary.txt") then
			   Glossary_Set.Line_Last := 0;
			   Load_File(Glossary_Set, "data" & Dir_Separator & "glossary.txt", Glossary_Last);
			   if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			   end if;
			end if;
			Glossary_Loaded := True;
		  end case;
		  Total := 0;
		  
		  if Print_Set(Center.Response_Buffer, Center.Frames(Resp).Win, 0, Total) then
		     null;
		  end if;
		  Total := 0;
		  if Print_Set(Center.Internal_Buffer, Center.Frames(Output).Win, 0, Total) then
		     null;
		  end if;
		  
		  Total_Lines := 0;
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index, Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;

		  
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Dialog_Lock.Release;
		  Clock_Tasking.Start;
	       end Wake_Up;
	    or
	       accept Halt do
		  End_Of_This_Task := True;
		  Clock_Tasking.Halt;
		  Deamon_Network.Halt;
	       end Halt;
	       exit;
	    or
	       accept Initialize(User_Text : in Wide_String) do
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Header).Win);
		  Draw_Window(Center.Frames(Global_View).Win);
		  Draw_Window(Center.Frames(Input).Win);
		  Draw_Window(Center.Frames(Resp).Win);
		  Draw_Window(Center.Frames(Output).Win);
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  Draw_Window(Center.Frames(Buffer).Win);
		  Draw_Window(Center.Frames(Right_View).Win);
		  -------------------------------------------------------------------
		  --                  Drawing the windows titles                   --
		  -------------------------------------------------------------------	    
		  Draw_Text(Center.Frames(Global_View).Win, 1, 1, Green, "Input");
		  Draw_Text(Center.Frames(Global_View).Win, 5, 1, Magenta, "Response");
		  Draw_Text(Center.Frames(Global_View).Win, 11, 1, Yellow, "Internal");	    
		  Draw_Text(Center.Frames(Global_View).Win, 17, 1, Cyan, "Dialog");
		  Draw_Text(Center.Frames(Global_View).Win, 5, Columns/4*3-8, Blue, "Samples/Glossary");
		  -------------------------------------------------------------------
		  --                  Drawing the buffer states                    --
		  -------------------------------------------------------------------
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  case View is
		     when False =>		  
			if Os_Lib.Is_Regular_File(Center.Set_Filename.all) then
			   Samples_Set.Line_Last := 0;
			   Load_File(Samples_Set, Center.Set_Filename.all, Samples_Last);
			elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & Center.Set_Filename.all) then	       
			   Load_File(Samples_Set, "data" & Dir_Separator & Center.Set_Filename.all, Samples_Last);
			end if;      	    
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
			Samples_Loaded := True;
		     when True =>
			if Os_Lib.Is_Regular_File("data" & Dir_Separator & "glossary.txt") then
			   Glossary_Set.Line_Last := 0;
			   Load_File(Glossary_Set, "data" & Dir_Separator & "glossary.txt", Glossary_Last);
			   if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			   end if;
			end if;
			Glossary_Loaded := True;
		  end case;
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  Dialog_Lock.Release;
	       end Initialize;
	    or
	       -------------------------------------------------------------------------------
	       --                    Dialog recheive an Wide_Character                      --
	       -------------------------------------------------------------------------------
	       accept Receive (Wchar : in Wide_Character) do
		  
		  End_Of_This_Task := False;
		  if Is_Control(Wchar) then
		     case Wchar is
			when Wide_Character'Val(1) =>
			   null; -- save (as);
			when Wide_Character'Val(9) =>
			   -- switch between sample and glossary
			   View := not View;
			   Dialog_Lock.Seize;
			   Draw_Window(Center.Frames(Right_View).Win);			   
			   case View is
			      when False =>
				 if not Samples_Loaded then
				    if Os_Lib.Is_Regular_File(Center.Set_Filename.all) then
				       
				       Samples_Set.Line_Last := 0;
				       Load_File(Samples_Set, Center.Set_filename.all, Samples_Last);
				       
				    elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & Center.Set_filename.all) then
				       Load_File(Samples_Set, "data" & Dir_Separator & Center.Set_Filename.all, Samples_Last);
				    end if;
				    Samples_Loaded := True;
				 end if;
				 
				 if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, 0, Total) then
				    Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
				 end if;
				 
			      when True =>
				 

				 if Os_Lib.Is_Regular_File("data" & Dir_Separator & "glossary.txt") then
				    
				    Glossary_Set.Line_Last := 0;
				    Load_File(Glossary_Set, "data" & Dir_Separator & "glossary.txt", Glossary_Last);
				    
				 end if;
				 Glossary_Loaded := True;
				 

				 if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, 0, Total) then
				    Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
				 end if;
				 
			   end case;

			   Dialog_Lock.Release;

			when Wide_Character'Val(10) | Wide_Character'Val(13) =>			
			   
			   --------------------------------------------------
			   --              Network intercation             --
			   --------------------------------------------------
			   if Wline /= null and then Wline'Length /= 0 then 
			      -----------------------------------------------------------
			      --                 Initilizeing neural input             --
			      Center.Input := String2input (Wline.all, Center.Words);  --
			      --						       --
			      -----------------------------------------------------------
			      
			      Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Wline.all);
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + Wline'Length/(Center.Frames(Dialog_View).Win.Win.W-2);
			      end if;
			   else
			      Center.Input := string2Input ("", Center.words);
			      
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "");			      
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			      end if;
			   end if;
			   
			   Wide_Free(Wline);
			   Wline := new Wide_String ' ("");
			   Current_Pos := 1;
			   
			   
			   declare
			      Request : Request_NN;
			   begin
			      Request.Respond (Center.Input, Center.Output);
			   exception
			      when others =>
				 null;
			   end;
			   
			   Center.Dialog_Response := new Wide_String ' (Output2string (Center.Output, Center.Words));
			   
			   declare
			      Request : Request_NN;
			   begin
			      Request.Respond (Neural_Retrain.Input_Type(Center.Output), Center.Feedback);
			   exception
			      when others =>
				 null;
			   end;
			   
			   Center.Internal := new Wide_String ' (Output2string(Center.Feedback, Center.Words));
			   
			   Dialog_Initialized := True;
			   
			   select
			      
			      Deamon_Network.Receive(Neural_Retrain.Input_Type(Center.Feedback),
						     Center.Input, Center.Output);
			      Neural_Retrain.Train(Center.Samples ' access, Epoch_Max, Conv, Reuse, Rms_Error);
			   		      
			   or
			      delay 1.0;
			   end select;
			      
			   Center.Input := Neural_Retrain.Input_Type(Center.Feedback);
			   
			   
			   
			   
			   --------------------------------------------------
			   --          Drawing the results                 --
			   --------------------------------------------------
			   
			   Dialog_Lock.Seize;
			   Draw_Window(Center.Frames(Resp).Win);
			   Draw_Window(Center.Frames(Output).Win);
			   Draw_Window(Center.Frames(Dialog_View).Win);
			   Draw_Window(Center.Frames(Buffer).Win);
			   if Center.Dialog_Response /= null and then Center.Dialog_Response'Length /= 0 then
			      Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Dialog_Response.all);			   			   
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + Center.Dialog_Response'Length/(Center.Frames(Dialog_View).Win.Win.W-2) + 1;
			      end if;
			      Total := 0;
			      Add_Line(Center.Response_Buffer, Center.Response_Buffer.Line_Last, Center.Dialog_Response.all);
			      if Print_Set(Center.Response_Buffer, Center.Frames(Resp).Win, 0, Total) then
				 null;
			      end if;
			   else
			      Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, "");	      			      
			      
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			      end if;
			      Draw_Text(Center.Frames(Resp).Win, 1, 1, White, "");
			   end if;
			   
			   
			   
			   if Center.Internal /= null and then Center.Internal'Length /= 0 then
			      Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Internal.all);
			      
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + Center.internal'Length/(Center.Frames(Dialog_View).Win.Win.W-2) + 1;
			      end if;
			      Total := 0;
			      Add_Line(Center.Internal_Buffer, Total, Center.Internal.all);
			      if Print_Set(Center.Internal_Buffer, Center.Frames(Output).Win, 0, Total) then
				 null;
			      end if;
			      
			   else
			      Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "");			      
			      if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
				 Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			      end if;
			      Draw_Text(Center.Frames(Output).Win, 1, 1, White, "");
			   end if;
			   
			   Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
			   Dialog_Lock.Release;
			when Wide_Character'Val(12)  =>
			   
			   Ansi_Console.Clear_Screen;
			   Dialog_Lock.Seize;
			   Draw_Window(Center.Frames(Header).Win);
			   Draw_Window(Center.Frames(Global_View).Win);
			   -------------------------------------------------------------------
			   --                  Drawing the windows titles                   --
			   -------------------------------------------------------------------	    
			   Draw_Text(Center.Frames(Global_View).Win, 1, 1, Green, "Input");
			   Draw_Text(Center.Frames(Global_View).Win, 5, 1, Magenta, "Response");
			   Draw_Text(Center.Frames(Global_View).Win, 11, 1, Yellow, "Internal");	    
			   Draw_Text(Center.Frames(Global_View).Win, 17, 1, Cyan, "Dialog");
			   Draw_Text(Center.Frames(Global_View).Win, 5, Columns/4*3-8, Blue, "Samples/Glossary");
			   
			   Draw_Window(Center.Frames(Input).Win);
			   Draw_Window(Center.Frames(Resp).Win);
			   Draw_Window(Center.Frames(Output).Win);
			   Draw_Window(Center.Frames(Dialog_View).Win);
			   Draw_Window(Center.Frames(Buffer).Win);
			   Draw_Window(Center.Frames(Right_View).Win);
			   case View is
			      when False =>			
				 if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
				    Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
				 end if;
			      when True =>
				 
				 if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
				    Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
				 end if;
			   end case;
			   Dialog_Lock.Release;
			when Wide_Character'Val(127) =>
			   if Wline /= null and then
			     Wline'Length > 0 then
			      if Current_Pos > 1 then
				 case Is_Overwrite is
				    when False =>
				       if Current_Pos > 2 then
					  if Wline'Length >= 2 then
					     Left_Buffer := new Wide_String ' (Wline(1..Current_Pos-2));
					     if Current_Pos <= Wline'Length then
						Wbuffer := new Wide_String ' (Left_Buffer.all & Wline(Current_Pos..Wline'Last));
					     else
						Wbuffer := new Wide_String ' (Left_Buffer.all);
					     end if;
					  else
					     Left_Buffer := new Wide_String ' (Wline(1..Current_Pos-2));
					     Wbuffer := new Wide_String ' (Left_Buffer.all);
					  end if;
				       elsif Current_Pos = 2 then
					  if Wline'Length >= 2 then
					     Left_Buffer := new Wide_String ' (Wline(Current_pos..Wline'Length));
					     Wbuffer := new Wide_String ' (Left_Buffer.all);
					  else
					     Wbuffer := new Wide_String ' ("");
					  end if;
				       end if;				       				       
				       Wide_Free(Wline);
				       Wline := new Wide_String ' (Wbuffer.all);
				       Wide_Free(Wbuffer);					  
				       Wide_Free(Left_Buffer);
				       Current_Pos := Current_Pos - 1;				       
				       Wline_Length := Wline_Length - 1;
				    when True =>
				       Wide_Fixed.Replace_Slice (Wline.all, Current_Pos-1, Current_Pos-1, " ");
				       Current_Pos := Current_Pos - 1;
				 end case;
			      else

				 W_Io.Put(Wide_Character'Val(7));
			      end if;			      
			   else
			      W_Io.Put(Wide_Character'Val(7));
			      -- beep...
			   end if;
			   
			when others =>
			   null;
		     end case;
		  elsif Is_Special(Wchar) then
		     case Wchar is
			when '!' | ';' | '.' | ''' | '?' | ' ' | '_' | '-' =>
			   if Wline /= null and then
			     Wline'Length < Wchar_Max then
			      if Current_Pos = Wline'Length+1 then
				 Wbuffer := new Wide_String ' (Wline.all & Wchar);
				 Wide_Free(Wline);
				 Wline := new Wide_String ' (Wbuffer.all);
				 Wide_Free(Wbuffer);
				 Wline_Length := Wline_Length + 1;
				 Current_Pos := Current_Pos + 1;
			      else
				 if Current_Pos > 2 then
				    if not Is_Overwrite then
				       Left_Buffer := new Wide_String ' (Wline(1..Current_Pos-1));
				       Wbuffer := new Wide_String ' (Left_Buffer.all & Wchar & Wline(Current_Pos..Wline'Last));
				       Wide_Free(Wline);
				       Wline := new Wide_String ' (Wbuffer.all);
				       Wide_Free(Wbuffer);					  
				       Wide_Free(Left_Buffer);
				       Wline_Length := Wline_Length + 1;
				       Current_Pos := Current_Pos + 1;
				    else
				       Wide_Fixed.Replace_Slice (Wline.all, Current_Pos, Current_Pos, Wchar & "");
				       Current_Pos := Current_Pos + 1;
				    end if;
				    
				    
				 elsif not Is_Overwrite then
				    
				    Wbuffer := new Wide_String ' (Wchar & Wline.all);
				    Wide_Free(Wline);
				    Wline := new Wide_String ' (Wbuffer.all);
				    Wide_Free(Wbuffer);
				    Wline_Length := Wline_Length + 1;
				    Current_Pos := Current_Pos + 1;
				 else
				    Wide_Fixed.Replace_Slice (Wline.all, Current_Pos, Current_Pos, Wchar & "");				       
				    Current_Pos := Current_Pos + 1;
				 end if;
			      end if;
			      	      
			   else
			      W_Io.Put(Wide_Character'Val(7));
			      -- beep...
			   end if;
			   
			      
			when Others =>
			   W_Io.Put(Wide_Character'Val(7));
			   -- beep...
		     end case;
		     
		  elsif Is_Digit(Wchar) then
		     W_Io.Put(Wide_Character'Val(7));
		     -- beep...
		     
		  else
		     if Wline /= null and then
		       Wline'Length < Wchar_Max then
			if Current_Pos = Wline'Length + 1 then

			   Wbuffer := new Wide_String ' (Wline.all & Wchar);
			   Wide_Free(Wline);
			   Wline := new Wide_String ' (Wbuffer.all);
			   Wide_Free(Wbuffer);			
			   Current_Pos := Current_Pos + 1;
			   Wline_Length := Wline_Length + 1;
			else
			   if Current_Pos > 1 then
			      if not Is_Overwrite then

				 Left_Buffer := new Wide_String ' (Wline(1..Current_Pos-1));
				 Wbuffer := new Wide_String ' (Left_Buffer.all & Wchar & Wline(Current_Pos..Wline'Last));
				 Wide_Free(Wline);
				 Wline := new Wide_String ' (Wbuffer.all);
				 Wide_Free(Wbuffer);					  
				 Wide_Free(Left_Buffer);
				 Wline_Length := Wline_Length + 1;				 
				 Current_Pos := Current_Pos + 1;
			      else

				 Wide_Fixed.Replace_Slice (Wline.all, Current_Pos, Current_Pos, Wchar & "");
				 Current_Pos := Current_Pos + 1;
			      end if;
			      
			      
			   elsif not Is_Overwrite then

			      Wbuffer := new Wide_String ' (Wchar & Wline.all);
			      Wide_Free(Wline);
			      Wline := new Wide_String ' (Wbuffer.all);
			      Wide_Free(Wbuffer);
			      Wline_Length := Wline_Length + 1;			      
			      Current_Pos := Current_Pos + 1;
			   else
			      Wide_Fixed.Replace_Slice (Wline.all, Current_Pos, Current_Pos, Wchar & ""); 
			      Current_Pos := Current_Pos + 1;
			   end if;
			end if;
		     else
			W_Io.Put(Wide_Character'Val(7));
			-- beep...
		     end if;
		  end if;
		  

		  -------------------------------------------------------------------
		  --                  Drawing the diallog buffer                   --
		  -------------------------------------------------------------------			   
		  Dialog_Lock.Seize;
		  
		  
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index, Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;

		  Draw_Window(Center.Frames(Buffer).Win);
		  -------------------------------------------------------------------
		  --                       Drawing the states                      --
		  -------------------------------------------------------------------

		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Word_Number := Word_Count(Wline.all, Integer(Sh_Max_Words));
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  Draw_Window(Center.Frames(Input).Win);

		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;

		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;

	       end Receive;
	       
	    or
	       accept Reset;
	    or
	       accept Lock do
		  Suspended := True;
		  Clock_Tasking.Suspend;
	       end Lock;
	       exit;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean) do
		  Center.On_Esc := Is_escape;
		  Dialog_Lock.Seize;
		  case Center.Current_Mode is
		     when Dialog_Mode | Quit_Mode => 
			if Center.On_Esc then
			   Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
			else
			   Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
			end if;
		     when others =>
			null;
		  end case;
		  Draw_Window(Center.Frames(Input).Win);
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Set_On_Esc;
	    or
	       accept Page_Down do
		  case View is
		     when False => -- samples view
			if Samples_Index < Samples_Last then
			   Samples_Index := Positive'Min(Samples_Index + Center.Frames(Right_View).Win.Win.H/2+1, Samples_Last);
			else
			   Samples_Index := Samples_Last;
			   T_Io.Put(Character'Val(7));
			end if;
		     when True => -- glossary view
			if Glossary_Index < Glossary_Last then
			   Glossary_Index := Positive'Min(Glossary_Index + Center.Frames(Right_View).Win.Win.H/2+1, Glossary_Last);
			else
			   Glossary_Index := Glossary_Last;
			   T_Io.Put(Character'Val(7));
			end if;
		  end case;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Right_View).Win);
		  case View is
		     when False =>			
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
		     when True =>			
			if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			end if;
		  end case;
		  
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;

		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Page_Down;
	    or
	       accept Page_Up do
		  case View is
		     when False => -- samples view
			if Samples_Index > Center.Frames(Right_View).Win.Win.H/2 then
			   Samples_Index := Positive'Min(Samples_Index - (Center.Frames(Right_View).Win.Win.H/2-1), Samples_Last);
			else
			   Samples_Index := 0;
			   T_Io.Put(Character'Val(7));
			end if;
			
		     when True => -- glossary view
			if Glossary_Index > Center.Frames(Right_View).Win.H then
			   Glossary_Index := Positive'Min(Glossary_Index - (Center.Frames(Right_View).Win.Win.H/2-1), Glossary_Last);
			else
			   Glossary_Index := 0;
			   T_Io.Put(Character'Val(7));
			end if;
		  end case;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Right_View).Win);
		  case View is
		     when False =>			
			if not Print_Set(Samples_Set, Center.Frames(Right_View).Win, Samples_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during samples printing !");
			end if;
		     when True =>
			
			if not Print_Set(Glossary_Set, Center.Frames(Right_View).Win, glossary_Index, Total) then
			   Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during glossary printing !");
			end if;
		  end case;
		  
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;

		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  
		  Dialog_Lock.Release;
	       end Page_Up;
	    or
	       accept Up_Arrow do
		  if Dialog_Buffer_index > 0 then
		     Dialog_Buffer_Index := Dialog_Buffer_Index - 1;
		  else
		     T_Io.Put(Character'Val(7));
		  end if;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index,
				   Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Up_Arrow;
	    or
	       accept Down_Arrow do
		  if Dialog_Buffer_Index < Dialog_Buffer_Last then
		     Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
		  else
		     T_Io.Put(Character'Val(7));
		  end if;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Dialog_View).Win);
		  if not Print_Set(Dialog_Buffer, 
				   Center.Frames(Dialog_View).Win,
				   Dialog_Buffer_Index,
				   Total_lines) then
		     Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		  end if;
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Down_Arrow;
	    or
	       accept Left_Arrow do
		  if Wline /= null and then Wline'Length > 0 then
		     if Current_Pos > 1 then
			
			Current_Pos := Current_Pos - 1;
			
			Dialog_Lock.Seize;
			Draw_Window(Center.Frames(Input).Win);
			if Wline /= null and then Wline'Length /= 0 then
			   Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
			else
			   Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
			end if;
			Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
			Dialog_Lock.Release;
			
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		  end if;
	       end Left_Arrow;
	    or
	       accept Right_Arrow do
		  if Wline /= null and then Wline'Length > 0 then
		     if Current_Pos < Wline'Length+1 then
			if Wline'Length >= 1 then
			   
			   Current_Pos := Current_Pos+1;
			   
			   Dialog_Lock.Seize;
			   Draw_Window(Center.Frames(Input).Win);
			   if Wline /= null and then Wline'Length /= 0 then
			      Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
			   else
			      Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
			   end if;
			   Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
			   Dialog_Lock.Release;
			else
			   T_Io.Put(Character'Val(7));
			end if;
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		  end if;
	       end Right_Arrow;
	    or
	       accept Full_Screen;
	    or
	       accept Del do
		  if Wline /= null and then
		    Wline'Length > 0 then
		     if Current_Pos < Wline'length then
			if Current_Pos > 1 then
			   Left_Buffer := new Wide_String ' (Wline(1..Current_Pos-1));
			   Wbuffer := new Wide_String ' (Left_Buffer.all & Wline(Current_Pos+1..Wline'Length));
			else 
			   Left_Buffer := new Wide_String ' (Wline(Current_Pos+1..Wline'length));
			   Wbuffer := new Wide_String ' (Left_Buffer.all);
			end if;
			Wide_Free(Wline);
			Wline := new Wide_String ' (Wbuffer.all);
			Wide_Free(Wbuffer);					  
			Wide_Free(Left_Buffer);		     
			Wline_Length := Wline_Length - 1;			
		     elsif Current_Pos = Wline'length then
			
			Left_Buffer := new Wide_String ' (Wline(Wline'First..Current_Pos-1));
			Wbuffer := new Wide_String ' (Left_Buffer.all);
			Wide_Free(Wline);
			Wline := new Wide_String ' (Wbuffer.all);
			Wide_Free(Wbuffer);					  
			Wide_Free(Left_Buffer);		     
			Wline_Length := Wline_Length - 1;			
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		     
		  else
		     T_Io.Put(Character'Val(7));
		  end if;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Input).Win);
		  Draw_Window(Center.Frames(Buffer).Win);
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Word_Number := Word_Count(Wline.all, Integer(Sh_Max_Words));
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  ------------------------------------------------------------------
		  --                   Drawing then dialog info                  --
		  ------------------------------------------------------------------
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;

		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Del;
	    or
	       accept Begin_Of do
		  Current_Pos := 1;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Input).Win);
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;		 
	       end Begin_Of;
	    or
	       accept End_Of do
		  Current_Pos := Wline'Length + 1;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Input).Win);
		  if Wline /= null and then Wline'Length /= 0 then
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, Wline.all);
		  else
		     Draw_Text(Center.Frames(Input).Win, 1, 1, White, "");
		  end if;
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;		 
	       end End_Of;
	    or
	       accept Overwrite do
		  Is_Overwrite := not Is_Overwrite;
		  Dialog_Lock.Seize;
		  Draw_Window(Center.Frames(Buffer).Win);
		  if Center.On_Esc then
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		  else
		     Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		  end if;
		  Word_Number := Word_Count(Wline.all, Integer(Sh_Max_Words));
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  
		  
		  Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		  Dialog_Lock.Release;
	       end Overwrite;
	    or
	       accept Screen_Print;
	    or
	       accept New_Evidence;
	    or
	       accept Shutdown do
		  null;
	       end shutdown;
	    or
	       accept Dialog do
		  
		  
		  if Dialog_Initialized then
		     
		     declare
			Request : Request_NN;		  
		     begin
			Request.Respond (Center.Input, Center.Output);
			null;
		     exception
			when others =>
			   null;
		     end;
		     
		     Center.Dialog_Response := new Wide_String ' (Output2string(Center.Output, Center.Words));
		     
		     declare
			Request : Request_NN;
		     begin		  
			Request.Respond (Neural_Retrain.Input_Type(Center.Output), Center.Feedback);
			null;
		     exception
			when others =>
			   null;
		     end;
		     
		     Center.Internal := new Wide_String ' (Output2String(Center.Feedback, Center.Words));
		     
		     Center.Input := Neural_Retrain.Input_Type(Center.Feedback);
		     
		     
		     
		     
		     --------------------------------------------------
		     --          Drawing the results                 --
		     --------------------------------------------------
		     Dialog_Lock.Seize;
		     Draw_Window(Center.Frames(Resp).Win);
		     Draw_Window(Center.Frames(Output).Win);
		     Draw_Window(Center.Frames(Dialog_View).Win);
		     
		     if Center.Dialog_Response /= null and then Center.Dialog_Response'Length /= 0 then
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Dialog_Response.all);			   			   
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + Center.Dialog_Response'Length/(Center.Frames(Dialog_View).Win.Win.W-2) + 1;
			end if;
			Total := 0;
			Add_Line(Center.Response_Buffer, Center.Response_Buffer.Line_Last, Center.Dialog_Response.all);
			if Print_Set(Center.Response_Buffer, Center.Frames(Resp).Win, 0, Total) then
			   null;
			end if;
		     else
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, "");	      			      
			
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			end if;
			Draw_Text(Center.Frames(Resp).Win, 1, 1, White, "");
		     end if;
		     if Center.Internal /= null and then Center.Internal'Length /= 0 then
			Add_Line(Dialog_Buffer, Dialog_Buffer.Line_Last, Center.Internal.all);
			
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + Center.internal'Length/(Center.Frames(Dialog_View).Win.Win.W-2) + 1;
			end if;
			Total := 0;
			Add_Line(Center.Internal_Buffer, Total, Center.Internal.all);
			if Print_Set(Center.Internal_Buffer, Center.Frames(Output).Win, 0, Total) then
			   null;
			end if;
			
		     else
			Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "");			      
			if Total_Lines+2 >= Center.Frames(Dialog_View).Win.Win.H-2 then
			   Dialog_Buffer_Index := Dialog_Buffer_Index + 1;
			end if;
			Draw_Text(Center.Frames(Output).Win, 1, 1, White, "");
		     end if;			   
		     
		     if not Print_Set(Dialog_Buffer, 
				      Center.Frames(Dialog_View).Win,
				      Dialog_Buffer_Index, Total_lines) then
			Add_Line(Dialog_Buffer, Dialog_Buffer_Last, "Error during dialog printing !");
		     end if;

		     Draw_Text(Center.Frames(Input).Win, 1, Current_Pos, White, "");
		     Dialog_Lock.Release;
		  end if;
	       end Dialog;
	    end select;	 
	 end loop;
	 
	 
      end loop;      
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Dialog halted");
      Center.Global_Lock.Release;
   end Dialog_Process;
   
   
   
   ---------------------------------------------------------------------------------------------
   --
   -- Up_Arrow, Down_Arrow, Left_Arrow and Right_Arrow for Edit in Devel mode and Multi mode. --
   --
   --
   ---------------------------------------------------------------------------------------------
   
   
   
   
   
   
   procedure Up_Arrow (Set : in Wide_String_Set;
		       Doc_Index : in out Natural;			  
		       Line_index : in out Natural; 
		       Wchar_Index : in out Natural;
		       Line_Cur : in out Line_Range;
		       Col_Cur : in out Column_Range; 
		       Height : in Line_Range;
		       Width : in Column_Range) is      
      
      
      Total_Lines : Natural := 0;       
   begin
      Total_Lines := Count(Set, Doc_Index, Line_Index, 
			  Wchar_Index, Height, Width);
      if Line_Cur <= 1 and 
	Doc_Index >= 1 then
	 Doc_Index := Doc_Index - 1;
	 Line_Index := Line_Index - 1;
	 Line_Cur := 1;
	 Col_Cur := 1;
	 Wchar_Index := 0;
	 
	    
      elsif Line_Cur > 1 then
	 
	 if Wchar_Index - Width >= 1 then	    	    	    
	    Wchar_Index := (Wchar_Index - Width);
	    Line_Cur := Line_Cur - 1;
	    if Wchar_Index >= Width then
	       Col_Cur := (Wchar_Index rem Width);
	    elsif Wchar_Index >= 0 then
	       Col_Cur := Wchar_Index;
	    else
	       Col_Cur := 1;
	    end if;
	 else
	    Line_Index := Line_Index - 1;
	    Line_Cur := Line_Cur - 1;
	    Col_Cur := 1;
	    Wchar_Index := 0;
	 end if;
      else
	 W_Io.Put(Wide_Character'Val(7));
      end if;
	 
   end Up_Arrow;
   
   procedure Down_Arrow (Set : in Wide_String_Set;
			  Doc_Index : in out Natural;			  
			  Line_index : in out Natural; 
			  Wchar_Index : in out natural;
			  Line_Cur : in out Line_Range;
			  Col_Cur : in out Column_Range; 
			  Height : in Line_Range;
			  Width : in Column_Range) is      
      
      Total_Lines : Natural := 0;
      
   begin
      
      
      if (Line_Cur >= (Height)) and ((Line_Index + 1) <= Set.Line_Last) then
	 Doc_Index := Doc_Index + 1;
	 Line_Index := Line_Index + 1;
	 Wchar_Index := 0;
	 
	 
	 Col_Cur := 1;
	 Total_Lines :=
	   Count(Set, Doc_Index, Line_Index, 
		 Wchar_Index, Height, Width);	 
	 Line_Cur := Total_Lines;
	 
      elsif Line_Index + 1 <= Set.Line_Last then
	 Line_Index := Line_Index + 1;
	 Line_Cur := Line_Cur + 1;
	 Wchar_Index := 0;
	 
	 Col_Cur := 1;
	 
      elsif Real_Length(Set.Lines(Line_Index).all) >= (Width) then
	 
	 
	 if (Wchar_Index + Width) <= Real_Length(Set.Lines(Line_Index).all) then
	    Line_Cur := Line_Cur + 1;
	    Wchar_Index := Wchar_Index + Width;	    
	    Col_Cur := (Wchar_Index rem Width);	    
	 elsif Real_Length(Set.Lines(Line_Index).all) > width then	    
	    Line_Cur := Line_Cur + 1;
	    Wchar_Index := Real_Length(Set.Lines(Line_Index).all);
	    Col_Cur := (Wchar_Index rem Width);	    
	 else
	    Wchar_Index := 0;
	    Col_Cur := 1;
	 end if;
      else
	 Put(Wide_Character'Val(7));
      end if;
   end Down_Arrow;
   
   
   
   procedure At_Begin (Set : in Wide_String_Set;
		       Doc_Index : in out Natural;			  
		       Line_index : in out Natural; 
		       Wchar_Index : in out Natural;
		       Line_Cur : in out Line_Range;
		       Col_Cur : in out Column_Range; 
		       Height : in Line_Range;
		       Width : in Column_Range) is
      Total_Lines : Natural := 0;
   begin
      
      Wchar_Index := 0;
      
      Col_Cur := 1;
      Total_Lines := Count(Set, Doc_Index, Line_Index, 
			   Wchar_Index, Height, Width);
      
      Line_Cur := Total_Lines;
      
   end At_Begin;
   
   procedure At_End (Set : in Wide_String_Set;
		     Doc_Index : in out Natural;			  
		     Line_index : in out Natural; 
		     Wchar_Index : in out Natural;
		     Line_Cur : in out Line_Range;
		     Col_Cur : in out Column_Range; 
		     Height : in Line_Range;
		     Width : in Column_Range) is
      Total_Lines : Natural := 0;
   begin
      
      
      Wchar_Index := Real_Length(Set.Lines(Line_Index).all);
      
      Total_Lines := Count(Set, Doc_Index, Line_Index, 
				 Wchar_Index, Height, Width);
      
      Col_Cur := Natural'Max(Wchar_Index mod Width, Wchar_Index) + 1;
      
      
      Line_Cur := Total_Lines;
      
   end At_End;
   
   

   procedure Right_Arrow (Set : in Wide_String_Set;
			  Doc_Index : in out Natural;			  
			  Line_index : in out Natural; 
			  Wchar_Index : in out Natural;
			  Line_Cur : in out Line_Range;
			  Col_Cur : in out Column_Range; 
			  Height : in Line_Range;
			  Width : in Column_Range) is      
      
      
         
      Done : Boolean := False;
      
      
      
      Total_Lines : Natural := 0;
      
   begin
      if Wchar_index + 1 > Real_Length(Set.Lines(Line_Index).all) then
	 if Line_Index + 1 > Set.Line_Last then
	    Put(Wide_Character'Val(7));
	 elsif Line_Cur = Height then
	    loop
	       
	       Total_Lines := Count(Set, Doc_Index, Line_Index + 1,
				    Wchar_Index, Height, Width);
	       if Total_Lines > height then
		  Doc_Index := Doc_Index + 1;		  
	       end if;
	       
	       exit when Total_Lines <= Height;
	       
	    end loop;
	    
	    Line_Index := Line_Index + 1;
	    Total_Lines := Count(Set, Doc_Index, Line_Index - 1,
				 Wchar_Index, Height, Width);
	    Col_Cur := 1;
	    Line_Cur := Total_Lines + 1;
	 else
	    
	    
	    
	    loop
	       
	       Total_Lines := Count(Set, Doc_Index, Line_Index + 1,
				    Wchar_Index, Height, Width);
	       if Total_Lines > height then
		  Doc_Index := Doc_Index + 1;		  
	       end if;
	       
	       exit when Total_Lines <= Height;
	       
	    end loop;
	    Line_Index := Line_Index + 1;
	    
	    Total_Lines := Count(Set, Doc_Index, Line_Index - 1,
				 Wchar_Index, Height, Width);
	    
	    Line_Cur := Total_Lines + 1;
	    Wchar_Index := 0;
	    Col_Cur := 1;
	 end if;
      elsif Wchar_Index < Set.Lines(Line_Index)'Length and (Wchar_Index + 1 = Width) then
	 Col_Cur := 1;
	 Line_Cur := Line_Cur + 1;
	 Wchar_Index := Wchar_Index + 1;
      elsif Wchar_Index < Set.Lines(Line_Index)'Length and (Wchar_Index + 1 > Width) then
	 Wchar_Index := Wchar_Index + 1;
	 if (Wchar_Index mod Width) = 0 then
	    Line_Cur := Line_Cur + 1;
	    Col_Cur := 1;
	 else
	    Col_Cur := (Wchar_Index mod Width) + 1;
	 end if;
      elsif Wchar_Index < Set.Lines(Line_Index)'Length then	 
	 Wchar_Index := Wchar_Index + 1;
	 Col_Cur := Col_Cur + 1;
      end if;
   end Right_Arrow;
   
   
   procedure Left_Arrow (Set : in Wide_String_Set;
			 Doc_Index : in out Natural;			  
			 Line_index : in out Natural; 
			 Wchar_Index : in out Natural;
			 Line_Cur : in out Line_Range;
			 Col_Cur : in out Column_Range; 
			 Height : in Line_Range;
			 Width : in Column_Range) is      
      
      Total_Lines : Natural := 0;      
   begin
      
      if (Col_Cur = 1) and Line_Cur = 1 then
	 if Doc_Index - 1 >= 0 then	       
	    Line_Index := Line_Index - 1;
	    Doc_Index := Doc_Index - 1;	       
	    Wchar_Index := Real_Length(Set.Lines(Line_Index).all);
	    
	    Total_Lines := Count(Set, Doc_Index, Line_Index, 
				 Wchar_Index, Height, Width);	    
	    Line_Cur := Total_Lines;
	    if (Wchar_Index mod Width) /= 0 then
	       Col_Cur := (Wchar_Index mod Width) + 1;
	    else
	       Col_Cur := 1;
	    end if;
	    
	    
	 else
	    W_Io.Put(Wide_Character'Val(7));
	 end if;
      elsif (Col_Cur = 1) and (Wchar_Index >= Width)  then	 
	 Wchar_index := Wchar_Index - 1;
	 Line_Cur := Line_Cur - 1;
	 if (Wchar_Index mod Width) /= 0 then
	    Col_Cur := (Wchar_Index mod Width) + 1;
	 else
	    Col_Cur := 1;
	 end if;
	 
      elsif Col_Cur = 1 then	 

	 loop
	    
	    Total_Lines := Count(Set, Doc_Index, Line_Index-1, 
				 Wchar_Index, Height, Width);
	    if Total_Lines > height then
	       Doc_Index := Doc_Index + 1;		  
	    end if;
	    
	    exit when Total_Lines <= Height;

	 end loop;
	 
	 Line_Index := Line_Index - 1;	 
	 Total_Lines := Count(Set, Doc_Index, Line_Index, 
			      Wchar_Index, Height, Width);	 
	 Line_Cur := Natural'Max(Line_Cur - 1, Total_Lines);
	 Wchar_Index := Real_Length(Set.Lines(Line_Index).all);
	 if (Wchar_Index mod Width) /= 0 then
	    Col_Cur := (Wchar_Index mod Width) + 1;
	 else
	    Col_Cur := 1;
	 end if;
	 
      elsif Wchar_index >= 1 then	 
	 Wchar_Index := Wchar_Index - 1;      
	 if (Wchar_Index mod Width) /= 0 then
	    Col_Cur := (Wchar_Index mod Width) + 1;
	 else
	    Col_Cur := 1;
	 end if;
	 
      else
	 W_Io.Put(Wide_Character'Val(7));
      end if;
	 
      
      
   end Left_Arrow;

   
   
   
   
   
   
   
   
   

   function Internal_Cmd_Value (Line : in Wide_String) return Internal_Cmd_Enum is
	 Cmd : Internal_Cmd_Enum  := Null_Cmd;
      First_Space : constant Natural := Index(Line, " ");
   begin
      
      if First_Space /= 0 then
	 begin
	    
	    Cmd := Internal_Cmd_Enum'Value(Handling.To_String(Line(Line'First..First_Space-1)));
	    
	 exception
	    when others =>
	       Cmd := Null_Cmd;
	 end;
	 
      elsif Index_Non_Blank(Line) /= 0 then
	 begin
	    Cmd := Internal_Cmd_Enum'Value(Handling.To_String(Line(Index_Non_Blank(Line)..Index_Non_Blank(Line, Backward))));
	 exception
	    when others =>
	       Cmd := Null_Cmd;
	 end;
	 
      end if;
      return Cmd;
   end Internal_Cmd_Value;
   
   
   function Add_Line (Ptr : in U_Array_Access; Line : in Wide_String) return U_Array_Access is      	 	 
      Buffer : U_Array_Access;
   begin
      if Ptr /= null then
	 
	 Buffer := new U_Array(1..(Ptr'Length)+1);
	 Buffer(Buffer'First..Buffer'Last-1) := Ptr.all;
	 
	 Buffer(Buffer'Last) := (+(Line));
      else
	 
	 Buffer := new U_Array(1..1);
	 
	 Buffer(Buffer'Last) := (+(Line));
	 
      end if;
      return Buffer;
   end Add_Line;           
   
   
   procedure Write_To_Append (Filename : in String; Wline : in Wide_String) is
      File : W_Io.File_Type;
   begin
      if Os_Lib.Is_Regular_File(Filename) then
	 W_Io.Open(File, W_Io.Append_File, Filename, Form => "WCEM=8");
      else
	 W_Io.Create(File, W_Io.Append_File, Filename, Form => "WCEM=8");
      end if;
      W_Io.Put_Line(File, Wline);
      W_Io.Close(File);
   end Write_To_Append;
   
   
   
   procedure Write_To_File (Buffer : in U_Array_Access; Filename : in String) is
      File : W_Io.File_Type;
   begin
      W_Io.Create(File, W_Io.Append_File, Filename, Form => "WCEM=8");      
      for Line in Buffer'Range loop
	 declare
	    Wline : constant Wide_String := -Buffer(Line);
	 begin
	    W_Io.Put_Line(File, Wline);
	 end;
      end loop;
      W_Io.Close(File);
   end Write_To_File;
   
   
   
   
   function Clear(Full_Path : in String) return Boolean is
      Success : Boolean := False;
   begin
      if Os_Lib.Is_Regular_File(Full_Path) then
	 Os_Lib.Delete_file(Full_Path, Success);
      end if;
      return Success;
   end Clear;
   
   
   procedure Alias (Line : in Wide_String; Alias_Set : in out Attributs; Result : in out Spawn_Result_Record) is
      
      Command : Gnat.Command_Line.Command_Line;
      Cmd : Internal_Cmd_Enum := Null_Cmd;
      Args_List : Os_Lib.Argument_List_Access;                  
      Buffer : U_Array_Access := Result.Wlines;
      Name   : Strings.String_Access;
      Image  : Strings.String_Access;
      Alias  : Attribut_Record;
   begin      
      Command_Line.Set_Command_Line(Command, Handling.To_String(Line));
      Gnat.Command_Line.Build(Command, Args_List, False);
      
      if Args_List'Last > 2 then
	 Buffer := Add_Line(Buffer, "alias : error : ");
	 Buffer := Add_Line(Buffer, "                -- To many argument in command --");
	 Buffer := Add_Line(Buffer, "                -- Try 'help cd' for more info --");
	 return;
      elsif Args_List'Last = 2 then
	 
	 if Args_List(2) /= null and then Args_List(2).all /= "" then
	    
	    Parse(Line, Name, Image);
	    
	    Alias := Make(Name.all, Image.all);
	    
	    Alias_Set.list(Alias_Set.Index + 1) := Alias;
	    Alias_Set.Index := Alias_Set.Index + 1;
	 else
	    
	    Buffer := Add_Line(Buffer, "alias : error : ");
	    Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 end if;
      else
	 
	 for Iter in 1..Alias_Set.Index loop
	    Buffer := Add_Line(Buffer, Handling.To_Wide_String(Attrib.Names(Alias_Set.list(Iter)) & '=' & Attrib.image(Alias_Set.list(Iter))));
	 end loop;
	 
      end if;
      Result.Wlines := Buffer;
      
   exception
      when others =>
	 Buffer := Add_Line(Buffer, "alias : error : ");
	 Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 Result.Wlines := Buffer;
   end Alias;
   
   procedure Unalias (Line : in Wide_String; Alias_Set : in out Attributs; Result : in out Spawn_Result_Record) is
      
      Command : Gnat.Command_Line.Command_Line;
      Cmd : Internal_Cmd_Enum := Null_Cmd;
      Args_List : Os_Lib.Argument_List_Access;                  
      Buffer : U_array_Access := Result.Wlines;
      Name   : Strings.String_Access;
      Image  : Strings.String_Access;
      Alias  : Attribut_Record;
   begin      
      Command_Line.Set_Command_Line(Command, Handling.To_String(Line));
      Gnat.Command_Line.Build(Command, Args_List, False);
      
      if Args_List'Last > 2 then
	 Buffer := Add_Line(Buffer, "alias : error : ");
	 Buffer := Add_Line(Buffer, "                -- To many argument in command --");
	 Buffer := Add_Line(Buffer, "                -- Try 'help cd' for more info --");
	 return;
      elsif Args_List'Last = 2 then	 
	 if Args_List(2) /= null and then Args_List(2).all /= "" then
	    for Iter in 1..Alias_Set.Index loop
	       declare
		  Alias : constant Attribut_Record := Alias_Set.List(Iter);
	       begin
		  if Names(Alias) = Args_List(2).all then
		     for Left in Iter..Alias_Set.Index loop
			Alias_Set.List(Left) := Alias_Set.List(Left+1);
			Alias_Set.Index := Alias_Set.Index - 1;
		     end loop;
		     exit;
		  end if;
	       end;
	    end loop;
	 else
	    Buffer := Add_Line(Buffer, "alias : error : ");
	    Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 end if;      
      end if;
      Result.Wlines := Buffer;
   exception
      when others =>
	 Buffer := Add_Line(Buffer, "alias : error : ");
	 Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 Result.Wlines := Buffer;
   end Unalias;
   
   
   procedure Internal_Help (Line : in Wide_string; Result : in out Spawn_Result_Record) is
      Command : Gnat.Command_Line.Command_Line;
      Cmd : Internal_Cmd_Enum := Null_Cmd;
      Args_List : Os_Lib.Argument_List_Access;                  
      Buffer : U_Array_Access := Result.Wlines;
   begin      
      Command_Line.Set_Command_Line(Command, Handling.To_String(Line));      
      Gnat.Command_Line.Build(Command, Args_List, False);
      if Args_List'Last > 1 then
	 if Args_List(2) /= null and then Args_List(2).all /= "" then
	    Cmd := Internal_Cmd_Enum'Value(Args_list(2).all);
	 else
	    Cmd := Null_Cmd;
	 end if;   
      else
	 Cmd := Null_Cmd;
      end if;
      case Cmd is
	 when Null_Cmd =>	    
	    Buffer := Add_Line(Buffer, "Command help : help");
	    Buffer := Add_Line(Buffer, "                -- help ? print command help for ? --");
	 when Clear =>
	    Buffer := Add_Line(Buffer, "Command help : clear");
	    Buffer := Add_Line(Buffer, "                -- Erase the shell file named "".shell_file.txt"" --");
	 when Alias =>
	    Buffer := Add_Line(Buffer, "Command help : alias");
	    Buffer := Add_Line(Buffer, "                -- Create an alias to any command --");
	    Buffer := Add_Line(Buffer, "                -- Ex : alias name=value          --");
	    
	 when Unalias =>
	    Buffer := Add_Line(Buffer, "Command help : unalias");
	    Buffer := Add_Line(Buffer, "                -- Delete an alias  --");
	    Buffer := Add_Line(Buffer, "                -- Ex : unalias name              --");
	 when Cd =>
	    Buffer := Add_Line(Buffer, "Command help : cd");
	    Buffer := Add_Line(Buffer, "                --  Change current working directory --");
	    Buffer := Add_Line(Buffer, "                --  Ex : cd Documents [ Enter ]      --");
	    Buffer := Add_Line(Buffer, "                --  Ex : cd [ Enter ]      --");
	 when Logout =>
	    Buffer := Add_Line(Buffer, "Command help : logout");
	    Buffer := Add_Line(Buffer, "                --  Quit the program --");
	 when Pwd =>
	    Buffer := Add_Line(Buffer, "Command help : pwd");
	    Buffer := Add_Line(Buffer, "                --  Return the current working directory --");
	 when Help =>
	    null;
      end case;
      
      Result.Wlines := Buffer;
   end Internal_Help;
   
   
   -------------------------------------------------------------
   --            Internal procedure for Command mode          --
   -------------------------------------------------------------
   
   
   procedure Change_directory (Line : in Wide_string; Result : in out Spawn_Result_Record) is
      Command : Gnat.Command_Line.Command_Line;
      Cmd : Internal_Cmd_Enum := Null_Cmd;
      Args_List : Os_Lib.Argument_List_Access;                  
      Buffer : U_Array_Access := Result.Wlines;
   begin      
      Command_Line.Set_Command_Line(Command, Handling.To_String(Line));
      Gnat.Command_Line.Build(Command, Args_List, False);
      if Args_List'Last > 2 then
	 Buffer := Add_Line(Buffer, "cd : error : ");
	 Buffer := Add_Line(Buffer, "                -- To many argument in command --");
	 Buffer := Add_Line(Buffer, "                -- Try 'help cd' for more info --");
	 return;
      elsif Args_List'Last = 2 then
	 if Args_List(2) /= null and then Args_List(2).all /= "" then
	    Change_Dir(Args_List(2).all);
	 else
	    Buffer := Add_Line(Buffer, "cd : error : ");
	    Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 end if;
      else
	 case Os_Name is
	    when System.Os_Constants.Windows =>
	       Change_Dir(Expand_Filename("HOMEPATH"));
	    when System.Os_Constants.Other_Os =>
	       Change_Dir(Expand_Filename("HOME"));
	 end case;
      end if;   
      Result.Wlines := Buffer;
   exception
      when others =>
	 Buffer := Add_Line(Buffer, "cd : error : ");
	 Buffer := Add_Line(Buffer, "                -- An error ocurred during operation. --");
	 Result.Wlines := Buffer;
   end Change_Directory;
   
   
   
      
   procedure New_Doc (Center : in Center_Record;
		      Lines_Table : in out Wide_String_Set; 
		      Docume : out Docume_Access; 		      
		      Filename : in String;		      
		      Version  : in Wide_String;
		      License : in Boolean;
		      Description : in Wide_string);
   
   procedure Open_Doc (Lines_Table : in out Wide_String_Set;
		       Docume : out Docume_Access;
		       Filename : in String);

   procedure Devel_Draw_User (User : in User_Access; Win : in Window_Record; Ready : in Boolean);
   procedure Devel_Draw_Docume (Docume : in Docume_Access; Win : in Window_Record; Initialized : in Boolean);
   procedure Devel_Draw_Main_Win (Center : access Center_Record);
   procedure Devel_Win_Init (Center : access Center_Record);
   
   
   
   
   procedure Save_Buffer(Docume : in Docume_Access; Lines_Table : in Wide_String_Set) is
   begin
      -- save buffer.
      declare
	 File : W_Io.File_Type;
      begin
	 
	 W_Io.Create(File, W_Io.Out_File, Docume.Filename.all, Form => "WCEM=8");
	 
	 for Line_Iter in 1..Lines_Table.Line_Last+1 loop
	    if Lines_Table.Lines(Line_Iter) /= null then
	       begin
		  
		  declare
		     
		     Line : constant Wide_String := 
		       Lines_Table.Lines(Line_Iter).all;
		     
		  begin
		     
		     W_Io.Put_Line(File, Line);
		  end;
	       end;
	    end if;
	 end loop;
	 W_Io.Close(File);
      end;
   end Save_Buffer;
   
   
   procedure Highlight (Docume : in Docume_Access; Lines : out Wide_String_Set) is
      Result_Highlight : Wide_Result_Access;  		     
      Deleted          : Boolean := False;      
      File             : W_Io.File_Type;
   begin
      Set_Free(Lines);
      Result_Highlight := new Spawn_Result_Record ' 
	(Spawn("sh -c ""highlight -O ANSI " & Docume.Filename.all 
		 & " > .highlighted""", ".Out_File.txt"));
      
      if Result_Highlight /= null and then
	Result_Highlight.Wlines /= null then
	 if  Index_Non_Blank (-Result_Highlight.Wlines(1)) = 0 then
	    if Os_Lib.Is_Regular_File(".highlighted") then
	       W_Io.Open(File, W_Io.In_File, ".highlighted", Form => "WCEM=8");
	       while not W_Io.End_Of_File(File) loop		  
		  Lines.Lines(Lines.Line_Last + 1) := new Wide_String ' (W_Io.Get_Line(File));	   
		  Lines.Line_Last := Lines.Line_Last + 1;
	       end loop;
	       W_Io.Close(File);
	    end if;
	 end if;
      end if;
      
      Os_Lib.Delete_file(".highlighted", deleted);      
      U_Array_Free(Result_Highlight.Wlines);
   end Highlight;
   
   procedure Open_Document (Doc_Name : in Wide_String; Table : in out Wide_String_Set; Line_Last : out natural) is      
      File : W_Io.File_Type;
   begin
      Set_Free(Table);
      W_Io.Open(File, W_Io.In_File, Handling.To_String(Doc_Name), Form => "WCEM=8");
      while not W_Io.End_Of_File(File) loop
	 declare
	    Wline : constant Wide_String := W_Io.Get_Line(File);
	 begin
	    Table.Lines(Line_Last + 1) := new Wide_String ' (Wline);
	    Line_Last := Line_Last + 1;
	 end;
      end loop;
      W_Io.Close(File);
      Table.Line_Last := Line_Last;
   end Open_Document;
   
   procedure Draw_Buffer (Spawn_Result : in Wide_Result_Access; Win : in Window_Record; File_Index : in Natural);
   
   
   function New_Wide_String (Title : in Wide_String;
   			     Label : in Wide_String;
   			     Length : in Positive;
   			     Line, Col : in Positive;
   			     Print : in Boolean) return Wide_String;
   
   procedure Doc_Init (Center : in Center_Record;
		       Lines_Table : in out Wide_String_Set);
   
   
   task body Devel_Process is
      
      Width : constant Column_Range := (Columns/2)-5;
      Height : constant Column_Range := ((Lines - 19) / 8) * 5-3;
      
      
      
      Current_State : Devel_Enum := Command;
      
      Screen_Lock : Binary_Semaphore(True, Default_Ceiling);
      
      Wline            : Wide_String_Access := new Wide_String ' ("");
      Wline_Length     : Natural := 0;
      Line_Current_Pos : Natural := 0;
      Wchar_Index  : Natural := 0;
      Line_Cur : Positive := 1;
      Col_Cur  : Natural := 1;
      
      Cmd_Line         : Wide_String_Access := new Wide_String ' ("");
      Cmd_Line_Length  : Natural := 0;
      Cmd_Current_Pos      : Natural := 1;      
      
      Is_Overwrite : Boolean := False;
      
      Suspended : Boolean := True;
      
      task Clocking is
	 entry Halt;
	 entry Refresh;
	 entry suspend;
	 entry start;
      end Clocking;
      task body Clocking is
	 --Title_Cur  : Positive := Center.Frames(Left_View).H/2-Title'Length/2;
	 --Sens       : Boolean := False;
	 End_Of_Task : Boolean := False;
	 Blink      : Boolean := False;	 
	 task Deamon is
	    entry Halt;
	 end Deamon;
	 task body Deamon is
	 begin
	    loop
	       select
		  accept Halt;
		  exit;
	       or
		  delay 0.33;
		  select
		     Clocking.Refresh;		  
		  or
		     delay 1.0;
		  end select;
	       end select;
	    end loop;
	 end deamon;
	 
      begin
	 accept Start;
	 while not End_Of_Task loop
	    loop
	       select		     		  
		  accept Halt do
		     End_Of_Task := True;
		     Deamon.Halt;
		  end Halt;
		  exit;
	       or
		  accept Suspend;
	       or
		  accept Refresh;
	       or
		  accept Start do
		     Suspended := False;
		  end Start;
		  exit;
	       end select;
	    end loop;
	    
	    
	    
	    
	    while not End_Of_Task loop
	       select		     		  
		  accept Halt do
		     End_Of_Task := True;
		     Deamon.Halt;
		  end Halt;
		  exit;
	       or
		  when not Suspended =>
		     accept Refresh do
			Screen_Lock.Seize;

			Put(Handling.To_Wide_String(Save_Cursor));
			
			Draw_Text(Center.Frames(Left_View).Win, 1, Center.Frames(Left_View).Win.Win.W-21, Cyan, Handling.To_Wide_String(Formatting.Image(Clock)));
			Put(Handling.To_Wide_String(Restore_Cursor));

			case Current_State is
			   when Document =>
			      
			      declare
				 Wchar : Wide_Character := Wide_Character ' Val(32);
			      begin
				 if Wline /= null and then
				   Wline'Length >= 1 then				 
				    
				    if Line_Current_Pos /= 0 then
				       
				       Wchar := Wline(Line_Current_Pos);
				       
				    else
				       
				       Wchar := Wline(Wline'First);
				       
				    end if;
				 end if;
				 
				 if not Blink then
				    if Is_Overwrite then
				       if Wline'Length /= 0 then
					  
					  Draw_Text(Center.Frames(Document).Win, Line_cur, Col_Cur, White, Handling.To_Wide_String(Reverse_Video & Handling.To_Character(Wchar)));
					  Draw_Text(Center.Frames(Document).Win, Line_cur, Col_Cur, White, Handling.To_Wide_String(Normal_Mode));				    
					  Draw_Text(Center.Frames(Document).Win, Line_Cur, Col_Cur, White, "");
					  
				       else
					  Draw_Text(Center.Frames(Document).Win, Line_cur, Col_Cur, White, Handling.To_Wide_String(Reverse_Video & ' '));
					  Draw_Text(Center.Frames(Document).Win, Line_cur, Col_Cur, White, Handling.To_Wide_String(Normal_Mode));				    
					  Draw_Text(Center.Frames(Document).Win, Line_Cur, Col_Cur, White, "");
				       end if;
				    else
				       Draw_Text(Center.Frames(Document).Win, Line_Cur, Col_Cur, White, "");
				    end if;
				 else
				    Draw_Text(Center.Frames(Document).Win, Line_Cur, Col_Cur, White, "");
				 end if;					  
				 Blink := not Blink;
			      end;
			   when Command =>
			      declare
				 Wchar : Wide_Character := Wide_Character ' Val(32);
			      begin
				 if Cmd_Line /= null and then
				   Cmd_Line'Length >= 1 then				 
				    if Cmd_Current_Pos > 1 then
				       Wchar := Cmd_Line(Cmd_Current_Pos-1);
				    else
				       Wchar := Cmd_Line(1);
				    end if;
				 end if;
				 
				 if not Blink then
				    if Is_Overwrite then
				       if Cmd_Line'Length /= 0 then
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, Handling.To_Wide_String(Reverse_Video & Handling.To_Character(Wchar)));
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, Handling.To_Wide_String(Normal_Mode));
				       else
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, Handling.To_Wide_String(Reverse_Video & ' '));
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, Handling.To_Wide_String(Normal_Mode));
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, "");
				       end if;
				    else
				       Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, "");
				    end if;
				 else
				    if Is_Overwrite then
				       
				       if Cmd_Line'Length /= 0 then
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, Wchar & "");
				       else
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, " ");
					  Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, "");
				       end if;
				    else
				       Draw_Text(Center.Frames(Cmd).Win, 1, Cmd_Current_Pos, White, "");
				    end if;
				 end if;
				 Blink := not Blink;
			      end;
			   when others =>
			      null;
			end case;
			
			Screen_Lock.Release;
		     end Refresh;
		     
	       or
		  accept Suspend do
		     Suspended := True;
		  end Suspend;
		  exit;
	       or
		  accept start;
		  --or

		  --  delay 0.5;
		  --  Title_Line := (others => Wide_Character'Val(32));
		  --  Replace_slice(Title_line, Title_Cur, 0, Title.all);
		  --  Screen_Lock.Seize;
		  --  Put(Handling.To_Wide_String(Save_Cursor));
		  --  Draw_Text(Center.Frames(Left_View), 1, Cols/4, Yellow, Title_Line);				 
		  --  Put(Handling.To_Wide_String(Restore_Cursor));
		  --  Screen_Lock.Release;
		  --  if Sens then
		  --     if Title_Cur + Title'Length+1 < Title_Line'length  then
		  --  	Title_Cur := Title_Cur + 1;
		  --     else
		  --  	Sens := not Sens;
		  --     end if;
		  --  else
		  --     if Title_Cur > 1 then
		  --  	Title_Cur := Title_Cur - 1;
		  --     else
		  --  	Sens := not Sens;
		  --     end if;
		  --  end if;		     		     
	       end select;
	    end loop;
	    
	 end loop;
      end Clocking;
      
      
      
      task Goal_Path_Deamon is
      	 entry Halt;
      	 entry Suspend;
      	 entry Refresh;
      	 entry Restart;
      end Goal_Path_Deamon;
      
      
      
      
      
      task body Goal_Path_Deamon is
      	 End_Of_Task : Boolean := False;
      begin
      	 while not End_Of_Task loop
      	    loop
      	       select
      		  accept Halt do
      		     End_Of_Task := True;
      		  end Halt;
      		  exit;
      	       or
      		  accept Suspend;
      	       or
      		  accept Refresh do
      		     
		     
		     
		     select
			Center.Astar_Proc.Goal_Path(Center.Goal_Path);
			
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);		  
			Devel_Ai_Draw_Goal_Path(Center);
			T_Io.Put(Restore_Cursor);
			Screen_Lock.Release;
		     or
			delay 0.5;
		     end select;
		     

      		  end Refresh;
      	       or
      		  accept Restart;
      		  exit;
      	       end select;
      	    end loop;
      	    while not End_Of_Task loop
      	       select
      		  accept Halt do
      		     End_Of_Task := True;
      		  end Halt;
      		  exit;
      	       or
      		  accept Suspend;
      		  exit;
      	       or
      		  accept Refresh do
      		     
		     select
			Center.Astar_Proc.Goal_Path(Center.Goal_Path);
			
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);		  
			Devel_Ai_Draw_Goal_Path(Center);
			T_Io.Put(Restore_Cursor);
			Screen_Lock.Release;
		     or
			delay 0.5;
		     end select;
		  
			
      		  end Refresh;
      	       or
      		  accept Restart;
      	       or
      		  delay 4.0;
      		  
		  
		  select
		     Center.Astar_Proc.Goal_Path(Center.Goal_Path);
		     
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);		  
		     Devel_Ai_Draw_Goal_Path(Center);
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
		  or
		     delay 1.0;
		  end select;
		  
		  
      	       end select;
      	    end loop;
      	 end loop;
      end Goal_Path_Deamon;
      
      
      
      
      
      
      
      
      
      
      Wchar_Max : Positive := Cols-4;
      
      
      Cmdbuffer        : Wide_String_Access;
      Wbuffer          : Wide_String_Access;
      
      Left_Buffer      : Wide_String_Access;
      Word_Number      : Word_Range := 0;
      Dialog_Initialized  : Boolean := False;
      Dialog_Buffer_Last  : Line_Index_Range := 0;
      
      
      End_Of_This_Task : Boolean := False;
      Line_Index   : Natural := 1;
      Doc_Index    : Natural := 0;
      
      
      Line_Last : Natural := 0;
      
      Lines_Table : Wide_String_Set(Lines_Max);
      Lines_Set : Wide_String_Set(Lines_Max);
      
      
      Result_Index   : natural := 0;
      File_Index     : Natural := 0;
      Result_Left    : constant Natural := ((Lines - 17) / 8) * 3;
      Result_Max_Lines : Natural := Result_left;
      
      Total_Line : Natural := 0;
      
      Is_Internal_Cmd : Boolean := False;
      Internal_Cmd    : Internal_Cmd_Enum := Null_Cmd;
      Aliazed         : Wide_String(1..256) := (others => Wide_Character'Val(32));
      Success : Boolean := False;
      
      History_Set   : Wide_String_Set(2048);
      History_Last  : Natural := 0;
      History_Index : Natural := 1;
      
      Completion_Result : Wide_Result_Access;
      
      
      
    begin
      while not End_Of_This_Task loop
	 loop
	    select

	       accept Halt do
		  End_Of_This_Task := True;
		  Clocking.Halt;
		  Goal_Path_Deamon.Halt;
	       end Halt;
	       exit;
	    or
   	       accept Initialize do
		  
		  Screen_Lock.Seize;
		  
		  Center.Header_Win.Initialize
		    (Lines/2-1,
		     Cols/2-20, 3, 40, Cyan, white);
		  Draw_Window(Center.Header_Win);
		  Window.Draw_Text(Center.Header_Win,
			1, (20-5), white, 
		     "Devel mode"
		    );
		  Screen_Lock.Release;
		  
		  
		  Devel_Win_Init(Center);
		  Doc_Init(Center.all, Lines_Table);
		  Initialize(Center.Frames(Buffer).Win, Lines-5, 2, 5, Cols/2-2,  Cyan, White);
		  
		  Screen_Lock.Seize;
		  
		  
		  Devel_Win_Init(Center);

		  Devel_Draw_Main_Win (Center);
		  Devel_Ai_Draw_Proposition(Center);
		  Devel_Ai_Draw_Neural_States(Center);
		  Draw_Window(Center.Frames(Buffer).Win);
		  case Current_State is
		     when Command =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
		     when Document =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
		     when Web_Search =>
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
		  end case;
		  
		  
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  if Center.User_Ready then
		     Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
		     
		  else
		     Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
		     
		  end if;
		  
		  Draw_Window(Center.Frames(Doc_states).Win);   
		  if Center.Current_Doc /= null and then
		    Center.Current_Doc.Filename /= null then		     
		     Center.Current_Doc.Openned := True;
		     if Center.Doc_Initialized then
			Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
		     else
			Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
		     end if;
		     
		     Set_Free(Lines_Set);
		     
		     Highlight(Center.Current_Doc, Lines_Set);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;		
		  end if;
		  
		  if not Center.Doc_Initialized then	
		     Set_Free(Lines_Table);
		     Doc_Init(Center.all, Lines_Table);
		     if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Program_Error;
		     end if;
		  else
		     Set_Free(Lines_Set);
		     Highlight(Center.Current_Doc, Lines_Set);		  
		     Draw_Window(Center.Frames(Document).Win);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;
		  end if;
		  
		  Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);	       
		  if Cmd_Line /= null then
		     Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
		  end if;
		  
		  
		  Screen_Lock.Release;
		  Clocking.Start;
		  
		  
	       end Initialize;
	       exit;
	    or
	       accept Lock;
	    or
	       accept Wake_Up do
		  
		  Screen_Lock.Seize;
		  Devel_Win_Init(Center);

		  Devel_Draw_Main_Win (Center);
		  Devel_Ai_Draw_Proposition(Center);
		  Devel_Ai_Draw_Neural_States(Center);
		  Devel_Ai_Draw_Top_Class (Center);
		  Devel_Ai_Draw_Class_Ranking (Center);	       
		  
		  Draw_Window(Center.Frames(Buffer).Win);
		  case Current_State is
		     when Command =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
		     when Document =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
		     when Web_Search =>
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
		  end case;
		  
		  
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  
		  if Center.User_Ready then
		     Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
		     
		  else
		     Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
		     
		  end if;
		  Draw_Window(Center.Frames(Doc_states).Win);   
		  if Center.Current_Doc /= null and then
		    Center.Current_Doc.Filename /= null then		     
		     Center.Current_Doc.Openned := True;
		     if Center.Doc_Initialized then
			Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			Set_Free(Lines_Set);
			Highlight(Center.Current_Doc, Lines_Set);
			if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			   raise Spec_Error;
			end if;		
		     else
			Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
		     end if;
		  else
		     Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
		     Set_Free(Lines_Table);
		     Doc_Init(Center.all, Lines_Table);
		     if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Program_Error;
		     end if;
		  end if;
		  
		  
		  Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);	       
		  if Cmd_Line /= null then
		     Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
		  end if;
		  
		  
		  Screen_Lock.Release;
		  Suspended := False;
		  Clocking.Start;
		  Goal_Path_Deamon.Restart;
		  Goal_Path_Deamon.Refresh;
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
	 
	 while not End_Of_This_Task loop	 
	    
	    select
	       accept Wake_Up do
		  null;
	       end Wake_Up;
	    or

	       accept Halt do
		  Clocking.Halt;
		  Goal_Path_Deamon.Halt;
	       end Halt;
	       exit;
	    or	    
	       accept Initialize do
		  
		  Screen_Lock.Seize;
		  Devel_Draw_Main_Win (Center);
		  Devel_Ai_Draw_Proposition(Center);
		  Devel_Ai_Draw_Neural_States(Center);
		  Draw_Window(Center.Frames(Buffer).Win);
		  case Current_State is
		     when Command =>

			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
		     when Document =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
		     when Web_Search =>
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
		  end case;
		  
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  if Center.User_Ready then
		     Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
		  else
		     Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
		  end if;
		  if Center.Doc_Initialized then
		     Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
		  else
		     Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
		  end if;
		  Set_Free(Lines_Table);
		  Doc_Init(Center.all, Lines_Table);
		  if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     raise Program_Error;
		  end if;
		  
		  Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);	       
		  if Cmd_Line /= null then
		     Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
		  end if;
		  
		  Screen_Lock.Release;
		  Clocking.Start;		  
		  Goal_Path_Deamon.Restart;
	       end Initialize;
	    or
	       accept Mode (Mode : out Devel_Enum) do
		  Mode := Current_State;
	       end Mode;
	    or
	       accept Switch (Mode : in Devel_Enum) do
		  Current_state := Mode;
		  
		  Draw_Window(Center.Frames(Doc_states).Win);   
		  case Current_State is
		     when Command =>
			
			
			if Center.Doc_Initialized then
			   
			   if Center.Current_Doc /= null then
			      Center.Current_Doc.Openned := False;			
			      
			      Set_Free(Lines_Set);
			      
			      Highlight(Center.Current_Doc, Lines_Set);		  
			      Draw_Window(Center.Frames(Document).Win);
			      if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				 raise Spec_Error;
			      end if;
			   end if;
			else
			   
			   Set_Free(Lines_Table);
			   Doc_Init(Center.all, Lines_Table);
			   if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			      raise Program_Error;
			   end if;
			end if;
			
			if Center.Doc_Initialized then
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			else
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			end if;
			   
			
			
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
			
		     when Document =>
			
			
			if Center.Doc_Initialized then
			   if Center.Current_Doc /= null and then
			     Center.Current_Doc.Filename /= null then		     
			      Center.Current_Doc.Openned := True;
			      
			      
			      
			   else
			      
			      Set_Free(Lines_Set);
			      Highlight(Center.Current_Doc, Lines_Set);
			      if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				 raise Spec_Error;
			      end if;		
			   end if;
			   
			end if;
			
			if Center.Doc_Initialized then
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			else
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			end if;
			   
			
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
			
			
		     when Web_Search =>
			
			if Center.Current_Doc /= null then
			   Center.Current_Doc.Openned := False;
			   			   
			end if;
			
			if Center.Doc_Initialized then
			      Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			   else
			      Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			   end if;
			   
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
		  end case;
		  Clocking.Refresh;
		  Goal_Path_Deamon.Refresh;
	       end Switch;
	    or
	       when Current_State = Web_Search =>
		  accept Receive (Wchar : in Wide_Character);
	    or
	       
	       when Current_State = Command =>
		  accept Receive (Wchar : in Wide_Character) do
		     if Is_control(Wchar) then	    
			case Wchar is
			   
			   when Wide_Character'Val(9) =>
			      
			      if Cmd_Line /= null then
				 -- Path completion...
				 declare
				    Line : Wide_String_Access := 
				      new Wide_String ' (Cmd_Line.all);
				    
				    Path_On_Line : Wide_String_Access;
				    buffer : Wide_string_Access := new Wide_String ' ("");
				 begin
				    if Completion_Result /= null then
				       U_Array_Free(Completion_Result.Wlines);
				    end if;
				    Completion(Handling.To_String(Line.all), Path_On_Line, Completion_Result);				    
				    if Path_On_Line /= null then
				       Buffer := new Wide_String ' (Path_On_Line.all);
				       Wide_Free(Path_On_Line);					  
				       Wide_Free(Cmd_Line);
				       Cmd_Line := new Wide_String ' (Buffer.all);
				       Wide_Free(Buffer);
				       Cmd_Line_Length := Cmd_Line'Length;
				       Wchar_Index := Cmd_Line'Length;
				       Cmd_Current_Pos := Wchar_Index + 1;
				       U_Array_Free(Completion_Result.Wlines);
				       Wide_Result_Free(Completion_Result);
				    end if;
				    
				 end;
			      end if;                        			      
			      
			   when Wide_Character'Val(10) =>
			      
			      Draw_Window(Center.Frames(Err).Win);
			      Draw_Window(Center.Frames(Cmd).Win);
			      
			      ------------------------------------------------------
			      --                        Internal                  --
			      ------------------------------------------------------
			      
			      History_Index := 1;
			      
			      Internal_Cmd := Internal_Cmd_Value(Cmd_Line.all);
			      Is_Internal_Cmd := True;
			      if Center.Devel_Spawn_Result = null then
				 Center.Devel_Spawn_Result := new Spawn_Result_Record;
				 Center.Devel_Spawn_Result.Wlines := new U_Array (1..0);
			      end if;
			      case Internal_Cmd is
				 when Null_Cmd =>
				    Is_Internal_Cmd := False;
				    
				 when Clear =>
				    
				    case Os_Name is
				       when System.Os_Constants.Windows =>
					  Success := 
					    Clear(Center.User_Path.all & Dir_Separator & ".Out_File.txt");
				       when System.Os_Constants.Other_Os =>
					  Success := 
					    Clear(Center.User_Path.all & Dir_Separator & ".Out_File.txt");
				    end case;
				    if not Success then
				       Center.Devel_Spawn_Result.Wlines := 
					 Add_Line(Center.Devel_Spawn_Result.Wlines, 
						  Handling.To_Wide_String
						    ("An error occured when attempt of """ & 
						       Center.User_Path.all & Dir_Separator & ".Out_File.txt"" deletion."));
				    else
				       Draw_Window(Center.Frames(Cmd).Win);
				       Draw_Window(Center.Frames(Cmd_Result).Win);
				       U_Array_Free(Center.Devel_Spawn_Result.Wlines);
				       Center.Devel_Spawn_Result.Wlines := new U_Array (1..0);
				    end if;
				 when Alias =>
				    
				    Alias(Cmd_Line.all, Center.User.Alias_Set, Spawn_Result_Record(Center.Devel_Spawn_Result.all));
				    
				 when Unalias =>
				    
				    Unalias(Cmd_Line.all, Center.User.Alias_Set, Spawn_Result_Record(Center.Devel_Spawn_Result.all));

				 when Cd =>
				    
				    Change_Directory(Cmd_Line.all, Spawn_Result_Record(Center.Devel_Spawn_Result.all));
				 when Pwd =>

				    Center.Devel_Spawn_Result.Wlines := 
				      Add_Line(Center.Devel_Spawn_Result.Wlines, Handling.To_Wide_String(String(Get_Current_Dir)));
				 when Help =>

				    Internal_Help(Cmd_Line.all, Spawn_Result_Record(Center.Devel_Spawn_Result.all));
				 when Logout =>
				    
				    Center.Logout := True;
				 when others =>
				    null;				    
			      end case;

			      case Internal_Cmd is
				 when Null_Cmd =>
				    Center.Devel_Spawn_Result.Return_code := 127;
				 when others =>

				    Add_Line(History_Set, History_Last, Cmd_Line.all);

				    Wide_Free(Cmd_Line);
				    Cmd_Line_Length := 0;				    
				    Cmd_Line := new Wide_String ' ("");				    
			      end case;

			      ------------------------------------------------------
			      --                   End internal                   --
			      ------------------------------------------------------
			      
			      if not Is_Internal_Cmd then
				 declare
				    Alias_Ptr   : Strings.String_Access;
				    Buffer      : Wide_String_Access;
				    Normal      : Strings.String_Access;				 
				 begin

				    for Iter in 1..Center.User.Alias_Set.Index loop
				       declare
					  Alias : constant Attribut_Record := Center.User.Alias_Set.List(Iter);
					  First_Space : Natural := 
					    Wide_Fixed.Index(Cmd_Line.all, " ");
					  Non_Blank   : Natural :=
					    Wide_Fixed.Index_Non_Blank(Cmd_Line.all, Backward);
				       begin	
					  if Non_Blank < First_Space then
					     First_Space := 0;
					  end if;
					  if First_Space /= 0 then
					     if Handling.To_Wide_String(Names(Alias)) =
					       Cmd_line(Cmd_Line'First..First_Space - 1) then					     
						Alias_Ptr := new String ' (Attrib.Image(Alias));
						exit;
					     end if;
					  else
					     if Handling.To_Wide_String(Names(Alias)) =
					       Cmd_Line(Cmd_Line'First..Index_Non_Blank(Cmd_Line.all, Backward)) then
						Alias_Ptr := new String ' (Attrib.Image(Alias));
						exit;
					     end if;
					  end if;
				       end;
				    end loop;

				    if Alias_Ptr /= null then
				       declare
					  First_Space : constant Natural := 
					    Wide_Fixed.Index(Cmd_Line.all, " ");
				       begin

					  if First_Space /= 0 then
					     Buffer := new Wide_String ' 
					       (Handling.To_Wide_String(Alias_Ptr.all) & ' ' & Cmd_Line(First_Space+1..Cmd_Line ' Last));

					  else
					     Buffer := new Wide_String ' (Handling.To_Wide_String(Alias_Ptr.all));
					  end if;
				       end;
				       Wide_Free(Cmd_Line);

				       Normal := new String ' (Normalize_Quoted_Argument(Handling.To_String(Buffer.all)));
				       Cmd_Line := new Wide_String ' (Handling.To_Wide_String(Normal.all));
				       Cmd_Line_Length := Cmd_line'Length;
				    end if;
				 end;
				 
				 U_Array_Free(Center.Devel_Spawn_Result.Wlines);
				 Wide_Result_Free(Center.Devel_Spawn_Result);
				 Center.Devel_Spawn_Result := new Spawn_Result_Record ' (Shell_Spawn ("sh -c """ & Handling.To_String(Cmd_Line.all) & """", ".Out_File.txt"));
				 Result_Max_Lines := Result_Left;
				 
			      end if;
			      Wide_Free(Cmd_Line);
			      Cmd_Line := new Wide_String ' ("");
			      Cmd_Current_Pos := 1;
			      Cmd_Line_Length := 0;
			      
			      Ansi_Console.Clear_Screen;				       
			      
			      Screen_Lock.Seize;
			      
			      T_Io.Put(Save_Cursor);
			      
			      
			      Devel_Draw_Main_Win (Center);
			      
			      Devel_Ai_Draw_Proposition(Center);
			      Devel_Ai_Draw_Neural_States(Center);
			      Devel_Ai_Draw_Top_Class (Center);
			      Devel_Ai_Draw_Class_Ranking (Center);	       
		  
			      Draw_Window(Center.Frames(Buffer).Win);			   
			      Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
			      
			      Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
			      Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
			      Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
			      Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
			      
			      
			      
			      case Current_State is
				 when Command =>
				    null;
				    
				 when Document =>
				    if Center.Doc_Initialized then
				       Draw_Window(Center.Frames(Document).Win);
				    end if;
				 when Web_Search =>
				    null;
			      end case;
			      
			      
			      if Center.User_Ready then
				 Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
			      else
				 Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
			      end if;
			      
			      
			      
			      if Completion_Result /= null and then
				Completion_Result.Wlines /= null then
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Completion_Result, Center.Frames(Cmd_Result).Win, Result_Index);		     
			      elsif Center.Devel_Spawn_Result /= null then
				 Draw_Window(Center.Frames(Cmd).Win);
				 
				 if Center.Devel_Spawn_Result.Success then
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Success");		     		     
				 else
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Error : " & Handling.To_Wide_String(Integer'Image(Center.Devel_Spawn_Result.Return_Code)));
				 end if;
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);
			      end if;
			      if Center.Doc_Initialized then
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			      else
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			      end if;
			      
			      if Center.Current_Doc /= null and then
				Center.Current_Doc.Filename /= null then
				 
				 Set_Free(Lines_Set);

				 Highlight(Center.Current_Doc, Lines_Set);		  
				 
				 if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Spec_Error;
				 end if;
			      else
				 Set_Free(Lines_Table);
				 Doc_Init(Center.all, Lines_Table);
				 if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Program_Error;
				 end if;
			      end if;
			      
			      
			      T_Io.Put(Restore_Cursor);
			      
			      Screen_Lock.Release;
			      
			      Clocking.Refresh;
			   when Wide_Character'Val(12) =>			
			      
			      Ansi_Console.Clear_Screen;				       
			      
			      Screen_Lock.Seize;
			      
			      T_Io.Put(Save_Cursor);
			      
			      
			      Devel_Draw_Main_Win (Center);
			      Draw_Window(Center.Frames(Buffer).Win);			   
			      Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
			      
			      Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
			      Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
			      Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
			      Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
			      
			      
			      
			      case Current_State is
				 when Command =>
				    null;
				    
				 
				 when Document =>
				       if Center.Doc_Initialized then
					  Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
				       else
					  Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
				       end if;

				 when Web_Search =>
				    null;
			      end case;
			      
			      
			      if Center.User_Ready then
				 Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
			      else
				 Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
			      end if;
			      
			      
			      
			      if Completion_Result /= null and then
				Completion_Result.Wlines /= null then
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Completion_Result, Center.Frames(Cmd_Result).Win, Result_Index);		     
			      elsif Center.Devel_Spawn_Result /= null then
				 Draw_Window(Center.Frames(Cmd).Win);
				 
				 if Center.Devel_Spawn_Result.Success then
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Success");		     		     
				 else
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Error : " & Handling.To_Wide_String(Integer'Image(Center.Devel_Spawn_Result.Return_Code)));
				 end if;
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);
			      end if;
			      if Center.Doc_Initialized then
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			      else
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			      end if;
			      
			      if Center.Current_Doc /= null and then
				Center.Current_Doc.Filename /= null then
				 
				 Set_Free(Lines_Set);

				 Highlight(Center.Current_Doc, Lines_Set);		  
				 
				 if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Spec_Error;
				 end if;
			      else
				 Set_Free(Lines_Table);
				 Doc_Init(Center.all, Lines_Table);
				 if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Program_Error;
				 end if;
			      end if;
			      
			      
			      T_Io.Put(Restore_Cursor);
			      
			      Screen_Lock.Release;
			      
			      Clocking.Refresh;
			      Goal_Path_Deamon.Refresh;
			   when Wide_Character'Val(127) =>
			      
			      Draw_Window(Center.Frames(Cmd).Win);
			      if Cmd_Line /= null and then
				Cmd_Line'Length > 0 then
				 if Cmd_Line /= null and then
				   Cmd_Line'Length > 0 then
				    if Cmd_Current_Pos > 1 then
				       case Is_Overwrite is
					  when False =>
					     if Cmd_Current_Pos > 2 then
						if Cmd_Line'Length >= 2 then
						   Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-2));
						   if Cmd_Current_Pos <= Cmd_Line'Length then
						      Wbuffer := new Wide_String ' (Left_Buffer.all & Cmd_Line(Cmd_Current_Pos..Cmd_Line'Last));
						   else
						      Wbuffer := new Wide_String ' (Left_Buffer.all);
						   end if;
						else
						   Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-2));
						   Wbuffer := new Wide_String ' (Left_Buffer.all);
						end if;
					     elsif Cmd_Current_Pos = 2 then
						if Cmd_Line'Length >= 2 then
						   Left_Buffer := new Wide_String ' (Cmd_Line(Cmd_Current_Pos..Cmd_Line'Length));
						   Wbuffer := new Wide_String ' (Left_Buffer.all);
						else
						   Wbuffer := new Wide_String ' ("");
						end if;
					     end if;				       				       
					     Wide_Free(Cmd_Line);
					     Cmd_Line := new Wide_String ' (Wbuffer.all);
					     Wide_Free(Wbuffer);					  
					     Wide_Free(Left_Buffer);
					     Wchar_Index := Wchar_Index - 1;
					     Cmd_Current_Pos := Cmd_Current_Pos - 1;				       
					     Cmd_Line_Length := Cmd_Line_Length - 1;
					  when True =>
					     Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos-1, Cmd_Current_Pos-1, " ");
					     Cmd_Current_Pos := Cmd_Current_Pos - 1;
				       end case;
				    else

				       W_Io.Put(Wide_Character'Val(7));
				    end if;			      
				 else
				    W_Io.Put(Wide_Character'Val(7));
				    -- beep...
				 end if;
				 
				 --  CmdBuffer := new Wide_String ' (Cmd_Line(Cmd_Line'First..Cmd_Line'Last-1));
				 --  Wide_Free(Cmd_Line);
				 --  Cmd_Line := new Wide_String ' (CmdBuffer.all);
				 --  Wide_Free(CmdBuffer);
			      else
				 Put(Wide_Character'Val(7));
			      end if;
			      
			   when others =>
			      null;
			end case;
			
			
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);
			
			if Completion_Result /= null and then
			  Completion_Result.Wlines /= null then
			   Draw_Window(Center.Frames(Cmd_Result).Win);
			   Draw_Buffer(Completion_Result, Center.Frames(Cmd_Result).Win, Result_Index);		     
			elsif Center.Devel_Spawn_Result /= null then
			   Draw_Window(Center.Frames(Cmd).Win);
			   
			   if Center.Devel_Spawn_Result.Success then
			      Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Success");		     		     
			   else
			      Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Error : " & Handling.To_Wide_String(Integer'Image(Center.Devel_Spawn_Result.Return_Code)));
			   end if;
			   Draw_Window(Center.Frames(Cmd_Result).Win);
			   Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);
			end if;
			Draw_Window(Center.Frames(Cmd).Win);
			if Cmd_Line /= null then
			   Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
			end if;
			
			T_Io.Put(Restore_Cursor);
			Screen_Lock.Release;
			Clocking.Refresh;
			
		     else
			
			
			if Cmd_Line /= null and then
			  Cmd_Line'Length = Width then
			   Put(Wide_Character'Val(7));
			else
			   if Cmd_Current_Pos = Cmd_Line'Length + 1 then
			      CmdBuffer := new Wide_String ' (Cmd_Line.all & Wchar);
			      Wide_Free(Cmd_Line);
			      Cmd_Line := new Wide_String ' (CmdBuffer.all);
			      Wide_Free(CmdBuffer);			
			      Cmd_Current_Pos := Cmd_Current_Pos + 1;
			      Wchar_Index := Wchar_Index + 1;
			      Cmd_Line_Length := Cmd_Line_Length + 1;
			   else
			      if Cmd_Current_Pos > 1 then
				 if not Is_Overwrite then
				    Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-1));
				    CmdBuffer := new Wide_String ' (Left_Buffer.all & Wchar & Cmd_Line(Cmd_Current_Pos..Cmd_Line'Last));
				    Wide_Free(Cmd_Line);
				    Cmd_Line := new Wide_String ' (CmdBuffer.all);
				    Wide_Free(CmdBuffer);					  
				    Wide_Free(Left_Buffer);
				    Cmd_Line_Length := Cmd_Line_Length + 1;				 
				    Cmd_Current_Pos := Cmd_Current_Pos + 1;
				    Wchar_Index := Wchar_Index + 1;
				 else
				    Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos, Cmd_Current_Pos, Wchar & "");				       
				    Cmd_Current_Pos := Cmd_Current_Pos + 1;
				 end if;
				 
				 
			      elsif not Is_Overwrite then
				 
				 CmdBuffer := new Wide_String ' (Wchar & Cmd_Line.all);
				 Wide_Free(Cmd_Line);
				 Cmd_Line := new Wide_String ' (CmdBuffer.all);
				 Wide_Free(CmdBuffer);
				 Cmd_Line_Length := Cmd_Line_Length + 1;			      
				 Cmd_Current_Pos := Cmd_Current_Pos + 1;
				 Wchar_Index := Wchar_Index + 1;
			      else
				 Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos, Cmd_Current_Pos, Wchar & "");				       
				 Cmd_Current_Pos := Cmd_Current_Pos + 1;
				 Wchar_Index := Wchar_Index + 1;
			      end if;
			   end if;
			   
			end if;
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);
			Draw_Window(Center.Frames(Cmd).Win);
			if Cmd_Line /= null then
			   Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
			end if;
			T_Io.Put(Restore_Cursor);
			Screen_Lock.Release;
		     end if;
		     
		     Clocking.Refresh;
		     --Draw_Text(Center.Frames(Doc_states).Win, 1, 30, White, "Line_Last : " & Handling.To_Wide_String(Natural'Image(Lines_Table.Line_Last)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 1, 50, White, "Line_index : " & Handling.To_Wide_String(Natural'Image(Line_index)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 2, 30, White, "Doc_index : " & Handling.To_Wide_String(Natural'Image(Doc_index)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 2, 50, White, "wchar_index : " & Handling.To_Wide_String(Natural'Image(Wchar_index)));
		     
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;

		     
		  end Receive;
	    or
	       
	       when Current_State = Document =>
		  accept Receive (Wchar : in Wide_Character) do
		     
		     if Is_control(Wchar) then	    
			case Wchar is
			   
			   when Wide_Character'Val(9) =>
			      
			      if Center.Current_Doc /= null then
				 
				 if Line_Current_Pos = Wline_Length then
				    Wbuffer := new Wide_String ' (Wline.all & "   ");
				    Wide_Free(WLine);
				    Wline := new Wide_String ' (Wbuffer.all);
				    Wide_Free(Wbuffer);
				    Wline_Length := Wline'Length;
				    Line_Current_Pos := Wline'Length;
				    
				    
				 else
				    Left_Buffer := new Wide_String ' (Wline(1..Line_Current_Pos));
				    Wbuffer := new Wide_String ' (Left_Buffer.all & "   " & Wline(Line_Current_Pos+1..Wline'Last));
				    Wide_Free(Wline);
				    Wline := new Wide_String ' (Wbuffer.all);
				    Wide_Free(Wbuffer);
				    Wline_Length := Wline'Length;
				    Line_Current_Pos := Line_Current_Pos + 3;
				    
				    
				 end if;
				 Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
				 if ((Line_Current_Pos) mod Width) /= 0 then
				    Col_Cur := ((Line_Current_Pos) mod Width) + 1;
				 else
				    Col_Cur := 1;
				 end if;
			      else
				 W_Io.Put(Wide_Character'Val(7));
			      end if;
			      
			   when Wide_Character'Val(10) =>
			      
			      if Lines_Table.Line_Last < Lines_Table.Lines'Length then
				 if Line_Current_Pos = Wline_Length then
				    if Line_Index = Lines_Table.Line_Last then
				       Lines_Table.Line_Last := Lines_Table.Line_Last + 1;
				       Line_Index := Line_Index + 1;
				       Wline := new Wide_String ' ("");
				       Lines_Table.Lines(Line_Index) := Wline;
				    else
				       for Line in reverse Line_Index .. Lines_Table.Line_Last loop
					  Lines_Table.Lines(Line + 1) := Lines_Table.Lines(Line);
				       end loop;
				       Line_Index := Line_Index + 1;
				       Wline := new Wide_String ' ("");
				       Lines_Table.Lines(Line_Index) := Wline;
				       Lines_Table.Line_Last := Lines_Table.Line_Last + 1;
				    end if;
				 elsif Line_Current_Pos > 0 and Line_Current_Pos < Wline'Length then
				    Left_Buffer := new Wide_String ' (Wline(1..Line_Current_Pos));
				    T_Io.New_Line;
				    
				    Wline := new Wide_String ' (Wline(Line_Current_Pos+1..Wline'Length));
				    Wide_Free(Lines_Table.Lines(Line_Index));
				    
				    Lines_Table.Lines(Line_Index) := Left_Buffer;				       
				    Line_Index := Line_Index + 1;					  
				    
				    if Lines_Table.Line_Last > Line_Index then
				       
				       for Line in reverse Line_Index .. Lines_Table.Line_Last loop
					  Lines_Table.Lines(Line + 1) := Lines_Table.Lines(Line);
				       end loop;
				       
				    end if;
				    
				    Lines_Table.Lines(Line_Index) := Wline;
				    Lines_Table.Line_Last := Lines_Table.Line_Last + 1;					  
				 else
				    Wbuffer := new Wide_String ' (Wline.all);
				    Wline := new Wide_String ' ("");
				    if Lines_Table.Line_Last > Line_Index then
				       for Line in reverse Line_Index .. Lines_Table.Line_Last loop
					  Lines_Table.Lines(Line + 1) := Lines_Table.Lines(Line);
				       end loop;
				    end if;
				    Lines_Table.Lines(Line_Index) := Wline ;
				    Line_Index := Line_Index + 1;					  
				    Lines_Table.Lines(Line_Index) := Wbuffer;
				    Lines_Table.Line_Last := Lines_Table.Line_Last + 1;					  
				 end if;
				 
				 Wline := Lines_Table.Lines(Line_index);
				 
				 Line_Current_Pos := 0;
				 Wline_Length := Wline'Length;
				 
				 
				 Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
				 Col_Cur := 1;
			      else
				 W_Io.Put(Wide_Character'Val(7));
			      end if;
			
			      
			      
			      Clocking.refresh;
			      
			      select
				 Goal_Path_Deamon.Refresh;
			      or
				 delay 0.1;
			      end select;
			      
			   when Wide_Character'Val(12) =>			
			      Ansi_Console.Clear_Screen;				       
			      Screen_Lock.Seize;
			      T_Io.Put(Save_Cursor);
			      
			      
			      Devel_Draw_Main_Win (Center);
			      Devel_Ai_Draw_Proposition(Center);
			      Devel_Ai_Draw_Neural_States(Center);
			      Devel_Ai_Draw_Top_Class (Center);
			      Devel_Ai_Draw_Class_Ranking (Center);	       
		  
			      Draw_Window(Center.Frames(Buffer).Win);
			      case Current_State is
				 when Command =>
				    
				    Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
				 when Document =>
				    
				    Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
				 when Web_Search =>
				    Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
			      end case;
			      
			      Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
			      Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
			      Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
			      Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
			      
			      
			      
			      
			      case Current_State is
				 when Command =>
				    null;
				    
				 when Document =>
				    if Center.Doc_Initialized then
				       Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
				    else
				       Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
				    end if;
				 when Web_Search =>
				    null;
			      end case;
			      
			      
			      if Center.User_Ready then
				 Devel_Draw_User(Center.User, Center.Frames(Visitor).Win, True);
			      else
				 Devel_Draw_User(null, Center.Frames(Visitor).Win, False);
			      end if;
			      
			      
			      
			      if Completion_Result /= null and then
				Completion_Result.Wlines /= null then
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Completion_Result, Center.Frames(Cmd_Result).Win, Result_Index);		     
			      elsif Center.Devel_Spawn_Result /= null then
				 Draw_Window(Center.Frames(Cmd).Win);
				 
				 if Center.Devel_Spawn_Result.Success then
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Success");		     		     
				 else
				    Draw_Text(Center.Frames(Err).Win, 1, 1, White, (-Center.Devel_Spawn_Result.Program_Name) & " Error : " & Handling.To_Wide_String(Integer'Image(Center.Devel_Spawn_Result.Return_Code)));
				 end if;
				 Draw_Window(Center.Frames(Cmd_Result).Win);
				 Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, Result_Index);
			      end if;
			      if Center.Doc_Initialized then
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			      else
				 Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			      end if;
			      
			      if Center.Current_Doc /= null and then
				Center.Current_Doc.Openned then
				 if Center.Current_Doc.Filename /= null then
				    
				    Save_Buffer(Center.Current_Doc, Lines_Table);
				    
				 end if;
				 
				 Set_Free(Lines_Set);

				 Highlight(Center.Current_Doc, Lines_Set);		  
				 
				 if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Spec_Error;
				 end if;
			      else
				 Set_Free(Lines_Table);
				 Doc_Init(Center.all, Lines_Table);
				 if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
				    raise Program_Error;
				 end if;
			      end if;
			      
			      
			      T_Io.Put(Restore_Cursor);
			      Screen_Lock.Release;
			      Clocking.Refresh;
			   when Wide_Character'Val(14) =>
			      Clocking.Suspend;			
			      Goal_Path_Deamon.Suspend;
			      Ansi_Console.Clear_Screen;
			      loop
				 begin
				    
				    declare
				       Filename : Wide_String := 
					 New_Wide_String ("New file", "Filename : ",  80, Lines/2-2, Cols/2-40, True);
				       
				       
				    begin
				       Ansi_Console.Clear_Screen;
				       declare
					  Response : constant Wide_String :=
					    (New_Wide_String 
					       ("License", "Include the Gnu Public Licence ? (y/o or n) : ",  50, Lines/2-2, Cols/2-25, True));
					  License  : Boolean := False;
				       begin
					  if Response(1) = 'y' or
					    Response(1) = 'Y'  or
					    Response(1) = 'o' or
					    Response(1) = 'O' then
					     License := True;
					  end if;
					  Ansi_Console.Clear_Screen;
					  
					  if License then				       
					     declare
						Description : Wide_String := 
						  New_Wide_String  ("Descrption", "Descr : ", 80, Lines/2-2, Cols/2-40, True);
						
					     begin
						Ansi_Console.Clear_Screen;
						declare
						   Version : Wide_String := 
						     New_Wide_String  ("Version", "Version : ", 20, Lines/2-2, Cols/2-10, True);
						begin
						   Set_Free(Lines_Table);
						   Ansi_Console.Clear_Screen;
						   Docume_Free(Center.Current_Doc);
						   New_Doc (Center => Center.all,
							    Lines_Table => Lines_Table,
							    Docume => Center.Current_Doc,
							    Filename => Handling.To_String(Filename),
							    Version  => Version,
							    License => True,
							    Description => Description);
						   Center.Current_Doc.Owner := new String ' (Center.User.Logname.all);
						   
						   Wline := new Wide_String ' ("");
						   
						   Lines_Table.lines(Lines_Table.Line_Last + 1) := Wline;
						   
						   Lines_Table.Line_Last := Lines_Table.Line_Last + 1;
						   
						   Center.Doc_Initialized := True;
						   
						   Doc_Index := 0;
						   
						   
						   Line_Cur := 1;
						   Col_Cur := 1;						
						   Line_Current_Pos := 0;
						   
						end;
					     end;
					  else
					     Ansi_Console.Clear_Screen;
					     declare
						Version : Wide_String := 
						  New_Wide_String  ("Version", "Version : ", 20, Lines/2-2, Cols/2-10, True);
					     begin
						Set_Free(Lines_Table);
						Ansi_Console.Clear_Screen;
						Docume_Free(Center.Current_Doc);
						New_Doc (Center => Center.all,
							 Lines_Table => Lines_Table,
							 Docume => Center.Current_Doc,
							 Filename => Handling.To_String(Filename),
							 Version => Version,
							 License => False,
							 Description => "");
						Center.Current_Doc.Owner := new String ' (Center.User.Logname.all);
						Wline := new Wide_String ' ("");
						Lines_Table.lines(Lines_Table.Line_Last + 1) := Wline;
						Lines_Table.Line_Last := Lines_Table.Line_Last + 1;
						Center.Doc_Initialized := True;
						Doc_Index := 0;
						Line_Cur := 1;
						Col_Cur := 1;
						Line_Current_Pos := 0;
						
					     end;
					     
					  end if;
					  
					  Current_State := Document;
				       end;
				    end;
				    Screen_Lock.Seize;
				    Devel_Draw_Main_Win (Center);
				    case Current_State is
				       when Command =>
					  
					  Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
				       when Document =>
					  
					  Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
				       when Web_Search =>
					  Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
				    end case;
				    Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
				    Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
				    Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
				    Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
				    Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
				    Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
				    Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
				    Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
				    Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
				    Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
				    Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
				    
				    Screen_Lock.Release;
				    exit;
				 exception
				    when W_Io.End_Error =>
				       exit;			   
				 end;
			      end loop;
			      Clocking.Start;		
			      Goal_Path_Deamon.Restart;
			   when Wide_Character'Val(15) =>			
			      Clocking.Suspend;			
			      Goal_Path_Deamon.Suspend;
			      Ansi_Console.Clear_Screen;
			      declare
				 Filename : Wide_String := 
				   New_Wide_String ("Open file", "Filename : ",  80, Lines/2-2, Cols/2-40, True);
			      begin
				 Ansi_Console.Clear_Screen;			   
				 Docume_Free(Center.Current_Doc);
				 
				 Set_Free(Lines_Table);

				 Open_Doc (Lines_Table => Lines_Table,
					   Docume => Center.Current_Doc,
					   Filename => Handling.To_String(Filename));
				 Center.Doc_Initialized := True;
				 
				 if Lines_Table.Line_Last /= 0 then
				    Wline := Lines_Table.Lines(1);
				    Wline_Length := Real_Length(Wline.all);
				 else
				    Lines_Table.Lines(1) := Wline;
				 end if;
				 
				 Line_Index := 1;
				 Doc_Index := 0;			   
				 Line_Current_Pos := 0;
				 Line_Cur := 1;
				 Col_Cur := 1;
				 
			      exception
				 when Spec_Error =>
				    null;
				    
			      end;			
			      Screen_Lock.Seize;
			      Devel_Draw_Main_Win (Center);
			      Current_State := Document;
			      
			      Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
			      Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
			      Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
			      Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
			      Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
			      Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
			      Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
			      Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
			      

			      
			      Screen_Lock.Release;
			      
			      Clocking.Start;
			      Goal_Path_Deamon.Restart;
			   when Wide_Character'Val(127) =>
			      if Lines_Table.Lines(Line_Index) /= null then
				 if Line_Current_Pos > 1 then
				    case Is_Overwrite is
				       when False =>
					  if Line_Current_Pos > 2 then
					     if Wline'Length >= 2 then
						Left_Buffer := new Wide_String ' (Wline(1..Line_Current_Pos-1));
						if Line_Current_Pos < Wline'Length then
						   Wbuffer := new Wide_String ' (Left_Buffer.all & Wline(Line_Current_Pos+1..Wline'Last));
						else
						   Wbuffer := new Wide_String ' (Left_Buffer.all);
						end if;
						
					     else
						Left_Buffer := new Wide_String ' (Wline(1..Line_Current_Pos-1));
						Wbuffer := new Wide_String ' (Left_Buffer.all);
					     end if;
					     
					  elsif Line_Current_Pos = 2 then
					     Left_Buffer := new Wide_String ' (Wline(1..1));
					     if Wline'Length >= 2 then
						Wbuffer := new Wide_String ' (Left_Buffer.all & Wline(Line_Current_Pos+1..Wline_Length));
					     else
						Wbuffer := new Wide_String ' (Left_Buffer.all);
					     end if;
					  elsif Wline'Length >= 2 then
					     Left_Buffer := new Wide_String ' (Wline(Line_Current_Pos+1..Wline'Length));
					     Wbuffer := new Wide_String ' (Left_Buffer.all);
					  end if;
					  Wide_Free(Wline);
					  Wline := new Wide_String ' (Wbuffer.all);
					  Wide_Free(Wbuffer);
					  Wide_Free(Left_Buffer);
					  Lines_Table.Lines(Line_Index) := Wline;
					  Line_Current_Pos := Line_Current_Pos - 1;
					  
					  Wline_Length := Wline'Length;
					  Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
					  if (Line_Current_Pos mod Width) /= 0 then
					     Col_Cur := (Line_Current_Pos mod Width) + 1;
					  else
					     Col_Cur := 1;
					  end if;
				       when True =>
					  Wide_Fixed.Replace_Slice (Wline.all, Line_Current_Pos, Line_Current_Pos, " ");
					  Line_Current_Pos := Line_Current_Pos - 1;
					  
					  Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
					  if (Line_Current_Pos mod Width) /= 0 then
					     Col_Cur := (Line_Current_Pos mod Width) + 1;
					  else
					     Col_Cur := 1;
					  end if;
				    end case;
				 elsif Line_Current_Pos = 0 then
				    
				    case Is_Overwrite is
				       when False =>
					  if Line_Index > 1 then
					     T_Io.New_Line;					     
					     if Wline'Length >= 1 then
						
						Left_Buffer := new Wide_String ' (Lines_Table.Lines(Line_Index - 1).all);
						
						if Line_Current_Pos < Wline'Length then
						   
						   Wbuffer := new Wide_String ' (Left_Buffer.all & Wline(Wline'First..Wline'Last));
						   
						else
						   Wbuffer := new Wide_String ' (Left_Buffer.all);
						   
						end if;
						
					     else
						
						Left_Buffer := new Wide_String ' (Lines_Table.Lines(Line_Index - 1).all);
						Wbuffer := new Wide_String ' (Left_Buffer.all);
						

					     end if;
					     Line_Current_Pos := Left_Buffer'Length;
					     
					     Wide_Free(Wline);
					     Wline := new Wide_String ' (Wbuffer.all);
					     Wide_Free(Wbuffer);
					     Wide_Free(Left_Buffer);
					     
					     Wide_Free(Lines_Table.Lines(Line_Index - 1));
					     Lines_Table.Lines(Line_Index - 1) := Wline;
					     
					     if Line_Index+1 < Lines_Table.Line_Last then
						
						for Line in Line_Index .. Lines_Table.Line_Last-1 loop
						   Lines_Table.Lines(Line) := Lines_Table.Lines(Line+1);
						end loop;
						
					     end if;
					     Line_Index := Line_Index - 1;
					     Lines_Table.Line_Last := Lines_Table.Line_Last - 1;
					     Wline_Length := Wline ' Length;
					     
					     Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
					     
					     if (Line_Current_Pos mod Width) /= 0 then
						Col_Cur := (Line_Current_Pos mod Width) + 1;
					     else
						Col_Cur := 1;
					     end if;
					     
					  else
					     W_Io.Put(Wide_Character'Val(7));
					  end if;
				       when True =>
					  if Line_Index > 1 then
					     
					     Left_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, width);
					     Wline_Length := Lines_Table.Lines(Line_Index)'Length;
					     Line_Current_Pos := Wline_Length;
					     
					     Wide_Fixed.Replace_Slice (Wline.all, Line_Current_Pos-1, Line_Current_Pos-1, " ");
					     Line_Current_Pos := Line_Current_Pos - 1;
					     
					     Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
					     if (Line_Current_Pos mod Width) /= 0 then
						Col_Cur := (Line_Current_Pos mod Width) + 1;
					     else
						Col_Cur := 1;
					     end if;
					     
					  else
					     W_Io.Put(Wide_Character'Val(7));
					  end if;
				    end case;
				 elsif Line_Current_pos = 1 then
				    case Is_Overwrite is
				       when False =>
					  
					  if Wline'Length > 1 then
					     
					     Wbuffer := new Wide_String ' (Wline(Line_Current_Pos+1..Wline'Last));
					     
					  else
					     
					     Wbuffer := new Wide_String ' ("");					     
					  end if;
					  
					  Wide_Free(Wline);
					  Wline := new Wide_String ' (Wbuffer.all);
					  Wide_Free(Wbuffer);
					  
					  Lines_Table.Lines(Line_Index) := Wline;
					  
					  Wline_Length := Lines_Table.Lines(Line_Index)'Length;
					  
					  Line_Current_Pos := 0;
					  Col_Cur := 1;
				       when True =>
					  Left_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, width);
					  Wline_Length := Lines_Table.Lines(Line_Index)'Length;
					  
					  
					  Wide_Fixed.Replace_Slice (Wline.all, Line_Current_Pos, Line_Current_Pos, " ");
					  Line_Current_Pos := 0;
					  
					  Line_Cur := Count (Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Height, Width);
					  Col_Cur := 1;
				    end case;
				 end if;
			      end if;
			   when others =>				 
			      null;
			end case;
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);
			if Center.Doc_Initialized then
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			else
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			end if;
			
			if Center.Current_Doc /= null and then
			  Center.Current_Doc.Openned then
			   if Center.Current_Doc.Filename /= null then
			      Save_Buffer(Center.Current_Doc, Lines_Table);
			   end if;
			   
			   Set_Free(Lines_Set);
			   Highlight(Center.Current_Doc, Lines_Set);		  
			   
			   Draw_Window(Center.Frames(Document).Win);
			   if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			      raise Spec_Error;
			   end if;
			   
			else
			   
			   Set_Free(Lines_Table);
			   Doc_Init(Center.all, Lines_Table);
			   
			   if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			      raise Program_Error;
			   end if;
			   
			end if;
			
			T_Io.Put(Restore_Cursor);
			
			Screen_Lock.Release;
			
			Clocking.Refresh;
			
		     else
			
			if Wline /= null and then
			  Wline'Length = 256 then
			   Put(Wide_Character'Val(7));
			else
			   if Wline'Length = 0 then
			      Wbuffer := new Wide_String ' (Wline.all & Wchar);
			      Wide_Free(Wline);
			      Wline := new Wide_String ' (Wbuffer.all);
			      Wide_Free(Wbuffer);			
			      
			      Line_Current_Pos := Wline'Length;

			      Wline_Length := Wline'Length;
			   elsif Line_Current_Pos = Wline'Length then
			      Wbuffer := new Wide_String ' (Wline.all & Wchar);
			      Wide_Free(Wline);
			      Wline := new Wide_String ' (Wbuffer.all);
			      Wide_Free(Wbuffer);			
			      Wline_Length := Wline'Length;
			      Line_Current_Pos := Wline_length;

			      
			      
			   else
			      if Line_Current_Pos > 1 then
				 if not Is_Overwrite then
				    Left_Buffer := new Wide_String ' (Wline(1..Line_Current_Pos));
				    if Wline'Length > Line_Current_Pos then
				       Wbuffer := new Wide_String ' (Left_Buffer.all & Wchar & Wline(Line_Current_Pos+1..Wline'Last));
				    else
				       Wbuffer := new Wide_String ' (Left_Buffer.all & Wchar);
				    end if;
				    Wide_Free(Wline);
				    Wline := new Wide_String ' (Wbuffer.all);
				    Wide_Free(Wbuffer);					  
				    Wide_Free(Left_Buffer);
				    Wline_Length := Wline_Length + 1;				 
				    Line_Current_Pos := Line_Current_Pos + 1;
				    
				 else
				    Wide_Fixed.Replace_Slice (Wline.all, Line_Current_Pos, Line_Current_Pos, Wchar & "");
				    Line_Current_Pos := Line_Current_Pos + 1;
				    
				 end if;
				 
				 
			      elsif not Is_Overwrite then
				 
				 Wbuffer := new Wide_String ' (Wchar & Wline.all);
				 Wide_Free(Wline);
				 Wline := new Wide_String ' (Wbuffer.all);
				 Wide_Free(Wbuffer);
				 Wline_Length := Wline_Length + 1;			      
				 Line_Current_Pos := Line_Current_Pos + 1;
				 
				 
			      elsif Is_Overwrite and Line_Current_Pos < Wline_Length then
				 Wide_Fixed.Replace_Slice (Wline.all, Line_Current_Pos + 1, Line_Current_Pos + 1, Wchar & "");
				 Line_Current_Pos := Line_Current_Pos + 1;
				 
				 
			      elsif Is_Overwrite and Wline_Length = 1 and Line_Current_Pos = 0 then
				 Wide_Fixed.Replace_Slice (Wline.all, 1, 1, Wchar & "");
				 Line_Current_Pos := Line_Current_Pos + 1;
				 
			      elsif Is_Overwrite and Wline_Length = 1 and Line_Current_Pos = 1 then
				 Wbuffer := new Wide_String ' (Wline.all & Wchar);
				 Wide_Free(Wline);
				 Wline := new Wide_String ' (Wbuffer.all);
				 Wide_Free(Wbuffer);			
				 
				 Line_Current_Pos := Wline'Length;
				 
				 Wline_Length := Wline'Length;
			      end if;
			   end if;
			   if ((Col_Cur + 1) mod Width) = 0 then
			      Line_Cur := Line_Cur + 1;
			      Col_Cur := 1;
			   else				 
			      Col_Cur := (Line_Current_Pos mod Width) + 1;
			   end if;
			end if;
			Lines_Table.Lines(Line_index) := Wline;
			
			Screen_Lock.Seize;
			T_Io.Put(Save_Cursor);
			if Center.Doc_Initialized then
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
			else
			   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, False);
			end if;
			
			if Center.Current_Doc /= null and then
			  Center.Current_Doc.Openned then
			   if Center.Current_Doc.Filename /= null then
			      Save_Buffer(Center.Current_Doc, Lines_Table);
			   end if;
			   Set_Free(Lines_Set);
			   Highlight(Center.Current_Doc, Lines_Set);		  
			   Draw_Window(Center.Frames(Document).Win);
			   if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			      raise Spec_Error;
			   end if;
			else
			   Set_Free(Lines_Table);
			   Doc_Init(Center.all, Lines_Table);
			   if not Print_Set(Lines_Table, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			      raise Program_Error;
			   end if;
			end if;
			T_Io.Put(Restore_Cursor);
			Screen_Lock.Release;
			Clocking.Refresh;
		     end if;
		     
		     --Draw_Text(Center.Frames(Doc_states).Win, 1, 30, White, "Line_Last : " & Handling.To_Wide_String(Natural'Image(Lines_Table.Line_Last)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 1, 50, White, "Line_index : " & Handling.To_Wide_String(Natural'Image(Line_index)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 2, 30, White, "Doc_index : " & Handling.To_Wide_String(Natural'Image(Doc_index)));
		     --Draw_Text(Center.Frames(Doc_states).Win, 2, 50, White, "wchar_index : " & Handling.To_Wide_String(Natural'Image(Wchar_index)));
		     
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;
		  end Receive;
	    or
	       accept Reset;
	    or
	       accept Lock do
		  suspended := True;	       
		  
		  Clocking.Suspend;
		  
		  Goal_Path_Deamon.Suspend;
		  
	       end Lock;
	       exit;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean) do
		  
		  null;
		  
	       end Set_On_Esc;
	    or
	       when Current_State = Document =>
		  accept Page_Down do
		     --  Screen_Lock.Seize;
		     --  T_Io.Put(Save_Cursor);
		     --  if Doc_Index + (Height /3 * 2) < Lines_Set.Line_Last then
		     --  	Doc_Index := Doc_Index + (Height/3*2);
		     --  	if Center.Doc_Initialized then
			   
		     --  	   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
		     --  	end if;
			
		     --  	Draw_Window(Center.Frames(Document).Win);
		     --  	if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     --  	   raise Spec_Error;
		     --  	end if;
		     --  else
		     --  	T_Io.Put(Character'Val(7));
		     --  end if;
		     --  T_Io.Put(Restore_Cursor);
		     --  Screen_Lock.Release;
		     null;
		  end Page_Down;
	    or
	       when Current_State = Document =>
		  accept Page_Up do
		     
		     --  if Doc_Index - (Height /3 * 2) > 0 then
		     --  	Doc_Index := Doc_Index - (Height/3*2);
		     --  	Screen_Lock.Seize;
		     --  	T_Io.Put(Save_Cursor);
		     --  	if Center.Doc_Initialized then
			   
		     --  	   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
		     --  	end if;
			
		     --  	Draw_Window(Center.Frames(Document).Win);
		     --  	if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     --  	   raise Spec_Error;
		     --  	end if;
		     --  elsif Doc_Index /= 0  then
		     --  	Doc_Index := 0;
		     --  	if Center.Doc_Initialized then
			   
		     --  	   Devel_Draw_Docume(Center.Current_Doc, Center.Frames(Doc_States).Win, True);
		     --  	end if;
			
		     --  	Draw_Window(Center.Frames(Document).Win);
		     --  	if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     --  	   raise Spec_Error;
		     --  	end if;
		     --  else
		     --  	T_Io.Put(Character'Val(7));
		     --  end if;
		     --  T_Io.Put(Restore_Cursor);
		     --  Screen_Lock.Release;
		     null;
		  end Page_Up;
	    or
	       when Current_State = Command =>
		  accept Page_Down do
		     if Center.Devel_Spawn_Result /= null and then
		       File_Index+(Result_Max_Lines-Result_Max_Lines/2) < Center.Devel_Spawn_Result.Wlines'Length  then
			File_Index := Positive'Min(File_Index+(Result_Max_Lines-Result_Max_Lines/2), 
						   Center.Devel_Spawn_Result.Wlines'Length);
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     Draw_Window(Center.Frames(Cmd_Result).Win);
		     Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, File_Index);
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
		  end Page_Down;
	    or
	       when Current_State = Command =>
		  accept Page_Up do
		     
		     if Center.Devel_Spawn_Result /= null and then
		       File_Index-(Result_Max_Lines-Result_Max_Lines/2) >= 0  then
			File_Index := File_Index - (Result_Max_Lines-Result_Max_Lines/2);
		     else
			T_Io.Put(Character'Val(7));
			File_Index := 0;
		     end if;	       
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     
		     Draw_Window(Center.Frames(Cmd_Result).Win);
		     Draw_Buffer(Center.Devel_Spawn_Result, Center.Frames(Cmd_Result).Win, File_Index);
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
		  end Page_Up;
	    or
	       
	       when Current_State = Document =>
		  accept Up_Arrow do

		     Up_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);

		     Wline := Lines_Table.Lines(Line_index);
		     Wline_Length := Real_Length(Wline.all);
		     
		     

		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     Draw_Window(Center.Frames(Document).Win);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;	
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
		     Clocking.Refresh;
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;
		  end Up_Arrow;
	    or
	       when Current_State = Document =>
		  accept Down_Arrow do
		     
		     Down_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);

		     
		     Wline := Lines_Table.Lines(Line_index);

		     Wline_Length := Real_Length(Wline.all);
		     
		     
		     
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     Draw_Window(Center.Frames(Document).Win);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;	
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;		     
		     Clocking.refresh;
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;
		  end Down_Arrow;
	    or
	       when Current_State = Document =>
		  accept Left_Arrow do
		     Left_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);
		     
		     
		     Wline := Lines_Table.Lines(Line_index);	       
		     Wline_Length := Real_Length(Wline.all);
		     
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     Draw_Window(Center.Frames(Document).Win);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;	
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
		     Clocking.refresh;
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;
		  end Left_Arrow;
	    or
	       when Current_State = Document =>
		  accept Right_Arrow do	       		  
		     Right_Arrow(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);
		     
		     
		     
		     
		     Wline := Lines_Table.Lines(Line_index);
		     Wline_Length := Real_Length(Wline.all);
		     
		     
		     Screen_Lock.Seize;
		     T_Io.Put(Save_Cursor);
		     Draw_Window(Center.Frames(Document).Win);
		     if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
			raise Spec_Error;
		     end if;	
		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;

		     Clocking.refresh;
		     select
			Goal_Path_Deamon.Refresh;
		     or
			delay 0.1;
		     end select;
		  end Right_Arrow;
		  
	    or
	       when Current_State = Command =>
		  accept Up_Arrow do
		     Draw_Window(Center.Frames(Cmd).Win);
		     if (History_Last > 0) and then
		       (History_Last - History_Index) >= 0 then
			Wide_Free(Cmd_Line);
			Cmd_Line := new Wide_String ' (History_Set.Lines(History_Last-(History_Index-1)).all);
			Cmd_Line_Length := Cmd_Line'Length;
			Cmd_Current_Pos := Cmd_Line_Length + 1;
			History_Index := History_Index + 1;
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		     if Cmd_Line /= null then
			Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
		     end if;
		  end Up_Arrow;
	    or
	       
	       when Current_State = Command =>
		  accept Down_Arrow do
		     Draw_Window(Center.Frames(Cmd).Win);
		     if (History_Index > 2) then
			Cmd_Line := new Wide_String ' (History_Set.Lines(History_Last-(History_Index-1)).all);
			Cmd_Line_Length := Cmd_Line'Length;
			Cmd_Current_Pos := Cmd_Line_Length + 1;
			History_Index := History_Index - 1;
		     elsif History_Index = 2 then
			Wide_Free(Cmd_Line);
			Cmd_Line := new Wide_String ' ("");
			Cmd_Line_Length := Cmd_Line'Length;
			Cmd_Current_Pos := Cmd_Line_Length + 1;
			History_Index := History_Index - 1;
		     else
			T_Io.Put(Character'Val(7));
		     end if;	       	       	       
		     if Cmd_Line /= null then
			Draw_Text(Center.Frames(Cmd).Win, 1, 1, White, Cmd_Line.all);
		     end if;
		  end Down_Arrow;
		  
	    or
	       
	       when Current_State = Command =>
		  accept Left_Arrow do
		     if Cmd_Line /= null and then Cmd_Line'Length > 0 then
			if Cmd_Current_Pos > 1 then
			   
			   Cmd_Current_Pos := Cmd_Current_Pos - 1;
			   
			else
			   T_Io.Put(Character'Val(7));
			end if;
		     end if;
		  end Left_Arrow;
	    or 
	       when Current_State = Command =>
		  accept Right_Arrow do
		  if Cmd_Line /= null and then Cmd_Line'Length > 0 then
		     if Cmd_Current_Pos < Cmd_Line'Length+1 then
			
			   
			Cmd_Current_Pos := Cmd_Current_Pos+1;
			   
			   
			
		     else
			T_Io.Put(Character'Val(7));
		     end if;
		  end if;
	       end Right_Arrow;
	    or
	       accept Full_Screen;
	    or
	       accept Del;
	    or
	       when Current_State = command =>		  
		  accept Begin_Of do
		     Cmd_Current_Pos := 1;
		  end Begin_Of;
		  
	    or
	       when Current_State = command =>
		  accept End_Of do
		     if Cmd_Line'Length >= 1 then
			Cmd_Current_Pos := Cmd_Line'Length + 1;
		     else
			Cmd_Current_Pos := 1;
		     end if;
		  end End_Of;
	    or
	       when Current_State = Document =>
	       accept Begin_Of do
		  
		  At_Begin(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);
		  
		  
		  Screen_Lock.Seize;
		  Draw_Window(Center.Frames(Document).Win);
		  if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     raise Spec_Error;
		  end if;	
		  Screen_Lock.Release;
		  Clocking.Refresh;
		  select
		     Goal_Path_Deamon.Refresh;
		  or
		     delay 0.1;
		  end select;
	       end Begin_Of;
	    or
	       when Current_State = Document =>
	       accept End_Of do
		  At_End(Lines_Table, Doc_Index, Line_Index, Line_Current_Pos, Line_Cur, Col_Cur, Height, Width);
		  
		  Wline := Lines_Table.Lines(Line_index);
		  
		  Line_Current_Pos := Wline'Last;
		  
		  Screen_Lock.Seize;
		  Draw_Window(Center.Frames(Document).Win);
		  if not Print_Set(Lines_Set, Center.Frames(Document).Win, Doc_Index, Total_Line) then
		     raise Spec_Error;
		  end if;	
		  Screen_Lock.Release;
		  Clocking.Refresh;
		  select
		     Goal_Path_Deamon.Refresh;
		  or
		     delay 0.1;
		  end select;
	       end End_Of;
	    or
	       accept Overwrite do
		  Is_Overwrite := not Is_Overwrite;
		  Screen_Lock.Seize;
		  Draw_Window(Center.Frames(Buffer).Win);
		  
		  T_Io.Put(Save_Cursor);
		  case Current_State is
		     when Command =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Command mode ");
		     when Document =>
			
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Document mode");
		     when Web_Search =>
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "Is Search mode  ");
		  end case;
		  
		  
		  Draw_Text(Center.Frames(Buffer).Win, 1, 22, white, " ; Alphabet");
		  Draw_Text(Center.Frames(Buffer).Win, 1, 33, red, " : "); 
		  Draw_Text(Center.Frames(Buffer).Win, 1, 36, yellow, "aàâäeéèêëiïîoôuùybcdfghjklmnpqrstvwxz.;?!'_ -");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 2, Cyan, "Chars max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 20, magenta, Handling.To_Wide_String(Natural'Image(Cols-5)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 2, Cyan, "Char(s) left");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 17, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 20, magenta, Handling.To_Wide_String(Natural'Image(((Cols- 5) - Wline_Length))));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 25, Cyan, " ; word max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Sh_Max_Words)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 25, Cyan, " ; Word number");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 39, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 43, magenta, Handling.To_Wide_String(Word_Range'Image(Word_Number)));
		  Draw_Text(Center.Frames(Buffer).Win, 2, 50, Cyan, " ; lines max");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 2, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(2880)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 50, Cyan, " ; Lines count");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 64, white, " : ");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 66, magenta, Handling.To_Wide_String(Line_Index_Range'Image(Dialog_Buffer_Last)));
		  Draw_Text(Center.Frames(Buffer).Win, 3, 71, Cyan, " ; Overwrite");
		  Draw_Text(Center.Frames(Buffer).Win, 3, 83, white, " : ");	    
		  Draw_Text(Center.Frames(Buffer).Win, 3, 86, magenta, Handling.To_Wide_String(Boolean'Image(Is_Overwrite)));
		  T_Io.Put(Restore_Cursor);
		  Screen_Lock.Release;
	       end Overwrite;
	    or
	       accept Screen_Print;
	    or
	       accept Shutdown;
	    or
	       accept New_Evidence do	       
		  Screen_Lock.Seize;
		  T_Io.Put(Save_Cursor);
		  Devel_Ai_Draw_Proposition(Center);
		  Devel_Ai_Draw_Neural_States(Center);
		  T_Io.Put(Restore_Cursor);
		  Screen_Lock.Release;
	       end New_Evidence;
	    or
	       accept New_Ranking(Objects : Rank_Array) do
		  
		  Center.Classes_Ranking := new  Rank_Array ' (Objects);
		  Sort_Init(Center.Classes_Ranking);
		  Screen_Lock.Seize;
		  T_Io.Put(Save_Cursor);
		  Devel_Ai_Draw_Top_Class (Center);
		  Devel_Ai_Draw_Class_Ranking (Center);	       
		  T_Io.Put(Restore_Cursor);
		  Screen_Lock.Release;
	       end New_Ranking;
	    end select;
	 end loop;
      end loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Devel halted");
      Center.Global_Lock.Release;
   end Devel_Process;
   
   
   use Expr_Vectors;
   use Prompt_Vectors;
   use Intelligency.Classes;
   
   task body Organizer_Process is
      
      
      Screen_Lock : Binary_Semaphore (True, Default_Ceiling);
      
      
      task type Clock_Task is
      	 entry Fresh;
      	 entry Stop;
      	 entry Start;
	 entry Halt;
      end Clock_Task;
      
      
      
      task body Clock_Task is
      
      	 task type Cross_task is
      	    entry Fresh;	
	    entry Halt;
      	 end Cross_task;
      
      	 task body Cross_Task is
      
      	    type Cross_Type is ('\', '|', '/','-');
      
      	    package Cross_Wrapping is new PragmARC.Wrapping(Cross_Type);
      	    use Cross_Wrapping;
      	    Cross : Cross_Type := '-';
      	 begin
      	    loop
	       select
		  accept Fresh do
		     
		     T_Io.Put(Save_Cursor);
		     Cross := Wrap_Succ(Cross);
		     Draw (Wins_Main.Main_Win_Ptr.All,
			   1,
			   (Columns/2)+22,
			   White, Handling.To_Wide_String("" & Character'Value(Cross_Type'Image(Cross))));
		     T_Io.Put(Restore_Cursor);
		     
		  end Fresh;
	       or
		  accept Halt;
		  exit;
	       end select;
      	    end loop;
      	 end Cross_Task;
      	 Speed_Cross : Cross_Task;
	 
	 End_Of_Task : Boolean := False;
	 
      begin
      	 while not End_Of_Task loop
	    loop
	       select
		  accept Stop;
	       or
		  accept Start;
		  exit;
	       or
		  accept Fresh;
	       or
		  accept Halt do
		     End_Of_Task := True;
		     Speed_Cross.Halt;
		  end Halt;
		  exit;
	       end select;
      	    end loop;	    
      	    while not End_Of_Task loop
      	       select
      		  accept Fresh do
		     Screen_Lock.Seize;
      		     T_Io.Put(Save_Cursor);
      		     Draw (Wins_Main.Main_Win_Ptr.All,
      				   1,
			   (Columns/2)-23,
			   cyan, Handling.To_Wide_String("elhoim verion " & El.To_String(El.Shared.Version) & " "& Formatting.Image(clock)(1..19)));
      		     T_Io.Put(Restore_Cursor);
		     Screen_Lock.Release;
      		  end Fresh;
      	       or
      		  accept Stop;
      		  exit;
      	       or
		  
		  accept Start;
	       or
		  
		  accept Halt do
		     End_Of_Task := True;
		     Speed_Cross.Halt;
		  end Halt;
		  exit;
	       or
      		  delay 0.1;
		  Screen_Lock.Seize;
      		  Speed_Cross.Fresh;
		  Screen_Lock.Release;
		  
      	       end select;
      	    end loop;
      	    
      	 end loop;
      end Clock_Task;
      
      
      
      Oclock : Clock_Task;
      
      
      Input_Line : Strings.Wide_String_Access := new Wide_String ' ("");
      Output_Line : Strings.Wide_String_Access := new Wide_String ' ("");
      Internal_Line : Strings.Wide_String_Access := new Wide_String ' ("");
      Internal_Input : Strings.Wide_String_Access := new Wide_String ' ("");
      
      
      Refresh : Boolean := False;
      
      
      
      Input : Input_Type := (others => 0.0);
      Feedback : Output_Type := (others => 0.0);
      saved : Output_Type := (others => 0.0);
      Output : Output_Type := (others => 0.0);
      ---------------------------
      ---------------------------
      State_Index : Natural := 0;
      Ai_Index : Natural := 0;
      
      Line_Index : Positive := 1;
      
      Subtitle : Wide_String(1..Columns/4-2) := (others => Wide_Character'Val(32));
      
      
      Input_Buffer : Strings.Wide_String_Access;
      
      WChar_Num : Natural := 0;
      
            
      Suspended : Boolean := False;
      
      End_Of_This_Task : Boolean := False;
      
      
   begin
      while not End_Of_This_Task loop
	 Window.Windows.Initialize;
	 loop
	    select

	       accept Halt (End_Of_Program : out Boolean) do
		  End_Of_This_Task := True;
		  End_Of_Program := End_Of_This_Task;
		  Oclock.Halt;
	       end Halt;
	       exit;
	    or
	       accept Initialize do
		  Center.Header_Win.Initialize
		    (Lines/2-1,
		     Cols/2-20, 3, 40, Cyan, white);
		  Draw_Window(Center.Header_Win);
		  Window.Draw_Text(Center.Header_Win,
		    1, (20-7), white, 
		     "Organizer mode"
		    );
		  Wins_Draw(Wins_Main);
		  
		  State_Index := 0;
		  Ai_Index := 0;
		  
		  if Center.T.Obj_Cur /= null then		  	 	 
		     Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		  else
		     Draw(Wins_Main.User_Objects_Tree_Win_Ptr.all, 1, 1, White, "Obj_Cur = null");
		  end if;
		  
		  if Center.Ai.Obj_Cur /= null then		  	 	 
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);	 
		  else
		     Draw(Wins_Main.Ai_Objects_Tree_Win_Ptr.all, 1, 1, White, "Obj_Cur = null");
		  end if;
		  Wide_Fixed.Move ("Subject", Center.Prompt);
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		  
		  
	       end Initialize;
	       exit;
	    or
	       accept Lock;
	    or
	       accept Wake_Up do
		  Wins_Draw(Wins_Main);
		  
		  State_Index := 0;
		  Ai_Index := 0;
		  Screen_Lock.Seize;
		  Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		  Screen_Lock.Release;
		  Oclock.Start;
		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
	 State_Index := 0;
	 
	 
	 while not End_Of_This_Task loop
	    
	    
	    select
	       accept Wake_Up do
		  Wins_Draw(Wins_Main);
		  
		  State_Index := 0;
		  Ai_Index := 0;
		  Screen_Lock.Seize;
		  Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		  Screen_Lock.Release;
		  Oclock.Start;
		  
	       end Wake_Up;
	       
	    or
	       accept Halt (End_Of_Program : out Boolean) do
		  Oclock.Halt;
		  Save(Center.T.Obj, Center.User_Filename.all);		     
		  Save(Center.Ai.Obj, Center.Ai_Filename.all);
		  End_Of_This_Task := True;
		  End_Of_Program := End_Of_This_Task;
	       end Halt;
	       exit;
	    or
	       accept Initialize do
		  Center.Header_Win.Initialize
		    (Lines/2-1,
		     Cols/2-20, 3, 40, Cyan, white);
		  Draw_Window(Center.Header_Win);
		  Window.Draw_Text(Center.Header_Win,
			1, (20-7), white, 
		     "Organizer mode"
		    );
		  Screen_Lock.Seize;
		  Wins_Draw(Wins_Main);
		  
		  State_Index := 0;
		  Ai_Index := 0;
		  
		  if Center.T.Obj_Cur /= null then		  	 	 
		     Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		  else
		     Draw(Wins_Main.User_Objects_Tree_Win_Ptr.all, 1, 1, White, "Obj_Cur = null");
		  end if;
		  
		  if Center.Ai.Obj_Cur /= null then		  	 	 
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);	 
		  else
		     Draw(Wins_Main.Ai_Objects_Tree_Win_Ptr.all, 1, 1, White, "Obj_Cur = null");
		  end if;
		  
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
		  Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		  Screen_Lock.Release;
		  Oclock.Start;
	       end Initialize;
	    or
	       
	       when (Center.View_Is_Goal_Path = True) and (Center.Object_Is_Goal = False) =>
		  accept Receive (Wchar : in Wide_Character) do
		     Screen_Lock.Seize;
		     End_Of_This_Task := False;
		     begin
			WChar_Num := Wide_Character'Pos(Wchar);		     
			case WChar_Num is
			   when 0 .. 31 =>
			      
			      case WChar_Num is
				 when 1 => -- Ctrl+A
				    null		    ;
				 when 14 => -- ctrl+N
				    
				    Center.Goal_T.Page_Index := Center.Goal_T.Page_Index + 1;
				    Center.Ai.Page_Index := Center.Ai.Page_Index + 1;
				    Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				    Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    
				 when 16 => -- Ctrl+P 
				    if Center.Goal_T.Page_Index > 0 then
				       Center.Goal_T.Page_Index := Center.Goal_T.Page_Index - 1;
				       Center.Ai.Page_Index := Center.Ai.Page_Index - 1;
				       Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				       Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    end if;

				    
				 when 8 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    end if;
				 when 9 =>
				    null;
				 when 11 =>
				    null;
				 when 27 =>
				    null;
				 when 12 => -- Ctrl+L;
				    Wins_Draw(Wins_Main);
				    
				    State_Index := 0;
				    Ai_Index := 0;
				    
				    Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				    Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				 when 10 | 13 =>
				    if Refresh then
				       Wins_Draw(Wins_Main);
				       Refresh := False;
				    end if;
				    State_Index := 0;
				    Ai_Index := 0;
				    
				    
				    if Center.Expr_Index > 0 then
				       
				       Update(Center.Goal_T.all, Center.Prompt, Center.Expr);
				       
				       Update(Center.Ai, Center.Prompt, Center.Expr);
				       begin
					  
					  Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
					  
					  Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
					  
					  
					  Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
					  
					  Center.Expr_List := Center.Expr_List & Center.Expr;	 
					  Center.Prompt_List := Center.Prompt_List & Center.Prompt;
					  Center.Expr := (others => Wide_Character'Val(32));
					  Center.Expr_Index := 0;				   
					  Center.Goal_T.Page_Index := 0;
					  Center.Ai.Page_Index := 0;
					  
					  
				       exception
					  
					  when Is_word =>
					     
					     
					     Wide_Free(Output_line);
					     
					     
					     
					     
					     Input := (others => 0.0);			      
					     --Input := String2Input(UTF_Encoding.Wide_Strings.Encode(Input_Line.all));   
					     Input := String2Input(Input_Line.all, Center.Words);
					     
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     
					     Saved := Output;
					     
					     
					     --Deamon_network.Initialize(Input_Type(saved));
					     
					     
					     Input := (others => 0.0);			      
					     Output_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     
					     Input := Neural_Retrain.Input_Type(Output);
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     Feedback := (others => 0.0);
					     Feedback := Output;
					     
					     
					     
					     Internal_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     Output := (others => 0.0);
					     
					     
					     
					     
					     
					     Center.Expr := (others => Wide_Character'Val(32));
					     Center.Expr_Index := 0;
					     Refresh := True;
					     
					     Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					     
					     
					  when others =>
					     Draw(Wins_Main.Main_Win_Ptr.All, 1, (Columns-2)/2-12, white, "Error : cannot exec cmd");
					     
					     Refresh := True;
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
					     
					     raise;
					     
					     
				       end;

				    elsif Center.Goal_T.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				       
				       if Center.Ai.Obj_Cur /= null then
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
				       end if;
				    elsif Center.Ai.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				       
				    end if;			   
				    
				    Wide_Free(Input_line);
				    Input_Line := new Wide_String ' ("");				 				 				
				    

				    
				    Classes.Windows.Draw_Window (Wins_Main.Output_Win_Ptr.all);
				    
				    if Output_Line'Length > 0 then
				       if Output_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line(1..Output_Line'length));
				       else
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line.all(1..Positive'Min(Output_Line'Length,
												Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 2, 1, white,
								Output_Line.all((Columns/2-4)+1 .. Positive'Min(output_Line'Length, Columns/2-4+Columns/2-4)));
				       end if;
				    end if;
				    Wide_Free(Output_Line);
				    Output_Line := new Wide_String ' ("");
				    Classes.Windows.Draw_Window (Wins_Main.Internal_Win_Ptr.all);            
				    if Internal_Line'Length > 0 then
				       
				       if Internal_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line(1..Internal_Line'length));

				       else
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line.all(1..Positive'Min(Internal_Line'length,
												  Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 2, 1, white,
								Internal_Line.all((Columns/2-4)+1 .. Positive'Min(Internal_Line'Length, Columns/2-4+Columns/2-4)));

				       end if;
				       
				    end if;
				    Wide_Free(Internal_Line);
				    Internal_Line := new Wide_String ' ("");
				    Classes.Windows.Draw_Window(Wins_Main.Input_Win_Ptr.all);				 
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, ": ");
				    
				    
				    
				 when others =>
				    null;
			      end case;
			   when others =>
			      case Wchar_Num is
				 
				 when 0..31 =>
				    null;
				    
				 when 32 =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);
				    end if;
				 when 33 .. 126 =>
				    if Center.Expr_Index < (Columns-26)  then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				 when 127 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) >= 1 then
				       Input_Buffer := new Wide_String ' (Input_Line.all(1..Input_Line'Length-1));
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    end if;
				 when others =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    
			      end case;
			      Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
			      Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
			      Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
			      
			end case;
			Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
		     exception
			when others =>
			   null;
		     end;
		     Screen_Lock.Release;
		  end Receive;
	    or
	       
	       
	       when (Center.Object_Is_Goal = True) and (Center.View_Is_Goal_Path = false) =>
		  accept Receive (Wchar : in Wide_Character) do
		     Screen_Lock.Seize;
		     End_Of_This_Task := False;
		     
		     begin
			WChar_Num := Wide_Character'Pos(Wchar);		     
			case WChar_Num is
			   when 0 .. 31 =>
			      
			      case WChar_Num is
				 when 1 => -- Ctrl+A
				    null;
				    
				 when 14 => -- ctrl+N
				    
				    Center.Goal_T.Page_Index := Center.Goal_T.Page_Index + 1;
				    Center.Ai.Page_Index := Center.Ai.Page_Index + 1;
				    Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				    Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    
				 when 16 => -- Ctrl+P 
				    if Center.Goal_T.Page_Index > 0 then
				       Center.Goal_T.Page_Index := Center.Goal_T.Page_Index - 1;
				       Center.Ai.Page_Index := Center.Ai.Page_Index - 1;
				       Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				       Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    end if;

				    
				 when 8 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    end if;
				 when 9 =>
				    null;
				 when 11 =>
				    null;
				 when 27 =>
				    null;
				 when 12 => -- Ctrl+L;
				    Wins_Draw(Wins_Main);
				    
				    State_Index := 0;
				    Ai_Index := 0;
				    
				    Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				    Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				 when 10 | 13 =>
				    
				    if Refresh then
				       Wins_Draw(Wins_Main);
				       Refresh := False;
				    end if;
				    State_Index := 0;
				    Ai_Index := 0;
				    
				    
				    
				    if Center.Expr_Index > 0 then
				       
				       begin
					  
					  
					  Update(Center.Goal_T.all, Center.Prompt, Center.Expr);
					  
					  Right_Exec(Center.Goal_T.all, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
					  
					  Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
					  
					  Center.Expr := (others => Wide_Character'Val(32));
					  Center.Expr_Index := 0;				   
					  Center.Goal_T.Page_Index := 0;
					  Center.Ai.Page_Index := 0;
					  
					  if Expr_vectors.Last_Element(Center.Goal_T.Expr_List)(1..6) = "create" then
					     
					     
					     
					     -----------------------------------------------------------------------------
					     --               Organizer Goal path search by Astar path finding          --
					     -----------------------------------------------------------------------------
					     --
					     --
					     Center.Goal := Abstracted_Vectors.First_Element(Center.Goal_T.Obj.Vector);
					     if Center.T.Obj_Cur /= null then
						Center.Start_node := Center.T.Obj_Cur;
					     else
						Center.Start_node := Center.T.Obj;
					     end if;
		     
					     Abstracted_Vectors.Clear (Center.Goal_Path);
					     Center.Astar_Proc.Search_Path(Center.Start_Node, Center.Goal);
					     --
					     --
					     -----------------------------------------------------------------------------
					     
					     
					     
					     Center.Object_Is_Goal := False;
					     
					     
					     
					     Refresh := True;
					  end if;
				       exception
					  
					  when Is_word =>
					     
					     
					     Wide_Free(Output_line);
					     
					     
					     
					     
					     Input := (others => 0.0);			      
					     --Input := String2Input(UTF_Encoding.Wide_Strings.Encode(Input_Line.all));   
					     Input := String2Input(Input_Line.all, Center.Words);
					     
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     
					     Saved := Output;
					     
					     
					     --Deamon_network.Initialize(Input_Type(saved));
					     
					     
					     Input := (others => 0.0);			      
					     Output_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     
					     Input := Neural_Retrain.Input_Type(Output);
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     Feedback := (others => 0.0);
					     Feedback := Output;
					     
					     
					     
					     Internal_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     Output := (others => 0.0);
					     
					     
					     
					     
					     
					     Center.Expr := (others => Wide_Character'Val(32));
					     Center.Expr_Index := 0;
					     Refresh := True;
					     
					     Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					     
					     
					  when others =>
					     Draw(Wins_Main.Main_Win_Ptr.All, 1, (Columns-2)/2-12, white, "Error : cannot exec cmd");
					     State_Index := 0;
					     Ai_Index := 0;
					     
					     Refresh := True;
					     
					     
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
					     
					     
					     raise;
					     
					     
				       end;
				       
				    elsif Center.Goal_T.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				       
				       if Center.Ai.Obj_Cur /= null then
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
				       end if;
				    elsif Center.Ai.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				       
				    end if;			   
				    
				    Wide_Free(Input_line);
				    Input_Line := new Wide_String ' ("");				 				 				
				    
				    
				    
				    Classes.Windows.Draw_Window (Wins_Main.Output_Win_Ptr.all);
				    
				    if Output_Line'Length > 0 then
				       if Output_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line(1..Output_Line'length));
				       else
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line.all(1..Positive'Min(Output_Line'Length,
												Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 2, 1, white,
								Output_Line.all((Columns/2-4)+1 .. Positive'Min(output_Line'Length, Columns/2-4+Columns/2-4)));
				       end if;
				    end if;
				    Wide_Free(Output_Line);
				    Output_Line := new Wide_String ' ("");
				    Classes.Windows.Draw_Window (Wins_Main.Internal_Win_Ptr.all);            
				    if Internal_Line'Length > 0 then
				       
				       if Internal_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line(1..Internal_Line'length));

				       else
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line.all(1..Positive'Min(Internal_Line'length,
												  Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 2, 1, white,
								Internal_Line.all((Columns/2-4)+1 .. Positive'Min(Internal_Line'Length, Columns/2-4+Columns/2-4)));

				       end if;
				       
				    end if;
				    Wide_Free(Internal_Line);
				    Internal_Line := new Wide_String ' ("");
				    
				    
				    Classes.Windows.Draw_Window(Wins_Main.Input_Win_Ptr.all);				 
				    
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, ": ");
				    
				    
				    
				    
				    
				 when others =>
				    null;		     
			      end case;
			   when others =>
			      
			      case Wchar_Num is
				 
				 when 0..31 =>
				    null;
				    
				 when 32 =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);
				    end if;
				 when 33 .. 126 =>
				    if Center.Expr_Index < (Columns-26)  then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				 when 127 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) >= 1 then
				       Input_Buffer := new Wide_String ' (Input_Line.all(1..Input_Line'Length-1));
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    end if;
				 when others =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    
			      end case;
			      
			      if not Center.Object_Is_Goal then
				 Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
				 Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
				 Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
				 Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
			      else
				 Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
				 Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Goal object");
				 Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
				 Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
			      end if;
			      
			end case;
			
			if not Center.Object_Is_Goal then
			   Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
			   Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
			   Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
			   Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
			else
			   Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
			   Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Goal object");
			   Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
			   Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
			end if;
		     exception
			when others =>
			   null;
			   
		     end;
		     Screen_Lock.Release;
		  end Receive;
		  
		  
		  
	    or
	       
	       when (Center.Object_Is_Goal = False) and (Center.View_Is_Goal_Path = False) =>
		  
		  accept Receive (Wchar : in Wide_Character) do
		     Screen_Lock.Seize;
		     End_Of_This_Task := False;
		     begin
			WChar_Num := Wide_Character'Pos(Wchar);		     
			case WChar_Num is
			   when 0 .. 31 =>
			      
			      case WChar_Num is
				 when 1 => -- Ctrl+A
				    
				    Save(Center.T);
				    
				    Save(Center.Ai);	
				    
				    
				    Save(Center.T.Obj, Center.User_Filename.all);
				    
				    Save(Center.Ai.Obj, Center.Ai_Filename.all);
				    
				    
				 when 14 => -- ctrl+N
				    
				    Center.T.Page_Index := Center.T.Page_Index + 1;
				    Center.Ai.Page_Index := Center.Ai.Page_Index + 1;
				    Right_Exec(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				    Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    
				 when 16 => -- Ctrl+P 
				    if Center.T.Page_Index > 0 then
				       Center.T.Page_Index := Center.T.Page_Index - 1;
				       Center.Ai.Page_Index := Center.Ai.Page_Index - 1;
				       Right_Exec(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
				       Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
				    end if;

				    
				 when 8 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    end if;
				 when 9 =>
				    null;
				 when 11 =>
				    null;
				 when 27 =>
				    null;
				 when 12 => -- Ctrl+L;
				    Wins_Draw(Wins_Main);
				    
				    State_Index := 0;
				    Ai_Index := 0;
				    
				    Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				    Global_Content_Print(Center.T.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				 when 10 | 13 =>
				    if Refresh then
				       Wins_Draw(Wins_Main);
				       Refresh := False;
				    end if;
				    State_Index := 0;
				    Ai_Index := 0;

				    
				    if Center.Expr_Index > 0 then
				       
				       Update(Center.T, Center.Prompt, Center.Expr);
				       
				       Update(Center.Ai, Center.Prompt, Center.Expr);
				       begin
					  
					  Right_Exec(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
					  
					  Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
					  
					  
					  Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
					  
					  Center.Expr_List := Center.Expr_List & Center.Expr;	 
					  Center.Prompt_List := Center.Prompt_List & Center.Prompt;
					  Center.Expr := (others => Wide_Character'Val(32));
					  Center.Expr_Index := 0;				   
					  Center.T.Page_Index := 0;
					  Center.Ai.Page_Index := 0;
					  
					  
				       exception
					  
					  when Is_word =>
					     
					     
					     Wide_Free(Output_line);
					     
					     
					     
					     
					     Input := (others => 0.0);			      
					     --Input := String2Input(UTF_Encoding.Wide_Strings.Encode(Input_Line.all));   
					     Input := String2Input(Input_Line.all, Center.Words);
					     
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     
					     Saved := Output;
					     
					     
					     --Deamon_network.Initialize(Input_Type(saved));
					     
					     
					     Input := (others => 0.0);			      
					     Output_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     
					     Input := Neural_Retrain.Input_Type(Output);
					     
					     
					     declare
						Request : Neural_Retrain.Request_Nn;
					     begin
						
						Request.Respond(Input, Output);
						
						abort Request;
						
					     end;
					     
					     Feedback := (others => 0.0);
					     Feedback := Output;
					     
					     
					     
					     Internal_Line := new Wide_String ' (Output2String(Output, Center.Words));
					     
					     Output := (others => 0.0);
					     
					     
					     
					     
					     
					     Center.Expr := (others => Wide_Character'Val(32));
					     Center.Expr_Index := 0;
					     Refresh := True;
					     
					     Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
					     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					     
					     
					  when others =>
					     State_Index := 0;
					     Ai_Index := 0;
					     
					     Refresh := True;
					     Draw(Wins_Main.Main_Win_Ptr.All, 1, (Columns-2)/2-12, white, "Error : cannot exec cmd");
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
					     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
					     
					     raise;
					     
					     
				       end;

				    elsif Center.T.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
				       
				       if Center.Ai.Obj_Cur /= null then
					  
					  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
					  
				       end if;
				    elsif Center.Ai.Obj_Cur /= null then
				       
				       Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
				       
				    end if;			   
				    
				    Wide_Free(Input_line);
				    Input_Line := new Wide_String ' ("");
				    

				    
				    Classes.Windows.Draw_Window (Wins_Main.Output_Win_Ptr.all);
				    
				    if Output_Line'Length > 0 then
				       if Output_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line(1..Output_Line'length));
				       else
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 1, 1, white,
								Output_Line.all(1..Positive'Min(Output_Line'Length,
												Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Output_Win_Ptr.all, 2, 1, white,
								Output_Line.all((Columns/2-4)+1 .. Positive'Min(output_Line'Length, Columns/2-4+Columns/2-4)));
				       end if;
				    end if;
				    Wide_Free(Output_Line);
				    Output_Line := new Wide_String ' ("");
				    Classes.Windows.Draw_Window (Wins_Main.Internal_Win_Ptr.all);            
				    if Internal_Line'Length > 0 then
				       
				       if Internal_Line'length <= (Columns/2-4) then
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line(1..Internal_Line'length));

				       else
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 1, 1, white,
								Internal_Line.all(1..Positive'Min(Internal_Line'length,
												  Columns/2-4)));
					  Classes.Windows.Draw (Wins_Main.Internal_Win_Ptr.all, 2, 1, white,
								Internal_Line.all((Columns/2-4)+1 .. Positive'Min(Internal_Line'Length, Columns/2-4+Columns/2-4)));

				       end if;
				       
				    end if;
				    Wide_Free(Internal_Line);
				    Internal_Line := new Wide_String ' ("");
				    Classes.Windows.Draw_Window(Wins_Main.Input_Win_Ptr.all);				 
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
				    Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, ": ");
				    
				    
				    
				 when others =>
				    null;
			      end case;
			   when others =>
			      case Wchar_Num is
				 
				 when 0..31 =>
				    null;
				    
				 when 32 =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);
				    end if;
				 when 33 .. 126 =>
				    if Center.Expr_Index < (Columns-26)  then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				 when 127 =>
				    if Center.Expr_Index > 0 then
				       Center.Expr(Center.Expr_Index) := Wide_Character'Val(32);
				       Center.Expr_Index := Center.Expr_Index - 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) >= 1 then
				       Input_Buffer := new Wide_String ' (Input_Line.all(1..Input_Line'Length-1));
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    end if;
				 when others =>
				    if Center.Expr_Index < (Columns-26) then
				       Center.Expr(Center.Expr_Index+1) := Wchar;
				       Center.Expr_Index := Center.Expr_Index + 1;
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    if Real_Length(Input_Line.all) < (Columns-26) then
				       Input_Buffer := new Wide_String ' (Input_Line.all & Wchar);
				       Wide_Free(Input_Line);
				       Input_Line := new Wide_String ' (Input_Buffer.all);		  
				    else
				       W_Io.Put(Wide_Character'Val(7));
				    end if;
				    
			      end case;
			      Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
			      Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
			      Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
			      
			end case;
			Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, " : ");
			Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
		     exception
			when others =>
			   null;
		     end;
		     Screen_Lock.Release;
		  end Receive;
	    or
	       when Center.Object_Is_Goal = False =>
		  accept New_Goal do
		     
		     
		     Abstracted_Vectors.Clear(Center.Goal_Path);
		     
		     Center.View_Is_Goal_Path := False;
		     Center.Object_Is_Goal := True;
		     
		     Terminal_Free(Center.Goal_T);
		     Center.Goal_T := new Terminal_Record;
		     
		     State_Index := 0;
		     Ai_Index := 0;
		     
		     Center.Goal_T.Obj := Id(universe);
		     
		     Center.Goal_T.Obj_Cur := Center.Goal_T.Obj;
		     
		     Wide_Fixed.Move("Universe", Center.Goal_T.Obj.Name);
		     
		     Screen_Lock.Seize;
		     Wins_Draw(Wins_Main);
		     
		     Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		     
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		     
		     Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Goal object");
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
		     Screen_Lock.Release;
		  end New_Goal;
	    or
	       when Center.Object_Is_Goal = True =>
		  accept New_Goal do
		     Center.Object_Is_Goal := False;
		     Center.View_Is_Goal_Path := False;
		     
		     State_Index := 0;
		     Ai_Index := 0;
		     
		     Screen_Lock.Seize;
		     Wins_Draw(Wins_Main);
		     
		     Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		     
		     Classes.Windows.Draw_Window(Wins_Main.Input_Win_Ptr.all);				 
		     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
		     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, ": ");
		     Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+4, white, Center.Expr(1..Center.Expr_index));
		     Screen_Lock.Release;
		  end New_Goal;
	    or
	       when Center.View_Is_Goal_Path = false =>
		  accept Goal_Path_Tree_View do
		     
		     
		     
		     Center.View_Is_Goal_Path := True;
		     
		     if not Abstracted_Vectors.Is_Empty (Center.Goal_Path) then
			
			
			
			if Center.Goal_T.Obj = null then			
			   Center.Goal_T.Obj := Abstracted_Vectors.First_Element(Center.Goal_Path);
			   
			   
			   if Abstracted_Vectors.Length(Center.Goal_Path) > 1 then
			      for Object in 2..Abstracted_Vectors.Length(Center.Goal_Path) loop
				 Abstracted_Vectors.Append(Center.Goal_T.Obj.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			      end loop;
			   end if;
			else
			   for Object in 1..Abstracted_Vectors.Length(Center.Goal_Path) loop
			      Abstracted_Vectors.Append(Center.Goal_T.Obj.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			   end loop;
			end if;
		     end if;
		     Center.Goal_T.Obj_Cur := Center.Goal_T.Obj;
		     Screen_Lock.Seize;
		     Wins_Draw(Wins_Main);
		     State_Index := 0;
		     Ai_Index := 0;
		     Global_Content_Print(Center.Goal_T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		     Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		     Screen_Lock.Release;
		  end Goal_Path_Tree_View;
	    or
	       when Center.View_Is_Goal_Path = True =>
		  accept Goal_Path_Tree_View do

		     
		     Center.View_Is_Goal_Path := False;
		     State_Index := 0;
		     Ai_Index := 0;
		     
		     Screen_Lock.Seize;
		     Wins_Draw(Wins_Main);
		     
		     Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		     
		     Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		     
		     Classes.Windows.Draw_Window (Wins_Main.Input_Win_Ptr.all);
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, "Subject");
		     Classes.Windows.Draw (Wins_Main.Input_Win_Ptr.all, 1, 17, white, " : ");
		     Screen_Lock.Release;
		     
		  end Goal_Path_Tree_View;
	    or
	       
	       accept Add_Goal_Path do		  
		  if not Abstracted_Vectors.Is_Empty (Center.Goal_Path) then 
		     if Center.Start_Node /= null then
			for Object in 1..Abstracted_Vectors.Length(Center.Goal_Path) loop			
			   Abstracted_Vectors.Append(Center.T.Obj_Cur.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			end loop;
		     elsif Center.T.Obj_Cur /= null then
			for Object in 1..Abstracted_Vectors.Length(Center.Goal_Path) loop			
			   Abstracted_Vectors.Append(Center.T.Obj_Cur.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			end loop;
		     elsif Center.T.Obj /= null then
			for Object in 1..Abstracted_Vectors.Length(Center.Goal_Path) loop			
			   Abstracted_Vectors.Append(Center.T.Obj.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			end loop;
			Center.T.Obj_Cur := Center.T.Obj;
		     else
			Center.T.Obj := Abstracted_Vectors.First_Element(Center.Goal_Path);
			if Abstracted_Vectors.Length(Center.Goal_Path) > 1 then
			      for Object in 2..Abstracted_Vectors.Length(Center.Goal_Path) loop
				 Abstracted_Vectors.Append(Center.T.Obj_Cur.vector, Abstracted_Vectors.Element(Center.Goal_Path, Integer(Object)));
			      end loop;
			end if;
			Center.T.Obj_Cur := Center.T.Obj;
		     end if;
		  end if;
		  
		  State_Index := 0;
		  Ai_Index := 0;
		  Screen_Lock.Seize;
		  Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
		  Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
		  Classes.Windows.Draw_Window(Wins_Main.Input_Win_Ptr.all);				 
		  Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, 1, Magenta, Center.Prompt);
		  Classes.Windows.Draw(Wins_Main.Input_Win_Ptr.all, 1, Center.Prompt'Length+1, white, ": ");
		  Screen_Lock.Release;
	       end Add_Goal_Path;
	    or
	       
	       accept Reset;
	    or
	       accept Lock do
		  Oclock.Stop;
		  Suspended := True;
	       end Lock;
	       exit;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean);      
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow;
	    or
	       accept Down_Arrow;
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Full_Screen;
	    or
	       accept Del;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Overwrite;
	    or
	       accept Screen_Print;
	    or
	       accept Shutdown;
	    or
	       delay 1.0;
	       Oclock.Fresh;
	    end select;
	 end loop;
      end loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Organizer halted");
      Center.Global_Lock.Release;
   end Organizer_Process;

   
   
   
   
   
   
   procedure Draw_Buffer (Spawn_Result : in Wide_Result_Access; Win : in Window_Record; File_Index : in Natural) is
      Line_Index : Natural := 0;
      Result_Max_Lines : Positive := Win.Win.H;
      Line_Count : Natural := 0;              
      Term_Width : constant Natural := Win.Win.W-2;
   begin
      
      
      if Spawn_Result /= null and then Spawn_result.Wlines /= null then                        

     Print_Loop:
	 
	 for I in File_Index+1..Positive'min(File_Index+((Result_Max_Lines-5)), 
					     Spawn_Result.Wlines'Length) loop
	    declare
	       tmp : constant Wide_String := 
		 (-Spawn_Result.Wlines(I));
	       Line : constant Wide_String := Tmp(Tmp'First..Wide_Fixed.Index_Non_Blank((Tmp), Backward));
	       
	       Chars_Max : constant Positive := Term_Width;
	       Lines_Max : constant Positive := (Result_Max_Lines-5);
	    begin	       
	       if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
		  for Line_Window in Line_Index..Integer'min(Real_Length(Line)/Chars_Max, Lines_Max) loop
		     
		     declare
			B_Inf : constant Positive := Line_Window*Chars_Max + 1;
			B_Sup : constant Positive := Integer'min((Line_Window + 1)*Chars_Max, Real_Length(Line));
		     begin
			if Real_length(Line) < Chars_max then
			   Draw_Text(Win, Line_Count+1, 1, White, Line);
			else
			   Draw_Text(Win, Line_Count+1, 1, White, Line(B_Inf..B_Sup));                            
			end if;                 
		     end;
		     Line_Count := Line_Count + 1;
		     exit Print_Loop When Line_Count > (Result_Max_Lines-1);
		  end loop;
	       else
		  Line_Count := Line_Count + 1;
	       end if;                           
	    end;
	    exit Print_Loop when Line_Count > (Result_Max_Lines-1);
	 end loop Print_Loop;

      end if;
      

   end Draw_Buffer;
   
   
   
   
   
   task body Multi_Process is
      End_Of_Task : Boolean := False;
      Suspended : Boolean := False;
      
      
   begin
      while not End_Of_Task Loop
	 
	 loop
	    select
	       accept Halt do
		  End_Of_Task := True;
		  Center.Command_Proc.Halt;
	       end Halt;
	       
	       exit;
	    or
	       accept Initialize do
		  Center.Command_Proc.Initialize;
		  Center.Prev_multi := Center.Multi_Mode;
		  Center.Multi_Mode := Command_Mode;
		  
	       end Initialize;
	       exit;
	    or
	       accept Lock;
	    or
	       accept Wake_Up do
		  Suspended := False;
		  Center.Command_Proc.Wake_Up;
		  
	       end Wake_Up;
	       exit;
	    end select;
	 end loop;
	 while not End_Of_Task  loop
	    select
	       accept Wake_Up do
		  Suspended := False;
		  Center.Command_Proc.Wake_Up;
		  
	       end Wake_Up;
	    or
	       accept Halt do
		  Center.Command_Proc.Halt;
	       end Halt;
	       exit;
	    or
	       
	       accept Initialize do
		  Center.Command_Proc.Initialize;
		  Center.Prev_multi := Center.Multi_Mode;
		  Center.Multi_Mode := Command_Mode;
	       end Initialize;
	       
	    or
	       when Center.Multi_Mode = Account_Mode =>
		  accept Receive (Wchar : in Wide_Character) do

		     null;

		  end Receive;

	       
	    or
	       when Center.Multi_Mode = Edit_Mode =>
		  accept Receive (Wchar : in Wide_Character) do

		     null;

		  end Receive;

	    or
	       
	       when Center.Multi_Mode = Command_Mode =>
		  accept Receive (Wchar : in Wide_Character) do

		     Center.Command_Proc.Receive(Wchar);

		  end Receive;
	    or
	       accept Mode(Mode : out Multi_Enum) do
		  Mode := Center.Multi_Mode;
	       end Mode;
	    or
	       accept Switch (Mode : in Multi_Enum) do
		  Center.Multi_Mode := Mode;	       
	       end Switch;
	    or
	       accept Reset;
	    or
	       accept Lock do
		  Suspended := True;
	       end Lock;
	       exit;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean) do
		  case Center.Multi_Mode is
		     when Command_Mode =>
			Center.Command_Proc.Set_On_Esc(Center.On_Esc);
			--when Edit_Mode =>
			--Center.Edit_Proc.Set_On_Esc(Center.On_Esc);
		     when others =>
			null;
		  end case;
	       end Set_On_Esc;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Page_Down do
		     Center.Command_Proc.Page_down;
		  end Page_Down;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Page_up do
		     Center.Command_Proc.Page_up;
		  end Page_up;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Up_arrow do
		     Center.Command_Proc.Up_arrow;
		  end up_Arrow;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Down_arrow do
		     Center.Command_Proc.Down_arrow;
		  end Down_arrow;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Right_arrow do
		     Center.Command_Proc.Right_arrow;
		  end Right_arrow;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Left_Arrow do
		     Center.Command_Proc.Left_arrow;
		  end Left_Arrow;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Begin_of do
		     Center.Command_Proc.Begin_of;
		  end Begin_of;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept End_of do
		     Center.Command_Proc.End_of;
		  end End_of;
	    or
	       when Center.Multi_Mode = Command_Mode =>
		  accept Full_screen do
		     Center.Command_Proc.Full_screen;
		  end Full_screen;
	    or
	       accept Del;
	    or
	       accept Overwrite;
	    or
	       accept Screen_Print;
	    or
	       accept Shutdown;
	    end select;
	 end loop;
      end loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Multi halted");
      Center.Global_Lock.Release;
   end Multi_Process;

   ------------------------------------------------------------------------------------------
   --                                      Shell mode                                      --
   ------------------------------------------------------------------------------------------
   
   task body Shell_Process is
      Suspended : Boolean := False;
      End_Of_This_Task : Boolean := False;
      
      Left : Window_Type := (1, 1, Lines, Cols, Single_Line_Frame, Cyan, Yellow);
      Right : Window_Type := (1, Cols/2+1, Lines, Cols/2, Single_Line_Frame, blue, White);
      
      File_Content : U_Array_Access;
      Current_Dialup : Dialup_Enum;
      
      Line_Count : Natural := 0;
      File_Index : Natural := 0;
      Result_Max_Lines : Positive := Lines-1;
      Term_Width : Positive := Cols-2;
      Line_Index : Natural := 0;
      Shell_Prompt_Line : Positive := 1;
      Command_Max_Lines : Positive := 3;
      Line : Wide_String_Access;
      
      
      
      
      Completion_Result : Wide_Result_Access;      
            
      
      History_Set   : Wide_String_Set(2048);
      History_Last  : Natural := 0;
      History_Index : Natural := 1;
      
      Command : Gnat.Command_Line.Command_Line;      
      Args_List : Os_Lib.Argument_List_Access;                  
      
      Success       : Boolean := False;
      
      Cmd_Line         : Wide_String_Access := new Wide_String ' ("");
      Cmd_Line_Length  : Natural := 0;
      Cmd_Current_Pos      : Natural := 1;      
      Left_Buffer      : Wide_String_Access;
      Wbuffer          : Wide_String_Access;
      Cmdbuffer          : Wide_String_Access;
      
      Is_Internal_Cmd : Boolean := False;
      Internal_Cmd    : Internal_Cmd_Enum := Null_Cmd;
      Aliazed         : Wide_String(1..256) := (others => Wide_Character'Val(32));
      
      Wchar_Index     : Natural := 0;
      
      Is_Overwrite    : Boolean := False;
      
      Result_Left    : constant Natural := ((Lines - 17) / 8) * 3;
      Width          : constant Column_Range := Cols;
      chars_Max      : constant Positive := Width;
   begin
      while not End_Of_This_Task loop
	 loop
	    select
	       accept Initialize (Dialup : Dialup_Enum) do	       
		  Center.Shells := new Shells_Record ' (New_Shells (Dialup));
		  case Dialup is
		     when Sys =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).System_Dialup.Wlines;
		     when Intern =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Intern_Dialup.Wlines;
		     when Neura =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Neural_Dialup.Wlines;
		     when Eng =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Engine_Dialup.Wlines;
		     when web =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Search_web.Wlines;
		  end case;	       
		  Current_Dialup := Sys;
	       end Initialize;
	       exit;
	    or
	       accept Halt do
		  End_Of_This_Task := True;
	       end Halt;
	       exit;
	    end select;
	 end loop;

	 
	 ----------------------------------------
   	 -- Printing GnuShell Spawn buffer.
	 Line_Count := 0;
	 
	 
   	 if File_Content /= null then
	    
   	GnuShell_init:
   	    for I in File_Index+1..Positive'min(File_Index+((Result_Max_Lines-2)), 
   						File_Content'Length) loop
   	       declare
   		  tmp : constant Wide_String := (-File_Content(I));
   		  Line : constant Wide_String := Tmp(Tmp'First..Index_Non_Blank((Tmp), Backward));
		  
   		  Chars_Max : constant Positive := Term_Width-5;
   		  Lines_Max : constant Positive := (Result_Max_Lines-2);
   	       begin
   		  if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
		     
   		     for Line_Window in Line_Index..Integer'min(Real_Length((Line))/Chars_Max, Lines_Max) loop
			
   			declare
   			   B_Inf : constant Positive := Line_Window*Chars_Max + 1;
   			   B_Sup : constant Positive := Integer'min((Line_Window + 1)*Chars_Max, Real_Length((Line)));
   			begin
   			   if Real_length((Line)) < Chars_max then
			      
			      
			      
			      
   			      Draw (left, Line_Count + 1, 2,  White, (Line));
			      
   			   else
			      
   			      Draw 
   				(Left, 
   				 Line_count + 1,
   				 2, 
   				 White, 
   				 (Line(B_inf..B_sup))
   				);						      
			      
   			   end if;                 
   			end;
   			Line_Count := Line_Count + 1;
   			exit GnuShell_init When Line_Count > (Result_Max_lines - 6 - Integer'min(Real_Length(Line)/Chars_Max, Lines_Max));
   		     end loop;
   		  else
   		     Line_Count := Line_Count + 1;
   		  end if;                           
   		  exit GnuShell_init when Line_Count > (Result_Max_lines - 6 - Integer'min(Real_Length(Line)/Chars_Max, Lines_Max));
   	       end;
	       
   	    end loop GnuShell_init;
	    
   	 end if;
   	 Move_Cursor_To(left, 1, 2);

	 
   	 -------------------------------------------
   	 -- Printing the user command line.
   	 --
	 
   	 if File_Content /= null then
	    
   	    Shell_Prompt_Line := Integer'Min (Line_Count + 1,
   					      (Command_Max_Lines - 5))+1;
   	 end if;
   	 if Line /= null then
   	    declare
	       
   	       Command_Line : constant Wide_String := Line.all;
   	       Local_Line_Count : Natural := 0;
	       
   	       Total_Lines_Of_Command : constant Natural := Real_Length(Command_Line)/Chars_Max;
	       
   	    begin
	       
   	       if Real_Length(Command_Line) > 0 then
   		  for Line_Window in 0..Integer'min(Real_Length(Command_Line)/Chars_Max, Lines_Max) loop
		     
		     
   		     if Local_Line_Count + (Local_Line_Count+Line_Window+2) >= (Command_Max_Lines - 4)  then
   			Shell_Prompt_Line := (Command_Max_Lines - Total_Lines_Of_Command) - (Local_Line_Count + 4);
   		     end if;
		     

		     
   		     declare
   			B_Inf : constant Positive := Line_Window*Chars_Max+1;
   			B_Sup : constant Positive := Integer'min((Line_Window+1)*Chars_max, Command_Line'last);
   		     begin
			
   			if Command_Line'last <= Chars_Max then					     
   			   Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# " & (Command_Line));
   			   Move_Cursor_To(Left, Shell_Prompt_Line, Command_Line'Last+4);

   			elsif Line_Window = 0 then

   			   Draw 
   			     (Left, 
   			      Shell_Prompt_Line, 
   			      2, 
   			      White, 
   			      "# " & (Command_Line(B_inf..B_sup))
   			     );
   			   Move_Cursor_To(Left, Shell_Prompt_Line, B_Sup+4);                                          
   			   Local_Line_Count := Local_Line_Count + 1;

   			else

   			   Draw 
   			     (Left, 
   			      Shell_Prompt_Line + (Line_Window) + 1 + (Local_Line_Count - 1),
   			      2, 
   			      White, 
   			      (Command_Line(B_inf..B_sup))
   			     );
   			   Move_Cursor_To(Left, Shell_Prompt_Line + (Line_Window) + 1 + (Local_Line_Count -1), (B_Sup-B_Inf)+3);
   			   Local_Line_Count := Local_Line_Count + 1;

   			end if;                                       
   		     end;
		     
   		     exit when Local_Line_Count > (Lines_Max-1);
		     
   		  end loop;
   	       else

   		  Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# ");
   		  Move_Cursor_To(Left, Shell_Prompt_Line, 4);

   	       end if;                              
   	    end;                        
   	 else

   	    Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# ");
   	    Move_Cursor_To(Left, Shell_Prompt_Line, 4);

   	 end if;			   
	 
	 while not End_Of_This_Task Loop
	    select
	       accept Wake_Up;
	    or
	       accept Halt do
		  End_Of_This_Task := True;
	       end Halt;
	       exit;
	    or
	       accept Initialize (Dialup : Dialup_Enum) do	       
		  
		  case Dialup is
		     when Sys =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).System_Dialup.Wlines;
		     when Intern =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Intern_Dialup.Wlines;
		     when Neura =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Neural_Dialup.Wlines;
		     when Eng =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Engine_Dialup.Wlines;
		     when web =>
			Shell_Process.File_content := Center.Shells.Shells_Set(Dialup).Search_web.Wlines;
		  end case;
		  
		  Current_Dialup := Dialup;
		  Draw_Window(Left);
	       end Initialize;
	    or
	       accept Receive (Wchar : in Wide_Character) do
		  
		  
		  
		  
		  --  if Is_control(Wchar) then	    
		  --     case Wchar is
			
		  --  	when Wide_Character'Val(9) =>
			   
		  --  	   if Cmd_Line /= null then
		  --  	      -- Path completion...
		  --  	      declare
		  --  		 Line : Wide_String_Access := 
		  --  		   new Wide_String ' (Cmd_Line.all);
				 
		  --  		 Path_On_Line : Wide_String_Access;
		  --  		 buffer : Wide_string_Access := new Wide_String ' ("");
		  --  	      begin
		  --  		 if Completion_Result /= null then
		  --  		    U_Array_Free(Completion_Result.Wlines);
		  --  		 end if;
		  --  		 Completion(Handling.To_String(Line.all), Path_On_Line, Completion_Result);				    
		  --  		 if Path_On_Line /= null then
		  --  		    Buffer := new Wide_String ' (Path_On_Line.all);
		  --  		    Wide_Free(Path_On_Line);					  
		  --  		    Wide_Free(Cmd_Line);
		  --  		    Cmd_Line := new Wide_String ' (Buffer.all);
		  --  		    Wide_Free(Buffer);
		  --  		    Cmd_Line_Length := Cmd_Line'Length;
		  --  		    Wchar_Index := Cmd_Line'Length;
		  --  		    Cmd_Current_Pos := Wchar_Index + 1;
		  --  		    U_Array_Free(Completion_Result.Wlines);
		  --  		    Wide_Result_Free(Completion_Result);
		  --  		 end if;
				 
		  --  	      end;
		  --  	   end if;                        			      
			   
		  --  	when Wide_Character'Val(10) =>
		  --  	   --W_Io.Put_Line("TUTU 1");
		  --  	   Draw_Window(Center.Frames(Err).Win);
		  --  	   Draw_Window(Center.Frames(Cmd).Win);
		  --  	   --W_Io.Put_Line("TUTU 2");
		  --  	   if Cmd_Line /= null then
		  --  	      ------------------------------------------------------
		  --  	      --                        Internal                  --
		  --  	      ------------------------------------------------------
		  --  	      --W_Io.Put_Line("TUTU 3");
		  --  	      History_Index := 1;
			      
		  --  	      Internal_Cmd := Internal_Cmd_Value(Cmd_Line.all);
		  --  	      Is_Internal_Cmd := True;
		  --  	      if Center.Shell_Result = null then
		  --  		 Center.Shell_Result := new Spawn_Result_Record;
		  --  		 Center.Shell_Result.Wlines := new U_Array (1..0);
		  --  	      end if;
		  --  	      case Internal_Cmd is
		  --  		 when Null_Cmd =>
		  --  		    Is_Internal_Cmd := False;
				    
		  --  		 when Clear =>
				    
		  --  		    case Os_Name is
		  --  		       when System.Os_Constants.Windows =>
		  --  			  Success := 
		  --  			    Clear(Center.User_Path.all & Dir_Separator & ".Out_File.txt");
		  --  		       when System.Os_Constants.Other_Os =>
		  --  			  Success := 
		  --  			    Clear(Center.User_Path.all & Dir_Separator & ".Out_File.txt");
		  --  		    end case;
		  --  		    if not Success then
		  --  		       Center.Shell_Result.Wlines := 
		  --  			 Add_Line(Center.Shell_Result.Wlines, 
		  --  				  Handling.To_Wide_String
		  --  				    ("An error occured when attempt of """ & 
		  --  				       Center.User_Path.all & Dir_Separator & ".Out_File.txt"" deletion."));
		  --  		    else
		  --  		       Draw_Window(Center.Frames(Cmd).Win);
		  --  		       Draw_Window(Center.Frames(Cmd_Result).Win);
		  --  		       U_Array_Free(Center.Shell_Result.Wlines);
		  --  		       Center.Shell_Result.Wlines := new U_Array (1..0);
		  --  		    end if;
		  --  		    W_Io.Put_Line("TUTU 4");
		  --  		 when Alias =>
				    
		  --  		    Alias(Cmd_Line.all, Center.User.Alias_Set, Spawn_Result_Record(Center.Shell_Result.all));
				    
		  --  		 when Unalias =>
				    
		  --  		    Unalias(Cmd_Line.all, Center.User.Alias_Set, Spawn_Result_Record(Center.Shell_Result.all));

		  --  		 when Cd =>
				    
		  --  		    Change_Directory(Cmd_Line.all, Spawn_Result_Record(Center.Shell_Result.all));
		  --  		 when Pwd =>

		  --  		    Center.Shell_Result.Wlines := 
		  --  		      Add_Line(Center.Shell_Result.Wlines, Handling.To_Wide_String(String(Get_Current_Dir)));
		  --  		 when Help =>

		  --  		    Internal_Help(Cmd_Line.all, Spawn_Result_Record(Center.Shell_Result.all));
		  --  		 when Logout =>
				    
		  --  		    Center.Logout := True;
		  --  		 when others =>
		  --  		    null;				    
		  --  	      end case;
		  --  	      --W_Io.Put_Line("TUTU 5");
		  --  	      case Internal_Cmd is
		  --  		 when Null_Cmd =>
		  --  		    Center.Shell_Result.Return_code := 127;
		  --  		 when others =>

		  --  		    Add_Line(History_Set, History_Last, Cmd_Line.all);

		  --  		    Wide_Free(Cmd_Line);
		  --  		    Cmd_Line_Length := 0;				    
		  --  		    Cmd_Line := new Wide_String ' ("");				    
		  --  	      end case;
		  --  	      --W_Io.Skip_Line;
		  --  	      --W_Io.Put_Line("TUTU 6");
			      
		  --  	      ------------------------------------------------------
		  --  	      --                   End internal                   --
		  --  	      ------------------------------------------------------
			      
		  --  	      if not Is_Internal_Cmd then
		  --  		 declare
		  --  		    Alias_Ptr   : Strings.String_Access;
		  --  		    Buffer      : Wide_String_Access;
		  --  		    Normal      : Strings.String_Access;				 
		  --  		 begin

		  --  		    for Iter in 1..Center.User.Alias_Set.Index loop
		  --  		       declare
		  --  			  Alias : constant Attribut_Record := Center.User.Alias_Set.List(Iter);
		  --  			  First_Space : Natural := 
		  --  			    Wide_Fixed.Index(Cmd_Line.all, " ");
		  --  			  Non_Blank   : Natural :=
		  --  			    Wide_Fixed.Index_Non_Blank(Cmd_Line.all, Backward);
		  --  		       begin	
		  --  			  if Non_Blank < First_Space then
		  --  			     First_Space := 0;
		  --  			  end if;
		  --  			  if First_Space /= 0 then
		  --  			     if Handling.To_Wide_String(Names(Alias)) =
		  --  			       Cmd_line(Cmd_Line'First..First_Space - 1) then					     
		  --  				Alias_Ptr := new String ' (Attrib.Image(Alias));
		  --  				exit;
		  --  			     end if;
		  --  			  else
		  --  			     if Handling.To_Wide_String(Names(Alias)) =
		  --  			       Cmd_Line(Cmd_Line'First..Index_Non_Blank(Cmd_Line.all, Backward)) then
		  --  				Alias_Ptr := new String ' (Attrib.Image(Alias));
		  --  				exit;
		  --  			     end if;
		  --  			  end if;
		  --  		       end;
		  --  		    end loop;

		  --  		    if Alias_Ptr /= null then
		  --  		       declare
		  --  			  First_Space : constant Natural := 
		  --  			    Wide_Fixed.Index(Cmd_Line.all, " ");
		  --  		       begin

		  --  			  if First_Space /= 0 then
		  --  			     Buffer := new Wide_String ' 
		  --  			       (Handling.To_Wide_String(Alias_Ptr.all) & ' ' & Cmd_Line(First_Space+1..Cmd_Line ' Last));

		  --  			  else
		  --  			     Buffer := new Wide_String ' (Handling.To_Wide_String(Alias_Ptr.all));
		  --  			  end if;
		  --  		       end;
		  --  		       Wide_Free(Cmd_Line);

		  --  		       Normal := new String ' (Normalize_Quoted_Argument(Handling.To_String(Buffer.all)));
		  --  		       Cmd_Line := new Wide_String ' (Handling.To_Wide_String(Normal.all));
		  --  		       Cmd_Line_Length := Cmd_line'Length;
		  --  		    end if;
				    
		  --  		 end;
				 
		  --  		 U_Array_Free(Center.Shell_Result.Wlines);
		  --  		 Wide_Result_Free(Center.Shell_Result);
				 
		  --  		 U_Array_Free(File_Content);
				 
		  --  		 Center.Shell_Result := new Spawn_Result_Record ' (Shell_Spawn (Handling.To_String(Cmd_Line.all), "shell_file.txt"));
				 
		  --  		 Result_Max_Lines := Result_Left;
				 
		  --  		 File_Content := Center.Shell_Result.Wlines;
				 
		  --  		 Wide_Free(Cmd_Line);
		  --  		 Cmd_Line := new Wide_String ' ("");
				 
		  --  	      end if;
		  --  	   end if;
			   
		  --  	when Wide_Character'Val(12) =>			
			   
		  --  	   Ansi_Console.Clear_Screen;				       
			   
			   
		  --  	when Wide_Character'Val(127) =>
			   
		  --  	   Draw_Window(Center.Frames(Cmd).Win);
		  --  	   if Cmd_Line /= null and then
		  --  	     Cmd_Line'Length > 0 then
		  --  	      if Cmd_Line /= null and then
		  --  		Cmd_Line'Length > 0 then
		  --  		 if Cmd_Current_Pos > 1 then
		  --  		    case Is_Overwrite is
		  --  		       when False =>
		  --  			  if Cmd_Current_Pos > 2 then
		  --  			     if Cmd_Line'Length >= 2 then
		  --  				Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-2));
		  --  				if Cmd_Current_Pos <= Cmd_Line'Length then
		  --  				   Wbuffer := new Wide_String ' (Left_Buffer.all & Cmd_Line(Cmd_Current_Pos..Cmd_Line'Last));
		  --  				else
		  --  				   Wbuffer := new Wide_String ' (Left_Buffer.all);
		  --  				end if;
		  --  			     else
		  --  				Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-2));
		  --  				Wbuffer := new Wide_String ' (Left_Buffer.all);
		  --  			     end if;
		  --  			  elsif Cmd_Current_Pos = 2 then
		  --  			     if Cmd_Line'Length >= 2 then
		  --  				Left_Buffer := new Wide_String ' (Cmd_Line(Cmd_Current_Pos..Cmd_Line'Length));
		  --  				Wbuffer := new Wide_String ' (Left_Buffer.all);
		  --  			     else
		  --  				Wbuffer := new Wide_String ' ("");
		  --  			     end if;
		  --  			  end if;				       				       
		  --  			  Wide_Free(Cmd_Line);
		  --  			  Cmd_Line := new Wide_String ' (Wbuffer.all);
		  --  			  Wide_Free(Wbuffer);					  
		  --  			  Wide_Free(Left_Buffer);
		  --  			  Wchar_Index := Wchar_Index - 1;
		  --  			  Cmd_Current_Pos := Cmd_Current_Pos - 1;				       
		  --  			  Cmd_Line_Length := Cmd_Line_Length - 1;
		  --  		       when True =>
		  --  			  Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos-1, Cmd_Current_Pos-1, " ");
		  --  			  Cmd_Current_Pos := Cmd_Current_Pos - 1;
		  --  		    end case;
		  --  		 else

		  --  		    W_Io.Put(Wide_Character'Val(7));
		  --  		 end if;			      
		  --  	      else
		  --  		 W_Io.Put(Wide_Character'Val(7));
		  --  		 -- beep...
		  --  	      end if;
			      
		  --  	      --  CmdBuffer := new Wide_String ' (Cmd_Line(Cmd_Line'First..Cmd_Line'Last-1));
		  --  	      --  Wide_Free(Cmd_Line);
		  --  	      --  Cmd_Line := new Wide_String ' (CmdBuffer.all);
		  --  	      --  Wide_Free(CmdBuffer);
		  --  	   else
		  --  	      Put(Wide_Character'Val(7));
		  --  	   end if;
			   
		  --  	when others =>
		  --  	   null;
		  --     end case;
		     
		  --     --W_Io.Put_Line("TUTU 6");
		  --     --W_Io.Skip_Line;
		     
		  --  else
		     
		     
		     
		  --     if Cmd_Current_Pos = Cmd_Line'Length + 1 then
		  --  	CmdBuffer := new Wide_String ' (Cmd_Line.all & Wchar);
		  --  	Wide_Free(Cmd_Line);
		  --  	Cmd_Line := new Wide_String ' (CmdBuffer.all);
		  --  	Wide_Free(CmdBuffer);			
		  --  	Cmd_Current_Pos := Cmd_Current_Pos + 1;
		  --  	Wchar_Index := Wchar_Index + 1;
		  --  	Cmd_Line_Length := Cmd_Line_Length + 1;
		  --     else
		  --  	if Cmd_Current_Pos > 1 then
		  --  	   if not Is_Overwrite then
		  --  	      Left_Buffer := new Wide_String ' (Cmd_Line(1..Cmd_Current_Pos-1));
		  --  	      CmdBuffer := new Wide_String ' (Left_Buffer.all & Wchar & Cmd_Line(Cmd_Current_Pos..Cmd_Line'Last));
		  --  	      Wide_Free(Cmd_Line);
		  --  	      Cmd_Line := new Wide_String ' (CmdBuffer.all);
		  --  	      Wide_Free(CmdBuffer);					  
		  --  	      Wide_Free(Left_Buffer);
		  --  	      Cmd_Line_Length := Cmd_Line_Length + 1;				 
		  --  	      Cmd_Current_Pos := Cmd_Current_Pos + 1;
		  --  	      Wchar_Index := Wchar_Index + 1;
		  --  	   else
		  --  	      Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos, Cmd_Current_Pos, Wchar & "");				       
		  --  	      Cmd_Current_Pos := Cmd_Current_Pos + 1;
		  --  	   end if;
			   
			   
		  --  	elsif not Is_Overwrite then
			   
		  --  	   CmdBuffer := new Wide_String ' (Wchar & Cmd_Line.all);
		  --  	   Wide_Free(Cmd_Line);
		  --  	   Cmd_Line := new Wide_String ' (CmdBuffer.all);
		  --  	   Wide_Free(CmdBuffer);
		  --  	   Cmd_Line_Length := Cmd_Line_Length + 1;			      
		  --  	   Cmd_Current_Pos := Cmd_Current_Pos + 1;
		  --  	   Wchar_Index := Wchar_Index + 1;
		  --  	else
		  --  	   Wide_Fixed.Replace_Slice (Cmd_Line.all, Cmd_Current_Pos, Cmd_Current_Pos, Wchar & "");				       
		  --  	   Cmd_Current_Pos := Cmd_Current_Pos + 1;
		  --  	   Wchar_Index := Wchar_Index + 1;
		  --  	end if;
		  --     end if;
		  --     --W_Io.Put_Line("TUTU 7");
		  --     --W_Io.Skip_Line;
		  --     --W_Io.Put_Line("TUTU 8");
		  --  end if;
		  --  Wide_Free(Line);
		  --  if Cmd_Line /= null and then (Cmd_Line'Length > 0) then
		  --     Line := new Wide_String ' (Cmd_Line.all);
		  --  else
		  --     Wide_Free(Line);
		  --  end if;
		  null;
		  
	       end Receive;
	    or
	       accept Mode(Mode : out Mode_Enum);
	    or
	       accept Reset;
	    or
	       accept Switch (Dialup : in Dialup_Enum);
	    or
	       accept Lock do
		  Suspended := True;
	       end Lock;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean);      
	    or
	       accept Page_Down;
	    or
	       accept Page_Up;
	    or
	       accept Up_Arrow;
	    or
	       accept Down_Arrow;
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Full_Screen;
	    or
	       accept Del;
	    or
	       accept Begin_Of;
	    or
	       accept End_Of;
	    or
	       accept Overwrite;
	    or
	       accept Screen_Print;
	    or
	       accept Shutdown;
	    end select;
	    
	    
	    ----------------------------------------
   	 -- Printing GnuShell Spawn buffer.
	 Line_Count := 0;
	 W_Io.New_Line;
	 W_Io.Put_Line("TOTO 1");
   	 if File_Content /= null then
	    W_Io.Put_Line("TOTO 2");
   	GnuShell_Loop:
   	    for I in File_Index+1..Positive'min(File_Index+((Result_Max_Lines-2)), 
   						File_Content'Length) loop
	       W_Io.Put_Line("TOTO 3");
   	       declare
   		  tmp : constant Wide_String := (-File_Content(I));
   		  Line : constant Wide_String := Tmp(Tmp'First..Index_Non_Blank((Tmp), Backward));
		  
   		  Chars_Max : constant Positive := Term_Width-5;
   		  Lines_Max : constant Positive := (Result_Max_Lines-2);
   	       begin
		  W_Io.Put_Line("TOTO 4");
   		  if Real_Length(Line) > 0 and Real_Length(Line) < (Chars_Max * Lines_Max) then
		     W_Io.Put_Line("TOTO 5");
   		     for Line_Window in Line_Index..Integer'min(Real_Length((Line))/Chars_Max, Lines_Max) loop
			
   			declare
   			   B_Inf : constant Positive := Line_Window*Chars_Max + 1;
   			   B_Sup : constant Positive := Integer'min((Line_Window + 1)*Chars_Max, Real_Length((Line)));
   			begin
   			   if Real_length((Line)) < Chars_max then
			      
			      
			      
			      
   			      Draw (left, Line_Count + 1, 2,  White, (Line));
			      
   			   else
			      
   			      Draw 
   				(Left, 
   				 Line_count + 1,
   				 2, 
   				 White, 
   				 (Line(B_inf..B_sup))
   				);						      
			      
   			   end if;                 
   			end;
   			Line_Count := Line_Count + 1;
   			exit GnuShell_Loop When Line_Count > (Result_Max_lines - 6 - Integer'min(Real_Length(Line)/Chars_Max, Lines_Max));
   		     end loop;
   		  else
   		     Line_Count := Line_Count + 1;
   		  end if;                           
   		  exit GnuShell_Loop when Line_Count > (Result_Max_lines - 6 - Integer'min(Real_Length(Line)/Chars_Max, Lines_Max));
   	       end;
	       
   	    end loop GnuShell_Loop;
	 else
	    W_Io.Put_Line("TOTO 2");
   	 end if;
   	 Move_Cursor_To(left, 1, 2);

	 
   	 -------------------------------------------
   	 -- Printing the user command line.
   	 --
	 
   	 if File_Content /= null then
	    
   	    Shell_Prompt_Line := Integer'Min (Line_Count + 1,
   					      (Command_Max_Lines - 3))+1;
   	 end if;
	 W_Io.Put_Line("TOTO 15");
   	 if Line /= null then
   	    declare
	       
   	       Command_Line : constant Wide_String := Line.all;
   	       Local_Line_Count : Natural := 0;
	       
   	       Total_Lines_Of_Command : constant Natural := Real_Length(Command_Line)/Chars_Max;
	       
   	    begin
	       
   	       if Real_Length(Command_Line) > 0 then
		  W_Io.Put_Line("TOTO 16");
   		  for Line_Window in 0..Integer'min(Real_Length(Command_Line)/Chars_Max, Lines_Max) loop
		     
		     W_Io.Put_Line("TOTO 17");
   		     
		     --Shell_Prompt_Line := (Command_Max_Lines - Total_Lines_Of_Command) - (Local_Line_Count + 3);
   		     
		     W_Io.Put_Line("TOTO 18");
		     
   		     declare
   			B_Inf : constant Positive := Line_Window*Chars_Max+1;
   			B_Sup : constant Positive := Integer'min((Line_Window+1)*Chars_max, Command_Line'last);
   		     begin
			W_Io.Put_Line("TOTO 19");
   			if Command_Line'last <= Chars_Max then					     
   			   Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# " & (Command_Line));
   			   Move_Cursor_To(Left, Shell_Prompt_Line, Command_Line'Last+4);

   			elsif Line_Window = 0 then
			   W_Io.Put_Line("TOTO 20");
   			   Draw 
   			     (Left, 
   			      Shell_Prompt_Line, 
   			      2, 
   			      White, 
   			      "# " & (Command_Line(B_inf..B_sup))
   			     );
   			   Move_Cursor_To(Left, Shell_Prompt_Line, B_Sup+4);                                          
   			   Local_Line_Count := Local_Line_Count + 1;
			   W_Io.Put_Line("TOTO 21");

   			else
			   W_Io.Put_Line("TOTO 8");
   			   Draw 
   			     (Left, 
   			      Shell_Prompt_Line + (Line_Window) + 1 + (Local_Line_Count - 1),
   			      2, 
   			      White, 
   			      (Command_Line(B_inf..B_sup))
   			     );
   			   Move_Cursor_To(Left, Shell_Prompt_Line + (Line_Window) + 1 + (Local_Line_Count -1), (B_Sup-B_Inf)+3);
   			   Local_Line_Count := Local_Line_Count + 1;
			   W_Io.Put_Line("TOTO 22");
   			end if;                                       
   		     end;
		     
   		     exit when Local_Line_Count > (Lines_Max-1);
		     
   		  end loop;
   	       else
		  W_Io.Put_Line("TOTO 23");
   		  Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# ");
   		  Move_Cursor_To(Left, Shell_Prompt_Line, 4);

   	       end if;                              
   	    end;                        
   	 else
	    W_Io.Put_Line("TOTO 24");
   	    Draw (Left, Shell_Prompt_Line, 2, Ansi_Console.White, "# ");
   	    Move_Cursor_To(Left, Shell_Prompt_Line, 4);
	    W_Io.Put_Line("TOTO 25");
   	 end if;			   

	    
	 end loop;
      end loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Shell halted");
      Center.Global_Lock.Release;
   end Shell_Process;
   
   
   --  task body Execut_Process is
   --     Suspended : Boolean := False;
   --     Emacs_Result : Spawn_Result_Record;
   --     End_Of_This_Task : Boolean := False;
      
      
   --  begin
   --     loop
   --  	 select
   --  	    accept Halt do
   --  	       End_Of_This_Task := True;	       
   --  	    end Halt;
   --  	    exit;
   --  	 or
   --  	    accept Initialize;
   --  	    exit;
   --  	 end select;
   --     end loop;
   --     while not End_Of_This_Task loop
   --  	 select
   --  	    accept Wake_Up;
   --  	 or
   --  	    accept Halt;
   --  	    exit;
   --  	 or
   --  	    accept Initialize;
   --  	 or
   --  	    accept Receive (Wchar : in Wide_Character) do
   --  	       if Is_Control(Wchar) then
   --  	       	  case Wchar is		     
   --  	       	     when Wide_Character'Val(10) | Wide_Character'Val(13) =>			
   --  	       		begin	       		   
			   
   --  	       		   Emacs_Result := Operat.Spawn("/usr/bin/emacs", ".Out_File");	       		   
   --  	       		end;
   --  	       	     when Wide_Character'Val(12) =>
   --  	       		Ansi_Console.Clear_Screen;
   --  	       	     when Wide_Character'Val(127) =>
   --  	       		null;
   --  	       	     when others =>
   --  	       		null;
   --  	       	  end case;
   --  	       elsif Is_Graphic(Wchar) then
   --  	       	 null;
   --  	       end if;
   --  	    end Receive;

   --  	 or
   --  	    accept Reset;
   --  	 or
   --  	    accept Lock do
   --  	       Suspended := True;
   --  	    end Lock;
   --  	 or
   --  	    accept Unlock;
   --  	 or
   --  	    accept Set_On_Esc(Is_Escape : in Boolean);      
   --  	 or
   --  	    accept Page_Down;
   --  	 or
   --  	    accept Page_Up;
   --  	 or
   --  	    accept Up_Arrow;
   --  	 or
   --  	    accept Down_Arrow;
   --  	 or
   --  	    accept Left_Arrow;
   --  	 or
   --  	    accept Right_Arrow;
   --  	 or
   --  	    accept Full_Screen;
   --  	 or
   --  	    accept Del;
   --  	 or
   --  	    accept Begin_Of;
   --  	 or
   --  	    accept End_Of;
   --  	 or
   --  	    accept Overwrite;
   --  	 or
   --  	    accept Screen_Print;
   --  	 or
   --  	    accept Shutdown;
   --  	 end select;
   --     end loop;
   --     Center.Global_Lock.Seize;
   --     W_Io.Put_Line("Execute halted");
   --     Center.Global_Lock.Release;
   --  end Execut_Process;

      
   --  task body Account_Process is
   --      Suspended : Boolean := False;
   --  begin
  
   --     loop
   --  	 select
   --       accept Wake_Up;
   --    or
   --  	    accept Halt (End_Of_Program : out Boolean);
   --  	    exit;
   --  	 or
   --  	    accept Initialize (Mode : Mode_Enum);
   --  	 or
   --  	    accept Receive (Wchar : in Wide_Character);
   --  	 or
   --  	    accept Mode(Mode : out Mode_Enum);
   --  	 or
   --  	    accept Reset;
   --  	 or
   --  	    accept Switch (Mode : in Mode_Enum);
   --  	 or
   --  	    accept Lock do
	    --     Suspended := True;
	    --  end Lock;
   --  	 or
   --  	    accept Unlock;
   --  	 or
   --  	    accept Set_On_Esc(Is_Escape : in Boolean);      
   --  	 or
   --  	    accept Page_Down;
   --  	 or
   --  	    accept Page_Up;
   --  	 or
   --  	    accept Up_Arrow;
   --  	 or
   --  	    accept Down_Arrow;
   --  	 or
   --  	    accept Left_Arrow;
   --  	 or
   --  	    accept Right_Arrow;
   --  	 or
   --  	    accept Full_Screen;
   --  	 or
   --  	    accept Del;
   --  	 or
   --  	    accept Begin_Of;
   --  	 or
   --  	    accept End_Of;
   --  	 or
   --  	    accept Overwrite;
   --  	 or
   --  	    accept Screen_Print;
   --  	 or
   --  	    accept Shutdown;
   --  	 end select;
   --     end loop;
   --  end Account_Process;

   --  task body Editor_Process is
   --      Suspended : Boolean := False;
   --  begin
  
   --     loop
   --  	 select
   --       accept Wake_Up;
   --    or
   --  	    accept Halt (End_Of_Program : out Boolean);
   --  	    exit;
   --  	 or
   --  	    accept Initialize (Mode : Mode_Enum);
   --  	 or
   --  	    accept Receive (Wchar : in Wide_Character);
   --  	 or
   --  	    accept Mode(Mode : out Mode_Enum);
   --  	 or
   --  	    accept Reset;
   --  	 or
   --  	    accept Switch (Mode : in Mode_Enum);
   --  	 or
   --  	    accept Lock do
	    --     Suspended := True;
	    --  end Lock;
   --  	 or
   --  	    accept Unlock;
   --  	 or
   --  	    accept Set_On_Esc(Is_Escape : in Boolean);      
   --  	 or
   --  	    accept Page_Down;
   --  	 or
   --  	    accept Page_Up;
   --  	 or
   --  	    accept Up_Arrow;
   --  	 or
   --  	    accept Down_Arrow;
   --  	 or
   --  	    accept Left_Arrow;
   --  	 or
   --  	    accept Right_Arrow;
   --  	 or
   --  	    accept Full_Screen;
   --  	 or
   --  	    accept Del;
   --  	 or
   --  	    accept Begin_Of;
   --  	 or
   --  	    accept End_Of;
   --  	 or
   --  	    accept Overwrite;
   --  	 or
   --  	    accept Screen_Print;
   --  	 or
   --  	    accept Shutdown;
   --  	 end select;
   --     end loop;
   --  end Editor_Process;
   
   --  task body Tasking_Process is
   --      Suspended : Boolean := False;
   --  begin
  
   --     loop
   --  	 select
   --       accept Wake_Up;
   --    or

   --  	    accept Halt (End_Of_Program : out Boolean);
   --  	    exit;
   --  	 or
   --  	    accept Initialize (Mode : Mode_Enum);
   --  	 or
   --  	    accept Receive (Wchar : in Wide_Character);
   --  	 or
   --  	    accept Mode(Mode : out Mode_Enum);
   --  	 or
   --  	    accept Reset;
   --  	 or
   --  	    accept Switch (Mode : in Mode_Enum);
   --  	 or
   --  	    accept Lock do
	    --     Suspended := True;
	    --  end Lock;
   --  	 or
   --  	    accept Unlock;
   --  	 or
   --  	    accept Set_On_Esc(Is_Escape : in Boolean);      
   --  	 or
   --  	    accept Page_Down;
   --  	 or
   --  	    accept Page_Up;
   --  	 or
   --  	    accept Up_Arrow;
   --  	 or
   --  	    accept Down_Arrow;
   --  	 or
   --  	    accept Left_Arrow;
   --  	 or
   --  	    accept Right_Arrow;
   --  	 or
   --  	    accept Full_Screen;
   --  	 or
   --  	    accept Del;
   --  	 or
   --  	    accept Begin_Of;
   --  	 or
   --  	    accept End_Of;
   --  	 or
   --  	    accept Overwrite;
   --  	 or
   --  	    accept Screen_Print;
   --  	 or
   --  	    accept Shutdown;
   --  	 end select;
   --     end loop;
   --  end Tasking_Process;
   
   task body Command_Process is
      End_Of_This_Task : Boolean := False;
      Suspended : Boolean := False;
      Current_Width : Column_Range := Columns/2;
      
      
      ----------------------------------
      --      Internal variables      --
      ----------------------------------
      
      Success           : Boolean := False;
      Is_Internal_Cmd : Boolean := False;
      Internal_Cmd    : Internal_Cmd_Enum := Null_Cmd;
      Aliazed         : Wide_String(1..256) := (others => Wide_Character'Val(32));
      ----------------------------------
      --      External Veriables      --
      ----------------------------------
      
      Ide_Line : Wide_String(1..256) := (others => Wide_Character'Val(32));
      Ide_Line_Last : Natural := 0;
      
      Line_Pos : Natural := 1;            
      
      
      Prev_Cmd : Wide_String_Access := new Wide_String ' ("");
      Prev_Arg : Wide_String_Access := new Wide_String ' ("");
      
      Cmd : Wide_String_Access;
      arg : Wide_String_Access;
      Command_Last : Natural;
      
      
      Line_Index : Natural := 0;
      File_Index : Natural := 0;      
      Total_Line : Natural := 0;
      Result_Max_Lines : Natural := (Lines-9)/2-2;
      
      Date : Time := Clock;            
      
      Completion_Result : Wide_Result_Access;      
            
      
      History_Set   : Wide_String_Set(2048);
      History_Last  : Natural := 0;
      History_Index : Natural := 1;
      
      Command : Gnat.Command_Line.Command_Line;      
      Args_List : Os_Lib.Argument_List_Access;                  
      
      
      Src : constant String := "hello.adb";
      Bin : constant String := "bin/hello";
      
      
      
   begin 
      while not End_Of_This_Task loop
	 loop
	    select
	       accept Initialize do
		  
		  Initialize(Center.Frames(Status).Win, 1, 1, 5, Cols/2, Yellow, Red);
		  Initialize(Center.Frames(Shell).Win, 6, 1, (Lines-9)/2, Cols/2, green, Yellow);
		  Initialize(Center.Frames(Edit).Win, 6+(Lines-9)/2, 1, (Lines-9)/2, Cols/2, Cyan, Red);
		  Initialize(Center.Frames(User).Win, Lines-3, 1, 3, Cols/2, Cyan, White);
		  Initialize(Center.Frames(Soldes).Win, 1, Cols/2+1, 5, Cols/2, Cyan, White);
		  Initialize(Center.Frames(Bank).Win, 6, Cols/2+1, Lines-9, Cols/2, blue, Yellow);
		  Initialize(Center.Frames(Deal).Win, Lines-3, Cols/2+1, 3, Cols/2, Cyan, Green);
		  Draw_Window(Center.Frames(Status).Win);
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Window(Center.Frames(User).Win);
		  Draw_Window(Center.Frames(Edit).Win);	       
		  Draw_Window(Center.Frames(Soldes).Win);
		  Draw_Window(Center.Frames(Bank).Win);
		  Draw_Window(Center.Frames(Deal).Win);	    
		  
		  if Os_Lib.Is_Regular_File(src) then		     
		     Center.Multi_Spawn_Result := 
		       new Spawn_Result_Record ' 
		       (Spawn("gnatmake -D obj -o " & Bin & ' ' & Src, ".Out_File.txt"));
		     
		     if Is_executable_File(bin) then
			
			declare
			   Hello_Result : Wide_Result_Access;
			begin
			   
			   Hello_Result := 
			     new Spawn_Result_Record ' 
			     (Spawn(Locate_Exec_On_Path(Bin).all, ".Out_File.txt"));
			   
			   Draw_Buffer(Hello_Result, Center.Frames(Shell).Win, File_Index);
			   Draw_Window(Center.Frames(User).Win);
			end;
		     else
			Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
			Draw_Window(Center.Frames(User).Win);
		     end if;
		  end if;
		  Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: ");
		  Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: ");
		  Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");

		  Draw_Window(Center.Frames(User).Win);
		  Draw_Text(Center.Frames(User).Win, 1, 1, White, "");

		  if not Center.Full_Screen then
		     Result_Max_Lines := (Lines-9)/2-2;
		  else
		     Result_Max_Lines := (Lines-9)-2;
		  end if;
		  if Ide_Line_Last /= 0 then
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
		  else
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
		  end if;
	       end Initialize;
	       exit;
	    or
	       accept Halt do
		  End_Of_This_Task := True;
		  W_Io.Put_Line("Halting Multi command");
	       end Halt;
	       exit;
	    or
	       accept Lock;
	    or
	       accept Wake_Up do
		  Suspended := False;
		  if Center.Full_Screen then
		     
		     Current_Width := Columns;		  
		     Result_Max_Lines := (Lines-9)-3;
		  else
		     
		     Current_Width := Columns/2;
		     Result_Max_Lines := (Lines-9)/2-2;
		     
		     Draw_Window(Center.Frames(Edit).Win);	       
		     Draw_Window(Center.Frames(Soldes).Win);
		     Draw_Window(Center.Frames(Bank).Win);
		     Draw_Window(Center.Frames(Deal).Win);
		  end if;
		  
		  
		  Center.Frames(Status).Win.win := 
		    (1, 1, 5, (Current_Width), Window.Windows.Single_Line_Frame, yellow, Red);
		  
		  Center.Frames(Shell).Win.Win :=
		    (6, 1, Result_Max_Lines+2, (Current_Width), Window.Windows.Single_Line_Frame, green, Yellow);
		  
		  Center.Frames(User).Win.Win := 
		    (Lines-3, 1, 3, (Current_Width), Window.Windows.Single_Line_Frame, cyan, white);
		  
		  
		  
		  Draw_Window(Center.Frames(Status).Win);
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Window(Center.Frames(User).Win);
		  
		  
		  
		  if Is_executable_File(bin) then
		     
		     declare
			Hello_Result : Wide_Result_Access;
		     begin
			
			Hello_Result := 
			  new Spawn_Result_Record ' 
			  (Spawn(Locate_Exec_On_Path(Bin).all, ".Out_File.txt"));
			Draw_Buffer(Hello_Result, Center.Frames(Shell).Win, File_Index);
			Draw_Window(Center.Frames(User).Win);
		     end;	       
		  end if;
		  Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: ");
		  Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: ");
		  Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");

		  if not Center.Full_Screen then
		     
		     Result_Max_Lines := (Lines-9)/2-2;
		  else
		     Result_Max_Lines := (Lines-9)-2;
		  end if;
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
		  if Ide_Line_Last /= 0 then
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
		  else
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
		  end if;
	       end Wake_Up;
	       exit;
	       
	    end select;
	 end loop;
	 
	 while not End_Of_This_Task loop
	    if Cmd /= null and then Cmd'Length > 0 then 
	       Prev_Cmd := new Wide_String ' (Cmd.all);	    	       
	    end if;

	    Command_Line.Set_Command_Line(Command, Handling.To_String(Ide_Line(Ide_Line'First..Ide_Line_last)));
	    Gnat.Command_Line.Build(Command, Args_List, False);	    

	    if Args_List /= null and then Args_List'Length /= 0 and then
	      Args_List(1) /= null then	   

	       if Args_List(1)'length /= 0 then

		  Wide_free(Prev_Arg);
		  Prev_Arg := new Wide_String ' (((Current_Width-2) * Wide_Space));

		  Wide_Free(Cmd);	       
		  Wide_Free(Arg);
		  Cmd := new Wide_String ' (Handling.To_Wide_String(Args_List(1).all));	    
		  
		  Arg := new Wide_String ' (ide_Line(Cmd'Last+1..ide_Line_Last));

		  if Arg /= null then
		     Wide_Free(Prev_Arg);
		     Prev_Arg := new Wide_String ' (Arg.all);
		  end if;
		  
		  Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " & 
			      Troncate(Cmd.all, Current_Width-8) & "     ");
		  Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & 
			      Troncate(Arg.all, Current_Width-8) & "     ");

	       end if;
	    else
	       Wide_Free(Prev_Arg);
	       Prev_Arg := new Wide_String ' (((Current_Width-2) * Wide_Space));
	       Wide_Free(Cmd);

	       If Ide_Line_Last > 0 then
		  Cmd := new Wide_String ' 
		    (ide_Line(Index_Non_Blank(ide_Line)..Index_Non_Blank
				(ide_Line(ide_Line'First..ide_Line_Last), backward)));
	       else
		  Cmd := new Wide_String ' ("");
	       end if;

	       Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " & 
			   Troncate(Cmd.all, Current_Width-8) & "     ");
	       Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & 
			   Troncate(Prev_Arg.all, Current_Width-8) & "     ");
	       

	    end if;

	    if Center.Multi_Spawn_Result /= null then
	       
	       Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: " & 
			   Handling.To_Wide_String(Integer'Image(Center.Multi_Spawn_Result.all.Return_Code)));
	    else
	       Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");
	    end if;

	    
	    Draw_Window(Center.Frames(User).Win);	 	       
	    if Ide_Line_Last /= 0 then
	       Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
	    else
	       Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
	    end if;      

	    select
	       accept Wake_Up do
		  Suspended := False;
		  if Center.Full_Screen then
		     
		     Current_Width := Columns;		  
		     Result_Max_Lines := (Lines-9)-3;
		  else
		     
		     Current_Width := Columns/2;
		     Result_Max_Lines := (Lines-9)/2-2;
		     
		     Draw_Window(Center.Frames(Edit).Win);	       
		     Draw_Window(Center.Frames(Soldes).Win);
		     Draw_Window(Center.Frames(Bank).Win);
		     Draw_Window(Center.Frames(Deal).Win);
		  end if;
		  
		  
		  Center.Frames(Status).Win.win := 
		    (1, 1, 5, (Current_Width), Window.Windows.Single_Line_Frame, yellow, Red);
		  
		  Center.Frames(Shell).Win.Win :=
		    (6, 1, Result_Max_Lines+2, (Current_Width), Window.Windows.Single_Line_Frame, green, Yellow);
		  
		  Center.Frames(User).Win.Win := 
		    (Lines-3, 1, 3, (Current_Width), Window.Windows.Single_Line_Frame, cyan, white);
		  
		  
		  
		  Draw_Window(Center.Frames(Status).Win);
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Window(Center.Frames(User).Win);
		  
		  
		  
		  if Is_executable_File(bin) then
		     
		     declare
			Hello_Result : Wide_Result_Access;
		     begin
			
			Hello_Result := 
			  new Spawn_Result_Record ' 
			  (Spawn(Locate_Exec_On_Path(Bin).all, ".Out_File.txt"));
			Draw_Buffer(Hello_Result, Center.Frames(Shell).Win, File_Index);
			Draw_Window(Center.Frames(User).Win);
		     end;	       
		  end if;
		  Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: ");
		  Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: ");
		  Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");

		  
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
		  if Ide_Line_Last /= 0 then
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
		  else
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
		  end if;
	       end Wake_Up;
	    or
	       
	       accept Initialize do
		  
		  Draw_Window(Center.Frames(Status).Win);
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Window(Center.Frames(User).Win);
		  Draw_Window(Center.Frames(Edit).Win);
		  Draw_Window(Center.Frames(Soldes).Win);
		  Draw_Window(Center.Frames(Bank).Win);
		  Draw_Window(Center.Frames(Deal).Win);	    
		  
		  if Is_executable_File(bin) then
		     
		     declare
			Hello_Result : Wide_Result_Access;
		     begin
			
			Hello_Result := 
			  new Spawn_Result_Record ' 
			  (Spawn(Locate_Exec_On_Path(Bin).all, ".Out_File.txt"));
			Draw_Buffer(Hello_Result, Center.Frames(Shell).Win, File_Index);
			Draw_Window(Center.Frames(User).Win);
		     end;	       
		  end if;
		  Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: ");
		  Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: ");
		  Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");

		  if not Center.Full_Screen then
		     
		     Result_Max_Lines := (Lines-9)/2-2;
		  else
		     Result_Max_Lines := (Lines-9)-2;
		  end if;
		  if Ide_Line_Last /= 0 then
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
		  else
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
		  end if;
	       end Initialize;
	    or
	       accept Halt do
		  W_Io.Put_Line("Halting Multi command");
		  End_Of_This_Task := True;
	       end Halt;
	       exit;
	    or
	       accept Receive (Wchar : in Wide_Character) do
		  
		  if Ada.Wide_Characters.Handling.Is_Control(Wchar) then
		     case Wchar is
			when Wide_Character'Val(0) =>
			   
			   null;
			when Wide_Character'Val(9) =>
			   if Ide_Line_Last > 0 then
			      -- Path completion...
			      declare
				 Path_On_Line : Wide_String_Access;
				 Line : Wide_String_Access := 
				   new Wide_String ' (ide_Line(ide_Line'First..ide_Line_Last));
				 buffer : Wide_string_Access := new Wide_String ' ("");
			      begin
				 Completion(Handling.To_String(Line.all), Path_On_Line, Completion_Result);
				 Center.Multi_Spawn_Result := Completion_Result;
				 if Path_On_Line /= null then
				    Buffer := new Wide_String ' (Path_On_Line.all);
				    Wide_Free(Line);
				    Line := new Wide_String ' (Buffer.all);
				    Wide_Free(Buffer);
				    
				    Ide_Line(Ide_Line'First..Ide_Line'First+Line'Length-1) := Line.all;
				    Ide_Line_Last := Line'Length;
				 end if;
				 Draw_Window(Center.Frames(Status).Win);
				 Draw_Window(Center.Frames(Shell).Win);                         
				 Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " &
					     Troncate(Prev_Cmd.all, Columns/2-2) & "    ");
				 Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & 
					     Troncate(Prev_Arg.all, Columns/2-2) & "    ");			      
				 File_Index := 0;
			      end;
			   end if;                        
			   
			when Wide_Character'Val(10) =>
			   if ide_Line_Last > 0 then
			      
			      History_Index := 1;
			      ------------------------------------------------------
			      --                        Internal                  --
			      ------------------------------------------------------
			      
			      Internal_Cmd := Internal_Cmd_Value(Ide_Line(Ide_Line'First..Ide_Line_Last));	
			      Is_Internal_Cmd := True;
			      case Internal_Cmd is
				 when Null_Cmd =>
				    Is_Internal_Cmd := False;
				 when Clear =>
				    case Os_Name is
				       when System.Os_Constants.Windows =>
					  Success := 
					    Clear(Center.User_Path.all & Dir_Separator & ".shell_file.txt");
				       when System.Os_Constants.Other_Os =>
					  Success := 
					    Clear(Center.User_Path.all & Dir_Separator & ".shell_file.txt");
				    end case;
				    if not Success then
				       Center.Multi_Spawn_Result.Wlines := 
					 Add_Line(Center.Multi_Spawn_Result.Wlines, "An error occured when attempt of "".shell_file.txt"" deletion.");
				       Center.Multi_Spawn_Result.Wlines := 
					 Add_Line(Center.Multi_Spawn_Result.Wlines, "Full path : "  & Handling.To_Wide_String(Center.User_Path.all & Dir_Separator & ".shell_file.txt"));
				       
				    else
				       U_Array_Free(Center.Multi_Spawn_Result.Wlines);
				       Center.Multi_Spawn_Result.Wlines := new U_Array (1..0);
				    end if;
				 when Alias =>
				    Alias(Ide_Line(Ide_Line'First..Ide_Line_Last), Center.User.Alias_Set, 
					  Spawn_Result_Record(Center.Multi_Spawn_Result.all));
				 when Unalias =>
				    Unalias(Ide_Line(Ide_Line'First..Ide_Line_Last), 
					    Center.User.Alias_Set, Spawn_Result_Record(Center.Multi_Spawn_Result.all));
				 when Cd =>
				    Change_Directory(Ide_Line(Ide_Line'First..Ide_Line_Last),
						     Spawn_Result_Record(Center.Multi_Spawn_Result.all));
				 when Pwd =>
				    Center.Multi_Spawn_Result.Wlines := 
				      Add_Line(Center.Multi_Spawn_Result.Wlines, Handling.To_Wide_String(String(Get_Current_Dir)));
				 when Help =>
				    Internal_Help(Ide_Line(Ide_Line'First..Ide_Line_Last), 
						  Spawn_Result_Record(Center.Multi_Spawn_Result.all));
				 when Logout =>
				    Center.Logout := True;
				 when others =>
				    null;
			      end case;
			      
			      case Internal_Cmd is
				 when Null_Cmd =>
				    Center.Multi_Spawn_Result.Return_code := 127;
				 when others =>
				    Add_Line(History_Set, History_Last, Ide_Line(Ide_Line'First..Ide_Line_Last));
				    ide_Line := (others => Wide_Character'Val(32));
				    ide_Line_Last := 0;
			      end case;
			      
			      ------------------------------------------------------
			      --                   End internal                   --
			      ------------------------------------------------------
			      
			      if not Is_Internal_Cmd then
				 declare
				    Alias_Ptr   : Strings.String_Access;
				    Buffer      : Wide_String_Access;
				    Normal      : Strings.String_Access;				 
				 begin
				    
				    for Iter in 1..Center.User.Alias_Set.Index loop
				       declare
					  Alias : constant Attribut_Record := Center.User.Alias_Set.List(Iter);
					  First_Space : Natural := 
					    Wide_Fixed.Index(Ide_Line(Ide_Line'First..Ide_Line_Last), " ");
					  Non_Blank   : Natural :=
					    Wide_Fixed.Index_Non_Blank(Ide_Line(Ide_Line'First..Ide_Line_Last), Backward);
				       begin	
					  if Non_Blank < First_Space then
					     First_Space := 0;
					  end if;
					  if First_Space /= 0 then
					     if Handling.To_Wide_String(Names(Alias)) =
					       Ide_Line(Ide_Line'First..First_Space - 1) then					     
						Alias_Ptr := new String ' (Attrib.Image(Alias));
						exit;
					     end if;
					  else
					     if Handling.To_Wide_String(Names(Alias)) =
					       Ide_Line(Ide_Line'First..Index_Non_Blank(Ide_Line, Backward)) then
						Alias_Ptr := new String ' (Attrib.Image(Alias));
						exit;
					     end if;
					  end if;
				       end;
				    end loop;
				    
				    
				    if Alias_Ptr /= null then
				       
				       declare
					  First_Space : constant Natural := 
					    Wide_Fixed.Index(Ide_Line(Ide_Line'First..Ide_Line_Last), " ");
				       begin
					  if First_Space /= 0 then
					     Buffer := new Wide_String ' 
					       (Handling.To_Wide_String(Alias_Ptr.all) & ' ' & Ide_Line(First_Space+1..Ide_Line_Last));
					  else
					     Buffer := new Wide_String ' (Handling.To_Wide_String(Alias_Ptr.all));
					  end if;
				       end;
				       Ide_Line := (others => Wide_Character'Val(32));
				       Normal := new String ' (Normalize_Quoted_Argument(Handling.To_String(Buffer.all)));
				       Wide_Fixed.Move (Handling.To_Wide_String(Normal.all), Ide_Line);
				       Ide_Line_Last := Buffer'Length;
				       
				    end if;
				 end;
				 
				 ------------------------------------------------------
				 --      Flush File content in shell file            --
				 ------------------------------------------------------
				 
				 Write_To_File(Center.Multi_Spawn_Result.Wlines, 
					       Center.Full_Shell_Filename.all);
				 
				 Center.Multi_Spawn_Result := 
				   new Spawn_Result_Record ' 
				   (Shell_Spawn("sh -c """ & Expand_Filename(Conversions.To_String(ide_Line(ide_Line'First..ide_Line_Last) & """")), ".shell_file.txt"));
				 if Center.Multi_Spawn_Result /= null and then Center.Multi_Spawn_Result.Wlines'length > (Result_Max_Lines - 3) then
				    File_Index := Center.Multi_Spawn_Result.Wlines'length - (Result_Max_Lines - (Result_Max_Lines-5)/2);
				 end if;
				 if Spawn_Result_Record(Center.Multi_Spawn_Result.All).Return_Code = 0 then			      
				    Add_Line(History_Set, History_Last, Ide_Line(Ide_Line'First..Ide_Line_Last));
				    ide_Line := (others => Wide_Character'Val(32));
				    ide_Line_Last := 0;			      
				 end if;
				 Draw_Window(Center.Frames(Status).Win);
				 Draw_Window(Center.Frames(Shell).Win);                         
				 Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " & Troncate(Prev_Cmd.all, Columns/2-2) & "    ");
				 Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & Troncate(Prev_Arg.all, Columns/2-2) & "    ");
			      else
				 if Center.Multi_Spawn_Result /= null and then Center.Multi_Spawn_Result.Wlines'length > (Result_Max_Lines - 3) then
				    File_Index := Center.Multi_Spawn_Result.Wlines'length - (Result_Max_Lines - (Result_Max_Lines-5)/2);
				 end if;
				 
			      end if;
			      Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
			   end if;
			when Wide_Character'Val(12) =>
			   null;
			when Wide_Character'Val(27) =>
			   null;
			   
			when Wide_Character'Val(127) =>
			   if ide_Line_Last > ide_Line'First then
			      W_Io.Put(Wide_Character'Val(8) & 
					 Wide_Character'Val(32) &                      
					 Wide_Character'Val(8));
			      ide_Line(ide_Line_Last) := Wide_Character'Val(32);
			      ide_Line_Last := ide_Line_Last - 1;
			   elsif Ide_Line_Last = Ide_Line'First then
			      W_Io.Put(Wide_Character'Val(8) & 
					 Wide_Character'Val(32) &                      
					 Wide_Character'Val(8));
			      ide_Line(ide_Line_Last) := Wide_Character'Val(32);
			      ide_Line_Last := ide_Line_Last - 1;
			      Cmd := new Wide_String ' (" ");
			   end if;
			   
			when others =>               
			   null;
		     end case;
		  elsif Ada.Wide_Characters.Handling.Is_Graphic(wchar) then
		     case wchar is
			when others => 
			   if not Center.Full_Screen then
			      if ide_Line_Last < (Columns/2-2) then
				 ide_Line(ide_Line_Last + 1) := wchar;
				 ide_Line_Last := ide_Line_Last + 1;
			      else
				 T_Io.Put(Character ' Val(7));
			      end if;
			      Prev_arg := new Wide_String ' (((Columns/2-2) * Wide_Space));
			   else
			      if ide_Line_Last < (Columns-2) then                
				 ide_Line(ide_Line_Last + 1) := wchar;
				 ide_Line_Last := ide_Line_Last + 1;
			      else
				 T_Io.Put(Character ' Val(7));
			      end if;
			      Prev_arg := new Wide_String ' (((Columns-2) * Wide_Space));
			   end if;                        
		     end case;
		  else
		     raise Program_Error;
		  end if;
		  
		  if Cmd /= null and then Cmd'Length > 0 then 
		     Prev_Cmd := new Wide_String ' (Cmd.all);	    	       
		  end if;

		  Command_Line.Set_Command_Line(Command, Handling.To_String(Ide_Line(Ide_Line'First..Ide_Line_last)));
		  Gnat.Command_Line.Build(Command, Args_List, False);	    

		  if Args_List /= null and then Args_List'Length /= 0 and then
		    Args_List(1) /= null then	   

		     if Args_List(1)'length /= 0 then

			Wide_free(Prev_Arg);
			Prev_Arg := new Wide_String ' (((Current_Width-2) * Wide_Space));

			Wide_Free(Cmd);	       
			Wide_Free(Arg);
			Cmd := new Wide_String ' (Handling.To_Wide_String(Args_List(1).all));	    
			
			Arg := new Wide_String ' (ide_Line(Cmd'Last+1..ide_Line_Last));

			if Arg /= null then
			   Wide_Free(Prev_Arg);
			   Prev_Arg := new Wide_String ' (Arg.all);
			end if;
			
			Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " & Troncate(Cmd.all, Current_Width-8) & "     ");
			Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & Troncate(Arg.all, Current_Width-8) & "     ");

		     end if;
		  else
		     Wide_Free(Prev_Arg);
		     Prev_Arg := new Wide_String ' (((Current_Width-2) * Wide_Space));
		     Wide_Free(Cmd);

		     If Ide_Line_Last > 0 then
			Cmd := new Wide_String ' 
			  (ide_Line(Index_Non_Blank(ide_Line)..Index_Non_Blank
				      (ide_Line(ide_Line'First..ide_Line_Last), backward)));
		     else
			Cmd := new Wide_String ' ("");
		     end if;

		     Draw_Text(Center.Frames(Status).Win, 1, 1, White, "cmd: " &
				 Troncate(Cmd.all, Current_Width-8) & "     ");
		     Draw_Text(Center.Frames(Status).Win, 2, 1, White, "arg: " & 
				 Troncate(Prev_Arg.all, Current_Width-8) & "     ");
		     

		  end if;

		  if Center.Multi_Spawn_Result /= null then
		     
		     Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: " &
				 Handling.To_Wide_String(Integer'Image(Center.Multi_Spawn_Result.all.Return_Code)));
		  else
		     Draw_Text(Center.Frames(Status).Win, 3, 1, White, "err: ");
		  end if;
		  
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);

		  Draw_Window(Center.Frames(User).Win);	 	       
		  if Ide_Line_Last /= 0 then
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, Ide_Line(1..Ide_Line_Last));
		  else
		     Draw_Text(Center.Frames(User).Win, 1, 1, White, "");
		  end if;
	       end Receive;
	    or
	       accept Reset;
	    or
	       accept Lock do
		  Suspended := True;
	       end Lock;
	       exit;
	    or
	       accept Unlock;
	    or
	       accept Set_On_Esc(Is_Escape : in Boolean);
	       
	    or
	       accept Page_Down do
		  if Center.Multi_Spawn_Result /= null and then
		    File_Index+(Result_Max_Lines-Result_Max_Lines/2) < Center.Multi_Spawn_Result.Wlines'Length  then
		     File_Index := Positive'Min(File_Index+(Result_Max_Lines-Result_Max_Lines/2), 
						Center.Multi_Spawn_Result.Wlines'Length);
		  else
		     T_Io.Put(Character'Val(7));
		  end if;
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
	       end Page_Down;
	    or
	       accept Page_Up do
		  
		  if Center.Multi_Spawn_Result /= null and then
		    File_Index-(Result_Max_Lines-Result_Max_Lines/2) > 0  then
		     File_Index := File_Index - (Result_Max_Lines-Result_Max_Lines/2);
		  else
		     T_Io.Put(Character'Val(7));
		     File_Index := 0;
		  end if;	       
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
	       end Page_Up;
	    or
	       accept Up_Arrow do
		  Draw_Window(Center.Frames(User).Win);
		  if (History_Last > 0) and then
		    (History_Last - History_Index) >= 0 then
		     
		     Move(History_Set.Lines(History_Last-(History_Index-1)).all, Ide_Line);
		     Ide_Line_Last := History_Set.Lines(History_Last-(History_Index-1))'Length;
		     History_Index := History_Index + 1;
		  else
		     T_Io.Put(Character'Val(7));
		  end if;	       
		  
	       end Up_Arrow;
	    or
	       accept Down_Arrow do
		  Draw_Window(Center.Frames(User).Win);
		  if (History_Index > 2) then
		     Move(History_Set.Lines(History_Index-1).all, Ide_Line);
		     Ide_Line_Last := History_Set.lines(History_Index-1)'Length;
		     History_Index := History_Index - 1;
		  elsif History_Index = 2 then
		     Ide_Line := (others => Wide_Character ' Val(32));
		     Ide_Line_Last := 0;
		     History_Index := History_Index - 1;
		  else
		     T_Io.Put(Character'Val(7));
		  end if;	       	       	       
	       end Down_Arrow;
	    or
	       accept Left_Arrow;
	    or
	       accept Right_Arrow;
	    or
	       accept Full_Screen do
		  if not Center.Full_Screen then
		     
		     Current_Width := Columns;		  
		     Result_Max_Lines := (Lines-9)-3;
		  else
		     
		     Current_Width := Columns/2;
		     Result_Max_Lines := (Lines-9)/2-2;
		     
		     Draw_Window(Center.Frames(Edit).Win);	       
		     Draw_Window(Center.Frames(Soldes).Win);
		     Draw_Window(Center.Frames(Bank).Win);
		     Draw_Window(Center.Frames(Deal).Win);
		  end if;
		  
		  
		  Center.Frames(Status).Win.win := 
		    (1, 1, 5, (Current_Width), Window.Windows.Single_Line_Frame, yellow, Red);
		  
		  Center.Frames(Shell).Win.Win :=
		    (6, 1, Result_Max_Lines+2, (Current_Width), Window.Windows.Single_Line_Frame, green, Yellow);
		  
		  Center.Frames(User).Win.Win := 
		    (Lines-3, 1, 3, (Current_Width), Window.Windows.Single_Line_Frame, cyan, white);
		  
		  Center.Full_Screen := not Center.Full_Screen;
		  
		  Draw_Window(Center.Frames(Status).Win);
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Window(Center.Frames(User).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
	       end Full_Screen;
	    or
	       accept Del;
	    or
	       accept Begin_Of do
		  File_Index := 0;
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
	       end Begin_Of;
	    or
	       accept End_Of do
		  if Center.Multi_Spawn_Result /= null and then Center.Multi_Spawn_Result.Wlines /= null then		  
		     if Center.Multi_Spawn_Result.Wlines'Length > (Result_Max_Lines-Result_Max_Lines/2) then
			File_Index :=Center.Multi_Spawn_Result.Wlines'Length - (Result_Max_Lines-Result_Max_Lines/2);
		     else
			File_Index := Center.Multi_Spawn_Result.Wlines'Length;
		     end if;
		  end if;
		  Draw_Window(Center.Frames(Shell).Win);
		  Draw_Buffer(Center.Multi_Spawn_Result, Center.Frames(Shell).Win, File_Index);
	       end End_Of;
	    or
	       accept Overwrite;
	    or
	       accept Screen_Print;   	
	    end select;	 
	 end loop;
      end loop;
      W_Io.Put_Line("Multi Command halted");
   end Command_Process;
   
   
   
    
   
   
   function Get_Lines (Int : in Center_Record) return Positive is
      Geometry_Parameter : constant Params.Parameter_Tagged_Record_Type :=
	Center.Get_Parameter(Int, Params.Geometry);
      Lines : Line_Range := 67;
      
   begin
      if Geometry_Parameter.Is_Setted then
	 Lines := Lines_Number(Geometry_Parameter.Value.all);	 
      end if;
      return Lines;
   end Get_Lines;
   
   function Get_Cols (Int : in Center_Record) return Positive is
      Geometry_Parameter : constant Params.Parameter_Tagged_Record_Type :=
	Center.Get_Parameter(Int, Params.Geometry);
      Cols  : Column_Range := 240;
      
   begin
      if Geometry_Parameter.Is_Setted then
	 Cols := Columns_Number(Geometry_Parameter.Value.all);     
      end if;
      return Cols;
   end Get_Cols;
   
   
   package Devel_Wrapp is new PragmARC.Wrapping(Devel_Enum);
   package Multi_Wrapp is new PragmARC.Wrapping(Multi_Enum);
   
   
   task body Astar_Process is
      End_Of_Program : Boolean := False;      
      Initialized : Boolean := False;
      
      
      Buffer : Classes.Abstracted_Vectors.Vector;
      
      Openned : Path_Finding.Element_List;
      Closed : Path_Finding.Element_List;
      Closed_Curs : Path_Finding.Element_Lists.Cursor;
   begin
      while not End_Of_Program loop
	 Initialized := False;
	 loop
	    select
	       
	       accept Search_Path(Open : in Classes.Abstracted_Access;
				  Goal : in Classes.Abstracted_Access) do
		  if Center.The_Server /= null then
		     Center.The_Server.Astar_Proc.Init(Open, Goal);
		  else
		     Path_Finding.Element_Lists.Clear(Openned);
		     Path_Finding.Element_Lists.Clear(Closed);
		     Path_Finding.Element_Lists.Append(Openned, Open);
		     Path_Finding.Element_Lists.Append(Openned, Goal);      
		     Path_Finding.Astar_Task.Init(Openned);
		     Config.Server.Engine.Path_Finding.Astar_Task.init(Openned);
		  end if;
		  Initialized := True;
		  Center.Goal_Path_Search_End_Of_Task := False;
	       end Search_Path;
	       exit;
	    or
	       
	       accept Halt do
		  End_Of_Program := True;
	       end Halt;
	       exit;
	    or
	       when Center.The_Server /= null =>
		  accept Goal_Path(Path : in out Classes.Abstracted_Vectors.Vector) do
		     --W_Io.Put_Line("Astar process : accept goal_path after search");
		     if Initialized then
			Abstracted_Vectors.Clear(Buffer);
			select
			   Center.The_Server.Astar_Proc.Close_Set(Buffer, Center.Goal_Path_Search_End_Of_Task);
			   Abstracted_Vectors.Clear(Path);
			   Path := Buffer;
			or
			   delay 1.0;
			end select;
		     end if;
		  end Goal_Path;
	    or
	       when Center.The_Server = null =>
		  accept Goal_Path(Path : in out Classes.Abstracted_Vectors.Vector) do
		     --W_Io.Put_Line("Astar process : accept goal_path after search");
		     if Initialized then
			
			select
			   
			   Path_Finding.Astar_Task.Close_Set(Closed, Center.Goal_Path_Search_End_Of_Task);
			   if Path_Finding.Element_Lists.Length(Closed) /= 0 then
			      
			      Abstracted_Vectors.Clear(Path);
			      Closed_Curs := Path_Finding.Element_Lists.First(Closed);
			      for El in 1..Path_Finding.Element_Lists.Length(Closed) loop
				 declare
				    E : Abstracted_Access := Path_Finding.Element_Lists.Element(Closed_Curs);
				 begin
				    Abstracted_Vectors.Append(Center.Goal_Path, E);
				    Path_Finding.Element_Lists.Next(Closed_Curs);
				 end;
			      end loop;
			      
			   end if;
			or
			   delay 1.0;
			end select;
		     
		     end if;
		  end Goal_Path;

	    end select;
	 end loop;
	 while not End_Of_Program loop
	    select
	      accept Halt do
		  End_Of_Program := True;
	       end Halt;
	       exit;
	    or
	       accept Search_Path(Open : in Classes.Abstracted_Access;
				  Goal : in Classes.Abstracted_Access) do
		  if Center.The_Server /= null then
		     Center.The_Server.Astar_Proc.Init(Open, Goal);
		  else
		     Path_Finding.Element_Lists.Clear(Openned);
		     Path_Finding.Element_Lists.Clear(Closed);
		     Path_Finding.Element_Lists.Append(Openned, Open);
		     Path_Finding.Element_Lists.Append(Openned, Goal);      
		     Path_Finding.Astar_Task.Init(Openned);
		     Config.Server.Engine.Path_Finding.Astar_Task.init(Openned);
		  end if;
		  Initialized := True;
		  Center.Goal_Path_Search_End_Of_Task := False;
	       end Search_Path;
	       
	    or
	       when Center.The_Server /= null =>
		  accept Goal_Path(Path : in out Classes.Abstracted_Vectors.Vector) do
		     --W_Io.Put_Line("Astar process : accept goal_path durring search");
		     if Initialized then
			Abstracted_Vectors.Clear(Buffer);
			select
			   Center.The_Server.Astar_Proc.Close_Set(Buffer, Center.Goal_Path_Search_End_Of_Task);
			   Abstracted_Vectors.Clear(Path);
			   Path := Buffer;
			or
			   delay 1.0;
			end select;
		     end if;
		  end Goal_Path;
		  if Center.Goal_Path_Search_End_Of_Task then
		     exit;
		  end if;
	    or
	       when Center.The_Server = null =>
		  accept Goal_Path(Path : in out Classes.Abstracted_Vectors.Vector) do
		     --W_Io.Put_Line("Astar process : accept goal_path durring search");
		     if Initialized then
			
			select
		
			   Path_Finding.Astar_Task.Close_Set(Closed, Center.Goal_Path_Search_End_Of_Task);
			   if Path_Finding.Element_Lists.Length(Closed) /= 0 then
			      
			      Abstracted_Vectors.Clear(Path);
			      Closed_Curs := Path_Finding.Element_Lists.First(Closed);
			      for El in 1..Path_Finding.Element_Lists.Length(Closed) loop
				 declare
				    E : Abstracted_Access := Path_Finding.Element_Lists.Element(Closed_Curs);
				 begin
				    Abstracted_Vectors.Append(Center.Goal_Path, E);
				    Path_Finding.Element_Lists.Next(Closed_Curs);
				 end;
			      end loop;
			      
			   end if;
			   
			or
			   delay 1.0;
			end select;
			
		     end if;	  
		  end Goal_Path;
		  if Center.Goal_Path_Search_End_Of_Task then
		     exit;
		  end if;
	    end select;	 	 
	 end loop;
      end loop;
      W_Io.Put_Line("Center : Astar process halted");
   end Astar_Process;
   
   
   ---------------------------------------------------------------------------------
   --                             Console process                                 --
   ---------------------------------------------------------------------------------
   task body Console_Process is
      
      package Center_Menu is new PragmARC.Menu_Handler (cols, lines);
      use Center_Menu;
      use Center_Menu.V_String;
   
      Halting_Menu_Choice  : Natural := 0;      
      
      Main_Menu : Center_Menu.Menu_Info := (4, True, To_Bounded_String("Choose the starting mode"),
					     (To_Bounded_String("Dialog"),
					      To_Bounded_String("Devel"),
					      To_Bounded_String("Organizer"),
					      To_Bounded_String("Multi")),
					      
					     1);
      Quit_Menu : Center_Menu.Menu_Info := (6, True, To_Bounded_String("Choose the halting mode"),
						    (To_Bounded_String("Quit"),
						     To_Bounded_String("Auth"),
						     To_Bounded_String("Dialog"),
						     To_Bounded_String("Devel"),
						     To_Bounded_String("Organizer"),
						     To_Bounded_String("Multi")),
						     
						    1);
      
      
      
      End_Of_This_Task, Verax, End_Of_Task, End_Of_Program : Boolean := False;
      
      
      Multi_State : Multi_Enum := Command_Mode;
      Devel_State : Devel_Enum := Command;
      
   begin
      
      loop	 
	 select	    
	    accept Halt (End_Of_Program : out Boolean) do
	       if Center.Current_Mode = Null_Mode then
		  Center.Auth_Proc.Auth;

		  Halting_Menu_Choice := Process(Quit_Menu);
		  
		  case Halting_Menu_Choice is
		     when 1 =>
			End_Of_This_Task := True;
		     when 2 =>
			Center.Current_Mode := Quit_Mode;
			End_Of_This_Task := False;
			Center.Auth_Proc.Auth;
		     when 3 =>
			Center.Auth_Proc.Auth;
			End_Of_This_Task := False;			
			Center.Current_Mode := Dialog_Mode;			
			Center.Dialog_Proc.Initialize(Center.User_Text.all);
		     when 4 =>
			Center.Auth_Proc.Auth;
			End_Of_This_Task := False;
			Center.Current_Mode := Devel_Mode;
			Center.Devel_Proc.Initialize;
		     when 5 =>
			Center.Auth_Proc.Auth;
			End_Of_This_Task := False;
			Center.Current_Mode := Organizer_Mode;
			Center.Organizer_Proc.Initialize;
		     when 6 =>
			Center.Auth_Proc.Auth;
			End_Of_This_Task := False;
			Center.Current_Mode := Multi_Mode;
			Center.Multi_Proc.Initialize;
		     when 7 =>
			End_Of_This_Task := False;
			Center.Current_Mode := Shell_Mode;
			--Center.Shell_Proc.Initialize(Sys);
		     when 8 =>
			End_Of_This_Task := False;
			Center.Current_Mode := Execut_Mode;
			--Center.Execut_Proc.Initialize;
		     when others =>			
			null;
		  end case;
		  
		  
		  End_Of_Program := End_Of_This_Task;		  
		  End_Of_Task := False;
	       else
		  End_Of_Program := false;
		  End_Of_This_Task := False;
	       end if;
	       
	       
	    end Halt;
	    if End_Of_This_task  Then
	       
	       End_Of_Task := False;
	       End_Of_Program := False;
	       Center.Current_Mode := Center.Prev_Mode;	       
	       W_Io.New_Line;	       	       
	       Center.Work_Proc.Initialize;
	    end if;	    
	    End_Of_Task := False;
	    
	 or
	    accept Initialize (Mode : Mode_Enum; User_Text : in Wide_String) do
	       case Mode is
		  when Auth_Mode =>
		     Ansi_Console.Clear_Screen;	    
		     Center.Auth_Proc.Auth;
		     
		     Ansi_Console.Clear_Screen;
		     
		     Center.Dialog_Proc.Initialize(User_Text);
		     Center.User_Text := new Wide_String ' (User_Text);
		     Center.Dialog_Proc.Lock;
		     delay 0.2;
		     
		     Ansi_Console.Clear_Screen;	       
		     Center.Devel_Proc.Initialize;
		     Center.Devel_Proc.Lock;
		     delay 0.2;
		     
		     Ansi_Console.Clear_Screen;
		     Center.Organizer_Proc.Initialize;
		     Center.Organizer_Proc.Lock;
		     delay 0.2;
		     
		     Ansi_Console.Clear_Screen;
		     Center.Multi_Proc.Initialize;
		     Center.Multi_Proc.Lock;		     
		     delay 0.2;
		     
		     Ansi_Console.Clear_Screen;
		     Center.Shell_Proc.Initialize(Sys);
		     Center.Shell_Proc.Lock;
		     delay 0.2;
		     
		     
		     Ansi_Console.Clear_Screen;
		     Center.Current_Mode := Mode_Enum'Val(Center_Menu.Process(Main_Menu)+2);
		     Ansi_Console.Clear_Screen;
		  when others =>
		     Center.Current_Mode := Mode;
	       end case;
	       --Center.Execut_Proc.Initialize;
	       --Center.Execut_Proc.Lock;
	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Wake_Up;
		  when Devel_Mode =>
		     Center.Devel_Proc.Wake_Up;
		  when Multi_Mode =>
		     Center.Multi_Proc.Wake_Up;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Wake_Up;
		  when others =>
		     null;
	       end case;
	    end Initialize;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  
		  End_Of_This_Task := False;
		  Center.Dialog_Proc.Receive(Wchar);		  
	       end Receive;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  
		  End_Of_This_Task := False;
		  Center.Devel_Proc.Receive(Wchar);
	       end Receive;
	       
	       or
	    when Center.Current_Mode = Organizer_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  
		  End_Of_This_Task := False;
		  Center.Organizer_Proc.Receive(Wchar);		  
	       end Receive;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  

		  End_Of_This_Task := False;
		  Center.Multi_Proc.Receive(Wchar);

	       end Receive;

	 or
	    when Center.Current_Mode = shell_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  
		  End_Of_This_Task := False;
		  Center.shell_Proc.Receive(Wchar);
	       end Receive;
	 or
	    when Center.Current_Mode = Execut_Mode =>
	       accept Receive (Wchar : in Wide_Character) do		  		  
		  End_Of_This_Task := False;
		  --Center.Execut_Proc.Receive(Wchar);
	       end Receive;
	       

	 or
	    accept Mode(Mode : out Mode_Enum);
	 or
	    accept Reset;
	 or
	    accept Multi_Switch (Mode : in Mode_Enum) do
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Lock;
		  when Devel_Mode =>
		     Center.Devel_Proc.Lock;
		  when Multi_Mode =>
		     Center.Multi_Proc.Lock;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Lock;
		  when Shell_Mode =>
		     Center.Shell_Proc.Lock;
		     null;
		  when Execut_Mode =>
		     --Center.Execut_Proc.Lock;
		     null;
		  when others =>		     
		     null;
	       end case;
	       Center.Prev_Mode := Center.Current_Mode;
	       Center.Current_Mode := Mode;
	       Ansi_Console.Clear_Screen;	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Wake_Up;
		  when Devel_Mode =>
		     Center.Devel_Proc.Wake_Up;
		     Center.Devel_Proc.Mode(Devel_State);
		     Center.Devel_Proc.Switch(Devel_Wrapp.Wrap_Succ(Devel_state));
		  when Multi_Mode =>
		     Center.Multi_Proc.Wake_Up;		     
		     Center.Multi_Proc.Mode(Multi_State);
		     Center.Multi_Proc.Switch(Multi_Wrapp.Wrap_Succ(Multi_State));
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Wake_Up;
		  when Shell_Mode =>
		     Center.Shell_Proc.Wake_up;
		     
		  when Execut_Mode =>
		     --Center.Execut_Proc.wake_up;
		     null;
		  when others =>		     
		     null;
	       end case;
	    end Multi_Switch;
	 or
	    accept Switch (Mode : in Mode_Enum) do
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Lock;
		  when Devel_Mode =>
		     Center.Devel_Proc.Lock;
		  when Multi_Mode =>
		     Center.Multi_Proc.Lock;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Lock;
		  when Shell_Mode =>
		     Center.Shell_Proc.Lock;
		     
		  when Execut_Mode =>
		     --Center.Execut_Proc.Lock;
		     null;
		  when others =>		     
		     null;
	       end case;
	       Center.Prev_Mode := Center.Current_Mode;
	       Center.Current_Mode := Mode;
	       Ansi_Console.Clear_Screen;	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Wake_Up;
		  when Devel_Mode =>
		     Center.Devel_Proc.Wake_Up;
		  when Multi_Mode =>
		     Center.Multi_Proc.Wake_Up;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Wake_Up;
		  when Shell_Mode =>
		     Center.Shell_Proc.Wake_up;
		     null;
		  when Execut_Mode =>
		     --Center.Execut_Proc.wake_up;
		     null;
		  when others =>		     
		     null;
	       end case;
	    end Switch;
	 or
	    accept Lock do
	       Center.Dialog_Proc.Lock;
	       Center.Devel_Proc.Lock;
	       Center.Organizer_Proc.Lock;
	       Center.Multi_Proc.Lock;		     
	    end Lock;
	 or
	    accept Wake_Up(Mode : in Mode_Enum) do
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Wake_Up;
		  when Devel_Mode =>
		     Center.Devel_Proc.Wake_Up;
		  when Multi_Mode =>
		     Center.Multi_Proc.Wake_Up;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Wake_Up;
		  when Shell_Mode =>
		     Center.Shell_Proc.Wake_up;
		     null;
		  when Execut_Mode =>
		     --Center.Execut_Proc.wake_up;
		     null;
		  when others =>		     
		     null;
	       end case;
	    end Wake_Up;
	 or
	    accept Unlock;
	 or
	    accept Set_On_Esc(Is_Escape : in Boolean) do
	       Center.On_Esc := Is_Escape;	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Set_On_Esc(Center.On_Esc);
		  when Quit_Mode =>
		     if Center.On_Esc then
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "one esc to quit");
		     else
			Draw_Text(Center.Frames(Buffer).Win, 1, 2, cyan, "two esc to quit");
		     end if;
		  when Devel_Mode =>
		     Center.Devel_Proc.Set_On_Esc(Center.On_Esc);
		  when others =>
		     null;
	       end case;	       
	    end Set_On_Esc;	    
	    
	 or
	    accept F9 do
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     
		     Center.Dialog_Proc.Lock;
		     
		  when Devel_Mode =>
		     
		     Center.Devel_Proc.Lock;
		     
		  when Multi_Mode =>
		     
		     Center.Multi_Proc.Lock;
		     
		  when Organizer_Mode =>
		     
		     Center.Organizer_Proc.Lock;
		     
		  when Shell_Mode =>
		     Center.Shell_Proc.Lock;		     
		  --  when Execut_Mode =>
		  --     --Center.Execut_Proc.Lock;
		     null;
		  when others =>		     
		     null;
	       end case;
	       
	       case Center.Current_Mode is
		  when Organizer_mode =>
		     null;
		  when others =>
		     
		     Center.Prev_Mode := Center.Current_Mode;
		     Center.Current_Mode := Organizer_Mode;
	       end case;
	       
	       Ansi_Console.Clear_Screen;	       
	       
	       Center.Organizer_Proc.Wake_Up;
	       
	       
	       Center.Organizer_Proc.New_goal;
	       
	       
	    end F9;
	       
	 or

	    accept F10 do
	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Lock;
		  when Devel_Mode =>
		     Center.Devel_Proc.Lock;
		  when Multi_Mode =>
		     Center.Multi_Proc.Lock;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Lock;
		  when Shell_Mode =>
		     Center.Shell_Proc.Lock;		     
		  --  when Execut_Mode =>
		  --     --Center.Execut_Proc.Lock;
		     null;
		  when others =>		     
		     null;
	       end case;
	       
	       case Center.Current_Mode is
		  when Organizer_mode =>
		     null;
		  when others =>
		     
		     Center.Prev_Mode := Center.Current_Mode;
		     Center.Current_Mode := Organizer_Mode;
	       end case;
	       Ansi_Console.Clear_Screen;	       
	       Center.Organizer_Proc.Wake_Up;
	       Center.Organizer_Proc.Goal_Path_Tree_View;
	       
	    end F10;
	 or
	    accept F11 do
	       
	       case Center.Current_Mode is
		  when Dialog_Mode =>
		     Center.Dialog_Proc.Lock;
		  when Devel_Mode =>
		     Center.Devel_Proc.Lock;
		  when Multi_Mode =>
		     Center.Multi_Proc.Lock;
		  when Organizer_Mode =>
		     Center.Organizer_Proc.Lock;
		  when Shell_Mode =>
		     Center.Shell_Proc.Lock;		     
		  --  when Execut_Mode =>
		  --     --Center.Execut_Proc.Lock;
		     null;
		  when others =>		     
		     null;
	       end case;
	       
	       case Center.Current_Mode is
		  when Organizer_mode =>
		     null;
		  when others =>
		     
		     Center.Prev_Mode := Center.Current_Mode;
		     Center.Current_Mode := Organizer_Mode;
	       end case;
	       Ansi_Console.Clear_Screen;	       
	       Center.Organizer_Proc.Wake_Up;
	       Center.Organizer_Proc.Add_Goal_Path;
	       
	    end F11;
      
	    
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Page_Down;
	       Center.Dialog_Proc.Page_Down;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Page_Up;
	       Center.Dialog_Proc.Page_Up;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Up_Arrow;
	       Center.Dialog_Proc.Up_Arrow;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Down_Arrow;
	       Center.Dialog_Proc.Down_Arrow;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Left_Arrow;
	       Center.Dialog_Proc.Left_Arrow;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Right_Arrow;
	       Center.Dialog_Proc.Right_Arrow;
	       
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Page_Down;
	       Center.Multi_Proc.Page_Down;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Page_Up;
	       Center.Multi_Proc.Page_Up;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Up_Arrow;
	       Center.Multi_Proc.Up_Arrow;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Down_Arrow;
	       Center.Multi_Proc.Down_Arrow;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Left_Arrow;
	       Center.Multi_Proc.Left_Arrow;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	       accept Right_Arrow;
	       Center.Multi_Proc.Right_Arrow;
	       
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Page_Down;
	       Center.Devel_Proc.Page_Down;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Page_Up;
	       Center.Devel_Proc.Page_Up;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Up_Arrow;
	       Center.Devel_Proc.Up_Arrow;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Down_Arrow;
	       Center.Devel_Proc.Down_Arrow;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Left_Arrow;
	       Center.Devel_Proc.Left_Arrow;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	       accept Right_Arrow;
	       Center.Devel_Proc.Right_Arrow;
	 
	 or

	    accept Up_Arrow;
	 or
	    accept Down_Arrow;
	 or
	    accept Left_Arrow;
	 or
	    accept Right_Arrow;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	      accept Full_Screen do
		 Center.Multi_Proc.Full_Screen;
	      end Full_Screen;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	      accept Del;
	      Center.Dialog_Proc.Del;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Begin_Of;
	       Center.Dialog_Proc.Begin_Of;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	      accept End_Of;
	      Center.Dialog_Proc.End_Of;
	 or
	    when Center.Current_Mode = Multi_Mode =>
		accept Begin_Of;
	       Center.Multi_Proc.Begin_Of;
	 or
	    when Center.Current_Mode = Multi_Mode =>
	      accept End_Of;
	      Center.Multi_Proc.End_Of;

	 or
	    when Center.Current_Mode = Dialog_Mode =>
	      accept Overwrite;
	      Center.Dialog_Proc.Overwrite;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept Begin_Of;
	       Center.Dialog_Proc.Begin_Of;
	 or
	    when Center.Current_Mode = Dialog_Mode =>
	       accept End_Of;
	      Center.Dialog_Proc.End_Of;
	 or
	    when Center.Current_Mode = Devel_Mode =>
		accept Begin_Of;
	       Center.Devel_Proc.Begin_Of;
	 or
	    when Center.Current_Mode = Devel_Mode =>
	      accept End_Of;
	      Center.Devel_Proc.End_Of;

	 or
	    when Center.Current_Mode = Dialog_Mode =>
	      accept Overwrite do
		 Center.Dialog_Proc.Overwrite;
	      end Overwrite;
	 or

	    when Center.Current_Mode = Devel_Mode =>
	       accept Overwrite do
	       Center.Devel_Proc.Overwrite;
	       end Overwrite;
	 or

	    
	      accept Del;
	      
	 or

	      accept Begin_Of;
	      
	 or
	    
	      accept End_Of;
	      
	 or
	    
	      accept Overwrite;
	 or
	 
	    accept Screen_Print;
	 or
	    accept Shutdown do
	       
	       
	       
	       W_Io.Put_Line("Halting Dialog");	     
	       Center.Dialog_Proc.Halt;
	       
	       
	       W_Io.Put_Line("Halting multi");
	       Center.Multi_Proc.Halt;
	       
	      --Center.Editor_Proc.Halt(Verax);
	       --Center.Command_Proc.Halt(Verax);
	       Center.Shell_Proc.Halt;
	       W_Io.Put_Line("Shell halted");
	       --Center.Execut_Proc.Halt;
	       --W_Io.Put_Line("Execute halted");
	       W_Io.Put_Line("Halting devel");
	       Center.Devel_Proc.Halt;
	       
	       --Center.Tasking_Proc.Halt(Verax);
	       W_Io.Put_Line("Halting Organizer");
	       Center.Organizer_Proc.Halt(End_Of_Task);
	       
	       
	       
	       
	       
	       
	       --Center.Account_Proc.Halt(Verax);
	       W_Io.Put_Line("Shutdown console...");
	    end Shutdown;
	    
	    exit;
	 or
	    delay 5.0;
	    if To_State(Center.Domain.Proof) /= Center.Domain.State then
	       case Center.Current_Mode is		  
		  when Devel_Mode =>
		     select
			Center.Devel_Proc.New_Evidence;
			Center.Devel_Proc.New_Ranking(El.Top_Classes.all);
		     or
			delay 0.05;
		     end select;
		  when Dialog_Mode =>
		     null;
		  when others =>
		     null;
	       end case;
	       
	    else
	       case Center.Current_Mode is
		  when Devel_Mode =>
		     select
			Center.Devel_Proc.New_Evidence;
			Center.Devel_Proc.New_Ranking(El.Top_Classes.all);
		     or
			delay 0.05;
		     end select;
		  when others =>
		     null;
	       end case;
	       
	    end if;
	    
	 end select;
	 
      end loop;            
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Console halted");
      Center.Global_Lock.Release;
   end Console_Process;
   
      
   ----------------------------------------------------
   --                   Auth mode                    --
   
   procedure Users_Read (Users : in out User_Set; filename : in string) is
      
      File : W_Io.File_Type;
      Iter : Natural := 0;
      
      
   begin
      W_Io.Open(File, W_Io.In_File, filename, Form => "WCEM=8");
      while not W_Io.End_Of_File(File) loop
	 --Player_Type := Player_Enum'Value(Handling.To_String(W_Io.Get_Line(File)));
   	 declare   	    
	    user : User_Record(Player_Type, Max_Players);
   	 begin
   	    User.Logname := new String ' (Handling.To_String(W_Io.Get_Line(File)));
   	    User.Password := new String '(Handling.To_String(W_Io.Get_Line(File)));
   	    User.Identity.Full_Name := new Wide_String ' (W_Io.Get_Line(File));
   	    User.Identity.address := new Wide_String ' (W_Io.Get_Line(File));
   	    User.Identity.City_Code := new Wide_String ' (W_Io.Get_Line(File));
   	    User.Identity.City_Name := new Wide_String ' (W_Io.Get_Line(File));
   	    User.Identity.email := new Wide_String ' (W_Io.Get_Line(File));
	    User.Identity.Web_Addr := new Wide_String ' (W_Io.Get_Line(File));
   	    Users.set(Iter + 1) := new User_Record ' (User);
	    Iter := Iter + 1;
   	 end;
      end loop;
      Users.User_Last := Iter;
      W_Io.Close(File);
   end Users_Read;
   
   procedure Users_write (Users : in out User_Set; filename: in String) is
      
      File : W_Io.File_Type;
      
   begin
      
      W_Io.Create(File, W_Io.Out_File, filename, Form => "WCEM=8");
      
      for Iter in 1..Users.User_Last loop
   	 declare
   	    User : constant User_Record := Users.Set(Iter).all;
   	 begin
	    
   	    if User.Logname /= null then
	       
   	       W_Io.Put_Line(File, Handling.To_Wide_String(User.Logname.all));
	       
   	       W_Io.Put_Line(File, Handling.To_Wide_String(User.Password.all));

   	       W_Io.Put_Line(File, User.Identity.Full_Name.all);

   	       W_Io.Put_Line(File, User.Identity.Address.all);

   	       W_Io.Put_Line(File, User.Identity.City_Code.all);

   	       W_Io.Put_Line(File, User.Identity.City_Name.all);

   	       W_Io.Put_Line(File, User.Identity.Email.all);

	       if User.Identity.Web_Addr /= null then

		  W_Io.Put_Line(File, User.Identity.Web_Addr.all);
	       else
		  W_Io.Put_Line(File, "");
	       end if;
   	    else
	       
   	       exit;
   	    end if;
	    
   	 end;		     
      end loop;
      W_Io.Close(File);
   end Users_write;
   
   
   
   function New_Wide_String (Title : in Wide_String;
   			     Label : in Wide_String;
   			     Length : in Positive;
   			     Line, Col : in Positive;
   			     Print : in Boolean) return Wide_String is
      
      Line_Win : Window_Type := 
   	(Line, Col, 5, Label'Length+3+Length+4, Single_Line_Frame, Cyan, Yellow);
      Wline     : Wide_String_Access := new Wide_String ' ("");
      Buff     : Wide_String_Access;
      Wchar    : Wide_Character := Wide_Character'Val(32);
   begin
      loop
   	 Draw_Window(line_Win);
   	 Draw(line_Win, 1, Line_Win.W/2-Title'length/2, yellow, Title);
   	 Draw(line_Win, 2, 2, cyan, Label);   	 
   	 if Wline /= null and then
   	   Wline'Length > 0 then
   	    if Print then
   	       Draw(line_Win, 2, 2+Label'Length, white, Wline.all);
   	    else
   	       Draw(line_Win, 2, 2+Label'Length, red, ((Wline'Length * '*')));
   	    end if;
   	 end if;
   	 W_Io.Get_Immediate(Wchar);
   	 if Is_control(Wchar) then	    
   	    case Wchar is
   	       when Wide_Character'Val(10) =>
   		  exit;
   	       when Wide_Character'Val(12) =>
   		  null;
   	       when Wide_Character'Val(27) =>
   		  raise W_Io.End_Error;
   	       when Wide_Character'Val(127) =>
   		  if Wline /= null and then
   		    Wline'Length > 0 then
   		     Buff := new Wide_String ' (Wline(Wline'First..Wline'Last-1));		     
   		     Wide_Free(Wline);
   		     Wline := new Wide_String ' (Buff.all);
   		     Wide_Free(Buff);		     
   		  else
   		     W_Io.Put(Wide_Character'Val(7));
   		  end if;
   	       when others =>
   		  null;
   	    end case;

   	 else
   	    if Wline /= null and then
   	      Wline'Length >= Length then
   	       W_Io.Put(Wide_Character'Val(7));
   	    else
   	       Buff := new Wide_String ' (Wline.all & Wchar);
   	       Wide_Free(Wline);
   	       Wline := new Wide_String ' (Buff.all);
   	       Wide_Free(Buff);
   	    end if;
   	 end if;
      end loop;
      return Wline.all;
   end New_Wide_String;
   
   
   
   function New_Auth return Auth_Record is
      Auth : Auth_Record;
      Logname : access String;
   begin
      
      
      Logname := Getenv("LOGNAME");
      if Logname /= null then
   	 Auth.Logname := new String ' (Logname.all);
      else
   	 Auth.Logname := 
   	   new String ' 
   	   (Handling.To_String(New_Wide_String("Authentification", "Logname : ", 24, Lines/2-3, Cols/2-19, True)));
      end if;
      
      Auth.password :=
   	Sha1.Digest
   	(Handling.To_String
   	   (New_Wide_String
   	      ("Authentification", "Password : ", 25, Lines/2-3, Cols/2-20,
   	       false)));
      return Auth;
   end New_Auth;
   

   
   use Os_Lib;
   
   task body Auth_Process is
      
      
      
      
      Denied_Access : Window_Type :=
   	(Lines/2-3, Cols/2-10, 5, 20, Single_Line_Frame, cyan, White);
      
      Id_Win        : Window_Type :=
      	(1, (Cols/4)*2+1, 8, Cols/2-1, Single_Line_Frame, Yellow, Green);
      
      Correct       : Wide_String_Access;
      User          : User_Record(Human_Player, 4);
   begin
      
      loop
   	 select
   	    accept Halt do
   	       W_Io.Put_Line("Halting Author...");
   	    end Halt;
   	    exit;
	    
   	 or
   	    accept Auth do
	       
   	       
   	       declare		  
   		  
		  Identity      : Identity_Record;
   		  Done          : Boolean := False;
   	       begin

   		  while not Done loop
   		     loop
   			begin
			   
   			   Center.User_Auth := New_Auth;
			   
   			   exit;
   			exception
   			   when others =>
   			      Ansi_Console.Clear_Screen;
   			      Draw_Window(Denied_Access);
   			      Draw(Denied_Access, 2, 3, Red, "Access denied");
   			      delay 2.0;
   			      Ansi_Console.Clear_Screen;
   			end;
   		     end loop;
		     Ansi_Console.Clear_Screen;
		     
   		     case System.Os_Constants.Target_Os is
   			when System.Os_Constants.Windows =>	    
   			   Center.User_path := Getenv("HOMEPATH");
   			when others =>
   			   Center.User_path := Getenv("HOME");
   		     end case;

   		     if Os_Lib.Is_Regular_File(Center.User_Path.all & Directory_separator & "users.txt") then
			
   			Users_Read(Center.Visitors.all, Center.User_Path.all & Directory_separator & "users.txt");
			if Center.Visitors /= null and then
			  Center.Visitors.User_Last /= 0 then
			   
			   for Iter in 1..Center.Visitors.User_Last loop
			      declare
				 User : User_Access := Center.Visitors.Set(Iter);
			      begin
				 
				 if User.Logname /= null then
				    if User.Logname.all = Center.User_Auth.Logname.all and then
				      User.Password.all = Center.User_Auth.password then
				       Identity := User.Identity;
				       Center.User := User;
				       Center.User_Ready := True;
				       Done := True;
				       Center.User_Ready := True;
				       Center.Full_Shell_Filename := new String ' 
					 (Center.User_Path.all & Directory_separator & Shell_file);

				       exit;
				    end if;
				 else
				    exit;
				 end if;
			      end;
			   end loop;	 
			   
			end if;
		     end if;
		     
   		     if not Done then
   			while not Done loop
   			   begin
   			      Ansi_Console.Clear_Screen;
   			      if Center.User_Auth.Password = 
   				Sha1.Digest
   				(Handling.To_String
   				   (New_Wide_String("Verify password", 
   						    "Password : ", 
   						    25, 
   						    Lines/2-3, 
   						    Cols/2-20, 
   						    false))) then
				 user.Logname := new String ' (Center.User_Auth.Logname.all);
				 User.password := new String ' (Center.User_Auth.Password);
   				 loop
   				    begin
   				       Ansi_Console.Clear_Screen;
   				       Identity.Full_Name := 
   					 new Wide_String ' 
   					 (New_Wide_String("Identification", 
   							  "Full name : ", 
   							  25, 
   							  Lines/2-3, 
   							  Cols/2-19, 
   							  True));
   				       Ansi_Console.Clear_Screen;
   				       Identity.Address := 
   					 new Wide_String ' 
   					 (New_Wide_String("Identification", 
   							  "Address : ", 48, 
   							  Lines/2-3, 
   							  Cols/2-30, 
   							  True));
   				       Ansi_Console.Clear_Screen;
   				       Identity.City_Code := 
   					 new Wide_String ' 
   					 (New_Wide_String
   					    ("Identification", 
   					     "City code : ", 
   					     10, 
   					     Lines/2-3, 
   					     Cols/2-11, 
   					     True));
   				       Ansi_Console.Clear_Screen;
   				       Identity.City_Name := 
   					 new Wide_String ' 
   					 (New_Wide_String("Identification", 
   							  "City : ", 
   							  25, 
   							  Lines/2-3, 
   							  Cols/2-16, 
   							  True));
   				       Ansi_Console.Clear_Screen;
   				       Identity.email := 
   					 new Wide_String ' 
   					 (New_Wide_String("Identification", 
   							  "Email address : ", 
   							  49, 
   							     Lines/2-3, 
   							  Cols/2-37, 
   							  True));
   				    exception
   				       when W_Io.End_Error =>
   					  raise;
   				    end;
   				    Ansi_Console.Clear_Screen;
   				    Draw_Window(Id_Win);
   				    Draw(Id_Win, 1, Id_Win.w/2-5, Magenta, "Informations");
   				    Draw(Id_Win, 2, 2, White, "Logname     : " & 
   					   Handling.To_Wide_String(User.Logname.all));
   				    Draw(Id_Win, 3, 2, White, "Full name   : " & 
   					   Identity.Full_Name.all);
   				    Draw(Id_Win, 4, 2, White, "Address     : " & 
   					   Identity.Address.all);
   				    Draw(Id_Win, 5, 2, White, "Code & City : " & 
   					   Identity.City_name.all & ' ' & Identity.City_Code.all);
   				    Draw(Id_Win, 6, 2, White, "Email       : " & 
   					   Identity.Email.all);
   				    Correct := new Wide_String ' 
   				      (New_Wide_String
   					 ("Verification", 
   					  "This informations are correct (y/o or n) : ", 
   					  3, Lines/2-3, Cols/2-25, True));

   				    if Correct /= null and then				
   				      Correct'Length = 1 then

   				       case Correct(1) is
   					  when 'y' | 'o' =>

					     User.Identity := Identity;
					     User.Logname := new String ' (Center.User_Auth.Logname.all);
					     User.Password := new String '(Center.User_Auth.Password);
					     if Center.Visitors = null then
						Center.Visitors := new User_Set(4);
					     end if;
					     
					     Center.Visitors.Set(Center.Visitors.User_Last + 1) := new User_Record ' (User);
						
					     Center.Visitors.User_Last := Center.Visitors.User_Last + 1;
					     
					     Center.User := Center.Visitors.Set(Center.Visitors.User_Last);
					     Center.User_Ready := True;
   					     
   					     Done := True;
					     
   					  when others =>
   					     null;
   				       end case;
   				       if Done then
   					  Correct := new Wide_String ' 
   					    (New_Wide_String
   					       ("Restering", 
   						"Do you want registerring this informations (y/o or n) : ", 
   						3, Lines/2-3, Cols/2-25, True));

   					  if Correct /= null and then				
   					    Correct'Length = 1 then

   					     case Correct(1) is
   						when 'y' | 'o' =>
						   if Center.Visitors = null then
						      raise Spec_Error;
						   end if;
      
   						   Users_Write
						     (Center.Visitors.all, Center.User_Path.all & 
							Directory_Separator & "users.txt");
						   
   						when others =>
   						   null;
   					     end case;
   					  end if;
   				       end if;
   				       if Done then
					  
   					  exit;
   				       end if;
   				    end if;
   				 end loop;
				 
   			      else
   				 Ansi_Console.Clear_Screen;
   				 Draw_Window(Denied_Access);
   				 Draw(Denied_Access, 2, 3, Red, "Access denied");
				 
   				 delay 2.0;
   				 Ansi_Console.Clear_Screen;
   				 exit;
   			      end if;
   			   exception
   			      when W_Io.End_Error =>
   				 Ansi_Console.Clear_Screen;
   				 exit;
   			   end;
   			end loop;

   		     end if;

   		  end loop;
   	       end;

   	    end Auth;
   	 end select;
      end loop;      
      W_Io.Put_Line("Author halted");
   end Auth_Process;

   
   

   task body Work_Process is
   begin
      loop
	 select
	    accept Halt do
	       W_Io.Put_Line("Halting Work process...");
	    end Halt;
	    exit;
	 or	    
	    accept Initialize;	    
	    accept wait;
	    
	 end select;
      end loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Working halted");
      Center.Global_Lock.Release;
   end Work_Process;

   task body Center_Process is
      Available : Boolean := False;
      End_Of_Line : Boolean := False;
      Wchar : Wide_Character := Wide_Character ' Val (32);
      Wchar_Buff : Wide_Character := Wide_Character ' Val (32);
      Esc_Wchar : Wide_Character := Wide_Character ' Val (32);
      End_Of_Task : Boolean := False;
      End_Of_Program : Boolean := False;
      Shutdown : Boolean := False;
      
      
   begin
      loop
	 select
	    accept Initialize (Mode : Mode_Enum; User_Text : in Wide_String) do
	       
	       Center.Console_Proc.Initialize(Mode, User_Text);
	    end Initialize;
	    exit;
	 or
	    accept Halt do	    
	       
	       Center.Console_Proc.Lock;
	       
	       Center.Config_Proc.Halt(End_Of_Program);
	       
	       if End_Of_Program then
		  
		  
		  
		  Center.Work_Proc.Halt;
		  
		  Center.Console_Proc.Shutdown;
		  
		  Center.Auth_Proc.Halt;
		  
		  Shutdown := True;
		  W_Io.Put_Line("Halting Center...");
	       end if;
	    end Halt;
	    if End_Of_Program then
	       exit ;
	    else
	       Center.Console_Proc.Wake_Up(Center.Current_Mode);
	    end if;
	 end select;
      end loop;
  Center_Loop :
      while not Shutdown loop
	 loop
	    select
	       accept Initialize (Mode : Mode_Enum; User_Text : in Wide_String) do
		  
		  Center.Console_Proc.Initialize(Mode, User_Text);
	       end Initialize;
	    or
	       accept Halt do	       
		  
		  Center.Config_Proc.Halt(End_Of_Program);

		  if End_Of_Program then
		     
		     W_Io.Put_Line("Haltin Astar process...");
		     Center.Astar_Proc.Halt;
	       
	       
		     
		     Center.Work_Proc.Halt;
		     
		     Center.Console_Proc.Shutdown;
		     
		     Center.Auth_Proc.Halt;
		     
		     Shutdown := True;
		     W_Io.Put_Line("Halting Center...");
		  end if;
	       end Halt;
	       if End_Of_Program then
		  exit Center_Loop;
	       end if;
	    or
	       delay 0.02;
	       
	       if Center.Logout then
		     Center.Multi_Proc.Lock;
		     for I in 1..1 loop
			Center.Console_Proc.Halt(End_Of_Program);
			if End_Of_Program then		  
			   exit;
			end if;
			
			case Center.Current_Mode is
			   when Null_Mode =>
			      
			      Center.Console_Proc.Set_On_Esc(True);
			      Center.Console_Proc.Switch(Quit_Mode);
			   when others =>
			      if not End_Of_Program then
				 Center.Console_Proc.Switch(Null_Mode);
				 Center.Console_Proc.Set_On_Esc(False);			
			      else
				 Center.Console_Proc.Switch(Quit_mode);
				 Center.Console_Proc.Set_On_Esc(False);
			      end if;
			end case;
		     end loop;
	       end if;
	       
	       case Center.Terminal is			 
		  --Text_Io.Put_Line("-- Si l'interface n'est en mode graphic :");
		  when Linux =>
		     begin
			
			Center.Console_Proc.Halt(End_Of_Task);
			if not End_Of_Program then
			   if End_Of_Task then
			      
			      exit;
			      
			   end if;
			end if;
			
			
			--Center.Console_Proc.Mode(Center.Current_Mode);
			
			W_Io.Get_Immediate(Wchar);		     	
			case Wchar is
		     	   when Wide_Character'Val(27) =>
			      Center.Console_Proc.Set_On_Esc(True);
		     	      
			      W_Io.Get_Immediate(Wchar);
			      
		     	      case Wchar is
		     		 when Wide_Character'Val(27) =>                             
		     		    if Center.Full_Screen then
		     		       Center.Console_Proc.Reset;
		     		    else                               
		     		       case Center.Current_Mode is                                                            
		     			  when Devel_Mode..Help_Mode =>
		     			     Center.Console_Proc.Reset;
		     			  when Null_Mode =>
					     
		     			     Center.Console_Proc.Lock;
		     			     Center.Console_Proc.Set_On_Esc(False);
		     			     Center.Console_Proc.Switch(Quit_Mode);                               
		     			     End_Of_Program := True;
		     			  when Auth_Mode =>
		     			     if End_Of_Program then                                             
		     				Center.Console_Proc.Switch(Quit_Mode);                                          
		     			     else
		     				Center.Console_Proc.Set_On_Esc(False);
		     				Center.Console_Proc.Switch(Quit_Mode);
		     				End_Of_Program := True;
		     			     end if;
		     			  when others =>
		     			     if End_Of_Program then
		     				Center.Console_Proc.Switch(Null_Mode);
		     				Center.Console_Proc.Set_On_Esc(False);
		     				End_Of_Program := False;
						
		     			     else
		     				Center.Console_Proc.Switch(Quit_mode);
		     				Center.Console_Proc.Set_On_Esc(False);
		     				End_Of_Program := True;
		     			     end if;
		     		       end case;       
				       
		     		    end if;				 
				    
		     		 when others =>
				    
		     		    case Center.Current_Mode is
		     		       when Quit_Mode =>
		     			  if End_Of_Program then         
					     
		     			     Center.Console_Proc.Switch(Quit_Mode);                                  
		     			  end if;
		     		       when others =>
					  
		     			  Center.Console_Proc.Set_On_Esc(False);
		     			  End_Of_Program := False;
		     		    end case;
		     	      end case;                     
			      
		     	      case Wchar is
		     		 when Wide_Character'Val(91) =>                           
		     		    W_Io.Get_Immediate(Wchar);
		     		    case Center.Current_Mode is
		     		       when Auth_Mode =>
		     			  null;
		     		       when others =>
		     			  Center.Console_Proc.Set_On_Esc(False);
		     			  case Wchar is
		     			     when Wide_Character'Val(49) =>
		     				W_Io.Get_Immediate(Wchar);
		     				case Wchar is
		     				   when Wide_Character'Val(55) =>
		     				      null;--  W_Io.Put_Line(" -- F6");

		     				      Center.Console_Proc.Switch(Execut_Mode);
		     				      Center.Current_Mode := Execut_Mode;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(56) =>
		     				      null;--  W_Io.Put_Line(" -- F7");
						      Center.Console_Proc.Switch(Shell_Mode);
		     				      Center.Current_Mode := Shell_Mode;
						      
						      --Center.Console_Proc.Switch(Execut_mode);
						      --   Center.Current_Mode := Execut_Mode;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(57) =>
		     				      --W_Io.Put_Line(" -- F8");
						      case Center.Current_Mode is
							 when Devel_Mode =>
							    Center.Console_Proc.Multi_Switch(Devel_Mode);
							 when Multi_Mode =>
							    Center.Console_Proc.Multi_Switch(Multi_Mode);
							 when others =>
							    null;
						      end case;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(126) =>                     
		     				      null;--  W_Io.Put_Line(" -- begin");
		     				      Center.Console_Proc.Begin_Of;
		     				   when others =>
		     				      null;
		     				end case;                      
		     			     when Wide_Character'Val(50) =>                    
		     				W_Io.Get_Immediate(Wchar);
		     				case Wchar is
		     				   when Wide_Character'Val(48) =>
		     				      null;--  W_Io.Put_Line(" -- F9");
						      Center.Console_Proc.F9;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(49) =>
		     				      null;--  W_Io.Put_Line(" -- F10");
						      Center.Console_Proc.F10;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(51) =>
		     				      null;--  W_Io.Put_Line(" -- F11");
						      Center.Console_Proc.F11;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(52) =>
		     				      null;--  W_Io.Put_Line(" -- F12");
		     				      Center.Console_Proc.Full_Screen;
		     				      W_Io.Get_Immediate(Wchar);
		     				   when Wide_Character'Val(53) =>                            
		     				      null;--  W_Io.Put_Line(" -- Maj+F1");
		     				   when Wide_Character'Val(54) =>                            
		     				      null;--  W_Io.Put_Line(" -- Maj+F2");
		     				   when Wide_Character'Val(56) =>                            
		     				      null;--  W_Io.Put_Line(" -- Maj+F3");
		     				   when Wide_Character'Val(57) =>
		     				      null;--  W_Io.Put_Line(" -- Maj+F4");
		     				   when Wide_Character'Val(126) =>
		     				      null;--  W_Io.Put_Line(" -- insert");
		     				      Center.Console_Proc.Overwrite;
		     				   when others =>
		     				      null;
		     				end case;                               
		     			     when Wide_Character'Val(51) =>                    
		     				null;--  W_Io.Put_Line(" -- del");
		     				W_Io.Get_Immediate(Wchar);
		     				Center.Console_Proc.Del;
		     			     when Wide_Character'Val(52) =>
		     				null;--  W_Io.Put_Line(" -- end");
		     				W_Io.Get_Immediate(Wchar);
		     				Center.Console_Proc.End_Of;
		     			     when Wide_Character'Val(53) =>
		     				null;--  W_Io.Put_Line(" -- page up");
		     				Center.Console_Proc.Page_Up;
		     				W_Io.Get_Immediate(Wchar);
		     			     when Wide_Character'Val(54) =>
		     				null;--  W_Io.Put_Line(" -- page down");
		     				Center.Console_Proc.Page_Down;
		     				W_Io.Get_Immediate(Wchar);
		     			     when Wide_Character'Val(65) =>
		     				null;--  W_Io.Put_Line(" -- up");
		     				Center.Console_Proc.Up_Arrow;
		     			     when Wide_Character'Val(66) =>
		     				null;--  W_Io.Put_Line(" -- down");
		     				Center.Console_Proc.Down_Arrow;
		     			     when Wide_Character'Val(67) =>
		     				null;--  W_Io.Put_Line(" -- right");
		     				Center.Console_Proc.Right_Arrow;
		     			     when Wide_Character'Val(68) =>
		     				null;--  W_Io.Put_Line(" -- left");
		     				Center.Console_Proc.Left_Arrow;
		     			     when Wide_Character'Val(91) =>
		     				W_Io.Get_Immediate(Wchar);
		     				case Wchar is
		     				   when Wide_Character'Val(65) =>
		     				      null;--  W_Io.Put_Line(" -- F1");
		     				      Center.Console_Proc.Switch(Dialog_Mode);
		     				      Center.Current_Mode := Dialog_Mode;
		     				   when Wide_Character'Val(66) =>
		     				      null;--  W_Io.Put_Line(" -- F2");
		     				      Center.Console_Proc.Switch(Devel_Mode);
		     				      Center.Current_Mode := Devel_Mode;
		     				   when Wide_Character'Val(67) =>
		     				      null;--  W_Io.Put_Line(" -- F3");
		     				      Center.Console_Proc.Switch(Organizer_Mode);
		     				      Center.Current_Mode := Organizer_Mode;
		     				   when Wide_Character'Val(68) =>
		     				      null;--  W_Io.Put_Line(" -- F4");
		     				      Center.Console_Proc.Switch(Multi_Mode);
		     				      Center.Current_Mode := Multi_Mode;
		     				   when Wide_Character'Val(69) =>
		     				      null;--  W_Io.Put_Line(" -- F5");
		     				      
		     				   when others =>
		     				      null;
		     				end case;
						
		     			     when others =>
						
		     				--Center.Console_Proc.Reset;            
						
		     				End_Of_Program := False;
		     				--Center.Console_Proc.Unlock;
		     				Center.Console_Proc.Receive(Wchar);
						
		     			  end case;
		     		    end case;
		     		 when others =>
		     		    Center.Console_Proc.Set_On_Esc(False);
		     		    --Center.Console_Proc.Unlock; 
		     	      end case;              
			      
		     	   when Wide_Character'Val(197) =>
		     	      W_Io.Get_Immediate(Wchar);
		     	   when others =>                    
			      
		     	      case Center.Current_Mode is
		     		 when Quit_Mode =>
				    
		     		    Center.Console_Proc.Switch(Center.Prev_mode);
				    
		     		 when others =>
				    
		     		    End_Of_Program := False;                        
			   
		     		    Center.Console_Proc.Receive(Wchar);
				    
		     	      end case;
		     	end case;

			
			   

		     exception
			
		     	-- A la saisie de Ctrl+D l'exception End_Error est levée.
			
		     	when W_Io.End_Error =>
		     	   null;		     
		     end;
		     
		  when fvwm1 =>
		     
		     begin
			
			Center.Console_Proc.Halt(End_Of_Task);
			if not End_Of_Program then
			   if End_Of_Task then
			      
			      exit;
			      
			   end if;
			end if;
			
			
			W_Io.Get_Immediate(Wchar);
			   
			case Wchar is
		     	   when Wide_Character'Val(27) =>
			      Center.Console_Proc.Set_On_Esc(True);		     	      
			      W_Io.Get_Immediate(Wchar);			      
		     	      case Wchar is
		     		 when Wide_Character'Val(27) =>                             
		     		    if Center.Full_Screen then
		     		       Center.Console_Proc.Reset;
		     		    else                               
		     		       case Center.Current_Mode is                                                            
		     			  when Devel_Mode..Help_Mode =>
		     			     Center.Console_Proc.Reset;
		     			  when Null_Mode =>
					     
		     			     Center.Console_Proc.Lock;
		     			     Center.Console_Proc.Set_On_Esc(False);
		     			     Center.Console_Proc.Switch(Quit_Mode);                               
		     			     End_Of_Program := True;
		     			  when Auth_Mode =>
		     			     if End_Of_Program then                                             
		     				Center.Console_Proc.Switch(Quit_Mode);                                          
		     			     else
		     				Center.Console_Proc.Set_On_Esc(False);
		     				Center.Console_Proc.Switch(Quit_Mode);
		     				End_Of_Program := True;
		     			     end if;
		     			  when others =>
		     			     if End_Of_Program then
		     				Center.Console_Proc.Switch(Null_Mode);
		     				Center.Console_Proc.Set_On_Esc(False);
		     				End_Of_Program := False;
						
		     			     else
		     				Center.Console_Proc.Switch(Quit_mode);
		     				Center.Console_Proc.Set_On_Esc(False);
		     				End_Of_Program := True;
		     			     end if;
				       end case;
				    end if;
				    
				 when Wide_Character'Val(79) =>
				    W_Io.Get_Immediate(Wchar);
				    case Wchar is
				       when Wide_Character'Val(80) =>
					  null;--  W_Io.Put_Line(" -- F1");
					  Center.Console_Proc.Switch(Dialog_Mode);
					  Center.Current_Mode := Dialog_Mode;
				       when Wide_Character'Val(81) =>
					  null;--  W_Io.Put_Line(" -- F2");
					  Center.Console_Proc.Switch(Devel_Mode);
					  Center.Current_Mode := Devel_Mode;
				       when Wide_Character'Val(82) =>
					  null;--  W_Io.Put_Line(" -- F3");
					  Center.Console_Proc.Switch(Organizer_Mode);
					  Center.Current_Mode := Organizer_Mode;
				       when Wide_Character'Val(83) =>
					  null;--  W_Io.Put_Line(" -- F4");
					  Center.Console_Proc.Switch(Multi_Mode);
					  Center.Current_Mode := Multi_Mode;				       
				       when others =>
					  null;
				    end case;
				    
				 when Wide_Character'Val(91) =>				    
				    
				    W_Io.Get_Immediate(Wchar);
				    
				    case Wchar is
				       when Wide_Character'Val(53) =>					  
					  null;--  W_Io.Put_Line(" -- page up");
					  Center.Console_Proc.Page_Up;
					  W_Io.Get_Immediate(Wchar);
				       when Wide_Character'Val(54) =>
					  null;--  W_Io.Put_Line(" -- page down");
					  Center.Console_Proc.Page_Down;
					  W_Io.Get_Immediate(Wchar);
				       when Wide_Character'Val(65) =>
					  null;--  W_Io.Put_Line(" -- up");
					  Center.Console_Proc.Up_Arrow;
					  
				       when Wide_Character'Val(66) =>
					  null;--  W_Io.Put_Line(" -- down");
					  Center.Console_Proc.Down_Arrow;
					  
				       when Wide_Character'Val(67) =>
					  null;--  W_Io.Put_Line(" -- right");
					  Center.Console_Proc.Right_Arrow;
					  
				       when Wide_Character'Val(68) =>
					  null;--  W_Io.Put_Line(" -- left");
					  Center.Console_Proc.Left_Arrow;
					  
				       when Wide_Character'Val(51) =>                    
					  
					  null;--  W_Io.Put_Line(" -- del");
					  W_Io.Get_Immediate(Wchar);
					  Center.Console_Proc.Del;
				       when Wide_Character'Val(70) =>
					  null;--  W_Io.Put_Line(" -- end");
					  
					  Center.Console_Proc.End_Of;
				       when Wide_Character'Val(72) =>
					  null;--  W_Io.Put_Line(" -- begin");
					  Center.Console_Proc.Begin_Of;				       
				       when Wide_Character'Val(49) =>
					  
					  W_Io.Get_Immediate(Wchar);
					  case Wchar is
					     when Wide_Character'Val(53) =>
						-- F5
						
						W_Io.Get_Immediate(Wchar);
					     when Wide_Character'Val(55) =>
						null;--  W_Io.Put_Line(" -- F6");
						
						Center.Console_Proc.Switch(Execut_Mode);
						Center.Current_Mode := Execut_Mode;
						W_Io.Get_Immediate(Wchar);
					     when Wide_Character'Val(56) =>
						null;--  W_Io.Put_Line(" -- F7");
						Center.Console_Proc.Switch(Shell_Mode);
						Center.Current_Mode := Shell_Mode;
						     --Center.Console_Proc.Switch(Execut_mode);
						     --   Center.Current_Mode := Execut_Mode;
						W_Io.Get_Immediate(Wchar);
					     when Wide_Character'Val(57) =>
						--W_Io.Put_Line(" -- F8");
						case Center.Current_Mode is
						   when Devel_Mode =>
						      Center.Console_Proc.Multi_Switch(Devel_Mode);
						   when Multi_Mode =>
						      Center.Console_Proc.Multi_Switch(Multi_Mode);
						   when others =>
						      null;
						end case;
						W_Io.Get_Immediate(Wchar);
						
					     when others =>
						null;
					  end case;                 
				       when Wide_Character'Val(50) =>                    
					  null;--  W_Io.Put_Line(" -- insert");
					  Center.Console_Proc.Overwrite;
					  
					  W_Io.Get_Immediate(Wchar);
					  case Wchar is
					     
					     when Wide_Character'Val(48) =>
						null;--  W_Io.Put_Line(" -- F9");
						
						Center.Console_Proc.F9;
						W_Io.Get_Immediate(Wchar);
					     when Wide_Character'Val(49) =>
						null;--  W_Io.Put_Line(" -- F10");
						
						Center.Console_Proc.F9;
						W_Io.Get_Immediate(Wchar);
					     when Wide_Character'Val(51) =>
						null;--  W_Io.Put_Line(" -- F11");
						
						Center.Console_Proc.F11;
						W_Io.Get_Immediate(Wchar);
						
					     when Wide_Character'Val(50) =>
						null;--                        inst
						Center.Console_Proc.Overwrite;
						
						W_Io.Get_Immediate(Wchar);
						
					     when Wide_Character'Val(52) =>
						null;--  W_Io.Put_Line(" -- F12");
						
						Center.Console_Proc.Full_Screen;
						
						W_Io.Get_Immediate(Wchar);
						--  when Wide_Character'Val(53) =>                            
						--     null;--  W_Io.Put_Line(" -- Maj+F1");
						--     W_Io.Get_Immediate(Wchar);
						--  when Wide_Character'Val(54) =>                            
						--     null;--  W_Io.Put_Line(" -- Maj+F2");
						--     W_Io.Get_Immediate(Wchar);
						--  when Wide_Character'Val(56) =>                            
						--     null;--  W_Io.Put_Line(" -- Maj+F3");
						--     W_Io.Get_Immediate(Wchar);
						--  when Wide_Character'Val(57) =>
						--     null;--  W_Io.Put_Line(" -- Maj+F4");
						--     W_Io.Get_Immediate(Wchar);							 
					     when others =>
						null;
					  end case;
					  
				       when others =>	
					  
					  W_Io.Get_Immediate(Wchar);
					  End_Of_Program := False;
					  
				    end case;
				    
				    
				 when Wide_Character'Val(197) =>
				    W_Io.Get_Immediate(Wchar);
				 when others =>
				    case Center.Current_Mode is
				       when Quit_Mode =>
					  
					  Center.Console_Proc.Switch(Center.Prev_mode);
					  
				       when others =>
					  
					  End_Of_Program := False;
					  
				    end case;
				    
				    End_Of_Program := False;
				    
				    Center.Console_Proc.Set_On_Esc(False);
			      end case;
			   
			   when others =>
			      
			      case Center.Current_Mode is
				 when Quit_Mode =>
				    if End_Of_Program then         
				       
				       Center.Console_Proc.Switch(Quit_Mode);                                  
				    end if;
				 when others =>
				    

				    
				       Center.Console_Proc.Receive(Wchar);
				       End_Of_Program := False;
				       Center.Console_Proc.Set_On_Esc(False);
			      end case;
			      
			end case;
			
			   

		     exception
			
		     	-- A la saisie de Ctrl+D l'exception End_Error est levée.
			
		     	when W_Io.End_Error =>
		     	   null;		     
		     end;
		     
		     
		  when Uncknow =>
		     null;
		  when Ms =>
		     null;
	       end case;
	    end select;	    
	 end loop;	 
	 accept Restart;	 
	 End_Of_Task := False;
	 End_Of_Program := False;
      end loop Center_Loop;
      Center.Global_Lock.Seize;
      W_Io.Put_Line("Center halted.");      
      Center.Global_Lock.Release;
   end Center_Process;
   

   
   procedure Devel_Draw_User (User : in User_access; Win : in Window_Record; Ready : in Boolean) is
      
   begin
      Draw_Text(Win, 1, 1, Yellow, "Loganme : ");
      if User /= null and then User.Logname /= null then
	 Draw_Text(Win, 1, 13, White, Handling.To_Wide_String(User.Logname.all));
      end if;
      Draw_Text(Win, 1, 13+Cols/8, Yellow, "Email : ");
      if User /= null and then User.Identity.Email /= null then
	 Draw_Text(Win, 1, 13+Cols/8+11, White, User.Identity.Email.all);
      end if;
      Draw_Text(Win, 1, 13+Cols/8*2+11, Yellow, "Alias nb : ");
      if User /= null then
	 Draw_Text(Win, 1, 13+Cols/8*2+23, White, Handling.To_Wide_String(Natural'Image(User.Alias_Set.Index)));
      end if;
      Draw_Text(Win, 2, 1, Yellow, "Home path : ");
      if User /= null and then User.Home_path /= null then
	 Draw_Text(Win, 2, 13, white, Handling.To_Wide_String(User.Home_Path.all));
      end if;
      Draw_Text(Win, 2, 13+Cols/8, Yellow, "Web addr : " );
      if User /= null and then User.Identity.Web_addr /= null then
	 Draw_Text(Win, 2, 13+Cols/8+11, white, User.Identity.Web_Addr.all);
      end if;      
      Draw_Text(Win, 2, 13+Cols/8*2+11, Yellow, "Var nb : ");
      if User /= null then
	 Draw_Text(Win, 2, 13+Cols/8*2+23, white, Handling.To_Wide_String(Natural'Image(User.Var_Set.Index)));
      end if;
      
   end Devel_Draw_User;
   
   
   procedure Devel_Draw_Docume (Docume : in Docume_Access; Win : in Window_Record; Initialized : in Boolean) is
   begin
      Draw_Text(Win, 1, 1, Yellow, "Title : ");
      if Docume /= null and then Docume.Title /= null then
	 Draw_Text(Win, 1, 9, White, Troncate(Handling.To_Wide_String(Docume.Title.all), Cols/4-10));
      end if;
      Draw_Text(Win, 1, 5+Cols/4+2, Yellow, "Unit : ");
      if Docume /= null and then Docume.Unit /= null then
	 Draw_Text(Win, 1, 10+Cols/4+9, white, Handling.To_Wide_String(Docume.Unit.all));
      end if;
      Draw_Text(Win, 1, 10+Cols/4+15, Yellow, "Created  : ");
      if Docume /= null then
	 Draw_Text(Win, 1, 10+Cols/4+26, White, Handling.To_Wide_String(Formatting.Image(Docume.Creation_Date)));
      end if;
      Draw_Text(Win, 2, 1, Yellow, "Name : ");
      if Docume /= null and then Docume.Filename /= null then
	 Draw_Text(Win, 2, 9,  White, Troncate(Handling.To_Wide_String(Docume.filename.all), Cols/4-10));
      end if;
      Draw_Text(Win, 2, 5+Cols/4+2, Yellow, "Openned : " );
      if Docume /= null and then Docume.Filename /= null then
	 Draw_Text(Win, 2, 5+Cols/4+12,white, Handling.To_Wide_String(Boolean'Image(Docume.Openned)));
      end if;
      Draw_Text(Win, 2, 10+Cols/4+15, Yellow, "Modified : ");
      if Docume /= null then
	 Draw_Text(Win, 2, 10+Cols/4+26, White, Handling.To_Wide_String(Formatting.Image(Docume.Last_Edition)));
      end if;
      Draw_Text(Win, 3, 1, Yellow, "Size : ");
      if Docume /= null then
	 Draw_Text(Win, 3, 9, white, Handling.To_Wide_String(Natural'Image(Docume.Size)) & " without end_of_line.");
      end if;
      Draw_Text(Win, 3,  2+Cols/4+5, Yellow, "Owner : ");
      if Docume /= null and then Docume.Owner /= null then
	 Draw_Text(Win, 3,  2+Cols/4+13, White, Handling.To_Wide_String(Docume.Owner.all));
      end if;

   end Devel_Draw_Docume;
   
   
   procedure Devel_Draw_Main_Win (Center : access Center_Record) is
   begin
      Draw_Window(Center.Frames(Left_view).Win);

      Draw_Window(Center.Frames(Doc_States).Win);

      Draw_Window(Center.Frames(Document).Win);
      
      Draw_Window(Center.Frames(Cmd).Win);

      Draw_Window(Center.Frames(Err).Win);
      
      Draw_Window(Center.Frames(Cmd_Result).Win);

      Draw_Window(Center.Frames(Visitor).Win);
      
      Draw_Window(Center.Frames(Proposition).Win);
      
      Draw_Window(Center.Frames(Neural_States).Win);
      
      Draw_Window(Center.Frames(Top_Class).Win);
      
      Draw_Window(Center.Frames(Class_Ranking).Win);
      
      Draw_Window(Center.Frames(Goal_Path).Win);
      
      Draw_Window(Center.Frames(Buffer).Win);      
   end;
   
   
   
   
   procedure Devel_Ai_Draw_Proposition(Center : access Center_Record) is
      
      Evidence : Preuve_Record := Center.Domain.Proof;
   begin
      Verifiing(Evidence);
      declare
	 Proof_text : String := To_String(Evidence);
      begin
	 Draw_Window(Center.Frames(Proposition).Win);
	 Draw_Text(Center.Frames(Proposition).Win, 1, 1, White, Handling.To_Wide_String(Proof_Text));
	 Draw_Text(Center.Frames(Proposition).Win, 2, 1, Yellow, "Veritable : ");	
	 Draw_Text(Center.Frames(Proposition).Win, 2, 16, white, Handling.To_Wide_String(Boolean'Image(Evidence.Veritable)));
	 Draw_Text(Center.Frames(Proposition).Win, 2, 35, Yellow, "Veracity : ");	
	 Draw_Text(Center.Frames(Proposition).Win, 2, 50, White, Handling.To_Wide_String(Boolean'Image(Evidence.Veracity)));
	 Draw_Text(Center.Frames(Proposition).Win, 2, 60, Yellow, "Verified : ");	
	 Draw_Text(Center.Frames(Proposition).Win, 2, 70, White, Handling.To_Wide_String(Boolean'Image(Evidence.Verified)));
	 Draw_Text(Center.Frames(Proposition).Win, 3, 1,  Green, "State     : ");	
	 if To_State(Evidence) then
	    Draw_Text(Center.Frames(Proposition).Win, 3, 16, Cyan, Handling.To_Wide_String(Boolean'Image(To_State(Evidence))));
	 else
	    Draw_Text(Center.Frames(Proposition).Win, 3, 16, white, Handling.To_Wide_String(Boolean'Image(To_State(Evidence))));
	 end if;
      end;
   end Devel_Ai_Draw_Proposition;
   
   procedure Devel_Ai_Draw_Neural_States(Center : access Center_Record) is
      Aiinfo : Aiinfo_Record := Aiinfo_Record(Center.Domain);
   begin
      Draw_Window(Center.Frames(Neural_States).Win);
      
      Draw_Text(Center.Frames(Neural_States).Win, 1, 1, Cyan, "Neural network layer 0 : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 1, 26, Yellow, "RMS_Erro : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 1, 37, White, Handling.To_Wide_String(Float'Image(Aiinfo.Layer_0.Rms_Error)));
      Draw_Text(Center.Frames(Neural_States).Win, 1, 50, Yellow, "Current epoch : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 1, 66, White, Handling.To_Wide_String(Natural'Image(Aiinfo.Layer_0.Current_Epoch)));
      Draw_Text(Center.Frames(Neural_States).Win, 1, 80, Yellow, "Converged : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 1, 92, White, Handling.To_Wide_String(Boolean'Image(Aiinfo.Layer_0.Converged)));
      Draw_Text(Center.Frames(Neural_States).Win, 2, 1, Cyan, "Neural network layer 2 : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 2, 26, Yellow, "RMS_Erro : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 2, 37, White, Handling.To_Wide_String(Float'Image(Aiinfo.Layer_2.Rms_Error)));
      Draw_Text(Center.Frames(Neural_States).Win, 2, 50, Yellow, "Current epoch : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 2, 66, White, Handling.To_Wide_String(Natural'Image(Aiinfo.Layer_2.Current_Epoch)));
      Draw_Text(Center.Frames(Neural_States).Win, 2, 80, Yellow, "Converged : ");	
      Draw_Text(Center.Frames(Neural_States).Win, 2, 92, White, Handling.To_Wide_String(Boolean'Image(Aiinfo.Layer_2.Converged)));
      
      
   end Devel_Ai_Draw_Neural_States;
   
   
   procedure Devel_Ai_Draw_Top_class(Center : access Center_Record) is
   begin
      
      Draw_Window(Center.Frames(Top_Class).Win);
      Draw_Text(Center.Frames(Top_Class).Win, 1, 1, Yellow, "Tag name : ");
      Draw_Text(Center.Frames(Top_Class).Win, 1, 12, White, Handling.To_Wide_String(El.Tag_Name ' Image (Center.Classes_Ranking(1).Name)));
      Draw_Text(Center.Frames(Top_Class).Win, 1, 30, Yellow, "Attrib Nb : ");
      Draw_Text(Center.Frames(Top_Class).Win, 1, 43, Cyan, Handling.To_Wide_String(Natural ' Image(Center.Classes_Ranking(1).Attributs_nb)));
      Draw_Text(Center.Frames(Top_Class).Win, 1, 52, Yellow, "Ucost : ");
      Draw_Text(Center.Frames(Top_Class).Win, 1, 60, white, Handling.To_Wide_String(Float ' Image(Center.Classes_Ranking(1).Ucost)));
      Draw_Text(Center.Frames(Top_Class).Win, 1, 87, Yellow, "Hcost : ");
      Draw_Text(Center.Frames(Top_Class).Win, 1, 95, White, Handling.To_Wide_String(Float ' Image(Center.Classes_Ranking(1).Hcost)));      
      
      Draw_Text(Center.Frames(Top_Class).Win, 2, 15, white, "Saved sold : ");
      Draw_Text(Center.Frames(Top_Class).Win, 2, 33, white, Handling.To_Wide_String(Float'Image(Center.Classes_Ranking(1).Saved_Sold)));
      Draw_Text(Center.Frames(Top_Class).Win, 2, 48, white, "Personal sold : ");
      Draw_Text(Center.Frames(Top_Class).Win, 2, 63, white, Handling.To_Wide_String(Float'Image(Center.Classes_Ranking(1).Perso_Sold)));
      Draw_Text(Center.Frames(Top_Class).Win, 2, 78, white, "Enteprise sold : ");
      Draw_Text(Center.Frames(Top_Class).Win, 2, 96, white, Handling.To_Wide_String(Float'Image(Center.Classes_Ranking(1).Entep_Sold)));
      
      if Center.Classes_Ranking(1).Is_Debit then
	 Draw_Text(Center.Frames(Top_Class).Win, 3, 1, red, "is debit");
      elsif Center.Classes_Ranking(1).Is_Credit then
	 Draw_Text(Center.Frames(Top_Class).Win, 3, 1, Green, "is creditt");
      elsif Center.Classes_Ranking(1).Is_Credit then
	 Draw_Text(Center.Frames(Top_Class).Win, 3, 1, Magenta, "is deal out");
      elsif Center.Classes_Ranking(1).Is_Credit then
	 Draw_Text(Center.Frames(Top_Class).Win, 3, 1, Yellow, "is deal in");
      else
	 Draw_Text(Center.Frames(Top_Class).Win, 3, 1, white, "no paiement");
      end if;	 
   end Devel_Ai_Draw_Top_class;
   
   
   procedure Devel_Ai_Draw_Class_ranking(Center : access Center_Record) is
   begin
      
      
      Draw_Window(Center.Frames(Class_Ranking).Win);
      Draw_Text(Center.Frames(Class_Ranking).Win, 1, 1, cyan, "Class name,        Attr Nb,  Ucost            Hcost               Total");
      Draw_Text(Center.Frames(Class_Ranking).Win, 2, 1, cyan, ((((Cols-2)/2 - 2 )) * '-'));
      
      
      for Class in Center.Classes_Ranking ' range loop
	 Draw_Text(Center.Frames(Class_Ranking).Win, Class + 2, 1, Yellow, Handling.To_Wide_String(El.Tag_Name ' Image (Center.Classes_Ranking(Class).Name)));
	 Draw_Text(Center.Frames(Class_Ranking).Win, Class + 2, 18, Cyan, Handling.To_Wide_String(Natural ' Image(Center.Classes_Ranking(Class).Attributs_nb)));
	 
	 Draw_Text(Center.Frames(Class_Ranking).Win, Class + 2, 30, White, Handling.To_Wide_String(Float ' Image(Center.Classes_Ranking(Class).Ucost)));
      
	 Draw_Text(Center.Frames(Class_Ranking).Win, Class + 2, 47, White, Handling.To_Wide_String(Float ' Image(Center.Classes_Ranking(Class).Hcost)));
	 Draw_Text(Center.Frames(Class_Ranking).Win, Class + 2, 67, White, Handling.To_Wide_String(Float ' Image(Center.Classes_Ranking(Class).Hcost - Center.Classes_Ranking(Class).Ucost)));
      end loop;
      
   end Devel_Ai_Draw_Class_Ranking;
   
   
   

   
   
   
   procedure Devel_Ai_Draw_Goal_Path (Center : access Center_Record) is
      
      
      Pos : Natural := 0;
   begin
      T_Io.Put(Save_Cursor);
      Draw_Window(Center.Frames(Goal_Path).Win);
      Draw_Text(Center.Frames(Goal_Path).Win, 1, 1, cyan, "Class name,        Attr Nb,  Ucost            Hcost               Total                  Name              End : " & 
		  Handling.To_Wide_String(Boolean'Image(Center.Goal_Path_Search_End_Of_Task)));
      Draw_Text(Center.Frames(Goal_Path).Win, 2, 1, Blue, ((((Cols-2)/2 - 2 )) * '-'));
      
      --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 1, Yellow, Handling.To_Wide_String(.Tag_Name ' Image (Center.Start_Node.Tag)));
      --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 18, Cyan, Handling.To_Wide_String(Natural ' Image(Center.Start_Node.Def.index)));
      
      --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 28, White, Handling.To_Wide_String(Float ' Image(Center.Start_Node.Ucost)));
      
      --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 40, White, Handling.To_Wide_String(Float ' Image(Center.Start_Node.Hcost)));
      --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 54, White, Handling.To_Wide_String(Float ' Image(Center.Start_Node.Hcost - Center.Start_Node.Ucost)));
      
      
      if not Abstracted_Vectors.Is_Empty(Center.Goal_Path) then
	 
	 for Iter in 1..Natural'Min(Natural(Abstracted_Vectors.Length(Center.Goal_Path)), Lines-52) loop
	    declare
	       E : Abstracted_Access := Abstracted_Vectors.Element(Center.Goal_Path, Iter);
	    begin	       	       	       
	       
	       if E /= null then
		  Pos := Pos + 1;
		  
		  
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 1, Yellow, Handling.To_Wide_String(El.Tag_Name ' Image (E.Tag)));
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 18, Cyan, Handling.To_Wide_String(Natural ' Image(E.Def.index)));
		  
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 30, White, Handling.To_Wide_String(Float ' Image(E.Ucost)));
		  
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 47, White, Handling.To_Wide_String(Float ' Image(E.Hcost)));
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 67, White, Handling.To_Wide_String(Float ' Image(E.Hcost - E.Ucost)));
		  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 90, Yellow, E.Name);
	       end if;
	       
	    end;
	 end loop;
	 
	 --  Pos := Pos + 1;
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 1, Yellow, Handling.To_Wide_String(El.Tag_Name ' Image (Center.Goal.Tag)));
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 18, Cyan, Handling.To_Wide_String(Natural ' Image(Center.Goal.Def.index)));
	 
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 28, White, Handling.To_Wide_String(Float ' Image(Center.Goal.Ucost)));
	 
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 40, White, Handling.To_Wide_String(Float ' Image(Center.Goal.Hcost)));
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 54, White, Handling.To_Wide_String(Float ' Image(Center.Goal.Hcost - Center.Goal.Ucost)));
	 --  Draw_Text(Center.Frames(Goal_Path).Win, Pos + 2, 70, Yellow, Center.Goal.Name);
      elsif not Center.Goal_Path_Search_End_Of_Task then
	 Draw_Text(Center.Frames(Goal_Path).Win, Center.Frames(Goal_Path).Win.H-2, 1, Green, "Goal path searching...");
      else
	 Draw_Text(Center.Frames(Goal_Path).Win, Center.Frames(Goal_Path).Win.H-2, 1, Green, "no goal path.");
      end if;
      
      
   end Devel_Ai_Draw_Goal_Path;
   
   
   procedure Devel_Win_Init (Center : access Center_Record) is
      
      Doc_Height : Positive := ((Lines - 19) / 8) * 5;
      Result_Height : Positive := Lines - (Doc_Height + 19);
	 
   begin
      Initialize(Center.Frames(Left_view).Win, 1, 1, Lines, Cols/2, Blue, White);

      Initialize(Center.Frames(Doc_States).Win, 3, 2, 5, Cols/2-2, Yellow, cyan);
      
      Initialize(Center.Frames(Document).Win, 8, 2, Doc_Height, Cols/2-2, White, Red);

      Initialize(Center.Frames(Cmd).Win, Doc_Height+8, 2, 3, Cols/2-2, White, Red);
      
      Initialize(Center.Frames(Err).Win,  Doc_Height+11, 2, 3, Cols/2-2, White, Red);
      
      Initialize(Center.Frames(Cmd_Result).Win, Doc_Height+14, 2, Result_Height, Cols/2-2, White, Red);

      Initialize(Center.Frames(Visitor).Win, 1, Cols/2+1, 7, Cols/2, Yellow, cyan);

      Initialize(Center.Frames(Proposition).Win, 8, Cols/2+1, 5, Cols/2, white, Yellow);

      Initialize(Center.Frames(Neural_States).Win, 13, Cols/2+1, 5, Cols/2, blue, White);
      
      Initialize(Center.Frames(Top_class).Win, 18, Cols/2+1, 5, Cols/2, Red, white);
      
      
      Initialize(Center.Frames(Goal_Path).Win, 23, Cols/2+1, Lines-45, Cols/2, Blue, white);
      
      Initialize(Center.Frames(Class_Ranking).Win, Lines-22, Cols/2+1, 23, Cols/2, Cyan, Red);
      
      
   end Devel_Win_Init;
   
   
   function Set_Size (Lines_Table : in Wide_String_Set) return Natural is
      Count : Natural := 0;
   begin
      for Wline in 1..Lines_Table.Line_Last loop
	 Count := Count + Lines_Table.Lines(Wline)'Length;
      end loop;
      return Count;
   end Set_Size;
	    
   
   
   procedure Open_Doc (Lines_Table : in out Wide_String_Set;
		       Docume : out Docume_Access;
		       Filename : in String) is
      
      File : W_Io.File_Type;
      Line_Last    : Natural := 0;
      Dot_Index    : Natural := 0;      
      Owner_result : Wide_Result_Access;      
      Success      : Boolean := False;
      
      Errno        : Integer := 0;
   begin
      
      Line_Last := 0;
      Set_Free(Lines_Table);
      if Os_Lib.Is_Regular_File(Filename) then
	 
	 W_Io.Open(File, W_Io.In_File, Filename, Form => "WCEM=8");
	 
	 while not W_Io.End_Of_File(File) loop
	    declare
	       Wline : constant Wide_String_Access := new Wide_String ' (W_Io.Get_Line(File));
	    begin
	       Lines_Table.Lines(Line_Last + 1) := Wline;
	       Line_Last := Line_Last + 1;
	    end;
	 end loop;	 
	 
	 Lines_Table.Line_Last := Line_Last;	 
	 
	 W_Io.Close(File);
	 
	 
	 
	 Docume := new Docume_Record;
	 
	 Docume.Creation_Date := Time_Of(1970, 1, 1, 0.0);
	 Docume.Last_Edition := Clock;
	 
	 Docume.Filename := new String ' (Filename);
	 declare
	    Dir : Dir_Name_Str := Dir_Name (Filename);
	 begin
	    case System.Os_Constants.Target_Os is
	       when System.Os_Constants.Other_Os =>
		  
		  Docume.Fullpath := new String ' (Dir & Dir_Separator & Base_Name(Filename));
		  
		  
		  Owner_Result := new Spawn_Result_Record ' 
		    (Spawn("ls -l " & Docume.Filename.all, ".Out_File.txt"));
		  
		  
		  
		  if ((Owner_Result /= null) and then
			(Owner_Result.Wlines /= null)) then
		     declare
			Line : constant Wide_String := (-Owner_Result.Wlines(1));
			Top  : Natural := 0;
			Bot  : Natural := 0;
		     begin
			for I in 1..2 loop
			   Top := Index(Line(Top+1..Line'Last), " ");
			end loop;
			Bot := Index(Line(Top+1..Line'Last), " ");
			Docume.Owner := new String ' (Handling.To_String(Line(Top+1..Bot-1)));
		     end;
		  end if;
		  
	       when System.Os_Constants.Windows =>
		  
		  Docume.Fullpath := new String ' (Dir & Dir_Separator & Base_Name(Filename));
	       when others =>
		  null;
	    end case;
	 
	 end;
	 
	 Dot_Index := Fixed.Index(Filename, ".");
	 
	 Docume.Title := new String ' (Base_Name(Filename, Filename(Dot_Index..Filename'Last)));
	 
	 Docume.Unit := new String ' (Filename(Dot_Index+1..Filename'Last));
	 
	 Docume.Openned := True;	 
	 
	 Docume.Size := Set_Size(Lines_Table);
      else
	 
	 
	 declare
	    Not_Found : constant Wide_String := New_Wide_String("Open document : document not found.", "Press 'enter' to continue.", 30, lines/2-1, cols/2-15, False);
	 begin
	    null;
	 end;
	 raise Spec_Error;
      end if;
      
   end Open_Doc;
   
   
   procedure New_Doc (Center : in Center_Record;
		      Lines_Table : in out Wide_String_Set; 
		      Docume : out Docume_Access; 		      
		      Filename : in String;		      
		      Version  : in Wide_String;
		      License : in Boolean;
		      Description : in Wide_string) is
      Dot_Index    : Natural := Fixed.Index(Filename, ".", Backward);
   begin      
      for Line in 1..Lines_Table.Line_Last+1 loop
	 Wide_Free(Lines_Table.Lines(Line));
      end loop;

      Docume := new Docume_Record;
      --W_Io.Put_Line("TOTO 2 : " & Filename & ' ' & Filename(Dot_Index..Filename'Last));
      Docume.Title := new String ' (Base_Name(Filename, Filename(Dot_Index..Filename'Last)));      

      Docume.Filename := new String ' (Filename);
      Docume.Creation_Date := Clock;
      Docume.Last_Edition := Clock;
      Dot_Index := Fixed.Index(Filename, ".");

      Docume.Unit := new String ' (Filename(Dot_Index+1..Filename'Last));
      
      Docume.Openned := True;	 
      if License then
	 Lines_Table.Lines(1) := new Wide_String ' ("--" & ((80 - 27) * ' ') & Handling.To_Wide_String(Formatting.Image(Clock)));

	 Lines_Table.Lines(2) := new Wide_String ' (("-- " & Handling.To_Wide_String(Docume.Title.all) & ' ' & Description & ' ' & (((80 - 8) - (Docume.Title'Length + Description'length + 2)) * ' ') & "--"));

	 Lines_Table.Lines(3) := 
	   new Wide_String ' ("-- " & Handling.To_Wide_String(Docume.Title.all) & ' ' & "Copyright (C)" & 
				Handling.To_Wide_String(Year_Number'Image(Year(Clock))) & ' ' &
				Handling.To_Wide_String(Center.User_Auth.Logname.all));

	 Lines_Table.Lines(4) := new Wide_String ' (((80-3) * '-'));
	 Lines_Table.Lines(5) := new Wide_String ' ("-- This program is free software; you can redistribute it and/or modify");
	 
	 Lines_Table.Lines(6) := new Wide_String ' ("-- it under the terms of the GNU General Public License as published by");
	 Lines_Table.Lines(7) := new Wide_String ' ("-- the Free Software Foundation; either version 2 of the License, or");
	 Lines_Table.Lines(8) := new Wide_String ' ("-- (at your option) any later version.");
	 Lines_Table.Lines(9) := new Wide_String ' ("--");
	 Lines_Table.Lines(10) := new Wide_String ' ("-- This program is distributed in the hope that it will be useful,");
	 Lines_Table.Lines(11) := new Wide_String ' ("-- but WITHOUT ANY WARRANTY; without even the implied warranty of");
	 Lines_Table.Lines(12) := new Wide_String ' ("-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
	 Lines_Table.Lines(13) := new Wide_String ' ("-- GNU General Public License for more details.");
	 Lines_Table.Lines(14) := new Wide_String ' ("");
	 Lines_Table.Lines(15) := new Wide_String ' ("-- You should have received a copy of the GNU General Public License");
	 Lines_Table.Lines(16) := new Wide_String ' ("-- along with this program; if not, write to the Free Software");
	 Lines_Table.Lines(17):= new Wide_String ' ("-- Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA");
	 Lines_Table.Lines(18) := new Wide_String ' (((80-3) * '-'));
	 Lines_Table.Lines(19) := new Wide_String ' ("-- Date    := """ & Handling.To_Wide_String(Formatting.Image(Docume.Creation_Date)) & """");
	 Lines_Table.Lines(20) := new Wide_String ' ("-- Version := """ & Version & """");
      end if;            
      
      Lines_Table.Line_Last := 20;
   end New_Doc;
   
   
   procedure Doc_Init (Center : in Center_Record;
		       Lines_Table : in out Wide_String_Set) is
      Color     : constant String := Character'Val(27) & "[01;36m";
      Normal    : constant String := Character'Val(27) & "[00m";
   
   begin
      Lines_Table.Lines(1) := new Wide_String ' ("Hello " & Handling.To_Wide_String(Color & Center.User_Auth.Logname.all & Normal) &
						   " and welcome to Devel mode of Elhoim program.");

      Lines_Table.Lines(2) := new Wide_String ' ("For to create an new document, you must try Crtl+N.");
      Lines_Table.Lines(3) := new Wide_String ' ("For to open an existing document, you must try Ctrl+O.");
      Lines_Table.Lines(4) := new Wide_String ' ("For to save an openned document, you must try Ctrl+A.");
      Lines_Table.Lines(5) := new Wide_String ' ("For to change state in Command or in Document, you must try F8.");
      Lines_Table.Lines(6) := new Wide_String ' ("");
      Lines_Table.Lines(7) := new Wide_String ' ("When you create a new document, you must to give :");
      Lines_Table.Lines(8) := new Wide_String ' ("  - A name of file with 65 characters maximum.");
      Lines_Table.Lines(9) := new Wide_String ' ("  - If you want to initialize you document with an Gnu Public license,");
      Lines_Table.Lines(10) := new Wide_String ' ("    you must to give the description for your document.");
      Lines_Table.Lines(11) := new Wide_String ' ("The Gnu Public license is in Ada programming language commented text: ");
      Lines_Table.Lines(12) := new Wide_String ' ("");
      Lines_Table.Lines(13) := new Wide_String ' (((80-3) * '-'));
	 Lines_Table.Lines(14) := new Wide_String ' ("-- This program is free software; you can redistribute it and/or modify");
	 
	 Lines_Table.Lines(15) := new Wide_String ' ("-- it under the terms of the GNU General Public License as published by");
	 Lines_Table.Lines(16) := new Wide_String ' ("-- the Free Software Foundation; either version 2 of the License, or");
	 Lines_Table.Lines(17) := new Wide_String ' ("-- (at your option) any later version.");
	 Lines_Table.Lines(18) := new Wide_String ' ("--");
	 Lines_Table.Lines(19) := new Wide_String ' ("-- This program is distributed in the hope that it will be useful,");
	 Lines_Table.Lines(20) := new Wide_String ' ("-- but WITHOUT ANY WARRANTY; without even the implied warranty of");
	 Lines_Table.Lines(21) := new Wide_String ' ("-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the");
	 Lines_Table.Lines(22) := new Wide_String ' ("-- GNU General Public License for more details.");
	 Lines_Table.Lines(23) := new Wide_String ' ("");
	 Lines_Table.Lines(24) := new Wide_String ' ("-- You should have received a copy of the GNU General Public License");
	 Lines_Table.Lines(25) := new Wide_String ' ("-- along with this program; if not, write to the Free Software");
	 Lines_Table.Lines(26):= new Wide_String ' ("-- Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA");
	 Lines_Table.Lines(27) := new Wide_String ' (((80-3) * '-'));
	 
	 Lines_Table.Line_Last := 27;
   end Doc_Init;
   
   
   
         
   
   --------------------------------------------------------------------------------------
   --
   --                               Initialize
   --
   --------------------------------------------------------------------------------------
   
   
   
   
   procedure Initialize (Center : in out Center_Record; Users_Max : in Positive;Lexic_Filename : in string) is
      Width_Line : Word_Range := 85;
      Hiddens    : Layer_Width_Range := 0;
      Epochs_Max : Positive := 1;
      Converge  : Float := 0.0;
      Samples_Max : Positive := 64;
      Max_words   : Lexical_Range := 2043;
      RMS_Error   : Real := 0.0;
      
      State_Index : Natural := 0;
      Ai_Index    : Natural := 0;
   begin      
      
      
      Config.Server.Engine.Intelligency.Cyborg.Initialize(Lexic_Filename);
      
      Interf.Initialize(Interf.Interf_Record(Center));
      
      if Center.Parameters_Kit.All_parameters(As_Server).Is_Setted then
	 Config.Initialize(Config.Config_Record(Center), Interf.Interf_Record(Center), True, Users_Max);
      end if;
      
      
      if Center.Parameters_Kit.All_parameters(Username).Is_Setted then
	 
	 Center.Given_Username := new Wide_String ' (Center.Parameters_Kit.All_parameters(Username).Value.all);
	 
      end if;
      
      if Center.Parameters_Kit.All_parameters(Address).Is_Setted then
	 Center.Given_Hostname := new Wide_String ' (Center.Parameters_Kit.All_parameters(Address).Value.all);
      end if;
      
      if Center.Parameters_Kit.All_parameters(Input_Text).Is_Setted then
	 Center.Given_Text := new Wide_String ' (Center.Parameters_Kit.All_parameters(Input_Text).Value.all);
      end if;
      
      if Center.Parameters_Kit.All_parameters(Train).Is_Setted then
	 begin
	    if Center.Parameters_Kit.All_parameters(Max_epochs).Is_Setted then
	       Epochs_Max := Positive'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Max_epochs).Value.all));
	       Sh_Epochs_Max := Epochs_Max;
	    end if;
	    
	 exception
	    when others =>
	       null;
	 end;

	 begin
	    if Center.Parameters_Kit.All_parameters(Converged).Is_Setted then
	       Converge := Float'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Converged).Value.all));
	       Sh_Converged := Converge;
	    end if;
	    
	 exception
	    when others =>
	       null;
	 end;
	 begin
	    if Center.Parameters_Kit.All_parameters(Reuse).Is_Setted then	    
	       Sh_Reused := True;
	    end if;
	    
	 exception
	    when others =>
	       null;
	 end;
	 
	 begin
	    if Center.Parameters_Kit.All_parameters(Num_samples).Is_Setted then
	       Samples_max := integer'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Num_samples).Value.all));
	       Sh_Samples_Max := Samples_Max;
	    end if;
	    
	 exception
	    when others =>
	       null;
	 end;
	 
	 begin
	    if Center.Parameters_Kit.All_parameters(Num_Hidden).Is_Setted then
	       Hiddens := integer'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Num_Hidden).Value.all));
	       Sh_Hiddens := Hiddens;
	       
	    end if;
	 exception
	    when others =>
	       null;
	 end;
	 begin
	    if Center.Parameters_Kit.All_parameters(Line_Width).Is_Setted then
	       Width_Line := integer'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Line_width).Value.all));
	       Sh_Line_width := Width_Line;
	       
	    end if;
	 exception
	    when others =>
	       null;
	 end;
	 
	 begin
	    if Center.Parameters_Kit.All_parameters(Words_Max).Is_Setted then
	       Max_Words := integer'Value
		 (Handling.To_String(Center.Parameters_Kit.All_parameters(Words_Max).Value.all));
	       Sh_Max_Words:= Max_Words;
	       
	    end if;
	 exception
	    when others =>
	       null;
	 end;
      end if;
      begin
	 if Center.Parameters_Kit.All_parameters(Term_Name).Is_Setted then
	    Center.terminal := Term_Enum'value
	      (Handling.To_String(Center.Parameters_Kit.All_parameters(Term_Name).Value.all));

	    
	 end if;
      exception
	 when others =>
	    null;
      end;
      
      -- Look for Samples filename :
      if Center.Parameters_Kit.All_parameters(Samples_filename).Is_Setted then
	 if (Center.Parameters_Kit.All_parameters(Samples_Filename).Value'Length /= 0) and then	   
	   Os_Lib.Is_Regular_File(Handling.To_String
				    (Center.Parameters_Kit.All_parameters(Samples_Filename).Value.all)) then	    
	    
	    Center.Set_filename := new String '
	      (Handling.To_String(Center.Parameters_Kit.All_parameters(Samples_Filename).Value.all));
	    
	 elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & "samples_set.txt") then
	    
	    Center.Set_Filename := new String ' ("data" & Dir_Separator & "samples_set.txt");
	    
	 else
	    
	    W_Io.Put("Enter samples filename : ");	       
	    
	    Center.Set_Filename := new String ' (T_Io.Get_Line);
	    
	 end if;
      elsif Os_Lib.Is_Regular_File("data" & Dir_Separator & "samples_set.txt") then
	 
	 Center.Set_Filename := new String ' ("data" & Dir_Separator & "samples_set.txt");
	 
      else
	 
	 W_Io.Put("Enter samples filename : ");	       
	 Center.Set_Filename := new String ' (T_Io.Get_Line);
	 
      end if;
      Center.Domain.Proof := From_Line(En_Veritable);
      Center.Elhoim_Task := new Elhoim_Task(new Terminal_Record ' (Center.T));
      
      
      Words_Glossary.Initialize(Center.Words, "data" & Dir_Separator & "glossary.txt");
      
      
      
      if Center.Parameters_Kit.All_parameters(Train).Is_Setted then
	 
	 W_Io.Put_Line("Going to train neural_network...");
	 W_Io.Put_Line("Max words : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Positive'Image(Sh_Max_Words)));
	 W_Io.Put_Line("Words size :");
	 W_Io.Put_Line(Handling.To_Wide_String(Natural'Image(Word_Size)));
	 W_Io.Put_Line("Glossary filename : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Glossary_Filename));
	 W_Io.Put_Line("Max epochs : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Natural'Image(Sh_Epochs_Max)));
	 W_Io.Put_Line("Max Max samples : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Natural'Image(Sh_Samples_Max)));
	 W_Io.Put_Line("Samples filename : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Center.Set_Filename.all));
	 W_Io.Put_Line("Line width : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Positive'Image(Sh_Line_Width)));
	 W_Io.Put_Line("Reuse network :");
	 W_Io.Put_Line(Handling.To_Wide_String(Boolean'Image(Sh_Reused)));
	 W_Io.Put_Line("convergency : ");
	 W_Io.Put_Line(Handling.To_Wide_String(Float'Image(Sh_Converged)));
	 W_Io.Put_Line("Going to initialization of samples_set...");
	 Init_Samples(Center.Samples_Set, Center.Words, Center.Set_Filename.all);
	 W_Io.Put_Line("Going to train of file : " & Handling.To_Wide_String(Network_Filename));
	 
	 Neural_Train.Train     (Samples   => Center.Samples_Set ' access,
				 Max_Epochs => Sh_Epochs_Max,
				 Converged => Real(Sh_Converged),
				 Reuse     => Sh_reused,
				 RMS_Error => RMS_Error);
	 
	 if Center.Domain.Layer_0.Rms_Error <= Sh_Converged then
	   Center.Domain.Layer_0.Converged := True;
	 end if;
	 Center.Domain.Layer_0.Rms_Error := Float(RMS_Error);
	 W_Io.Put_Line("Done.");
	 W_Io.Put_Line("Copying file");
	 declare
	    Success :Boolean := False;
	 begin
	    Copy_File(Network_Filename & ".wgt1", Network_Filename & ".wgt2", success);
	 end ;
	 
      end if;
      
      
      
      
      
      
      
      
      
      
      if Argument_Count /= 0 then
	 if Argument(1) = "-le" then      
	    if Os_Lib.Is_Regular_File("organizer-listing.txt") then
	       -- loading file
	       declare
		  File : Ada.Wide_Text_Io.File_Type;
		  
	       begin
		  begin
		     Ada.Wide_Text_Io.Open(File, Ada.Wide_Text_Io.In_File, "prehander-listing.txt");
		     while not Ada.Wide_Text_Io.End_Of_File(File) loop
			declare
			   E : constant Expression_type := Ada.Wide_Text_Io.Get_Line(File);
			begin
			   Center.Expr_List := Center.Expr_List & E;
			end;
		     end loop;	       
		     Ada.Wide_Text_Io.Close(File);
		  exception
		     when Ada.Wide_Text_Io.Name_Error =>
			null;
		  end;	    
	       end;
	       
	       
	       -- executing list
	       if Expr_Vectors.length(Center.Expr_List) /= 0 then
		  
		  for I in Expr_Vectors.First_Index(Center.Expr_List)..Expr_Vectors.Last_Index(Center.Expr_List) loop
		     
		     declare
			
		     begin	       
			
			declare
			   Expr : constant Expression_Type := Expr_Vectors.Element(Center.Expr_List, I);
			begin
			   
			   Center.Prompt_List := Center.Prompt_List & Center.Prompt;
			   Update(Center.T, Center.Prompt, Expr);
			   Right_Exec(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
			   Center.Prompt_List := Center.Prompt_List & Center.Prompt;	 
			   
			end;
		     end;
		  end loop;	 
	       end if;	    
	    end if;
	 end if;
	 
	 
      end if;
      
      if Os_Lib.Is_Regular_File(Center.User_Filename.all) then
	 begin
	    Restore(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, 
		    Wins_Main.User_Objects_Tree_Win_Ptr.all, Center.User_Filename.all);
	    --Draw_Window(Wins_Main.User_Objects_View_Win_Ptr.all);
	    --Draw_Window(Wins_Main.User_Objects_Tree_Win_Ptr.all);
	    Center.T.Obj_Cur := Center.T.Obj;
	    Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
	 end;
      end if;
      if Center.T.Obj = null then
	 Wide_Fixed.Move("Create universe", Center.Expr, Error, Left, ' ');
	 Update(Center.T, Center.Prompt, Center.Expr);
	 begin
	    Right_Exec(Center.T, Wins_Main.User_Objects_View_Win_Ptr.all, Center.Prompt);
	    --Draw_Window(Wins_Main.User_Objects_Tree_Win_Ptr.all);
	    --Global_Content_Print(Center.T.Obj_Cur, Wins_Main.User_Objects_Tree_Win_Ptr.all, State_Index);
	    Center.Expr_List := Center.Expr_List & Center.Expr;	 
	    Center.Prompt_List := Center.Prompt_List & Center.Prompt;
	    Center.Expr := (others => Wide_Character'Val(32));		     
	    Center.T.Page_Index := 0;			      			      
	 end;
	 Save(Center.T);
	 Save(Center.T.Obj, Center.User_Filename.all);
      end if;
      
      if Os_Lib.Is_Regular_File(Center.Ai_Filename.all) then
	 begin	    
	    Restore(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, 
		    Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Center.Ai_Filename.all);
	    --Draw_Window(Wins_Main.Ai_Objects_View_Win_Ptr.all);
	    --Draw_Window(Wins_Main.Ai_Objects_Tree_Win_Ptr.all);
	    Center.Ai.Obj_Cur := Center.Ai.Obj;
	    --Global_Content_Print(Center.Ai.Obj_Cur, Wins_Main.Ai_Objects_Tree_Win_Ptr.all, Ai_Index);
	 end;
      end if;

      if Center.Ai.Obj = null then
	 Wide_Fixed.Move("Create universe", Center.Expr, Error, Left, ' ');
	 Update(Center.Ai, Center.Prompt, Center.Expr);
	 begin
	    Right_Exec(Center.Ai, Wins_Main.Ai_Objects_View_Win_Ptr.all, Center.Prompt);
	    --Draw_Window(Wins_Main.Ai_Objects_Tree_Win_Ptr.all);	    
	    Center.Ai.Page_Index := 0;			      			      
	 end;
	 
	 Save(Center.Ai);		     
	 Save(Center.Ai.Obj, Center.Ai_Filename.all);
	 
      end if;	 
      
      W_Io.Put_Line("initialization done.");
   end Initialize;
   
   
   --------------------------------------------------------------------------------------
   --
   --                        End initialize
   --
   --------------------------------------------------------------------------------------

end El.Center ; 
